/* -*- mode:C; indent-tabs-mode:t; tab-width:8; c-basic-offset:8; -*- */


/* gxmessage - an xmessage clone using GTK2
 *
 * $Id: gxmessage.c,v 1.13 2004/04/15 11:25:46 trm Exp trm $
 *
 * Copyright 2001 - 2004, Timothy Musson <trmusson@ihug.co.nz>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * http://www.gnu.org/copyleft/gpl.html
 *
 */



#include <config.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <gtk/gtk.h>


#ifdef ENABLE_NLS
#  include <libintl.h>
#  define _(String) gettext(String)
#  ifdef gettext_noop
#    define N_(String) gettext_noop(String)
#  else
#    define N_(String) (String)
#  endif
#else
#  define _(String) (String)
#  define N_(String) (String)
#  define textdomain(String) (String)
#  define gettext(String) (String)
#  define dgettext(Domain,String) (String)
#  define dcgettext(Domain,String,Type) (String)
#  define bindtextdomain(Domain,Directory) (Domain)
#endif /* ENABLE_NLS */


#define FNAME_ICON    PACKAGE_DATA_DIR G_DIR_SEPARATOR_S \
                      "pixmaps" G_DIR_SEPARATOR_S "gxmessage.png"


typedef struct _ButtonList ButtonList;



struct GXMessage {
	gchar             *message_text;
	gint              message_length;
	ButtonList        *button_list;
	const gchar       *default_str;
	const gchar       *title_str;
	const gchar       *geom_str;
	const gchar       *font_str;
	const gchar       *color_fg;
	const gchar       *color_bg;
	const gchar       *encoding;
	gint              timeout;
	guint             timeout_id;
	gboolean          do_iconify;
	gboolean          do_print;
	gboolean          do_entry;
	gboolean          do_buttons;
	gboolean          do_borderless;
	GtkWrapMode       wrap_mode;
	GtkWindowPosition window_position;
	GtkWidget         *entry_widget;
	gint              exit_code;
} gx;


struct _ButtonList {
	gboolean          is_default;
	gint              value;
	gchar             *label_str;
	ButtonList        *prev;
	ButtonList        *next;
};


struct Option {
	gint     min_len; /* support -bu, -but, -butt, etc., as with xmessage */
	gboolean requires_arg;
	gchar    *opt_str;
};


static struct Option option[] = { /* min_len, requires_arg, opt_str */
	{2, TRUE,  "buttons"},
	{1, FALSE, "center"},
	{2, TRUE,  "default"},
	{2, TRUE,  "file"},
	{2, FALSE, "nearmouse"},
	{1, FALSE, "print"},
	{3, TRUE,  "timeout"},
	{2, TRUE,  "fn"},
	{2, TRUE,  "font"},
	{1, TRUE,  "geometry"},
	{3, TRUE,  "title"},
	{2, TRUE,  "bg"},
	{2, TRUE,  "fg"},
	{2, TRUE,  "bd"},
	{2, TRUE,  "bw"},
	{1, FALSE, "iconic"},
	{2, TRUE,  "xrm"},
	{2, FALSE, "rv"},
	{2, FALSE, "reverse"},
	{2, TRUE,  "selectionTimeout"},
	{2, TRUE,  "xnllanguage"},
	{2, FALSE, "borderless"},
	{1, FALSE, "wrap"},
	{3, TRUE,  "encoding"},
	{1, FALSE, "?"},
	{1, FALSE, "help"},
	{1, FALSE, "version"},
	{1, FALSE, "entry"}
};
#define OPT_BUTTONS           0
#define OPT_CENTER            1
#define OPT_DEFAULT           2
#define OPT_FILE              3
#define OPT_NEARMOUSE         4
#define OPT_PRINT             5
#define OPT_TIMEOUT           6
#define OPT_FN                7
#define OPT_FONT              8
#define OPT_GEOMETRY          9
#define OPT_TITLE             10
#define OPT_BG                11
#define OPT_FG                12
#define OPT_BD                13
#define OPT_BW                14
#define OPT_ICONIC            15
#define OPT_XRM               16
#define OPT_RV                17
#define OPT_REVERSE           18
#define OPT_SELECTIONTIMEOUT  19
#define OPT_XNLLANGUAGE       20
#define OPT_BORDERLESS        21
#define OPT_WRAP              22
#define OPT_ENCODING          23
#define OPT_HELP_Q            24
#define OPT_HELP              25
#define OPT_VERSION           26
#define OPT_ENTRY             27
#define N_OPTS                28

#define OPT_IS_MISSING_ARG    -1
#define OPT_IS_UNKNOWN        -2



void prog_cleanup (void);



ButtonList*
button_first (ButtonList *button)
{
	if (button == NULL) return NULL;

	while (button->prev != NULL) button = button->prev;
	return button;
}



ButtonList*
button_new (void)
{
	ButtonList *button = g_new0 (ButtonList, 1);

	if (button == NULL) prog_cleanup ();
	return button;
}



void
button_free (ButtonList *button)
{
	ButtonList *button_next;

	button = button_first (button);
	while (button != NULL) {
		button_next = button->next;
		g_free (button->label_str);
		g_free (button);
		button = button_next;
	}
}



ButtonList*
button_append (ButtonList *button, ButtonList *new_button)
{
	if (button != NULL) {
		new_button->prev = button;
		button->next = new_button;
	}
	return new_button;
}



ButtonList*
button_new_from_str (const gchar *str, gint default_value, gint len)
{
	/*
	 * Given a string containing "label" or "label:value",
	 * fill in a new button structure and return it.
	 */
	ButtonList *button = button_new ();
	gboolean value_from_str = FALSE;
	gint i = len;


	/* parse "label:value" format in str */
	while (i != 0 && !value_from_str) value_from_str = str[--i] == ':';
	if (value_from_str) {
		/* get label and value from str */
		button->label_str = g_strndup (str, i);
		if (button->label_str == NULL) prog_cleanup ();
		button->value = atoi (&str[i+1]);
	}
	else {
		/* get label from str, value from default_value */
		button->label_str = g_strndup (str, len);
		if (button->label_str == NULL) prog_cleanup ();
		button->value = default_value;
	}

	/* xmessage will only draw the buttons if at least one of
	 * them has a label. To emulate that, keep track of it here:
	 */
	gx.do_buttons = gx.do_buttons || len > 0;

	return button;
}



ButtonList*
button_list_from_str (const gchar *str)
{
	/*
	 * Given a string containing a comma-separated list of
	 * "label" and/or "label:value", build and return a list
	 * of buttons. The return points to the last button in
	 * the list.
	 */
	ButtonList *button = NULL;
	ButtonList *b;
	gint len;
	gint i = 0;
	gint startpos = 0;
	gint value = 101;


	if (str == NULL) return NULL;

	len = strlen (str);
	while (i < len) {
		if (str[i] == ',') {
			b = button_new_from_str (&str[startpos], value, i - startpos);
			button = button_append (button, b);
			startpos = i + 1;
			value++;
		}
		i++;
	}

	b = button_new_from_str (&str[startpos], value, i - startpos);
	button = button_append (button, b);

	return button;
}



void
button_set_default (ButtonList *button, const gchar *str)
{
	/*
	 * Make button->is_default TRUE for any button whose label
	 * matches str. (If there's more than one, only the last
	 * one found will actually be used as the default - that
	 * happens automatically in window_create.)
	 */
	if (str == NULL) return;

	button = button_first (button);
	while (button != NULL) {
		button->is_default = strcmp (button->label_str, str) == 0;
		button = button->next;
	}
}



gint
my_get_opt (const gchar *str, gboolean is_last)
{
	/*
	 * Try to identify the command line option in str,
	 * returning a unique option/error code. The
	 * is_last variable specifies whether or not the
	 * current option was followed by something else
	 * (a value or another option) on the command line.
	 */
	gint opt;
	gint len;
	gint offset = 1;


	if (strcmp (str, "+rv") == 0) return OPT_RV;
	if (str[0] != '-') return OPT_IS_UNKNOWN;

	len = strlen (&str[offset]);
	if (len > 0) {
		if (str[1] == '-') {
			offset++;
			len--;
		}
		if (len > 0) {
			for (opt = 0; opt < N_OPTS; opt++) {
				if (len >= option[opt].min_len && strncmp (&str[offset], option[opt].opt_str, len) == 0) {
					if (!option[opt].requires_arg || is_last) {
						return opt;
					}
					else {
						return OPT_IS_MISSING_ARG;
					}
				}
			}
		}
	}
	return OPT_IS_UNKNOWN;
}



void
cb_window_destroy (GtkWidget *widget, GdkEvent *event, gpointer data)
{
	gtk_main_quit ();
}



void
cb_button_clicked (GtkWidget *widget, gpointer data)
{
	gx.exit_code = ((ButtonList*)data)->value;

	if (gx.do_print) {
		g_print ("%s\n", ((ButtonList*)data)->label_str);
	}
	else if (gx.do_entry) {
		g_print ("%s\n", gtk_entry_get_text (GTK_ENTRY(gx.entry_widget)));
	}
	gtk_main_quit ();
}



void
cb_entry_activated (GtkWidget *widget, gpointer data)
{
	gx.exit_code = 0;
	g_print ("%s\n", gtk_entry_get_text (GTK_ENTRY(gx.entry_widget)));
	gtk_main_quit ();
}



gint
cb_timeout (gpointer data)
{
	static gint counter = 0;

	if (++counter >= (gint)data) {
		gx.exit_code = 0;
		gtk_main_quit ();
	}
	return TRUE;
}



gint
geometry_get_number (const gchar **geometry)
{
	gint value = 0;
	gint mult  = 1;


	if (**geometry == '-') {
		mult = -1;
		(*geometry)++;
	}
	while ( **geometry && isdigit(**geometry) ) {
		value = value * 10 + (**geometry - '0');
		(*geometry)++;
	}
	return value * mult;
}



gboolean
geometry_parse_string (const gchar *geometry,
                       gint *x, gint *y, gint *w, gint *h)
{
	/*
	 * Parse geometry string into x, y, w and h. Return TRUE
	 * if successful, else FALSE. (This code is taken straight
	 * from the GNOME 1.x gnome-geometry.c)
	 */

	gint subtract;


	g_return_val_if_fail (x != NULL, FALSE);
	g_return_val_if_fail (y != NULL, FALSE);
	g_return_val_if_fail (w != NULL, FALSE);
	g_return_val_if_fail (h != NULL, FALSE);

	*x = *y = *w = *h = -1;

	if (geometry == NULL) return FALSE;

	if (*geometry == '=') geometry++;
	if (!*geometry) return FALSE;

	if (isdigit (*geometry)) {
		*w = geometry_get_number (&geometry);
	}
	if (!*geometry) {
		return TRUE;
	}
	if (*geometry == 'x') {
		geometry++;
		*h = geometry_get_number (&geometry);
	}
	if (!*geometry) {
		return TRUE;
	}


	if (*geometry == '+') {
		subtract = 0;
		geometry++;
	}
	else if (*geometry == '-') {
		subtract = gdk_screen_width ();
		geometry++;
	}
	else {
		return FALSE;
	}


	*x = geometry_get_number (&geometry);
	if (subtract) {
		*x = subtract - *x;
	}
	if (!*geometry) {
		return TRUE;
	}


	if (*geometry == '+') {
		subtract = 0;
		geometry++;
	}
	else if (*geometry == '-') {
		subtract = gdk_screen_height ();
		geometry++;
	}
	else {
		return FALSE;
	}


	*y = geometry_get_number (&geometry);
	if (subtract) {
		*y = subtract - *y;
	}

	return TRUE;
}



void
window_create (void)
{
	GtkWidget  *window;
	GtkWidget  *vbox, *vbox2;
	GtkWidget  *scroller;
	GtkWidget  *sep = NULL;
	GtkWidget  *button_box = NULL;
	GtkWidget  *widget = NULL;
	GtkWidget  *message_widget;
	ButtonList *button_list;
	GdkColor   color;
	PangoFontDescription *font_desc;
	GtkRequisition size_req;
	GtkTextBuffer *buf;
	gint win_w, win_h, scr_w, scr_h, max_default_w, max_default_h;
	gint tmp_w, tmp_h, tmp_x, tmp_y;


	window = gtk_window_new (GTK_WINDOW_TOPLEVEL);


	g_signal_connect (GTK_OBJECT(window),
	                  "destroy",
	                  GTK_SIGNAL_FUNC(cb_window_destroy),
	                  NULL);


	if (gx.title_str != NULL) {
		gtk_window_set_title (GTK_WINDOW(window), gx.title_str);
	}

	if (gx.do_iconify) {
		gtk_window_iconify (GTK_WINDOW(window));
	}

	gtk_window_set_icon_from_file (GTK_WINDOW(window), FNAME_ICON, NULL);

	if (gx.do_borderless) {
		gtk_window_set_decorated (GTK_WINDOW(window), 0);
	}


	/* window contents */
	gtk_container_set_border_width (GTK_CONTAINER(window), 6);

	vbox = gtk_vbox_new (FALSE, 0);
	gtk_container_add (GTK_CONTAINER(window), vbox);

	vbox2 = gtk_vbox_new (FALSE, 0);
	gtk_box_pack_start (GTK_BOX(vbox), vbox2, TRUE, TRUE, 0);
	gtk_container_set_border_width (GTK_CONTAINER(vbox2), 0);

	scroller = gtk_scrolled_window_new (NULL, NULL);
	gtk_container_set_border_width (GTK_CONTAINER(scroller), 0);
	gtk_scrolled_window_set_policy (GTK_SCROLLED_WINDOW(scroller), GTK_POLICY_AUTOMATIC, GTK_POLICY_AUTOMATIC);
	gtk_scrolled_window_set_shadow_type (GTK_SCROLLED_WINDOW(scroller), GTK_SHADOW_ETCHED_IN);
	gtk_scrolled_window_set_placement (GTK_SCROLLED_WINDOW(scroller), GTK_CORNER_TOP_LEFT);
	gtk_box_pack_start (GTK_BOX(vbox2), scroller, TRUE, TRUE, 0);


	/* the message */
	message_widget = gtk_text_view_new ();
	gtk_text_view_set_editable (GTK_TEXT_VIEW(message_widget), FALSE);
	gtk_text_view_set_cursor_visible (GTK_TEXT_VIEW(message_widget), FALSE);
	gtk_text_view_set_wrap_mode (GTK_TEXT_VIEW(message_widget), gx.wrap_mode);
	gtk_text_view_set_justification (GTK_TEXT_VIEW(message_widget), GTK_JUSTIFY_LEFT);
	gtk_text_view_set_left_margin (GTK_TEXT_VIEW(message_widget), 2);
	gtk_text_view_set_right_margin (GTK_TEXT_VIEW(message_widget), 2);
	gtk_text_view_set_pixels_above_lines (GTK_TEXT_VIEW(message_widget), 2);
	gtk_text_view_set_pixels_below_lines (GTK_TEXT_VIEW(message_widget), 2);
	gtk_text_view_set_pixels_inside_wrap (GTK_TEXT_VIEW(message_widget), 0);
	gtk_text_view_set_indent (GTK_TEXT_VIEW(message_widget), 0);

	if (gx.font_str != NULL) {
		font_desc = pango_font_description_from_string (gx.font_str);
	}
	else {
		/* default to a monospaced font */
		font_desc = pango_font_description_from_string ("monospace");
	}
	gtk_widget_modify_font (message_widget, font_desc);
	pango_font_description_free (font_desc);

	gtk_container_add (GTK_CONTAINER(scroller), message_widget);

	buf = gtk_text_view_get_buffer (GTK_TEXT_VIEW(message_widget));
	gtk_text_buffer_set_text (buf, gx.message_text, gx.message_length);

	if (gx.color_fg != NULL) {
		gdk_color_parse (gx.color_fg, &color);
		gtk_widget_modify_text (message_widget, GTK_STATE_NORMAL, &color);
	}
	if (gx.color_bg != NULL) {
		gdk_color_parse (gx.color_bg, &color);
		gtk_widget_modify_base (message_widget, GTK_STATE_NORMAL, &color);
	}


	/* text entry */
	if (gx.do_entry) {
		gx.entry_widget = gtk_entry_new ();
		gtk_editable_set_editable (GTK_EDITABLE(gx.entry_widget), TRUE);
		gtk_box_pack_start (GTK_BOX(vbox), gx.entry_widget, FALSE, FALSE, 5);
		gtk_widget_grab_focus (gx.entry_widget);
		if (!gx.do_buttons) {
			/* allow hitting <RETURN> to close the window */
			g_signal_connect (GTK_OBJECT(gx.entry_widget),
			                  "activate",
			                  GTK_SIGNAL_FUNC(cb_entry_activated),
			                  (gpointer)0);
		}
	}


	/* add buttons */
	if (gx.do_buttons) {

		sep = gtk_hseparator_new ();
		gtk_box_pack_start (GTK_BOX(vbox), sep, FALSE, FALSE, 5);

		button_list = button_first (gx.button_list);

		button_box = gtk_hbutton_box_new ();
		gtk_box_pack_end (GTK_BOX(vbox), button_box, FALSE, FALSE, 0);

		while (button_list != NULL) {

			widget = gtk_button_new_with_label (button_list->label_str);
			g_signal_connect (GTK_OBJECT(widget),
			                  "clicked",
			                  GTK_SIGNAL_FUNC(cb_button_clicked),
			                  (gpointer)button_list);
			gtk_box_pack_start (GTK_BOX(button_box), widget, FALSE, FALSE, 0);

			if (button_list->is_default) {
				gtk_widget_grab_focus (widget);
			}

			button_list = button_list->next;

		}

	}


	/*
	 * figure out window geometry
	 */

	/* start by getting the width and height of the GtkTextView's contents */

	gtk_widget_realize (message_widget);
#if 0
	/* FIXME: this only gets the _displayed_ geometry, but I need the full
	 * size of the GtkTextView widget's "contents" */
	/* gdk_window_get_geometry (GDK_WINDOW(message_widget->window), NULL, NULL, &win_w, &win_h, NULL); */
	{
		GtkRequisition req;
		gtk_widget_size_request (message_widget, &req);
		g_print ("%d %d\n", req.width, req.height);
		win_w = req.width;
		win_h = req.height;
	}
#else
	/* FIXME: these are temporary defaults */
	win_w = 300;
	win_h = 80;
#endif

	if (gx.do_entry) {
		/* get entry height + padding */
		gtk_widget_realize (gx.entry_widget);
		gtk_widget_size_request (gx.entry_widget, &size_req);
		win_h = win_h + size_req.height + 10;
	}
	if (gx.do_buttons) {
		/* get button height + separator height + padding */
		gtk_widget_size_request (button_box, &size_req);
		win_h = win_h + size_req.height;
		gtk_widget_size_request (sep, &size_req);
		win_h = win_h + size_req.height + 10;
	}

	/* a bit of extra padding */
	win_w = win_w + 30;
	win_h = win_h + 10;

	/* no bigger than 75% screen width x 75% screen height by default */
	scr_w = gdk_screen_width ();
	scr_h = gdk_screen_height ();
	max_default_w = scr_w * 0.75;
	max_default_h = scr_h * 0.75;
	if (win_w > max_default_w) win_w = max_default_w;
	if (win_h > max_default_h) win_h = max_default_h;


	/* if a geometry string was given, try to use it for width and height */
	if (gx.geom_str != NULL) {
		if (geometry_parse_string (gx.geom_str, &tmp_x, &tmp_y, &tmp_w, &tmp_h)) {
			if (win_w != -1) {
				win_w = tmp_w;
				win_h = tmp_h;
			}
		}
	}


	/* set height and position */
	gtk_window_set_default_size (GTK_WINDOW(window), win_w, win_h);
	gtk_window_set_position (GTK_WINDOW(window), gx.window_position);

	/* open the window */
	gtk_widget_show_all (window);


	/* begin timeout */
	if (gx.timeout != 0) {
		gx.timeout_id = gtk_timeout_add (1000, (GtkFunction)cb_timeout,
		                                 (gpointer)gx.timeout);
	}
}



gchar*
read_stdin (void)
{
	GString *text;
	gchar *str;
	gint ch;

	text = g_string_new ("");
	if (text == NULL) prog_cleanup ();

	while ( (ch = getc (stdin)) != EOF ) {
		if (g_string_append_c (text, ch) == NULL) prog_cleanup ();
	}
	str = text->str;
	g_string_free (text, FALSE);
	return str;
}



gchar*
message_to_utf8 (const gchar *str)
{
	gchar *result;
	GError *error = NULL;

	if (gx.encoding == NULL) {
		/* assume message encoding matches current locale */
		result = g_locale_to_utf8 (str, -1, NULL, NULL, NULL);
	}
	else {
		/* use encoding specified on command line */
		result = g_convert_with_fallback (str, -1, "UTF-8", gx.encoding,
		                                  NULL, NULL, NULL, NULL);
	}

	if (result == NULL) {
		/* fall back to ISO-8859-1 as source encoding */
		result = g_convert_with_fallback (str, -1, "UTF-8", "ISO-8859-1",
		                                  NULL, NULL, NULL, &error);
		if (result == NULL) {
			if (error != NULL && error->message != NULL) {
				g_printerr (PACKAGE ": %s\n", error->message);
			}
			prog_cleanup ();
		}
	}
	return result;
}



void
option_kludge (gint *argc, gchar ***argv)
{
	/*
	 * Change -display, -name and -synchronous to
	 * --display, --name and --sync, ready for gtk_init.
	 * (This function is based on code in GTK 1.2)
	 */

	gint i, j, k, len;


	for (i = 1; i < *argc;) {

		len = strlen ((*argv)[i]);

		if (len > 2) {

			if (strncmp ("-display", (*argv)[i], len) == 0) {
				(*argv)[i] = "--display";
				if ((i + 1) < *argc && (*argv)[i + 1]) {
					i += 1;
				}
			}
			else if (strncmp ("-name", (*argv)[i], len) == 0) {
				(*argv)[i] = "--name";
				if ((i + 1) < *argc && (*argv)[i + 1]) {
					i += 1;
				}
			}
			else if (strncmp ("-synchronous", (*argv)[i], len) == 0) {
				(*argv)[i] = "--sync";
			}
		}

		i += 1;

	}

	for (i = 1; i < *argc; i++) {
		for (k = i; k < *argc; k++) {
			if ((*argv)[k] != NULL) {
				break;
			}
		}
		if (k > i) {
			k -= i;
			for (j = i + k; j < *argc; j++) {
				(*argv)[j-k] = (*argv)[j];
			}
			*argc -= k;
		}
	}
}



void
usage (void)
{
	g_print(_("\n%s - a GTK-based xmessage clone\n\n"), PACKAGE);
	g_print(_("Usage: %s [OPTIONS] message ...\n"), PACKAGE);
	g_print(_("       %s [OPTIONS] -file FILENAME\n\n"), PACKAGE);
	g_print(_("xmessage options:\n" \
	        "  -file FILENAME         Get message text from file, '-' for stdin\n" \
	        "  -buttons BUTTON_LIST   List of \"LABEL:EXIT_CODE\", comma separated\n" \
	        "  -default LABEL         Give keyboard focus to the specified button\n" \
	        "  -print                 Send the selected button's LABEL to stdout\n" \
	        "  -center                Try to open window in the centre of the screen\n" \
	        "  -nearmouse             Try to open window near the mouse pointer\n" \
	        "  -timeout SECONDS       Exit with code 0 after SECONDS seconds\n\n"));
	g_print(_("X-like options:\n" \
	        "  -display DISPLAY       X display to use\n" \
	        "  -fn FONT | -font FONT  Set message font (works with GTK font names)\n" \
	        "  -fg COLOUR             Set message font colour\n" \
	        "  -bg COLOUR             Set message background colour\n" \
	        "  -geometry GEOMETRY     Set window size (position will be ignored)\n" \
	        "  -iconic                Start iconified\n" \
	        "  -name NAME             Program name as used by the window manager\n" \
	        "  -title TITLE           Set window title to TITLE\n\n"));
	g_print(_("%s options:\n" \
	        "  -borderless            Try to open window without border decoration\n" \
	        "  -encoding CHARSET      Expect CHARSET as the message encoding\n" \
	        "  -entry                 Prompt for text to be sent to stdout\n" \
	        "  -wrap                  Wrap lines of text to fit window width\n" \
	        "  -help | -?             Show this usage information\n" \
	        "  -version               Show gxmessage version information\n\n"), PACKAGE);

	prog_cleanup ();
}



void
gxmessage_init (void)
{
	gx.message_text    = NULL;
	gx.message_length  = 0;
	gx.button_list     = NULL;
	gx.default_str     = NULL;
	gx.title_str       = NULL;
	gx.geom_str        = NULL;
	gx.font_str        = NULL;
	gx.color_fg        = NULL;
	gx.color_bg        = NULL;
	gx.encoding        = NULL;
	gx.timeout         = 0;
	gx.timeout_id      = 0;
	gx.do_print        = FALSE;
	gx.do_iconify      = FALSE;
	gx.do_buttons      = FALSE;
	gx.do_borderless   = FALSE;
	gx.wrap_mode       = GTK_WRAP_NONE;
	gx.window_position = GTK_WIN_POS_NONE;
	gx.exit_code       = 1;
}



void
prog_cleanup (void)
{
	button_free (gx.button_list);
	if (gx.message_text != NULL) {
		g_free (gx.message_text);
	}
	if (gx.timeout_id != 0) {
		gtk_timeout_remove (gx.timeout_id);
	}

	exit (gx.exit_code);
}



int
main (gint argc, gchar *argv[])
{
	GString *gstr = NULL;
	gboolean input_is_file = FALSE;
	gchar *ch = NULL, *tmpstr;
	gint opt, arg = 1;


#ifdef ENABLE_NLS
	bindtextdomain (PACKAGE, PACKAGE_LOCALE_DIR);
	bind_textdomain_codeset (PACKAGE, "UTF-8");
	textdomain (PACKAGE);
#endif

	option_kludge (&argc, &argv);
	gtk_init (&argc, &argv);
	gxmessage_init ();


	while (arg < argc) {
		opt = my_get_opt (argv[arg], arg+1 < argc);
		switch (opt) {
		case OPT_HELP :
		case OPT_HELP_Q :
			gx.exit_code = 0;
			usage ();
			break;
		case OPT_VERSION :
			g_print (PACKAGE "-" VERSION "\n");
			exit (0);
			break;
		case OPT_ENTRY :
			if (gx.do_print) {
				g_printerr (_("%s: can't have both -entry and -print\n"), PACKAGE);
				prog_cleanup ();
			}
			if (gx.timeout) {
				g_printerr (_("%s: can't have both -entry and -timeout\n"), PACKAGE);
				gx.timeout = 0;
			}
			gx.do_entry = TRUE;
			break;
		case OPT_BUTTONS :
			button_free (gx.button_list);
			gx.button_list = button_list_from_str (argv[++arg]);
			break;
		case OPT_CENTER :
			gx.window_position = GTK_WIN_POS_CENTER;
			break;
		case OPT_DEFAULT :
			gx.default_str = argv[++arg];
			break;
		case OPT_FILE :
			if (input_is_file) {
				g_printerr (_("%s: only one -file allowed\n"), PACKAGE);
				prog_cleanup ();
			}
			arg++;
			input_is_file = TRUE;
			if (gstr != NULL) {
				g_printerr (_("%s: can't get message from both -file and command line\n"), PACKAGE);
				prog_cleanup ();
			}
			if (strcmp ("-", argv[arg]) == 0) {
				tmpstr = read_stdin ();
			}
			else if (!g_file_get_contents (argv[arg], &tmpstr, NULL, NULL)) {
				g_printerr (_("%s: unable to read file\n"), PACKAGE);
				prog_cleanup ();
			}
			gx.message_text = message_to_utf8 (tmpstr);
			gx.message_length = strlen (gx.message_text);
			g_free (tmpstr);
			break;
		case OPT_NEARMOUSE :
			/* -center takes priority over -nearmouse */
			if (gx.window_position != GTK_WIN_POS_CENTER) {
				gx.window_position = GTK_WIN_POS_MOUSE;
			}
			break;
		case OPT_PRINT :
			if (gx.do_entry) {
				g_printerr (_("%s: can't have both -entry and -print\n"), PACKAGE);
				prog_cleanup ();
			}
			gx.do_print = TRUE;
			break;
		case OPT_TIMEOUT :
			gx.timeout = strtol (argv[++arg], &ch, 10);
			if (*ch) {
				g_printerr (_("%s: integer -timeout value expected\n"), PACKAGE);
			}
			if (gx.timeout < 0) {
				/* xmessage doesn't complain here, just disables timeout */
				gx.timeout = 0;
			}
			if (gx.do_entry) {
				g_printerr (_("%s: can't have both -entry and -timeout\n"), PACKAGE);
				gx.timeout = 0;
			}
			break;
		case OPT_TITLE :
			gx.title_str = argv[++arg];
			break;
		case OPT_GEOMETRY :
			gx.geom_str = argv[++arg];
			break;
		case OPT_FN :
		case OPT_FONT :
			gx.font_str = argv[++arg];
			break;
		case OPT_RV :
		case OPT_REVERSE :
			/* not implemented - ignore */
			break;
		case OPT_BG :
			gx.color_bg = argv[++arg];
			break;
		case OPT_FG :
			gx.color_fg = argv[++arg];
			break;
		case OPT_BD :
		case OPT_BW :
		case OPT_XRM :
		case OPT_SELECTIONTIMEOUT :
		case OPT_XNLLANGUAGE :
			/* not implemented - ignore and skip arg */
			arg++;
			break;
		case OPT_ICONIC :
			gx.do_iconify = TRUE;
			break;
		case OPT_BORDERLESS :
			gx.do_borderless = TRUE;
			break;
		case OPT_WRAP :
			gx.wrap_mode = GTK_WRAP_WORD;
			break;
		case OPT_ENCODING :
			gx.encoding = argv[++arg];
			break;
		case OPT_IS_MISSING_ARG :
			/* in this case, xmessage treats the "option" as normal text */
		case OPT_IS_UNKNOWN :
		default:
			if (input_is_file) {
				g_printerr (_("%s: can't get message from both -file and command line\n"), PACKAGE);
				prog_cleanup ();
			}
			if (gstr == NULL) {
				gstr = g_string_new ("");
				if (gstr == NULL) prog_cleanup ();
			}
			else {
				gstr = g_string_append_c (gstr, ' ');
				if (gstr == NULL) prog_cleanup ();
			}
			gstr = g_string_append (gstr, argv[arg]);
			if (gstr == NULL) prog_cleanup ();
			break;
		}
		arg++;
	}


	if (gstr != NULL) {
		gx.message_text = message_to_utf8 (gstr->str);
		gx.message_length = strlen (gx.message_text);
		g_string_free (gstr, TRUE);
	}

	if (gx.message_text == NULL) {
		g_printerr (_("%s: message text is required\n"), PACKAGE);
		g_printerr (_("Try `%s --help' for more information\n"), PACKAGE);
		prog_cleanup ();
	}

	if (gx.button_list == NULL) {
		gx.button_list = button_append (NULL, button_new_from_str ("okay:0", 0, 6));
	}


	button_set_default (gx.button_list, gx.default_str);


	window_create ();
	gtk_main ();


	prog_cleanup ();
	return 0;
}
