/*--------------------------------------------------------------------
 *    $Id: ps2raster.c,v 1.26 2008/05/01 22:41:00 remko Exp $
 *
 *    Copyright (c) 2006-2008 by J. Luis
 *    See README file for copying and redistribution conditions.
 *--------------------------------------------------------------------*/
/*
 *
 * ps2raster converts one or several PostScript file(s) to other formats using GhostScript.
 * It works by modifying the page size in order that the image will have a size
 * which is specified by the BoundingBox.
 * As an option, a tight BoundingBox may be computed.
 * ps2raster uses the ideas of the EPS2XXX.m from Primoz Cermelj published in MatLab Central
 * and of psbbox.sh of Remko Scharroo.
 *
 *
 *--------------------------------------------------------------------*/
/*
 * Authors:	Joaquim Luis and Remko Scharroo
 * Created:	15-FEB-2005
 * Modified:	15-NOV-2005 (It used to fail too many times in detecting if is GMT_ps)
 *		01-DEC-2005 Changed the generated script name to not overlap with -N
 *		27-DEC-2005 Integrated into GMT's misc supplement
 *		07-SEP-2006 Added EPS format; no more GMT specific; high-quality PDF output
 *		 1-APR-2007 Moved to GMT src directory
 *		 6-JUN-2007 Added -P option to force Portrait mode
 *		14-SEP-2007 Reduced bbox rendering from the default 4000 to 720 DPI. This
 *			    produces about 50% speedup for the -A option.
 *		 4-NOV-2007 Added -D to select alternative directory for output. Also added -V
 *			    and modifier -Au to remove GMT time-stamps.
 *		 6-NOV-2007 Can now process crappy Illu*er EPS files with its mix of \r and \n
 */

#include "gmt.h"

#ifdef WIN32	/* Special for Windows */
#include <process.h>
#define getpid _getpid
#endif

#define N_GS_DEVICES		6	/* Number of supported GS output devices */
#define GS_DEV_EPS		0
#define GS_DEV_PDF		1
#define GS_DEV_JPG		2
#define GS_DEV_PNG		3
#define GS_DEV_PPM		4
#define GS_DEV_TIF		5

struct PS2RASTER_CTRL {
	struct A {	/* -A[u] [Adjust boundingbox] */
		BOOLEAN active;
		BOOLEAN strip;	/* Remove the -U time-stamp */
	} A;
	struct D {	/* -D<dir> */
		BOOLEAN active;
		char *dir;
	} D;
	struct E {	/* -E<resolution> */
		BOOLEAN active;
		int dpi;
	} E;
	struct G {	/* -G<GSpath> */
		BOOLEAN active;
		char *file;
	} G;
	struct L {	/* -L<listfile> */
		BOOLEAN active;
		char *file;
	} L;
	struct N {	/* -N */
		BOOLEAN active;
	} N;
	struct P2 {	/* -P */
		BOOLEAN active;
	} P;
	struct S {	/* -S */
		BOOLEAN active;
	} S;
	struct T {	/* -T */
		BOOLEAN active;
		BOOLEAN eps;	/* TRUE if we want to make EPS (possibly in addition to another format) */
		int device;
	} T;
	struct V2 {	/* -V */
		BOOLEAN active;
	} V;
};

int main (int argc, char **argv) {
	BOOLEAN error = FALSE;

	int i, j, k, len, w, h, r, xt, yt, x0 = 0, x1 = 612, y0 = 0, y1 = 828, n_files = 0;
	int got_BB, got_BBatend, got_orient, landscape, setup, pos_file, pos_ext, nr, nl;

	size_t n_alloc = GMT_SMALL_CHUNK;

	char ps_file[BUFSIZ];
	char **ps_names, *no_U_file = NULL, *clean_PS_file = NULL, *tmp_file = NULL, *out_file = NULL, *BB_file = NULL;
	char line[BUFSIZ], c[BUFSIZ], *p, c1[20], c2[20], c3[20], c4[20], cmd[BUFSIZ];
	char *gs_params = NULL, *gs_BB = NULL;
	char *device[N_GS_DEVICES] = {"", "pdfwrite", "jpeg", "png16m", "ppmraw", "tiff24nc"};
	char *ext[N_GS_DEVICES] = {".eps", ".pdf", ".jpg", ".png", ".ppm", ".tif"};
#ifdef WIN32
	char at_sign[2] = "@";
#else
	char at_sign[2] = "";
#endif

	FILE *fp = NULL, *fpo = NULL, *fpb = NULL, *fpl = NULL, *fp2 = NULL;

	struct PS2RASTER_CTRL *Ctrl;

	void *New_Ps2raster_Ctrl (), Free_Ps2raster_Ctrl (struct PS2RASTER_CTRL *C);

	Ctrl = (struct PS2RASTER_CTRL *)New_Ps2raster_Ctrl ();	/* Allocate and initialize a new control structure */

	/* Parameters for all the formats available */

	gs_params = "-q -dNOPAUSE -dBATCH -dUseFlateCompression=true -dPDFSETTINGS=/prepress -dEmbedAllFonts=true -dMonoImageFilter=/FlateEncode -dAutoFilterGrayImages=false -dGrayImageFilter=/FlateEncode -dAutoFilterColorImages=false -dColorImageFilter=/FlateEncode";
	gs_BB = "-q -dNOPAUSE -dBATCH -sDEVICE=bbox -r720";

	/* Check and interpret the command line arguments */

	GMT_program = argv[0];
	for (i = 1; i < argc; i++) {
		if (argv[i][0] == '-') {
			switch(argv[i][1]) {
				case '\0':
					GMT_give_synopsis_and_exit = TRUE;
					break;

				/* Supplemental parameters */
				case 'A':	/* Adjust BoundingBox */
					Ctrl->A.active = TRUE;
					if (argv[i][2] == 'u') Ctrl->A.strip = TRUE;
					break;
				case 'D':	/* Change output directory */
					Ctrl->D.active = TRUE;
					Ctrl->D.dir = strdup (&argv[i][2]);
					break;
				case 'G':	/* Set GS path */
					Ctrl->G.active = TRUE;
					free ((void *)Ctrl->G.file);
					Ctrl->G.file = strdup (&argv[i][2]);
					break;
				case 'E':	/* Set output dpi */
					Ctrl->E.active = TRUE;
					Ctrl->E.dpi = atoi(&argv[i][2]);
					break;
				case 'L':	/* Give list of files to convert */
					Ctrl->L.active = TRUE;
					Ctrl->L.file = strdup (&argv[i][2]);
					break;
				case 'N':	/* Do NOT remove auxillary files used with GS */
					Ctrl->N.active = TRUE;
					break;
				case 'P':	/* Force Portrait mode */
					Ctrl->P.active = TRUE;
					break;
				case 'S':	/* Write the GS command to STDOUT */
					Ctrl->S.active = TRUE;
					break;
				case 'T':	/* Select output format (optionally also request EPS) */
					Ctrl->T.active = TRUE;
					for (j = 2; argv[i][j]; j++) {
						switch (argv[i][j]) {
							case 'e':	/* EPS */
								Ctrl->T.eps = TRUE;
								break;
							case 'f':	/* PDF */
								Ctrl->T.device = GS_DEV_PDF;
								break;
							case 'j':	/* JPEG */
								Ctrl->T.device = GS_DEV_JPG;
								break;
							case 'g':	/* PNG */
								Ctrl->T.device = GS_DEV_PNG;
								break;
							case 'm':	/* PPM */
								Ctrl->T.device = GS_DEV_PPM;
								break;
							case 't':	/* TIFF */
								Ctrl->T.device = GS_DEV_TIF;
								break;
							default:
								fprintf (stderr, "%s: GMT ERROR: Unrecognized option %s\n", GMT_program, argv[i]);
								error = TRUE;
								break;
						}
					}
					break;
				case 'V':	/* Verbose  */
					Ctrl->V.active = TRUE;
					break;
				/* Options not recognized */
				default:
					error = TRUE;
					break;
			}
		}
		else {
			strcpy (ps_file, argv[i]);
			n_files++;
		}
	}

	if (argc == 1 || GMT_give_synopsis_and_exit || error) {
		fprintf (stderr,"ps2raster %s - Converts one or several [E]PS file(s) to raster formats using GhostScript.\n\n", GMT_VERSION);
		fprintf(stderr,"usage: ps2raster <psfile1> <psfile2> <...> [-A[u]] [-D<dir>] [-E<resolution>]\n");
		fprintf(stderr,"       [-G<ghost_path>] [-L<listfile>] [-N] [-P] [-Te|f|g|j|m|t] [-V]\n\n");
		if (GMT_give_synopsis_and_exit) exit (EXIT_FAILURE);

		fprintf (stderr,"Works by modifying the page size in order that the resulting\n");
		fprintf (stderr,"image will have the size specified by the BoundingBox.\n");
		fprintf (stderr,"As an option, a tight BoundingBox may be computed.\n\n");
		fprintf (stderr,"	<psfile(s)> postscript file(s) to be converted.\n");
		fprintf(stderr,"\n\tOPTIONS:\n");
		fprintf (stderr,"\t-A Adjust the BoundingBox to the minimum required by the image contents\n");
		fprintf (stderr,"\t   Append u to strip out time-stamps (produced by GMT -U options)\n");
		fprintf (stderr,"\t-D Sets an alternative output directory (which must exist) [Default is same directory as PS files]\n");
		fprintf (stderr,"\t   Use -D. to place the output in the current directory.\n");
		fprintf (stderr,"\t-E Set raster resolution in dpi [default = 720 for PDF, 300 for others]\n");
		fprintf (stderr,"\t-G Full path to your ghostscript executable.\n");
		fprintf (stderr,"\t   NOTE: Under Unix systems this is generally not necessary.\n");
		fprintf (stderr,"\t   Under Windows, ghostscript is not added to the system's path.\n");
		fprintf (stderr,"\t   So either you do it yourself, or give the full path here.\n");
		fprintf (stderr,"\t   (e.g. -Gc:\\programs\\gs\\gs7.05\\bin\\gswin32c).\n");
		fprintf (stderr,"\t-L The <listfile> is an ASCII file with names of ps files to be converted\n");
		fprintf (stderr,"\t-N OBSOLETE. Use -S and/or -Te instead.\n");
		fprintf (stderr,"\t-P Force Portrait mode. All Landscape mode plots will be rotated back\n");
		fprintf (stderr,"\t   so that they show unrotated in Portrait mode.\n");
		fprintf (stderr,"\t   This is practical when converting to image formats or preparing\n");
		fprintf (stderr,"\t   EPS or PDF plots for inclusion in documents.\n");
		fprintf (stderr,"\t-S Apart from executing it, also writes the ghostscript cammand to standard output.\n");
		fprintf (stderr,"\t-T Set output format [default is jpeg]\n");
		fprintf (stderr,"\t   e means EPS\n");
		fprintf (stderr,"\t   f means PDF\n");
		fprintf (stderr,"\t   g means PNG\n");
		fprintf (stderr,"\t   j means JPEG\n");
		fprintf (stderr,"\t   m means PPM\n");
		fprintf (stderr,"\t   t means TIF\n");
		fprintf (stderr,"\t   The EPS format can be combined with any of the other formats.\n");
		fprintf (stderr,"\t   For example, -Tef creates both an EPS and PDF file.\n");
		fprintf (stderr,"\t-V Provides progress report [default is silent]\n");
		exit (EXIT_FAILURE);
	}

	if (Ctrl->N.active) {
		fprintf (stderr, "%s: GMT WARNING: Option -N is obsolete. Will run with -S -Te.\n", GMT_program);
		Ctrl->S.active = Ctrl->T.eps = TRUE;
	}

	if (!Ctrl->T.active) Ctrl->T.device = GS_DEV_JPG;	/* Default output device if none is specified */

	if (error) exit (EXIT_FAILURE);

	if (n_files > 1 && Ctrl->L.active) {
		fprintf (stderr, "%s: GMT SYNTAX ERROR: Cannot handle both a file list and multiple ps files in input\n", GMT_program);
		error = TRUE;
	}

	if (Ctrl->L.active && (fpl = fopen (Ctrl->L.file, "r")) == NULL) {
		fprintf (stderr, "%s: GMT ERROR: Cannot to open list file %s\n", GMT_program, Ctrl->L.file);
		error = TRUE;
	}

	/* Use default DPI if not already set */
	if (Ctrl->E.dpi <= 0) Ctrl->E.dpi = (Ctrl->T.device == GS_DEV_PDF) ? 720 : 300;

	/* Multiple files in a file with their names */
	if (Ctrl->L.active) {
		ps_names = (char **) GMT_memory (VNULL, n_alloc, sizeof (char *), GMT_program);
		while (fgets (line, BUFSIZ, fpl) != NULL) {
			ps_names[n_files] = (char *) GMT_memory (VNULL, (size_t)BUFSIZ, sizeof (char), GMT_program);
			if (line[0] == '#' || line[0] == '\n' || line[0] == '\r') continue;
			sscanf (line, "%s",ps_names[n_files]);
			n_files++;
			if (n_files > (int)n_alloc) {
				n_alloc <<= 1;
				ps_names = GMT_memory ((void *)ps_names, n_alloc, sizeof (char *), GMT_program);
			}
		}
		fclose (fpl);
	}

	/* Multiple files given on command line */

	else if (n_files > 1) {
		ps_names = (char **) GMT_memory (VNULL, n_alloc, sizeof (char *), GMT_program);
		j = 0;
		for (k = 1; k < argc; k++) {
			if (argv[k][0] == '-') continue;
			ps_names[j] = (char *) GMT_memory (VNULL, (size_t)BUFSIZ, sizeof (char), GMT_program);
			ps_names[j] = argv[k];
			j++;
			if (n_files > (int)n_alloc) {
				n_alloc <<= 1;
				ps_names = GMT_memory ((void *)ps_names, n_alloc, sizeof (char *), GMT_program);
			}
		}
	}
	else {				/* Single file */
		ps_names = (char **) GMT_memory (VNULL, (size_t)1, sizeof (char *), GMT_program);
		ps_names[0] = ps_file;
	}

	/* Loop over all input files */

	for (k = 0; k < n_files; k++) {
		strcpy(ps_file,ps_names[k]);
		if ((fp = fopen (ps_file, "r")) == NULL) {
			fprintf (stderr, "%s: Cannot to open file %s\n", GMT_program, ps_file);
			continue;
		}
		if (Ctrl->V.active) fprintf (stderr, "%s: Processing %s:", GMT_program, ps_file);

		/* Check for JUNKY Illustrator EPS crap with mix of \r and \n */
		fgets (line, BUFSIZ, fp);	/* Get a single line - this will be a full buffer if crap */
		for (j = nr = nl = 0; j < (int)strlen (line); j++) {
			if (line[j] == '\r') nr++;	/* Number of CR found */
			if (line[j] == '\n') nl++;	/* Number of LF found */
		}
		rewind (fp);	/* Go back to beginning */
		if (nl == 0 && nr) {	/* Must create temporary file without the \r shit */
			int byte;
			if (Ctrl->V.active) fprintf (stderr, " Clean EPS...");
			clean_PS_file = (char *) GMT_memory (VNULL, (size_t)BUFSIZ, sizeof(char), GMT_program);
			sprintf (clean_PS_file, "%s%cps2raster_%da.eps", Ctrl->D.dir, DIR_DELIM, (int)getpid());
			if ((fp2 = fopen (clean_PS_file, "w+")) == NULL) {
				fprintf (stderr, "%s: Unable to create a temporary file\n", GMT_program);
				exit (EXIT_FAILURE);
			}
			while ((byte = fgetc (fp)) != EOF) {
				if (byte == '\r') byte = '\n';	/* Replace the CR with LF */
				fputc (byte, fp2);
			}
			fclose (fp);	/* Close original PS file */
			rewind (fp2);	/* Rewind new clean PS file */
			fp = fp2;	/* Set original file pointer to this file instead */
		}

		if (Ctrl->A.strip) {	/* Must strip off the GMT timestamp stuff */
			BOOLEAN dump = TRUE;
			if (Ctrl->V.active) fprintf (stderr, " Strip GMT time-stamp...");
			no_U_file = (char *) GMT_memory (VNULL, (size_t)BUFSIZ, sizeof(char), GMT_program);
			sprintf (no_U_file, "%s%cps2raster_%db.eps", Ctrl->D.dir, DIR_DELIM, (int)getpid());
			if ((fp2 = fopen (no_U_file, "w+")) == NULL) {
				fprintf (stderr, "%s: Unable to create a temporary file\n", GMT_program);
				exit (EXIT_FAILURE);
			}
			while (fgets (line, BUFSIZ, fp) != NULL) {
				if (!strncmp (line, "% Begin GMT time-stamp", (size_t)22)) dump = FALSE;
				if (dump) fprintf (fp2, "%s", line);
				if (!strncmp (line, "% End GMT time-stamp", (size_t)20)) dump = TRUE;
			}
			fclose (fp);	/* Close original PS file */
			rewind (fp2);	/* Rewind new file without timestamp */
			fp = fp2;	/* Set original file pointer to this file instead */
		}

		got_BB = got_BBatend = got_orient = landscape = setup = FALSE;

		len = strlen(ps_file);
		j = len - 1;
		pos_file = -1;
		pos_ext = -1;	/* In case file has no extension */
		for (i = 0; i < len; i++, j--) {
			if (pos_ext < 0 && ps_file[j] == '.') pos_ext = j;	/* Beginning of file extension */
			if (pos_file < 0 && (ps_file[j] == '/' || ps_file[j] == '\\')) pos_file = j + 1;	/* Beginning of file name */
		}
		if (pos_ext == -1) pos_ext = len - 1;	/* File has no extension */
		if (!Ctrl->D.active || pos_file == -1) pos_file = 0;	/* File either has no leading directory or we want to use it */

		/* Adjust to a tight BoundingBox if user requested so */

		if (Ctrl->A.active) {
			char *psfile_to_use;
			if (Ctrl->V.active) fprintf (stderr, " Find BoundingBox ");
			BB_file = (char *) GMT_memory (VNULL, (size_t)BUFSIZ, sizeof(char), GMT_program);
			sprintf (BB_file, "%s%cps2raster_%dc.bb", Ctrl->D.dir, DIR_DELIM, (int)getpid());
			psfile_to_use = (Ctrl->A.strip) ? no_U_file : ((clean_PS_file) ? clean_PS_file : ps_file);
			sprintf (cmd, "%s%s %s %s 2> %s", at_sign, Ctrl->G.file, gs_BB, psfile_to_use, BB_file);
			system (cmd);		/* Execute the command that computes the tight BB */
			if ((fpb = fopen (BB_file, "r")) == NULL) {
				fprintf (stderr, "%s: Unable to open file %s\n", GMT_program, BB_file);
				exit (EXIT_FAILURE);
			}
			while (fgets (line, BUFSIZ, fpb) != NULL && !got_BB) {
				if ((p = strstr (line,"%%BoundingBox:"))) {
					sscanf (&p[14], "%s %s %s %s", c1, c2, c3, c4);
					x0 = atoi (c1);		y0 = atoi (c2);
					x1 = atoi (c3);		y1 = atoi (c4);
					if (x1 <= x0 || y1 <= y0) {
						fprintf (stderr, "%s: Unable to decode BoundingBox file %s\n", GMT_program, BB_file);
						exit (EXIT_FAILURE);
					}
					got_BB = TRUE;
				}
			}
			fclose (fpb);
			remove (BB_file);	/* Remove the file with BB info */
			GMT_free ((void *)BB_file);
			if (got_BB && Ctrl->V.active) fprintf (stderr, "[%d %d %d %d]...", x0, y0, x1, y1);
		}

		/* Open temporary file to be processed by ghostscript. When -Te is used, tmp_file is for keeps */

		if (Ctrl->V.active && Ctrl->T.eps) fprintf (stderr, " Format EPS file...");
		tmp_file = (char *) GMT_memory (VNULL, (size_t)BUFSIZ, sizeof(char), GMT_program);
		if (Ctrl->T.eps) {
			if (Ctrl->D.active) sprintf (tmp_file, "%s%c", Ctrl->D.dir, DIR_DELIM);	/* Use specified output directory */
			strncat (tmp_file, &ps_file[pos_file], (size_t)(pos_ext - pos_file));
			strcat (tmp_file, ext[GS_DEV_EPS]);
			if ((fpo = fopen (tmp_file, "w")) == NULL) {
				fprintf (stderr, "%s: Unable to open file %s for writing\n", GMT_program, tmp_file);
				exit (EXIT_FAILURE);
			}
		}
		else {
			sprintf (tmp_file, "%s%cps2raster_%dd.eps", Ctrl->D.dir, DIR_DELIM, (int)getpid());
			if ((fpo = fopen (tmp_file, "w+")) == NULL) {
				fprintf (stderr, "%s: Unable to create a temporary file\n", GMT_program);
				exit (EXIT_FAILURE);
			}
		}

		/* Scan first 20 lines of input file for BoundingBox and Orientation statements */

		i = 0;
		while ((fgets (line, BUFSIZ, fp) != NULL) && i < 20 && !(got_BB && got_orient)) {
			i++;
			if (!line[0] || line[0] != '%') continue;/* Skip empty and non-comment lines */
			if (!got_BB && (p = strstr (line, "%%BoundingBox:"))) {
				sscanf (&p[14], "%s %s %s %s",c1,c2,c3,c4);
				if (strncmp (c1, "(atend)", (size_t)7)) {	/* Got actual numbers */
					x0 = atoi (c1);		y0 = atoi (c2);
					x1 = atoi (c3);		y1 = atoi (c4);
					got_BB = TRUE;
				}
				else
					got_BBatend++;
			}
			else if (!got_orient && (p = strstr (line, "%%Orientation:"))) {
				sscanf (&p[14], "%s", c1);
				if (!strncmp (c1, "Landscape", (size_t)9)) landscape = TRUE;
				got_orient = TRUE;
			}
			if (got_BBatend == 1 && got_orient) {	/* Now is the time to look at the end of the file */
				got_BBatend++;			/* Avoid jumping more than once to the end */
				if (!fseek (fp, (long)-132, SEEK_END)) i = 0;
			}
		}

		/* Can not proceed without knowing the BoundingBox */

		if (!got_BB) {
			fprintf (stderr, "%s: GMT FATAL ERROR: The file %s has no BoundingBox in the first or last 20 lines. Use -A option.\n", GMT_program, ps_file);
			exit (EXIT_FAILURE);
		}

		/* Do the math on the BoundingBox and translation coordinates */

		if (Ctrl->P.active && landscape)
			xt = -x1, yt = -y0, w = y1-y0, h = x1-x0, x0 = y0 = 0, r = -90;
		else
			xt = -x0, yt = -y0, w = x1-x0, h = y1-y0, x0 = y0 = r = 0;

		/* Rewind the input file and start copying and replacing */

		rewind (fp);
		while (fgets (line, BUFSIZ, fp) != NULL) {
			if (line[0] != '%') {	/* Copy any non-comment line, except one containing /PageSize in the Setup block */
				if (setup && strstr(line,"/PageSize") != NULL) continue;
				fprintf (fpo, "%s", line);
				continue;
			}
			sscanf (line, "%s",c);
			if (!strncmp(c, "%%BoundingBox:", (size_t)14)) {
				if (got_BB) fprintf (fpo, "%%%%BoundingBox: %d %d %d %d\n", x0, y0, x0+w, y0+h);
				got_BB = FALSE;
				continue;
			}
			else if (Ctrl->P.active && landscape && !strncmp(c, "%%Orientation:", (size_t)14)) {
				fprintf (fpo, "%%%%Orientation: Portrait\n");
				landscape = FALSE;
				continue;
			}
			else if (!strncmp(c, "%%BeginSetup", (size_t)12))
				setup=TRUE;
			else if (!strncmp(c, "%%EndSetup", (size_t)10))
				setup=FALSE;
			else if (!strncmp(c, "%%EndComments", (size_t)13)) {
				fprintf (fpo, "%s", line);
				if (r != 0) fprintf (fpo, "%d rotate\n", r);
				if (xt != 0 || yt != 0) fprintf (fpo, "%d %d translate\n", xt, yt);
				xt = yt = r = 0;
				continue;
			}
			fprintf (fpo, "%s", line);
		}

		fclose (fpo);
		fclose (fp);

		/* Build the converting ghostscript command and execute it */

		if (Ctrl->T.device != GS_DEV_EPS) {
			char tag[16];
			strcpy (tag, &ext[Ctrl->T.device][1]);
			GMT_str_toupper (tag);
			if (Ctrl->V.active) fprintf (stderr, " Convert to %s...", tag);
			out_file = (char *) GMT_memory (VNULL, (size_t)BUFSIZ, sizeof(char), GMT_program);
			if (Ctrl->D.active) sprintf (out_file, "%s%c", Ctrl->D.dir, DIR_DELIM);		/* Use specified output directory */
			strncat (out_file, &ps_file[pos_file], (size_t)(pos_ext - pos_file));
			strcat (out_file, ext[Ctrl->T.device]);
			w = (int)irint (w / 72.0 * Ctrl->E.dpi);
			h =  (int)irint (h / 72.0 * Ctrl->E.dpi);
			sprintf (cmd, "%s%s %s -sDEVICE=%s -g%dx%d -r%d -sOutputFile=%s -f%s", at_sign, Ctrl->G.file, gs_params, device[Ctrl->T.device],
				w, h, Ctrl->E.dpi, out_file, tmp_file);
			system (cmd);		/* Execute the GhostScript command */
			if (Ctrl->S.active) fprintf (stdout, "%s\n", cmd);
			GMT_free ((void *)out_file);
		}
		if (Ctrl->V.active) fprintf (stderr, " Done.\n");

		if (!Ctrl->T.eps) remove (tmp_file);
		if (no_U_file) remove (no_U_file);
		if (clean_PS_file) remove (clean_PS_file);
		GMT_free ((void *)tmp_file);
		GMT_free ((void *)clean_PS_file);
		GMT_free ((void *)no_U_file);
	}

	Free_Ps2raster_Ctrl (Ctrl);	/* Deallocate control structure */

	GMT_free ((void *)ps_names);

	exit (EXIT_SUCCESS);
}

void *New_Ps2raster_Ctrl () {	/* Allocate and initialize a new control structure */
	struct PS2RASTER_CTRL *C;

	C = (struct PS2RASTER_CTRL *) GMT_memory (VNULL, (size_t)1, sizeof (struct PS2RASTER_CTRL), "New_Ps2raster_Ctrl");

	/* Initialize values whose defaults are not 0/FALSE/NULL */
#ifdef WIN32
	C->G.file = strdup ("gswin32c");
#else
	C->G.file = strdup ("gs");
#endif
	C->D.dir = strdup (".");

	return ((void *)C);
}

void Free_Ps2raster_Ctrl (struct PS2RASTER_CTRL *C) {	/* Deallocate control structure */
	if (C->D.dir) free ((void *)C->D.dir);
	if (C->G.file) free ((void *)C->G.file);
	if (C->L.file) free ((void *)C->L.file);
	GMT_free ((void *)C);
}
