------------------------------------------------------------------------------
--  Ada95 Interface to Oracle RDBMS                                         --
--  Copyright (C) 2000-2003 Dmitriy Anisimkov.                              --
--  License agreement and authors contact information are in file oci.ads   --
------------------------------------------------------------------------------

--  $Id: oci-thick-notifications.adb,v 1.3 2003/07/25 10:01:20 vagul Exp $

with Ada.Exceptions;
with Ada.Text_IO;

with OCI.Lib;
with OCI.Thread;
with OCI.Environments;
with Interfaces.C;
with System.Address_To_Access_Conversions;
with System.Address_Image;

package body OCI.Thick.Notifications is

   use type System.Address;
   use OCI.Lib;

   To_UB4 : array (Namespace_Enum) of UB4 :=
      (AQ        => OCI_SUBSCR_NAMESPACE_AQ,
       Anonymous => OCI_SUBSCR_NAMESPACE_Anonymous);

   function OCISubscriptionNotify
      (pCtx        : in DVoid;
       pSubscrHp   : in OCISubscription;
       pPayload    : in DVoid;
       iPayloadLen : in Ub4;
       pDescriptor : in DVoid;
       iMode       : in Ub4)
       return      SWord;

   pragma Convention (C, OCISubscriptionNotify);

   ------------
   -- Create --
   ------------

   procedure Create
     (Item      : in out Subscription;
      Name      : in     String;
      Namespace : in     Namespace_Enum) is
   begin
      if Item.Handle /= OCISubscription (Empty_Handle) then
         -- Avoid memory and handles leak.

         raise Already_Registered;
      end if;

      Item.Handle := OCISubscription
                       (Alloc_Handle (Thread.Environment,
                                      OCI_HTYPE_SUBSCRIPTION));

      Set_Attr
        (OCIHandle (Item.Handle),
         OCI_HTYPE_SUBSCRIPTION,
         Name,
         OCI_ATTR_SUBSCR_NAME);

      Set_Attr
        (OCIHandle (Item.Handle),
         OCI_HTYPE_SUBSCRIPTION,
         To_UB4 (Namespace)'Address,
         OCI_ATTR_SUBSCR_NAMESPACE);
   end Create;

   -------------
   -- Disable --
   -------------

   procedure Disable (Item : in Subscription) is
      RC : SWord := OCISubscriptionDisable
                      (Item.Handle, Thread.Error, OCI_DEFAULT);
   begin
      Check_Error (RC);
   end Disable;

   ------------
   -- Enable --
   ------------

   procedure Enable (Item : in Subscription)is
      RC : SWord := OCISubscriptionEnable
                      (Item.Handle, Thread.Error, OCI_DEFAULT);
   begin
      Check_Error (RC);
   end Enable;

   --------------
   -- Finalize --
   --------------

   procedure Finalize (Object : in out Subscription) is
   begin
      if Object.Handle /= OCISubscription (Empty_Handle) then
         Free (H => OCIHandle (Object.Handle),
               HType => OCI_HTYPE_SUBSCRIPTION);
      end if;
   end Finalize;

   ------------
   -- Notify --
   ------------

   procedure Notify
     (Item   : in out Subscription;
      Buffer : in     Buffer_Type) is
   begin
      null;
   end Notify;

   ---------------------------
   -- OCISubscriptionNotify --
   ---------------------------

   function OCISubscriptionNotify
      (pCtx        : in DVoid;
       pSubscrHp   : in OCISubscription;
       pPayload    : in DVoid;
       iPayloadLen : in Ub4;
       pDescriptor : in DVoid;
       iMode       : in Ub4)
       return      SWord
   is
      package Converter is new
         System.Address_To_Access_Conversions (Subscription'Class);

      use Converter;
      use Ada.Text_IO;
      Subscr : Object_Pointer := To_Pointer (pCtx);
      --  Queue_Name : String := Get_Attr (pDescriptor,
      --     OCI_DTYPE_AQNFY_DESCRIPTOR, OCI_ATTR_QUEUE_NAME);
      --  CONSUMER_Name : String := Get_Attr (pDescriptor,
      --     OCI_DTYPE_AQNFY_DESCRIPTOR, OCI_ATTR_CONSUMER_NAME);
      Buffer : Buffer_Type := (Ptr => pPayload, Length => Natural (iPayloadLen));
   begin
      Notify (Subscr.all, Buffer);
      return OCI_CONTINUE;
   exception
      when E : others =>
         Put_Line (Current_Error, Ada.Exceptions.Exception_Information (E));
      return OCI_ERROR;
   end OCISubscriptionNotify;

   ----------
   -- Post --
   ----------

   procedure Post
     (Item    : in out Subscription;
      Connect : in     Connection;
      Data    : in     String)
   is
      RC : SWord;
      Buffer : C.Char_Array := C.To_C (Data);
   begin
      RC := OCIAttrSet
        (Trgthndlp  => OCIHandle (Item.Handle),
         Trghndltyp => OCI_HTYPE_SUBSCRIPTION,
         Attributep => Buffer,
         Size       => Data'Length,
         Attrtype   => OCI_ATTR_SUBSCR_PAYLOAD,
         Errhp      => Thread.Error);

      Check_Error (RC);

      RC := OCISubscriptionPost
        (svchp     => OCISvcCtx (Handle (Connect)),
         subscrhpp => Item.Handle'Access,
         count     => 1,
         errhp     => Thread.Error,
         mode      => OCI_DEFAULT);

      Check_Error (RC);

      Item.Connect := Connect;

      RC := OCIAttrSet
        (Trgthndlp  => OCIHandle (Item.Handle),
         Trghndltyp => OCI_HTYPE_SUBSCRIPTION,
         Attributep => System.Null_Address,
         Size       => 0,
         Attrtype   => OCI_ATTR_SUBSCR_PAYLOAD,
         Errhp      => Thread.Error);

      Check_Error (RC);
   end Post;

   --------------
   -- Register --
   --------------

   procedure Register
     (Item    : in out Subscription;
      Connect : in     Connection)
   is
      RC : SWord;
      Connect_Handle : OCIHandle := Handle (Connect);
   begin
      Set_Attr (OCIHandle (Item.Handle),
         OCI_HTYPE_SUBSCRIPTION,
         OCISubscriptionNotify'Address,
         OCI_ATTR_SUBSCR_CALLBACK);

      Set_Attr (OCIHandle (Item.Handle),
         OCI_HTYPE_SUBSCRIPTION,
         Item'Address,
         OCI_ATTR_SUBSCR_CTX);

      RC := OCISubscriptionRegister
            (svchp     => OCISvcCtx (Connect_Handle),
             subscrhpp => Item.Handle'Access,
             count     => 1,
             errhp     => Thread.Error,
             mode      => OCI_DEFAULT);

      Check_Error (RC);
      Item.Connect := Connect;
   end Register;

   ---------------
   -- To_String --
   ---------------

   function To_String (Buffer : in Buffer_Type) return String is
      use Interfaces.C;
      type Data_Type is new Char_Array (0 .. size_t (Buffer.Length) - 1);
      package Converter is new System.Address_To_Access_Conversions (Data_Type);
      use Converter;
      Ptr : Object_Pointer := To_Pointer (Buffer.Ptr);
   begin
      if Ptr = null then
         return "";
      else
         return To_Ada (Ptr.all, Trim_Nul => False);
      end if;
   end To_String;

   ----------------
   -- Unregister --
   ----------------

   procedure Unregister (Item : in out Subscription) is
      RC : SWord := OCISubscriptionUnRegister
                      (svchp    => OCISvcCtx (Handle (Item.Connect)),
                       subscrhp => Item.Handle,
                       errhp    => Thread.Error,
                       mode     => OCI_DEFAULT);
   begin
      Check_Error (RC);
   end Unregister;

begin
   OCI.Environments.Set_Create_Mode_Flag (OCI_EVENTS);
end OCI.Thick.Notifications;

