------------------------------------------------------------------------------
--  Ada95 Interface to Oracle RDBMS                                         --
--  Copyright (C) 2000-2006 Dmitriy Anisimkov.                              --
--  License agreement and authors contact information are in file oci.ads   --
------------------------------------------------------------------------------

--  $Id: oci-thread.adb,v 1.13 2007/02/08 08:53:35 vagul Exp $

with Interfaces.C;

package body OCI.Thread is

   Global_Env : Thread_Environment := Init_Environment;

   procedure Create (Error : in out Error_Type);

   procedure Ignore_Error (Code : in SWord);
   pragma Inline (Ignore_Error);

   ------------
   -- Create --
   ------------

   procedure Create (Error : in out Error_Type) is
      HErr : aliased OCIHandle := Empty_Handle;
      Error_Handle_Creation_Error         : exception;
      Environment_Should_Be_Created_First : exception;

      use type Interfaces.C.int;

   begin
      if Error.Environment.Handle = OCIEnv (Empty_Handle) then
         raise Environment_Should_Be_Created_First;
      end if;

      if Error.Handle /= OCIError (Empty_Handle) then
         Ignore_Error
           (OCIHandleFree (OCIHandle (Error.Handle), OCI_HTYPE_ERROR));
      end if;

      if OCIHandleAlloc
           (Parenth => OCIHandle (Error.Environment.Handle),
            Hndlpp  => HErr'Access,
            Htype   => OCI_HTYPE_ERROR) /= OCI_SUCCESS
      then
         raise Error_Handle_Creation_Error;
      end if;

      Error.Handle := OCIError (HErr);
   end Create;

   -----------------
   -- Environment --
   -----------------

   function Environment return OCIEnv is
      Env : constant OCIEnv := Attributes.Reference.Environment.Handle;
   begin
      if Env = OCIEnv (Empty_Handle) then
         return Synch.Environment.Handle;
      else
         return Env;
      end if;
   end Environment;

   -----------
   -- Error --
   -----------

   function Error return OCIError is
      Ptr   : constant Attributes.Attribute_Handle := Attributes.Reference;
      Dummy : OCIEnv;
      pragma Warnings (Off, Dummy);
   begin
      if Ptr.Handle = OCIError (Empty_Handle) then
         Dummy := Environment;
         Create (Ptr.all);
      end if;

      return Ptr.Handle;
   end Error;

   --------------
   -- Finalize --
   --------------

   procedure Finalize (Object : in out Error_Type) is
   begin
      if Object.Handle /= OCIError (Empty_Handle) then
         Ignore_Error
           (OCIHandleFree (OCIHandle (Object.Handle), OCI_HTYPE_ERROR));
      end if;
   end Finalize;

   ------------------
   -- Ignore_Error --
   ------------------

   procedure Ignore_Error (Code : in SWord) is
      pragma Unreferenced (Code);
   begin
      null;
   end Ignore_Error;

   -----------
   -- Synch --
   -----------

   protected body Synch is

      -----------------
      -- Environment --
      -----------------

      function Environment return Thread_Environment is
         Envr : OCIEnv renames
            Thread.Attributes.Reference.Environment.Handle;
      begin
         if Envr = OCIEnv (Empty_Handle) then
            if Global_Env.Handle = OCIEnv (Empty_Handle) then
               Global_Env := Create;
            end if;

            Attributes.Reference.Environment := Global_Env;
         end if;

         return Attributes.Reference.Environment;
      end Environment;

      --------------------------------
      -- Task_Dedicated_Environment --
      --------------------------------

      function Task_Dedicated_Environment return Boolean is
      begin
         if Thread.Attributes.Reference.Environment.Handle
            /= OCIEnv (Empty_Handle)
         then
            return False;
         else
            Attributes.Reference.Environment := Create;
            return True;
         end if;
      end Task_Dedicated_Environment;

   end Synch;

   ---------------
   -- Set_Error --
   ---------------

   procedure Set_Error (Env : OCIEnv; Err : OCIError) is
   begin
      Attributes.Reference.Environment
         := (RF.Controlled_Reference with Handle => Env);
      Attributes.Reference.Handle := Err;
   end Set_Error;

end OCI.Thread;
