with Ada.Text_IO;      use Ada.Text_IO;
with Projects.Registry;    use Projects, Projects.Registry;
with Projects.Editor;      use Projects.Editor;
with Src_Info;             use Src_Info;
with Src_Info.ALI;         use Src_Info.ALI;
with Src_Info.CPP;         use Src_Info.CPP;
with Language_Handlers;       use Language_Handlers;
with Language_Handlers.Glide; use Language_Handlers.Glide;
with Traces;               use Traces;
with Prj;
with VFS;                  use VFS;

package body Test_Utils is

   function Predefined_Source_Path return String;
   function Predefined_Object_Path return String;
   --  Return the predefined source paths for the current compiler

   ----------------------------
   -- Predefined_Source_Path --
   ----------------------------

   function Predefined_Source_Path return String is
   begin
      return "/usr/local/gnat/lib/gcc-lib/i686-pc-linux-gnu/2.8.1/adainclude";
   end Predefined_Source_Path;

   ----------------------------
   -- Predefined_Object_Path --
   ----------------------------

   function Predefined_Object_Path return String is
   begin
      return "/usr/local/gnat/lib/gcc-lib/i686-pc-linux-gnu/2.8.1/adalib";
   end Predefined_Object_Path;

   ------------------
   -- Load_Project --
   ------------------

   procedure Load_Project
     (Name : String;
      Handler : access Language_Handlers.Language_Handler_Record'Class;
      Registry : in out Project_Registry'Class;
      Project : out Project_Type)
   is
      procedure Report_Error (S : String);

      procedure Report_Error (S : String) is
      begin
         Put_Line ("**Error**: " & S);
      end Report_Error;

      New_Project_Loaded : Boolean;
   begin
      Load (Registry, Name, Report_Error'Unrestricted_Access,
            New_Project_Loaded);
      Project := Get_Root_Project (Registry);

      Set_Predefined_Source_Path (Registry, Predefined_Source_Path);
      Set_Predefined_Object_Path (Registry, Predefined_Object_Path);

      Recompute_View (Registry, Report_Error'Unrestricted_Access);

      Reset
        (CPP_LI_Handler (Get_LI_Handler_By_Name
                         (Glide_Language_Handler (Handler), "c/c++")),
         Project);
   end Load_Project;

   ------------------
   -- Load_LI_File --
   ------------------

   procedure Load_LI_File
     (Source_Info_List : in out Src_Info.LI_File_List;
      Handler          : Language_Handlers.Language_Handler;
      Registry         : in out Project_Registry'Class;
      Source_Filename  : VFS.Virtual_File;
      LI               : out Src_Info.LI_File_Ptr)
   is
      File_Project : Project_Type;
   begin
      LI := No_LI_File;
      --  This code is extracted from Locate_From_Source_And_Complete

      File_Project := Get_Project_From_File
        (Registry, Source_Filename);

      --  Create and parse the LI file
      Create_Or_Complete_LI
        (Handler                => Get_LI_Handler_From_File
           (Glide_Language_Handler (Handler), Source_Filename),
         File                   => LI,
         Source_Filename        => Source_Filename,
         List                   => Source_Info_List,
         Project                => File_Project);

      if LI = No_LI_File then
         Put_Line ("*** Null LI_File_Ptr returned.");
         return;
      end if;
   end Load_LI_File;

   -------------------------
   -- Create_Lang_Handler --
   -------------------------

   function Create_Lang_Handler
     (Registry : access Project_Registry'Class)
      return Language_Handlers.Language_Handler
   is
      Handler : Glide_Language_Handler;
   begin
      Gtk_New (Handler);

      Register_LI_Handler
        (Handler, "Ada", new Src_Info.ALI.ALI_Handler_Record);
      Register_LI_Handler
        (Handler, "c/c++", new Src_Info.CPP.CPP_LI_Handler_Record);

      declare
         Msg : constant String := Set_Executables
           (CPP_LI_Handler (Get_LI_Handler_By_Name (Handler, "c/c++")));
      begin
         if Msg /= "" then
            Put_Line ("Error: " & Msg);
         end if;
      end;

      Register_Language (Handler, "ada", null);
      Set_Language_Handler
        (Handler, "ada", Get_LI_Handler_By_Name (Handler, "Ada"));
      Prj.Register_Default_Naming_Scheme
        (Language            => Get_String ("ada"),
         Default_Spec_Suffix => Get_String (".ads"),
         Default_Body_Suffix => Get_String (".adb"));

      Register_Language (Handler, "c",   null);
      Set_Language_Handler
        (Handler, "c", Get_LI_Handler_By_Name (Handler, "c/c++"));
      Prj.Register_Default_Naming_Scheme
        (Language            => Get_String ("c"),
         Default_Spec_Suffix => Get_String (".h"),
         Default_Body_Suffix => Get_String (".c"));

      Register_Language (Handler, "c++", null);
      Set_Language_Handler
        (Handler, "c++", Get_LI_Handler_By_Name (Handler, "c/c++"));
      Prj.Register_Default_Naming_Scheme
        (Language            => Get_String ("c++"),
         Default_Spec_Suffix => Get_String (".h"),
         Default_Body_Suffix => Get_String (".cc"));

      Set_Registry (Handler, Registry);

      return Language_Handler (Handler);
   end Create_Lang_Handler;

begin
   Parse_Config_File (".gnatdebug");
   Projects.Registry.Initialize;
end Test_Utils;
