/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
 ///////////////////////////////////////////////////////////////////////////////
/*
BinauralBeat.cpp: implementation of the CBinauralBeat class.
    Author: Bret Logan
    Date: Nov. 3, 2005
FUNCTION: 
    CBinauralBeat outputs stereo sound data iin 44.1khz PCM (.WAV-style) format, coresponding to a Left-channel
	sine wave mixed with pink noise, and a Right-channel sine wave mixed with pink noise. Why? Because we love you.
HOW TO USE:
    Create a CBinauralBeat object. That'll run InitBinauralBeat(), which in turn will
    check to see if you have have a gnaural_schedule.txt (the class' default name)
    file. If you do, it will be used, if you don't, a default internal one will used and
    also written to a text file calle gnauralGNAURAL_WIN32uture editing.
	If you just want to create a .WAV file and be done with it, call WriteWAVFile(char *szFilename). 
	WriteWAVFileToSTDOUT does basically the same thing, but to stdout -- so you can pipe the 
	output through another program (like LAME, for instance). Example: 
     	 gnaural | lame - MyMeditationFile.mp3
	But if you want to send the output directly to a sound server, pass a 
	buffer to UserSoundProc() --  a simple int array is OK, ready for Gnaural's 16-bit stereo 
	format -- left-channel short int/right-channel short int.
	UserSoundProc() acts like a "friend on the inside" that can contact  
	MainLoop (void *pSoundBuffer, long bufferLen). MainLoop does the filling,
	so you can in turn feed whatever sound server you use.
*/  
///////////////////////////////////////////////////////////////////////////////
  
#include "BinauralBeat.h"
#include <ctype.h>
  
//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////
  
/*
//#######################################
//basefreq - the base frequency used to create the beats
//tmpbuff - a set of triplets defining a schedule 
//loop - how many times to do schedule (0 means forever!)
CBinauralBeat::CBinauralBeat(float basefreq, char *tmpbuff, int _loop) {
 
 if (tmpbuff) {
  SchedBuff=tmpbuff;
  //System.out.println("BrainJav got the schedule");
  strcpy(info,"BrainJav got the schedule");
  msgtype|=INFO;
 }
 else  {//System.out.println("BrainJav didn't get the schedule");
  SchedBuff=DefaultSchedBuff;
  strcpy(error,"No schedule, using default");
  msgtype|=BB_ERROR;
 }
 
 FreqBase=basefreq;
 if (FreqBase<5) FreqBase=5;
 loops=_loop;
 InitBinauralBeat();
}
*/ 
  
				//======================================
  CBinauralBeat::CBinauralBeat (char *sched_filename)
{
  SchedBuff = DefaultSchedBuff;//just so it has something to run if no file is found; SchedFilenameToSchedule() will fill SchedBuff otherwise  
//  SchedFilename = DefaultSchedFilename;//default name for user's schedule file:
  SchedFilename = sched_filename;//sched_filename defaults to DefaultSchedFilename in constructor if nothing is provided
  InitBinauralBeat ();

}




				//======================================
  void
CBinauralBeat::InitBinauralBeat ()
{
  
SeedRand (3676, 2676862);	//anything but 0 on either or twins seems OK
  me = this;
  
Schedule = NULL;
  
SetDefaultValues ();
  InfoFlag = 0;

  //see if there is a file, if not use default schedule and write a default file:
  if (!SchedFilenameToSchedule ())
    {
      SchedBuffToSchedule (SchedBuff);
    }
}				//end InitBinauralBeat()


//======================================
  void
CBinauralBeat::SetDefaultValues ()
{

  FreqBase = 140.f;

  Volume_Tone = 13926;		// 0 to 12288 to 16384 are feasible
  Volume_Noiz = .2f;

//these next two are what 50% vols would be:
// Volume_Tone=8192;// 0 to 12288 to 16384 are feasible
// Volume_Noiz=.5f;
  StereoNoiz = true;

  FileFlag = false;

  ManualFreqOffsetControl = false;

  WriteStop = false;

  ManualFreqLOffset = ManualFreqROffset = 0.f;

  loops = loopcount = 1;

  TotalSampleCount = TotalSampleCountLooped = 0;

}


				//======================================
  CBinauralBeat::~CBinauralBeat () 
{
  
if (Schedule != NULL)
    delete[]CBinauralBeat::Schedule;

}



//======================================
//*note*: in porting this from Java, I now need to delete[] some stuff 
//made in this function
void CBinauralBeat::SchedBuffToSchedule (char *str)
{
    //the philosophy here: take a string with commas separating the
    //floats, hunt for commas. When one is found, start a new string
    //beginning with next character. Take every following character up 
    //to (but not including) the next comma. Then store that float, dispose
    //of the tempstring, and hunt some more until length of string is 
    //covered.
  char tmpstr[256];
float *tmpfloat = NULL;
char substr[256];
    //first count how many floats are in the string:
  int floatcount = 1;
unsigned long i;
for (i = 0; i < strlen (str); i++)
    {
if (str[i] == ',')
	{
floatcount++;
}			// end if comma
    }				//end for i
    //do it all again, now that I know how many floats:
    tmpfloat = new float[floatcount];
floatcount = 0;
int j = 0;
for (i = 0; i < strlen (str); i++)
    {
	//first extract a number to a string:
	if ((str[i] >= '0' && str[i] <= '9') || str[i] == '.'   || str[i] == '-')   tmpstr[j++] = str[i];

	//if I found a comma, end the string:
	else if (str[i] == ',')
	{
 tmpstr[j] = '\0';
 strcpy (substr, tmpstr);
 tmpfloat[floatcount] = (float) atof (substr);
 j = 0;
 floatcount++;
}			// end if comma
    }				//end for i
  if (j != 0)
    {				//there should be one more float to get:
      tmpstr[j] = '\0';
 strcpy (substr, tmpstr);
 tmpfloat[floatcount] = (float) atof (substr);
 ++floatcount;
}				// end if j!=0
 
ScheduleEntriesCount = floatcount / 3;

if (Schedule != NULL)    delete[]Schedule;
  
Schedule = new CItinerary[ScheduleEntriesCount];
  
int listcount = -1;
int buffcount = 0;
ScheduleTime = 0;

    //now load key variables in to Schedule vars:
    while ((++listcount) < ScheduleEntriesCount)
    {
Schedule[listcount].FreqLStart = (float) tmpfloat[buffcount++];
Schedule[listcount].FreqRStart = (float) tmpfloat[buffcount++];
Schedule[listcount].Duration = (float) tmpfloat[buffcount++];
ScheduleTime += Schedule[listcount].Duration;
Schedule[listcount].AbsoluteEndTime_100 =	(unsigned int) ScheduleTime *100;
} 

//now figure Schedule's FreqEnds and Spreads:
listcount = -1;
int previnlist, nextinlist;
while ((++listcount) < ScheduleEntriesCount)
{
if ((nextinlist = listcount + 1) >= ScheduleEntriesCount) 	nextinlist = 0;
Schedule[listcount].FreqLEnd = Schedule[nextinlist].FreqLStart;
Schedule[listcount].FreqREnd = Schedule[nextinlist].FreqRStart;
Schedule[listcount].FreqLSpread = Schedule[listcount].FreqLEnd - Schedule[listcount].FreqLStart;
Schedule[listcount].FreqRSpread = Schedule[listcount].FreqREnd - Schedule[listcount].FreqRStart;
if ((previnlist = listcount - 1) < 0) 	Schedule[listcount].PrevAbsoluteEndTime_100 = 0;
else 	Schedule[listcount].PrevAbsoluteEndTime_100 =  Schedule[previnlist].AbsoluteEndTime_100;
}
  
//code to write a schedule to a file:
 if (!FileFlag)
 {
   WriteScheduleToFile ();
 }				//end file writing routine
delete[]tmpfloat;
}				//end of extractfloats()




//======================================
//################################################
//START PSEUDORANDOM NUMBER GENERATOR SECTION
//Make a choice. 
/*
//=============================================
          The McGill Super-Duper Random Number Generator
             G. Marsaglia, K. Ananthanarayana, N. Paul
  
    Incorporating the Ziggurat method of sampling from decreasing
              or symmetric unimodal density functions.
                      G. Marsaglia, W.W. Tsang
  
                 Rewritten into C by E. Schneider 
                 [Date last modified: 05-Jul-1997]
//=============================================
*/
#define ML_MULT 69069L
//Need to also define these in .h file: 
//unsigned long mcgn, srgn;
//IMPORTANT NOTE:when using a fixed i2, for some reason Seed pairs 
//for i1 like this: 
// even 
// even+1 
//produce idential sequences when r2 returned (r1 >> 12).
//Practically, this means that 2 and 3 
//produce one landscape; likewise 6 and 7, 100 and 101, etc.
//This is why i do the dopey "add 1" to i2
//ALSO, JUST DON'T USE 0 FOR i1 or i2. PLEASE
  void
CBinauralBeat::SeedRand (unsigned int i1, unsigned int i2) 
//void rstart (long i1, long i2)
{
  
if (i2 == (unsigned int) -1)
    i2 = i1 + 1;		//yech
  mcgn = (unsigned long) ((i1 == 0L) ? 0L : i1 | 1L);
  
srgn = (unsigned long) ((i2 == 0L) ? 0L : (i2 & 0x7FFL) | 1L);

} 


//======================================
//returns int from -2^31 to +2^31; "PM" means "PlusMinus"
  int
CBinauralBeat::rand () 
{
  unsigned long r0, r1;
  

r0 = (srgn >> 15);
  
r1 = srgn ^ r0;
  
r0 = (r1 << 17);
  
srgn = r0 ^ r1;
  
mcgn = ML_MULT * mcgn;
  
r1 = mcgn ^ srgn;
  
return r1;

}


/*
//THERE WAS NOTHING SONICALLY WRONG WITH THE 
//FOLLOWING HOMEMADE PRNG FUNCTIONS
//And they don't have the problem with low or close seeds causing
//paterned output (unlike the one above). But they supposedly repeat sooner
//than the one above (haven't tested, though). Not sure if that makes them
//worse. I just like the idea of not repeating often, because 
//it is possible that the user can, with enough exposure, start to hear the
//repetition on very short cycle loops.
//
//returns a signed integer noise value:
int CBinauralBeat::rand() {
static int Seed=9182736;//this is just here for uncommenting-conveinence; real seedable one is in BinauralBeat.h
 //to make any of the following noise generators positive-
 //only returns, simply add &0x7FFFFFFF to the end to negate the 
 //sign.
 //all of these make perfectly acceptable audio noise (my criteria: "no discernable patterns"):
 //return (Seed=(Seed<<3)-Seed+11);
 //return Seed=(201*Seed+11);
 //return ++Seed*=3;
 //new formula, related to MTwister:
 ++Seed ^= (Seed >> 11);
 Seed ^= (Seed <<  7) & 0x9D2C5680U;
 return Seed ^= (Seed << 15) & 0xEFC60000U;

 //curiosities- these are amazing sequences, but
 //not very random sounding!:
 //pulsar:
 //return (Seed^=++Seed<<3);
 //machine shop:
 //return (Seed^=++Seed<<1);
}
*/
//END PSEUDORANDOM NUMBER GENERATOR SECTION
//################################################


				//======================================
  CBinauralBeat * CBinauralBeat::me;



				//======================================
  void
CBinauralBeat::UserSoundProc (void *pSoundBuffer, long bufferLenInBytes) 
{
  //Saves me from indirecting every goddamn variable in the processing!
  //NOTE: Important to remember that bufferLenInBytes is literally the size of 
  //the buffer you give UserSoundProc in bytes -- a common error is to send it
  //whatever the size of the buffer is in it's native units (int, short, etc.)
  me->MainLoop (pSoundBuffer, bufferLenInBytes);

} 

/*
//======================================
//////////////////////////////////////////////////
//UserSoundProc(void *pSoundBuffer,long bufferLen)
//use the following to fill your sound buffers. 
//it will automatically keep
//time, by using sample counts instead of a 
//real-time clock. The
//basis for time is 0.01 seconds (1/100th of 
//a second frequency updates)
void CBinauralBeat::MainLoop (void *pSoundBuffer, long bufferLen) 
{
  //need these statics as bookmarks for where I am in the sound in each call:
 static float sinPosL = 0.f;
 static float sinPosR = 0.f;
 static int noizval1 = 1;
 static int noizval2 = 1;
 static int samplerezvar = 1;	//critical: initialize as 1
 static int oldj = -1; //I think I have a bug concerning this variable
 
 float factor;
 float noiz1;
 float noiz2;

 // Convert params, assuming we create a 16bits, mono waveform.
 signed short *pSample = (signed short *) pSoundBuffer;

 //Generally speaking, this is what I am doing:
 //long    nbSample = bufferLen / (sizeof(int));
 //But since I always know bufferlen in in chars, I just divide by four:
 long nbSample = bufferLen >> 2;

//-------------------------------------------
// START Fill sound buffer
 //for every sample in pSoundBuffer I need to fill, do this:
 for (int i = 0; i < nbSample; i++)
 {
  //START 1/100th sec. updates
  //Philosophy: I update the frequency to match user's schedule directives 
  //100 times a second, literally by doing this routine every 441 samples (i.e.,
  //that's once a second when you take 44.1k samples per second).
  //I also increment TotalSampleCount every 100th of a sec. in order to
  //use it as an index as to where we are in the schedule.
  if ((--samplerezvar) == 0)
 	{
   samplerezvar = INTEGRATION_REZ;	//441
   if (!ManualFreqOffsetControl)
   {
    TotalSampleCount++;

    //START which entry loop
   	//Now figure out which ScheduleCount I'm at:
	   int j;
	   for (j = 0; j < ScheduleEntriesCount; j++)
	   {
	    //asking "is total samples less than sched entry's endtime?"
	    if (TotalSampleCount < Schedule[j].AbsoluteEndTime_100)
	     {
	      //answer: it is, therefore I know the ScheduleCount:
 	     ScheduleCount = j;

       //if this is not the old entry, let the world know one time:       
       if (j != oldj)
       {
         InfoFlag |= BB_NEWENTRY;
         oldj = j;
 	      }

        //by dividing exact point in period by total period time, I have a factor
        //by which to calculate proper frequency:
        factor =	(TotalSampleCount - Schedule[ScheduleCount].PrevAbsoluteEndTime_100) / (Schedule[ScheduleCount].Duration * 100.f);
		      
        //Left Freq is equal to frequency spread from Left Start to Left End (next
        //schedule's Left Start) multiplied by above factor. Then add FreqBase.
        FreqL =	(Schedule[ScheduleCount].FreqLSpread * factor) +	FreqBase + Schedule[ScheduleCount].FreqLStart;
        FreqR = (Schedule[ScheduleCount].FreqRSpread * factor) + FreqBase + Schedule[ScheduleCount].FreqRStart;
        break;	//exit the "figuring which entry" loop
 	     }
      }
//END which entry loop
	      
		//following tests to see if Schedule was done:
		if (j >= ScheduleEntriesCount)
		{
   InfoFlag |= BB_NEWLOOP;
   TotalSampleCountLooped += TotalSampleCount;
   ScheduleCount = TotalSampleCount = 0;	//repeat is default for now
	  if (--loopcount == 0)
   {		//end this sucker
     InfoFlag |= BB_COMPLETED;
    }
   }
 } //end !ManualFreqOffsetControl
 else
 {//Start ManualFreqOffsetControl
   //it is manual control now:
	  FreqL = ManualFreqLOffset + FreqBase;
   FreqR = ManualFreqROffset + FreqBase;
  }
}
//END 1/100th sec. updates
      
 //Now use the info taken from the 1/100th sec. update to actually make the sound:
 //first pink some noise:
 noizval1 = ((noizval1 * 31) + (rand () >> 15)) >> 5;
 noiz1 = noizval1 * Volume_Noiz;
 if (!StereoNoiz) noiz2 = noiz1; 
 else
 {
  noizval2 = ((noizval2 * 31) + (rand () >> 15)) >> 5;
  noiz2 = noizval2 * Volume_Noiz;
 }

	//now do sine waves:
//	sinPosL += (TWO_PI * FreqL * SAMPLE_FACTOR);	//SAMPLE_FACTOR=1 over sample rate
// sinPosR += (TWO_PI * FreqR * SAMPLE_FACTOR);
	sinPosL += (FreqL * BB_SAMPLE_FACTOR);	//SAMPLE_FACTOR=1 over sample rate
 sinPosR += (FreqR * BB_SAMPLE_FACTOR);
 if (sinPosL >= BB_TWO_PI)	sinPosL -= BB_TWO_PI;
 if (sinPosR >= BB_TWO_PI)	sinPosR -= BB_TWO_PI;
 *pSample++ = (signed short) (Volume_Tone * sin (sinPosL) + noiz1);
 *pSample++ = (signed short) (Volume_Tone * sin (sinPosR) + noiz2);
}
// END Fill sound buffer
//-------------------------------------------
} 
*/


//======================================
//////////////////////////////////////////////////
//UserSoundProc(void *pSoundBuffer,long bufferLen)
//use the following to fill your sound buffers. 
//it will automatically keep
//time, by using sample counts instead of a 
//real-time clock. The
//basis for time is 0.01 seconds (1/100th of 
//a second frequency updates)
void CBinauralBeat::MainLoop (void *pSoundBuffer, long bufferLen) 
{
  //need these statics as bookmarks for where I am in the sound in each call:
 static BB_PRECISION_TYPE sinPosL = 0.f;
 static BB_PRECISION_TYPE sinPosR = 0.f;
 static int noizval1 = 1;
 static int noizval2 = 1;
 static int samplerezvar = 1;	//critical: initialize as 1
 static int oldj = -1; //I think I have a bug concerning this variable
 
 BB_PRECISION_TYPE factor;
 float noiz1;
 float noiz2;
 //20060331: next two were added to remove extra multiplications from speed loop:
 static BB_PRECISION_TYPE factoredFreqL=0;
 static BB_PRECISION_TYPE factoredFreqR=0;

 // Convert params, assuming we create a 16bits, mono waveform.
 signed short *pSample = (signed short *) pSoundBuffer;

 //Generally speaking, this is what I am doing:
 //long    nbSample = bufferLen / (sizeof(int));
 //But since I always know bufferlen in in chars, I just divide by four:
 long nbSample = bufferLen >> 2;

//-------------------------------------------
// START Fill sound buffer
 //for every sample in pSoundBuffer I need to fill, do this:
 for (int i = 0; i < nbSample; i++)
 {
  //START 1/100th sec. updates
  //Philosophy: I update the frequency to match user's schedule directives 
  //100 times a second, literally by doing this routine every 441 samples (i.e.,
  //that's once a second when you take 44.1k samples per second).
  //I also increment TotalSampleCount every 100th of a sec. in order to
  //use it as an index as to where we are in the schedule.
  if ((--samplerezvar) == 0)
 	{
   samplerezvar = BB_INTEGRATION_REZ;	//BB_INTEGRATION_REZ should always equal 441
   if (!ManualFreqOffsetControl)
   {
    TotalSampleCount++;

    //START which entry loop
   	//Now figure out which ScheduleCount I'm at:
	   int j;
	   for (j = 0; j < ScheduleEntriesCount; j++)
	   {
	    //asking "is total samples less than sched entry's endtime?"
	    if (TotalSampleCount < Schedule[j].AbsoluteEndTime_100)
	     {
	      //answer: it is, therefore I know the ScheduleCount:
 	     ScheduleCount = j;

       //if this is not the old entry, let the world know one time:       
       if (j != oldj)
       {
         InfoFlag |= BB_NEWENTRY;
         oldj = j;
 	      }

        //by dividing exact point in period by total period time, I have a factor
        //by which to calculate proper frequency:
        factor =	(TotalSampleCount - Schedule[ScheduleCount].PrevAbsoluteEndTime_100) / (Schedule[ScheduleCount].Duration * 100.f);
		      
        //Left Freq is equal to frequency spread from Left Start to Left End (next
        //schedule's Left Start) multiplied by above factor. Then add FreqBase.
        FreqL =	(Schedule[ScheduleCount].FreqLSpread * factor) +	FreqBase + Schedule[ScheduleCount].FreqLStart;
        factoredFreqL=FreqL * BB_SAMPLE_FACTOR;
        FreqR = (Schedule[ScheduleCount].FreqRSpread * factor) + FreqBase + Schedule[ScheduleCount].FreqRStart;
        factoredFreqR=FreqR * BB_SAMPLE_FACTOR;
        break;	//exit the "figuring which entry" loop
 	     }
      }
//END which entry loop
	      
		//following tests to see if Schedule was done:
		if (j >= ScheduleEntriesCount)
		{
   InfoFlag |= BB_NEWLOOP;
   TotalSampleCountLooped += TotalSampleCount;
   ScheduleCount = TotalSampleCount = 0;	//repeat is default for now
	  if (--loopcount == 0)
   {		//end this sucker
     InfoFlag |= BB_COMPLETED;
    }
   }
 } //end !ManualFreqOffsetControl
 else
 {//Start ManualFreqOffsetControl
   //it is manual control now:
	  FreqL = ManualFreqLOffset + FreqBase;
   factoredFreqL= FreqL * BB_SAMPLE_FACTOR;	//SAMPLE_FACTOR=1 over sample rate
   FreqR = ManualFreqROffset + FreqBase;
   factoredFreqR=FreqR * BB_SAMPLE_FACTOR;
  }
}
//END 1/100th sec. updates
      
 //Now use the info taken from the 1/100th sec. update to actually make the sound:
 //First, pink some noise:
 noizval1 = ((noizval1 * 31) + (rand () >> 15)) >> 5;
 noiz1 = noizval1 * Volume_Noiz;
 if (!StereoNoiz) noiz2 = noiz1; 
 else
 {
  noizval2 = ((noizval2 * 31) + (rand () >> 15)) >> 5;
  noiz2 = noizval2 * Volume_Noiz;
 }

	//Second, do sine waves:
	sinPosL += factoredFreqL;	
 if (sinPosL >= BB_TWO_PI)	sinPosL -= BB_TWO_PI;
 sinPosR += factoredFreqR;
 if (sinPosR >= BB_TWO_PI)	sinPosR -= BB_TWO_PI;
 *pSample++ = (signed short) (Volume_Tone * sin (sinPosL) + noiz1);
 *pSample++ = (signed short) (Volume_Tone * sin (sinPosR) + noiz2);

//PS this should be the equivalent of above:
// *pSample = (signed short) (Volume_Tone * sin (sinPosL) + noiz1); ++pSample;
// *pSample = (signed short) (Volume_Tone * sin (sinPosR) + noiz2); ++pSample;
}
// END Fill sound buffer
//-------------------------------------------
} 









//======================================
//opens whatever is in SchedFilename, dumps it in to SchedBuff, 
//then uses SchedBuffToSchedule to fill Schedule
int CBinauralBeat::SchedFilenameToSchedule () 
{
  char * filename=SchedFilename;//this line simply gives the global to a local var, in order to make this function easy to make independent
FILE * stream;
  
char *str;
  
if ((stream = fopen (filename, "r")) == NULL)
    {				//didn't find a schedule file, so return (could try to create one at this point with WriteScheduleToFile()
      return 0;
    }
  
FileFlag = true;
  
char tmpchar;
  
unsigned int i = 0;
  
while (feof (stream) == 0)
    
    {
      
tmpchar = fgetc (stream);
      
if (tmpchar == '#')
	{
	  
while (feof (stream) == 0 && tmpchar != '\n')
	    
tmpchar = fgetc (stream);
	
}
      


//  if (tmpchar>='0' && tmpchar <='9') i++;
//  else if (tmpchar==',' || tmpchar=='.') i++;
	
if ((tmpchar >= '0' && tmpchar <= '9') || 
tmpchar == ','
	     || 
tmpchar == '.' || 
tmpchar == '-')
	i++;
    

}				//end while
  
str = new char[++i];
  
rewind (stream);
  
i = 0;
  
while (feof (stream) == 0)
    
    {
      
tmpchar = fgetc (stream);
      

	//deal with comments:
	if (tmpchar == '#')
	{			//throw whole line away:
	  while (feof (stream) == 0 && tmpchar != '\n')
	    
tmpchar = fgetc (stream);
	
}
      

	//deal with command strings:
	else if (tmpchar == '[')
	{
	  
ParseCmd (stream);
	
}
      

	//if it is a number, add it to the monster string:
	else if ((tmpchar >= '0' && tmpchar <= '9') || 
tmpchar == ','
		 || 
tmpchar == '.' || 
tmpchar == '-')
	{
	  
str[i] = tmpchar;
	  
i++;
	
}
    
}
  
fclose (stream);
  
    //put end on the string!!
    str[i] = '\0';
  
SchedBuffToSchedule (str);
  
delete[]str;
  
return 1;

}


//======================================
//default name of user's schedule; user can change it by pointing 
//the actual variable used to open files (BinauralBeat::SchedFilename)
//to some other string:
char   CBinauralBeat::DefaultSchedFilename[] = "gnaural_schedule.txt";


//======================================
//this works great with 110Hz basefreq and 85% tone 20% noiz
  char
  CBinauralBeat::DefaultSchedBuff[] = 
"0,0,9,\
6,-6,45,\
4,-4,60,\
3,-3,60,\
2.5,-2.5,120,\
2.15,-2.15,180,\
2,-2,180,\
1.95,-1.95,6,\
3.5,-3.5,6,\
1.95,-1.95,360,\
2.1,-2.1,6,\
3.5,-3.5,6,\
1.95,-1.95,180,\
2,-2,180,\
1.95,-1.95,6,\
3.5,-3.5,6,\
1.95,-1.95,340,\
2.1,-2.1,6,\
3.5,-3.5,6,\
2,-2,180,\
2.1,-2.1,180,\
1.9,-1.9,6,\
3.5,-3.5,6,\
1.95,-1.95,400,\
2.1,-2.1,6,\
3.5,-3.5,6,\
2.1,-2.1,180,\
1.95,-1.95,180,\
2,-2,6,\
3.5,-3.5,6,\
2,-2,300,\
1.9,-1.9,6,\
3.5,-3.5,6,\
1.95,-1.95,180,\
2.05,-2.05,180,\
1.95,-1.95,6,\
3.5,-3.5,6,\
1.95,-1.95,360,\
2.05,-2.05,6,\
3.5,-3.5,6,\
1.95,-1.95,180,\
1.8,-1.8,180,\
2,-2,6,\
3.5,-3.5,6,\
2.15,-2.15,64";




/*
//THIS ONE WAS IN USE FOR 2 YEARS; tried-and-true, but replaced
//by the above since the above should be the same and possibly
//better/
//this works great with 110Hz basefreq and 85% tone 20% noiz
char CBinauralBeat::DefaultSchedBuff[]=
"0,0,9,\
12,0,45,\
8,0,60,\
6,0,60,\
5,0,120,\
4.3,0,180,\
4,0,180,\
3.9,0,6,\
7,0,6,\
3.9,0,360,\
4.2,0,6,\
7,0,6,\
3.9,0,180,\
4.0,0,180,\
3.9,0,6,\
7,0,6,\
3.9,0,340,\
4.2,0,6,\
7,0,6,\
4.0,0,180,\
4.2,0,180,\
3.8,0,6,\
7,0,6,\
3.9,0,400,\
4.2,0,6,\
7,0,6,\
4.2,0,180,\
3.9,0,180,\
4.0,0,6,\
7,0,6,\
4.0,0,300,\
3.8,0,6,\
7,0,6,\
3.9,0,180,\
4.1,0,180,\
3.9,0,6,\
7,0,6,\
3.9,0,360,\
4.1,0,6,\
7,0,6,\
3.9,0,180,\
3.6,0,180,\
4.0,0,6,\
7,0,6,\
4.3,0,64";
*/ 
  




				//======================================
  void
CBinauralBeat::next ()
{

  TotalSampleCount = Schedule[ScheduleCount].AbsoluteEndTime_100;

  if (++ScheduleCount >= this->ScheduleEntriesCount)
    ScheduleCount = 0;

}


//======================================
void CBinauralBeat::previous ()
{
/*
 if (ScheduleCount>0) {
  TotalSampleCount=Schedule[ScheduleCount].AbsoluteEndTime_100-Schedule[ScheduleCount-1].AbsoluteEndTime_100;
 }
 else Reset();
 */

  if ((--ScheduleCount) < 0)
    {
      Reset ();
    }
  else   TotalSampleCount = Schedule[ScheduleCount].AbsoluteEndTime_100;
}



//======================================
void CBinauralBeat::ParseCmd (FILE * stream) 
{
char strNum[128];
char strCmd[128];
char tmpchar;
int inum = 0, icmd = 0;

strNum[0] = strCmd[0] = '\0';
  
while ((tmpchar = fgetc (stream)) != ']' && feof (stream) == 0)
    {
	//eat up any extra characters
	//while (tmpchar=='=' || ' ') tmpchar=fgetc( stream );
	
	//if it is a number, put it in the number string:
	if ((tmpchar >= '0' && tmpchar <= '9') || 
        tmpchar == '.'	    || 
        tmpchar == '-')
	{
  strNum[inum] = tmpchar;
  inum++;
 }
      

	//if it is a string, put it in Cmd string:
	else if ((tmpchar >= 'A' && tmpchar <= 'Z')  || (tmpchar >= 'a' && tmpchar <= 'z'))
	{
   strCmd[icmd] = toupper (tmpchar);
   icmd++;
 }
      

 else if (tmpchar == '#')
	{			//user put a comment after a command!
	  while (feof (stream) == 0 && tmpchar != '\n') tmpchar = fgetc (stream);
   return;
  }
    
}				//end of parsing line:
  
strNum[inum] = '\0';
  
strCmd[icmd] = '\0';
  
if (!strcmp (strCmd, "BASEFREQ"))
    {
      
FreqBase = (float) atof (strNum);
      
if (FreqBase < 40)
	FreqBase = 40;
      
if (FreqBase > 1000)
	FreqBase = 1000;
    
}
  

if (!strcmp (strCmd, "TONEVOL"))
    {
      
float ftemp = (float) atof (strNum);
      
if (ftemp < 0)
	ftemp = 0;
      
if (ftemp > 100)
	ftemp = 100;
      
Volume_Tone = (int) (163.84 * ftemp + 0.5);
    
}
  

if (!strcmp (strCmd, "NOISEVOL"))
    {
      
Volume_Noiz = (float) atof (strNum);
      
if (Volume_Noiz < 0)
	Volume_Noiz = 0;
      
if (Volume_Noiz > 100)
	Volume_Noiz = 100;
      
Volume_Noiz = (float) (Volume_Noiz * 0.01);
    
}
  

if (!strcmp (strCmd, "LOOPS"))
    {
      
loops = atoi (strNum);
      
if (loops < 0)
	loops = 0;
      
loopcount = loops;
    
}
  
 if (!strcmp (strCmd, "STEREONOISE"))
 {
  if (atoi (strNum) > 0)	StereoNoiz = true;
  else 	StereoNoiz = false;
 }
}


//======================================
void CBinauralBeat::WriteScheduleToFile () 
{
WriteScheduleToFile (SchedFilename);
}
void CBinauralBeat::WriteScheduleToFile (char *file_name) 
{
FILE * stream;
if ((stream = fopen (file_name, "w")) == NULL)    return;
fprintf (stream, "# This is a Gnaural schedule file\n");
fprintf (stream, "# Gnaural ignores lines starting with \"#\"\n\n");
fprintf (stream, "# Base frequency for tones: (40-1000)\n");
fprintf (stream, "[BASEFREQ=%g]\n\n", FreqBase);
fprintf (stream, "# Noise Volume: (0-100)\n");
fprintf (stream, "[NOISEVOL=%g]\n\n", Volume_Noiz * 100.0f);
fprintf (stream, "# Tone Volume: (0-100)\n");
fprintf (stream, "[TONEVOL=%g]\n\n", (float) (Volume_Tone / 163.84));
fprintf (stream, "# Times to repeat schedule: (0 means infinite)\n");
fprintf (stream, "[LOOPS=%d]\n\n", loops);
fprintf (stream, "# Use Stereo Noise 1=true, 0=false\n");
fprintf (stream, "[STEREONOISE=%d]\n\n", StereoNoiz ? 1 : 0);
fprintf (stream, "# Runtime: %d min. %d sec. (%g sec.), %d Entries\n", 
	    (int) ScheduleTime / 60, (int) ScheduleTime % 60, ScheduleTime,
	    ScheduleEntriesCount);
fprintf (stream, "#FreqLStart\tFreqRStart\tDuration\n");
  
int listcount = -1;
  
while ((++listcount) < (ScheduleEntriesCount - 1))
  {
//fprintf(stream,"%.1f,\t\t %.1f,\t\t %.1f,\n",Schedule[listcount].FreqLStart,Schedule[listcount].FreqRStart,Schedule[listcount].Duration);
	  fprintf (stream, "%g,\t\t%g,\t\t%g,\n",
		 Schedule[listcount].FreqLStart,
		 Schedule[listcount].FreqRStart,
		 Schedule[listcount].Duration);
   }
  
// fprintf(stream,"%.1f,\t\t %.1f,\t\t %.1f\n",Schedule[listcount].FreqLStart,Schedule[listcount].FreqRStart,Schedule[listcount].Duration);
   fprintf (stream, "%g,\t\t%g,\t\t%g\n", Schedule[listcount].FreqLStart,
	  Schedule[listcount].FreqRStart, Schedule[listcount].Duration);
  
fclose (stream);

}




//======================================
void CBinauralBeat::WriteDefaultScheduleToFile () 
{
 SetDefaultValues ();
 SchedBuffToSchedule (DefaultSchedBuff);//does a call to WriteScheduleToFile() at end
} 


//======================================
void CBinauralBeat::Mute (bool act)
{
  
static float oldnoiz = Volume_Noiz;
  
static int oldtone = Volume_Tone;
  
if (act)
    {
      
oldnoiz = Volume_Noiz;
      
oldtone = Volume_Tone;
      
Volume_Noiz = 0.f;
      
Volume_Tone = 0;
    
}
  
  else
    {
      
Volume_Noiz = oldnoiz;
      
Volume_Tone = oldtone;
    
}

}


void CBinauralBeat::WAVheader_fill()
{    
wavh.ckID[0] = 'R';
wavh.ckID[1] = 'I';
wavh.ckID[2] = 'F';
wavh.ckID[3] = 'F';
wavh.ckSize = FileByteCount + sizeof (WAVheader) - 8;
wavh.wave_ckID[0] = 'W';
wavh.wave_ckID[1] = 'A';
wavh.wave_ckID[2] = 'V';
wavh.wave_ckID[3] = 'E';
wavh.fmt_ckID[0] = 'f';
wavh.fmt_ckID[1] = 'm';
wavh.fmt_ckID[2] = 't';
wavh.fmt_ckID[3] = ' ';
wavh.fmt_ckSize = 16;
wavh.formatTag = 1;
wavh.nChannels = 2;
wavh.nSamplesPerSec = 44100;
wavh.nAvgBytesPerSec = 176400;  //nAvgBytesPerSec= nSamplesPerSec* nBitsPerSample/8* nChannels
wavh.nBlockAlign = 4;
wavh.nBitsPerSample = 16;
wavh.data_ckID[0] = 'd';
wavh.data_ckID[1] = 'a';
wavh.data_ckID[2] = 't';
wavh.data_ckID[3] = 'a';
wavh.data_ckSize = FileByteCount;	//arbitrary for now, fill in right later
}



//======================================
// I need to add two size parameters when I am done- in the simplified case of 44100
// 2 channel, I can basically sum the size of WAVheader in to total data size, then 
// subtract 8 (god knows why - RIFF is insane)
bool CBinauralBeat::WriteWAVFile (char *szFilename) 
{
  FILE *    stream;  
if ((stream = fopen (szFilename, "wb")) == NULL)
{
    return false;
}

WriteWAVFileToStream(stream);

//now that data has been written, write those two damn header values correctly this time:
rewind (stream);  

wavh.data_ckSize = FileByteCount;
wavh.ckSize = FileByteCount + sizeof (WAVheader) - 8;
fwrite (&wavh, sizeof (WAVheader), 1, stream);
fclose (stream);
}


//======================================
bool CBinauralBeat::WriteWAVFileToSTDOUT ()
{
 //Apparently, Win32 sets stdout to text mode, while in Linux is binary. And yet,
 //Linux is easy to change, while Win32 proprietary/wierd to change. I think to change
 //Win32, do something like this: setmode(fileno(stdout), O_BINARY);
 //A more general solution to make Linux happy:
//http://www.contrib.andrew.cmu.edu/~twm/stuff/binary-redirect.html
/* 
 //for POSIX compliant compilers, this is the approach:
   FILE *stream = fdopen (fileno(stdout), "wb");
//   if (stream==NULL) return false;
   if (stream==NULL) stream=stdout; //give it a try anyway...
    WriteWAVFileToStream (stream);

#ifdef WIN32
  _setmode (_fileno (stdout), O_BINARY);
  _setmode (_fileno (stdin), O_BINARY);
#endif
*/
//NOTE: the above is too platform specific for this class; user should do that elsewhere.
 WriteWAVFileToStream(stdout);
}

//======================================
//it is up to the user to ensure that the stream is writable and in binary mode.
bool
CBinauralBeat::WriteWAVFileToStream (FILE *stream)
{
//  FILE *stream;
//  stream = stdout; 
 
//20051129 - I need to estimate final size of file for stream/piping use,
//and so something is in there in case user cancels mid-write:
 FileByteCount = (unsigned int)(loops * ScheduleTime * 176400);
 WAVheader_fill();
 
  
    //create the header:
    fwrite (&wavh, sizeof (WAVheader), 1, stream);
    FileByteCount = 0; //now reset FileByteCount of estimate in order to have it count real samples
    WriteStop = false;
    ManualFreqOffsetControl = false;
  
signed short
    buff[4];
  
Reset ();			//make sure we start at absolute beginning
  while (!(InfoFlag & BB_COMPLETED) && !WriteStop)
    {
//  fputc((unsigned char)((val)&0xff), stream);
//  fputc((unsigned char)((val>>8)&0xff), stream);
//  fputc((unsigned char)(0), stream);
//  fputc((unsigned char)(0), stream);
	MainLoop (buff, 4);
 fwrite (buff, 4, 1, stream);
 FileByteCount += 4;
}
  
// InfoFlag|=BB_COMPLETED;//just in case WriteStop was called
// WriteStop=false;
    return true;
}




/*
//======================================
bool
CBinauralBeat::WriteWAVFileToStream (FILE *stream)
{
//  FILE *stream;
//  stream = stdout; 
 
 //added 20051129 after noticing that STDOUT was not necessarily by default binary:
  if (stream==NULL) {
  stream = fdopen (fileno(stdout), "wb");
   if (stream==NULL) return false;
   }
 
WAVheader  wavh;


// I need to add two size parameters when I am done- in the simplified case of 44100
// 2 channel, I can basically sum the size of WAVheader in to total data size, then 
// subtract 8 (god knows why - RIFF is insane)

  wavh.ckID[0] = 'R';
  wavh.ckID[1] = 'I';
  wavh.ckID[2] = 'F';
  wavh.ckID[3] = 'F';
  wavh.ckSize = sizeof (WAVheader) - 8; //this will be the size of data plus size of WAVHeader... minu
  wavh.wave_ckID[0] = 'W';
  wavh.wave_ckID[1] = 'A';
  wavh.wave_ckID[2] = 'V';
  wavh.wave_ckID[3] = 'E';
  wavh.fmt_ckID[0] = 'f';
  wavh.fmt_ckID[1] = 'm';
  wavh.fmt_ckID[2] = 't';
  wavh.fmt_ckID[3] = ' ';
  wavh.fmt_ckSize = 16;
  wavh.formatTag = 1;
  wavh.nChannels = 2;
  wavh.nSamplesPerSec = 44100;
  wavh.nAvgBytesPerSec = 176400;
  wavh.nBlockAlign = 4;
  wavh.nBitsPerSample = 16;
  wavh.data_ckID[0] = 'd';
  wavh.data_ckID[1] = 'a';
  wavh.data_ckID[2] = 't';
  wavh.data_ckID[3] = 'a';
  wavh.data_ckSize = 0;		//can't tell big the file will be

  //create the header:
  fwrite (&wavh, sizeof (WAVheader), 1, stream);


  WriteStop = false;

  ManualFreqOffsetControl = false;

  FileByteCount = 0;

  signed short buff[4];

  Reset ();			//make sure we start at absolute beginning
  while (!(InfoFlag & BB_COMPLETED) && !WriteStop)
    {

//  fputc((unsigned char)((val)&0xff), stream);
//  fputc((unsigned char)((val>>8)&0xff), stream);
//  fputc((unsigned char)(0), stream);
//  fputc((unsigned char)(0), stream);
      MainLoop (buff, 4);

      fwrite (buff, 4, 1, stream);

      FileByteCount += 4;

    }

  return true;

}
*/


//======================================
void CBinauralBeat::Reset () 
{
 TotalSampleCount = TotalSampleCountLooped = ScheduleCount = 0;
 InfoFlag &= ~BB_COMPLETED;
 loopcount = loops;
}
