/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

//Gnaural - a Binaural Beat and pink noise generator by Bret Logan (c) 2006
// ***Entrain Your Brain***

//TODO: 
//   - have a Graph menu entry for "Go To selected point", which allows user to progress to a selected point.
//   - allow arrow keys and mouse scroller to do fine adjustments of points
//- consider a balance slider for Tone, because percieved loudness can be stereo-lopsided with small shifts at low base frequencies
// - make volume controls "audio-taper" (noise slider in particular is too sensitive at very low values)
// - when you do "Save As, make sure new filename is now the "Currently Using..." (currently, it saves to new file, but a Save will still go to original file)
// [graph] -  create some functions to process graph data:
//                -  show somewhere on status window the location of the mouse in Hz and Duration 
//                -  a means to set the endtime of the schedule without having to monkey with values (that accepts minute values too)
//- create a means of mixing OGG, MP3 or WAV files in to the sound, with their own sliders 
//- offer a Tone control (since so many people use open-headphones, which over accentuate the hi-freq noise)
//- implement longer Undo history (currently just holds last change; Redo/Undo just toggles current and last-change data)
//- create a "most recently used" list of files under File menu, maybe also have Gnaural open last used file. Maybe make it a dropdown list on main interface?
//- add a way in schedule file to provide a description
//- solve issue of overly persistent statusbar messages
//- offer an (optional/backward-compatible) way to modulate Noise volume.
// [win32] implement "Make MP3" 
// make "disable screensaver" option actually work
// [win32] give a desktop default icon
//- fix the inconsistency that when a user saves current setup with Base on manual, reloads setup, then toggles Base back to auto, Gnaural reverts to a "pre-saved" Base freq
//- examine all DBGOUT, ERROUT and stray fprinfs to see if each is really optional or crucial info (maybe replace some with an "INFOOUT" define
//-create real GUI help dialog
//-GUI: figure out how to left justify text and not make boxes not resize according to text size

//IDEAS:
//-GUI: add some tooltips
//-GUI: let user hold either Rewind or FF to literally move multiple times (instead of having to do lots of clicks). [keyboard shortcuts work for now]
//-allow user to set ranges for the manual sliders in the user file
//- have mainInit() to return a value to main() to let main() decide to exit
//-check disk space for user before write (use statfs, I guess?)

//BUGS:
//- calling system() [as I do with with gedit and MP3 tasks] keep "things" persistent even after program termination
//     if user doesn't exit those tasks, disallowing a new copy of Gnaural to open properly (sound won't work).

//NOTES:
//Platform differences hinge on GNAURAL_WIN32. I used to use G_PLATFORM_WIN32, but found some unclear issues regarding it.

#ifdef HAVE_CONFIG_H
#  include <config.h>
#endif

#include <unistd.h> //NOTE: Linux sleep() calls were replaced 20051126 with g_usleep() code. But this is still needed for getopt
#include <sys/stat.h>
#include <errno.h>
#include <signal.h>
#include <gtk/gtk.h>
#include <glib.h>
//#include <pthread.h> //switched to Glib threads on 20051126

#include "src/portaudio.h"

#include "interface.h"
#include "support.h"
#include "BinauralBeat.h"
#include "ScheduleGUI.h"
#include "main.h"
#include "gnaural.h"


//NOTE: if you are compiling this for Linux and want to use ALSA instead of OSS for sound, 
//see gnaural-0.2.20051103; that was the last ALSA version. I switched to OSS only because 
//PortAudio still uses that (plans to eventually switch, though).

#define LABEL_FLOATPRECISION  "% g"
//#define LABEL_FLOATPRECISION  "%8.5f"

#define DBGOUT(a)   if (cmdline_d==true) fprintf(stderr,"Gnaural: %s\n",a)
//#define DBGBEEP  putc('\a',stderr);
#define ERROUT(a)   fprintf(stderr,"Gnaural: #Error# %s\n",a) 

//global Path/File variables:
char szDir_CurrentWorking[PATH_MAX + 1]; //whatever directory user called gnaural from
char szDir_Home[PATH_MAX + 1];                          //  ~/                                               -- not meaningfull in Win32
char szDir_Gnaural[PATH_MAX + 1];                       // ~/.gnaural/                                  -- not meaningfull in Win32
char szFile_gnaural_schedule_txt[PATH_MAX + 1]; // ~/.gnaural/gnaural_schedule.txt  --
char szPath_Temp[PATH_MAX+PATH_MAX + 1];	//used to copy the other paths and strcat filenames-to
char * szFile_ExecutablePath;//so program can call itself to make MP3


//START main()======================================
int main (int argc, char *argv[])
{
  szFile_ExecutablePath=argv[0];//so program can call itself to make MP3

 //do command line parsing:
 bbl_ParseCmdLine (argc,argv);

      #ifdef GNAURAL_WIN32
 //this ensures that any potential Windows stdout activity is in binary mode:
  _setmode (_fileno (stdout), O_BINARY);
//  _setmode (_fileno (stdin), O_BINARY);
      #endif 
 
  //do all the non-GUI and non-Sound initializations here:
  bbl_mainInit ();

 //added 20051126 to support GThread cross-compatibility. Yes, it is supposed to be called before gtk_init()
 // call  echo `pkg-config --libs gthread-2.0` to see the libs to link to; I was segfaulting endlessly because I was linking
 //with old libs! THis is the proper:
//g++  -g -O2  -o gnaural  main.o BinauralBeat.o support.o interface.o callbacks.o pa_lib.o pa_unix.o pa_unix_oss.o -lgtk-x11-2.0 -lgdk-x11-2.0 -latk-1.0 -lgdk_pixbuf-2.0 -lm -lpangoxft-1.0 -lpangox-1.0 -lpangoft2-1.0 -lpango-1.0 -lgobject-2.0 -lgmodule-2.0 -ldl -lglib-2.0 -pthread -lgthread-2.0
//if (!g_thread_supported ()) g_thread_init (NULL);
 g_thread_init(NULL);
 gdk_threads_init();
 gdk_threads_enter (); //if I don't do this, the Win32 version blocks! The documentation is amazingly vague.
 
//sort the args:
//The main question: Do I need a GUI? Basically boils down to these two categories:
//THINGS AFFECTING BOTH GUI AND TERM VERSION: 
//- cmdline_a: Tell Gnaural which sound HW to use
//- cmdline_d: Show debugging information
//THINGS AFFECTING SOLEY TERM VERSION:
//- cmdline_h: Print "Help" [already did this by this point] DOES NOT REQUIRE SOUND SYSTEM
//- cmdline_o: dump a .WAV file to sdtout DOES NOT REQUIRE SOUND SYSTEM
//- cmdline_w: dump a .WAV file to a file DOES NOT REQUIRE SOUND SYSTEM
//- cmdline_s: create a fresh gnaural_schedule.txt file DOES NOT REQUIRE SOUND SYSTEM
//- cmdline_i: show terminal-style GUI info
//- cmdline_p: run the schedule directly through the soundsystem

if ((cmdline_i+cmdline_p+cmdline_o+cmdline_w+cmdline_s)>0) 
 {//do the command line version then exit:
 DBGOUT("Entering console mode");
    ++gnaural_guiflag;
     bbl_RunCmdLineMode();
     bbl_mainCleanup();
     exit (EXIT_SUCCESS);      
}


    //Set-up sound for GUI mode:
  if (EXIT_FAILURE==bbl_SoundInit()) 
   {
    DBGOUT("Couldn't intialize the sound system, running without sound");
    Pa_Terminate();
    gnaural_pa_stream=NULL;//this is how the rest of the program knows if we're using sound or not.
  }

  DBGOUT("Entering GUI mode");

  
  //START glade insertions:
#ifdef ENABLE_NLS
  bindtextdomain (GETTEXT_PACKAGE, PACKAGE_LOCALE_DIR);
  bind_textdomain_codeset (GETTEXT_PACKAGE, "UTF-8");
  textdomain (GETTEXT_PACKAGE);
#endif
  gtk_set_locale ();
  gtk_init (&argc, &argv);
  add_pixmap_directory (PACKAGE_DATA_DIR "/" PACKAGE "/pixmaps");
  window1 = create_window1 ();
//    GTK_WIDGET_SET_FLAGS (lookup_widget (GTK_WIDGET (window1), "drawingarea1"), GTK_CAN_FOCUS);

  bbl_SetIcon();//added 20051202; bruteforce method to avoid nasty GTK path/function inconsistencies

  
  gtk_widget_show (window1);
  //END glade insertions
  

 //GUI is initialized, so now I can call this to update it with real info:
  bbl_InitGUI ();

  //add display update timer:
  g_timeout_add (DISPLAY_REFRESH_TIME, (GtkFunction) bbl_UpdateGUI_Labels,
		 NULL);

  if (gnaural_pa_stream==NULL)  {
   bbl_UpdateGUI_Statusbar("Couldn't intialize the sound system","Running without sound");
  }

  bbl_UpdateGUI_Filename();

//enter full GUI mode: must call gtk_main_quit() from somewhere to exit gtk_main()
  gtk_main ();


//clean up my allotted resources:
  bbl_mainCleanup ();

//I have no idea where this should legally go:
  gdk_threads_leave();

return EXIT_SUCCESS;
}
//END main()======================================


//======================================
void bbl_UpdateGUI_Statusbar(char *msg1, char *msg2)
{
static gint context_id = gtk_statusbar_get_context_id(StatusBar_main, "SC");
 gtk_statusbar_pop(StatusBar_main,context_id);
 sprintf(gnaural_tmpstr, "%s %s",msg1,msg2);
 gtk_statusbar_push (StatusBar_main,context_id,gnaural_tmpstr);
}


//======================================
void bbl_UpdateGUI_ProgramStatus(char *msg1, char *msg2)
{
 sprintf(gnaural_tmpstr, "%s %s",msg1,msg2);
 gtk_label_set_text (LabelProgramStatus, gnaural_tmpstr); 
}


//======================================
void bbl_UpdateGUI_Filename()
{
   sprintf(gnaural_tmpstr,"Using %s",szFile_gnaural_schedule_txt);
  gtk_label_set_text (LabelFilename, gnaural_tmpstr);
}


//======================================
// PortAudio calls this:
static int patestCallback(   void *inputBuffer, void *outputBuffer,
                             unsigned long framesPerBuffer,
                             PaTimestamp outTime, void *userData )
{
    bb->UserSoundProc(outputBuffer, (framesPerBuffer<<2));//IMPORTANT: no matter what units the sndbuffer uses, CBinauralBeat wants sndbufsize in bytes
    return 0; //return 1 to stop sound server
}



//======================================
//I bet I could do this without actually writing to disk, but 
//have wrestled with so many bizarrely non-functional GTK 
//themed/builtin/window/named icon functions that
//after a full day of failing, I must simply do something 
//I understand (see icon_problems.txt):
void bbl_SetIcon()
{

#ifdef GNAURAL_WIN32 
 // Get the temp path.
    if (0!=GetTempPath(sizeof(szPath_Temp), szPath_Temp)) //NOTE: returned string ends in backslash
    {     //success
     strcat(szPath_Temp,"gnaural-icon.png");
    }
    else
    {     //fail
     sprintf(szPath_Temp,"gnaural-icon.png");
    }
 #endif
    
#ifndef GNAURAL_WIN32 
     sprintf(szPath_Temp,"%s/%s",szDir_Gnaural,"gnaural-icon.png");
#endif

    
FILE *    stream;  
if ((stream = fopen (szPath_Temp, "wb")) == NULL)
{
    return;
}
fwrite (&GnauralIcon, sizeof (GnauralIcon), 1, stream);
fclose (stream);

gtk_window_set_default_icon_from_file (szPath_Temp,  NULL);
}


//======================================
//returns zero if file or path DOES NOT exist
int
bbl_TestPathOrFileExistence (char *PathName)
{
  struct stat buf;
  return stat (PathName, &buf);
}


//======================================
 //following tries to be sure I don't end with a \ or / (which happens 
//only when getcwd returns a root like / or C:\, D:\   :
void bbl_FixPathTerminator(char * pth)
{
//  #ifdef GNAURAL_WIN32
 if (pth[(strlen(szDir_CurrentWorking)-1)]=='\\') {
  pth[(strlen(szDir_CurrentWorking)-1)]='\0';
 }
// #endif
else
// #ifndef GNAURAL_WIN32
if (szDir_CurrentWorking[(strlen(szDir_CurrentWorking)-1)]=='/') {
  szDir_CurrentWorking[(strlen(szDir_CurrentWorking)-1)]='\0';
 }
// #endif
 }


//======================================
void bbl_SetupPathsAndFiles ()
{
  //Current Working Directory:
  getcwd (szDir_CurrentWorking, PATH_MAX);
  bbl_FixPathTerminator(szDir_CurrentWorking);

 //DBGOUT(szDir_CurrentWorking); 

  //copy CWD to the other vars, just in case I forget to put anything in them:
  strcpy (szDir_Home, szDir_CurrentWorking);
  strcpy (szDir_Gnaural, szDir_CurrentWorking);

DBGOUT("This was the command to launch Gnaural:");
DBGOUT(szFile_ExecutablePath);
 
DBGOUT("This is the current working directory:");
DBGOUT(szDir_CurrentWorking);

  #ifdef GNAURAL_WIN32
//###########START WIN32-ONLY CODE#############    
 //In Win32, I try to work from wherever the executable is located:
 
//----- 
//START of determine Gnaural's home directory:
//HKEY_LOCAL_MACHINE\\Software\Microsoft\\Windows\CurrentVersion\\App Paths\\gnaural.exe
//C:\Program Files\Gnaural;c:\Program Files\Common Files\GTK\2.0\bin

HKEY hkey;
if (ERROR_SUCCESS==RegOpenKeyEx(HKEY_LOCAL_MACHINE, "Software\\Microsoft\\Windows\\CurrentVersion\\App Paths\\gnaural.exe",0, KEY_QUERY_VALUE, &hkey))
{ 
  DWORD type, count = sizeof(szPath_Temp);
  if (ERROR_SUCCESS==RegQueryValueExA( hkey, "Path", 0, &type, (LPBYTE) szPath_Temp, &count ))
  {
  //Looking for the first path in the list:
  char *tmpch;
   tmpch=strchr(szPath_Temp,';'); //search for first ';'
   if (tmpch!=NULL)  (*tmpch)='\0';
   strcpy(szDir_Gnaural, szPath_Temp);
   bbl_FixPathTerminator(szDir_Gnaural);
   }
   RegCloseKey( hkey );
} 
else
{
  DBGOUT ("No gnaural.exe registry entry, checking manually for dir");
 //try a bruteforce approach:
  if (0 != bbl_TestPathOrFileExistence ("C:\\Program Files\\Gnaural\\"))
   {
     strcpy(szDir_Gnaural,"C:\\Program Files\\Gnaural");
   }
}


/*
//NOTE: Original intent (below commented code) was to have all file creation in executable's home directory, but
//it failed because of "/" and "\" inconsistencies between WINE and Win32, rendering
//mixed paths, etc. Since getcwd() seems to be mostly consistent on each 
//platform, I eventually stuck with that -- but in the end, a WINE bug causing gcwd to return 
//(in my case) H:\ for both root and my home directory made me give-up and just 
//use the (goddamned) registry.
    
 strcpy (szPath_Temp, szFile_ExecutablePath);
 
 //Now make all slashes consistent with DOS:
 int i=strlen(szPath_Temp);
 while ((--i)>-1) if (szPath_Temp[i]=='/') szPath_Temp[i]='\\';

 //Now throw away executable name if there is any:
 char *tmpch;
 tmpch=strrchr(szPath_Temp,'\\'); //search for first slash from end
if (tmpch!=NULL) {
 (*tmpch)='\0';

 DBGOUT("This is the directory Gnaural is being run from:");
 DBGOUT(szPath_Temp);
 
//now return all slashes to DOS style:
 i=strlen(szPath_Temp);
 while ((--i)>-1) if (szPath_Temp[i]=='/') szPath_Temp[i]='\\';

//now change in to that directory [still necessary?!]
 chdir(szPath_Temp);
 
 //now ADD starting slash if it isn't there:
if (szPath_Temp[0]!='\\')
{
for (i=strlen(szPath_Temp);i>-1;i--)  szPath_Temp[i+1]=szPath_Temp[i];
szPath_Temp[0]='\\';
}
}
//If I got here, user is already working from Gnaural's directory:
else szPath_Temp[0]='\0';

//now attach this string to the end of CWD:
strcpy(szDir_Gnaural,szDir_CurrentWorking);
strcat(szDir_Gnaural,szPath_Temp);
*/

//END of determine Gnaural's home directory
//----- 

//Finally, setup file and path to gnaural_schedule.txt file:
  sprintf (szFile_gnaural_schedule_txt, "%s\\gnaural_schedule.txt", szDir_Gnaural);
  DBGOUT("Accessing schedule file: ");
  DBGOUT (szFile_gnaural_schedule_txt);
  return;
//###########END WIN32-ONLY CODE#############       
 #endif

 
   #ifndef GNAURAL_WIN32
//###########START 'NUX-ONLY CODE#############
 //In Windows, Gnaural just does everything in it's formal installation directory.
 //In 'nix, it works from ~/.gnaural
 
 //figure out the paths/directories needed for all file access:
  if (NULL != getenv ("HOME"))
    {
      //Home directory:
      strcpy (szDir_Home, getenv ("HOME"));
      //~/.gnaural directory:
      sprintf (szDir_Gnaural, "%s/.gnaural", szDir_Home);
    }
  else
    {//failed, so harmlessly put current directory in there...
     ERROUT("Couldn't determine home directory");
      strcat (szDir_Home, szDir_CurrentWorking);
      strcat (szDir_Gnaural, szDir_CurrentWorking);
    }

  //create the .gnaural directory if it doesn't exist yet:szDir_CurrentWorking
  if (0 != bbl_TestPathOrFileExistence (szDir_Gnaural))
    {
      DBGOUT ("Creating ~/.gnaural directory");

      if (0 != mkdir (szDir_Gnaural, 0777))
        {
	         fprintf (stderr, "Couldn't make ~/.gnaural directory %s: errno=%i\n", szDir_Gnaural, errno);
        }
    }
  else
  {
    DBGOUT ("Found ~/.gnaural directory; will use.");
  }
//setup file and path to gnaural_schedule.txt file:
  sprintf (szFile_gnaural_schedule_txt, "%s/gnaural_schedule.txt", szDir_Gnaural);
  DBGOUT("Accessing schedule file: ");
  DBGOUT (szFile_gnaural_schedule_txt);
  return;
   #endif
//###########END 'NUX-ONLY CODE#############

}



//======================================
//updates a terminal window via either stdout or stderr streams
void bbl_UpdateTerminalGUI(FILE * gstream)
{ 
 //-----------------------------------------------------------
//START stuff that gets updated EVERY second:
//
// current time point within schedule:
 fputs("\033[2;1H",gstream);//place cursor
  fprintf (gstream, "Current Progress: \t%d min. \t%d sec.",
	   (bb->TotalSampleCount + bb->TotalSampleCountLooped) / 6000,
	   ((bb->TotalSampleCount + bb->TotalSampleCountLooped) / 100) % 60);
 fputs("\033[K",gstream);//clear to end of line

  //remaining time in entry:
 fputs("\033[5;1H",gstream);//place cursor
  fprintf (gstream, "Time left: \t\t%d sec.",
	   ((bb->Schedule[bb->ScheduleCount].AbsoluteEndTime_100 -
	     bb->TotalSampleCount) / 100));
 fputs("\033[K",gstream);//clear to end of line

  //Left frequency:
 fputs("\033[9;1H",gstream);//place cursor
  fprintf (gstream, "Current Freq. Left: \t%g\n", 
  bb->FreqL);
 fputs("\033[K",gstream);//clear to end of line

  //Right frequency:
// fputs("\033[15;1H",gstream);//place cursor
  fprintf (gstream, "Current Freq. Right: \t%g\n", 
  bb->FreqR);
 fputs("\033[K",gstream);//clear to end of line

  //Beat frequency:
// fputs("\033[16;1H",gstream);//place cursor
  fprintf (gstream, "Current Beat Freq.: \t%g\n", 
  bb->FreqL - bb->FreqR);
 fputs("\033[K",gstream);//clear to end of line

//
  //END stuff that gets updated EVERY second:
  //-----------------------------------------------------------------------

  //-----------------------------------------------------------------------
//START check bb->InfoFlag
//now update things that BinauralBeat says are changed:
  if (bb->InfoFlag != 0)
    {

      //.........................
      //if schedule is done:
      if (((bb->InfoFlag) & BB_COMPLETED) != 0)
	{
 fputs("\033[7;1H",gstream);//place cursor
  fprintf (gstream, "Schedule Completed");
 fputs("\033[K",gstream);//clear to end of line
	}

      //.........................
      //if starting a new loop:
      if (((bb->InfoFlag) & BB_NEWLOOP) != 0)
	{
	  //reset the "new loop" bit of InfoFlag:
	  bb->InfoFlag &= ~BB_NEWLOOP;
	  //bbl_UpdateGUI_LoopInfo ();
	}

      //.........................
      //START things that get updated every new entry:
      //if new entry
      if (((bb->InfoFlag) & BB_NEWENTRY) != 0)
	{
	  //reset the "new entry" bit of InfoFlag:
	  (bb->InfoFlag) &= (~BB_NEWENTRY);
  
	  //current entry number:
  fputs("\033[3;1H",gstream);//place cursor
	  fprintf (gstream, "Current Entry: \t\t%d of %d",
		   bb->ScheduleCount + 1, bb->ScheduleEntriesCount);
 fputs("\033[K",gstream);//clear to end of line

	  //total time in entry: 
  fputs("\033[4;1H",gstream);//place cursor
	  fprintf (gstream, "Duration: \t\t%g sec.",
		   bb->Schedule[bb->ScheduleCount].Duration);
 fputs("\033[K",gstream);//clear to end of line

//fprintf(gstream,"%d",bb->Schedule[bb->ScheduleCount].AbsoluteEndTime_100/100);
//SetDlgItemText(lEndCount, gstream);

 fputs("\033[6;1H",gstream);//place cursor
	  fprintf (gstream, "Beat Range: \t\t%g to %g Hz\n",
		   bb->Schedule[bb->ScheduleCount].FreqLStart -
		   bb->Schedule[bb->ScheduleCount].FreqRStart,
		   bb->Schedule[bb->ScheduleCount].FreqLEnd -
		   bb->Schedule[bb->ScheduleCount].FreqREnd);
 fputs("\033[K",gstream);//clear to end of line

//left
// fputs("\033[7;1H",gstream);//place cursor
	  fprintf (gstream, "Left ear: \tStart:\t%g",
		   bb->Schedule[bb->ScheduleCount].FreqLStart);
 fputs("\033[K",gstream);//clear to end of line     
    fprintf (gstream, "\tEnd:\t%g\n",
		   bb->Schedule[bb->ScheduleCount].FreqLEnd);
 fputs("\033[K",gstream);//clear to end of line

//right
// fputs("\033[8;1H",gstream);//place cursor
	  fprintf (gstream, "Right ear: \tStart:\t%g",
		   bb->Schedule[bb->ScheduleCount].FreqRStart);
 fputs("\033[K",gstream);//clear to end of line     
	  fprintf (gstream, "\tEnd:\t%g",
		   bb->Schedule[bb->ScheduleCount].FreqREnd);
 fputs("\033[K",gstream);//clear to end of line
 
 //this is overdoing it, but need it updated sometime:
  
   if (bb->loops > 0)
   {
 fputs("\033[1;1H",gstream);//place cursor
    fprintf (gstream, "Projected Runtime: \t%d min. \t%d sec.",
	     (int) (bb->loops * bb->ScheduleTime) / 60,
	     (int) (bb->loops * bb->ScheduleTime) % 60);
 fputs("\033[K",gstream);//clear to end of line
   }
  else
  {
 fputs("\033[1;1H",gstream);//place cursor
    fprintf (gstream, "Projected Runtime:  FOREVER (Inf. Loop Mode)");
 fputs("\033[K",gstream);//clear to end of line
  }
  
	}
      //END things that get updated every new entry
}
}



//====================================== 
void bbl_ParseCmdLine (int argc, char *argv[])
{
  opterr = 0;			//this means I want to check the ? character myself for errors

   int c;
  //NOTE: options with ":" after them require arguments
  while ((c = getopt (argc, argv, GNAURAL_CMDLINEOPTS)) != -1)
    switch (c)
      {
      case 'h':		//print help
       fprintf(stdout,"Gnaural (ver. %s) - ",VERSION);
      
      fputs(szGnauralHelp, stdout);
	exit( EXIT_SUCCESS);//completely quit if the user asked for help.
	break;

     case 'i':		//show terminal-style gui info
    ++cmdline_i;
	break;

    case 'p':		//output directly to sound system
	++cmdline_p;
	break;

     case 'd':		//output debugging info to stderr
      ++cmdline_d;
    //may go back in to GUI mode, can't know from this
	break;

      case 'o':		//output a .WAV stream to stdout
	++cmdline_o;
	break;

      case 's':		//create default gnaural_schedule.txt file
	++cmdline_s;
      //CHANGE 051110:  to take care of weird but probably common possibility that user also used -w too:
      //NOTE: this fix also required a change in -w below.
      if (gnaural_writefilename!=NULL) 
       {        
        ERROUT("Freeing memory alotted by -w; -s has priority");
         delete gnaural_writefilename;
         gnaural_writefilename = NULL;
         cmdline_w=0;//just in case I missed something...
      }
	break;

    case 'a':		
	++cmdline_a;
   gnaural_pa_SoundDevice=atoi(optarg);
    //may go back in to GUI mode, can't know from this
  	break;

    case 'w':		//output a .WAV stream to a user specified .wav file
     if (cmdline_s>0) //CHANGE 051110: you don't want -s and -w in the same line
     {
      ERROUT("Ignoring -w; -s has priority");
      cmdline_w=0;//just in case I missed something...
      break;
     }
	     ++cmdline_w;
      gnaural_writefilename = new char[strlen (optarg) + 2];
      strcpy(gnaural_writefilename,optarg);
	   break;

    case ':':		//some option was missing its argument
	    fprintf (stderr, "Missing argument for `-%c'.\n", optopt);    
	    fputs(szGnauralHelpShort, stderr);
     exit( EXIT_FAILURE);//completely quit on error
    
   case '?':
	    fprintf (stderr, "Unknown option `-%c'.\n", optopt);
	    fputs(szGnauralHelpShort, stderr);
     exit( EXIT_FAILURE);//completely quit on error

      default:
	    fputs(szGnauralHelpShort, stderr);
        exit( EXIT_FAILURE);//completely quit on error
      break;
      }

//now see if there was any command-line stuff user inputted that made no sense:
  int nonoptionflag = 0;
  for (int index = optind; index < argc; index++)
    {
      fprintf (stderr, "Error: non-option argument \"%s\"\n", argv[index]);
      nonoptionflag = 1;
    }

    if (nonoptionflag != 0)
    {
	    fputs(szGnauralHelpShort, stderr);
     exit( EXIT_FAILURE);//completely quit on error
    }
  //END command line parsing
    return;
   }


//======================================
//this is basically just the command line version dropped in as a function-call in
//to the GUI version. 
void bbl_RunCmdLineMode ()
{
 //don't need GUI, so turn pause off:
  gnaural_pauseflag = 0;    
 
   //first do the things that don't require sound at all:
//- cmdline_s: create a fresh gnaural_schedule.txt file then exits DOES NOT REQUIRE SOUND SYSTEM
//- cmdline_o: dump a .WAV file to sdtout DOES NOT REQUIRE SOUND SYSTEM
//- cmdline_w: dump a .WAV file to a file DOES NOT REQUIRE SOUND SYSTEM
  
  // user asked to create a new default Schedule file;  do that then exit:
  if (cmdline_s > 0)
    {
      bb->WriteDefaultScheduleToFile ();
     DBGOUT("Wrote default gnaural_schedule.txt file");
     return;
    }

////////////////
//NOTE:
//from this point forward, I work on the premise that user may be piping 
//output to stdout to another program expecting a WAV file
////////////////

//these next three are mutually exclusive, and only cmdline_p needs sound system:
if (cmdline_w != 0)
    {
//old direct way of doing this:
//      fprintf (stderr, "Writing schedule %s to file %s... ",
//	       bb->SchedFilename, w_str);
//      bb->WriteWAVFile (w_str);
//      fprintf (stderr, "Done.\n");
//      return (EXIT_SUCCESS);

//threaded way of doing the above:
  if (cmdline_i!=0)  {
      fputs( "\033[2J",stderr);//clear entire screen
      fputs("\033[14;1H",stderr);//place cursor
      }
     
 fprintf (stderr, "Writing WAV data to file %s... \n", gnaural_writefilename);
  //create the sound engine thread (it shouldn't need GUI), it heeds only bb variables
     //bb->WriteStop=0;
      
//old pthread code (dead 20051126):      :      
//  if (pthread_create (&gnaural_thread_WriteEngineWAV, NULL, bbl_writethreadWAV, (void *) NULL) != 0)        
//    {
//      DBGOUT ("Couldn't create the WAV filewriting  thread.");
//    }

//new gthread code (new 20051126):      
 GError           *thread_err = NULL ;
 if ((gnaural_thread_WriteEngineWAV = g_thread_create((GThreadFunc)bbl_writethreadWAV, (void *)NULL, TRUE, &thread_err)) == NULL)
  {
     fprintf(stderr,"g_thread_create failed: %s!!\n", thread_err->message );
     g_error_free ( thread_err ) ;
  }

    while (gnaural_writefilename!=NULL && (((bb->InfoFlag) & BB_COMPLETED) == 0))
 {
     if (cmdline_i!=0)  
       {
       bbl_UpdateTerminalGUI(stdout);
      fflush(stdout);//must flush often when using stdout
      }
      g_usleep (1000);
     } 
     if (cmdline_i!=0)  fputs("\033[14;1H",stderr);//place cursor

  return; 
    }
    
    //----------

    
  else if (cmdline_o != 0)
    {
      bb->WriteWAVFileToSTDOUT ();
     return ;
    }
    
    //----------
    
      else   if (cmdline_p != 0)		//write to sound system:
    {
  //First, set up the sound system, or exit if not possible:
    //first set-up sound, since program may not return from parsing command line
  if (EXIT_FAILURE==bbl_SoundInit()) 
   {
    DBGOUT("Couldn't intialize the sound system, exiting");
    Pa_Terminate();
    gnaural_pa_stream=NULL;//this is how the rest of the program knows if we're using sound or not.
    return;
  }

  {
   DBGOUT("Writing to sound system");
   }

  if (cmdline_i!=0)  {
      fputs( "\033[2J",stderr);//clear entire screen
      fputs("\033[14;1H",stderr);//place cursor
      }

   Pa_StartStream( gnaural_pa_stream );

    while (gnaural_pa_stream!=NULL && (((bb->InfoFlag) & BB_COMPLETED) == 0))
     {
      if (cmdline_i!=0)  
       {
       bbl_UpdateTerminalGUI(stdout);
      fflush(stdout);//must flush often when using stdout
      }
       Pa_Sleep( 1000 );
     } ;
    if (cmdline_i!=0)  fputs("\033[14;1H",stderr);//place cursor

return ;
     }
}


//======================================
void bbl_InterceptCtrl_C (int sig)
{
  DBGOUT ("Caught Ctrl-C");

  if (gnaural_guiflag == GNAURAL_GUIMODE)   {
   //20060328: I used to just call gtk_main_quit() here, but it
   //is smarter to let the callback functions associated with window1
   //do their own cleanup:
   gtk_widget_destroy( window1 );
//   ScheduleGUI_delete_event
   DBGOUT ("Quit GUI mode");
  }
  bbl_mainCleanup ();
  exit(EXIT_SUCCESS);
}


//======================================
void bbl_OnRestoreDefaultFile ()
{
  if (true ==
      bbl_MessageDialogBox
      ("Click 'Yes' to replace your current gnaural_schedule.txt file with a default one.\n\nClick 'No' to leave it alone.",
       GTK_MESSAGE_WARNING, GTK_BUTTONS_YES_NO))
    {
     bbl_SetupPathsAndFiles (); //added 20060315 to solve possibly writing over user-named file instead of default file
     bb->WriteDefaultScheduleToFile ();
     bbl_OnReloadFile();
    }
}



//======================================
void bbl_OnReloadFile ()
{
  bbl_OnButton_Stop ();		//this puts it back at beginning -- conservative, but do I really want that?
  bb->SchedFilenameToSchedule ();
  bbl_InitGUI ();		//to be sure sliders reflect new user data
  bbl_UpdateGUI_Info ();
  bbl_UpdateGUI_Filename();
}


//======================================
void
bbl_OnButton_Stop ()
{
  bb->WriteStop = TRUE;		//try to kill a write if it is what's happening

  //first step is to pause:
  gnaural_pauseflag = 0;		//to be sure bbl_OnButton_Play() thinks it is was running
  bbl_OnButton_Play ();		// Simulate a user button push to enter paused state

  //now make schedule and everything else go back to begining
  bb->Reset ();

  bbl_UpdateGUI_Info ();
  gtk_label_set_text (LabelProgramStatus, "Program Stopped");
}


//======================================
//called whenever opening/re-initing a user file for data
void bbl_UpdateGUI_UserDataInfo ()
{
//start menu checks init
  gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM
				  (lookup_widget
				   (GTK_WIDGET (window1), "stereo_noise1")),
				  bb->StereoNoiz);

  gtk_check_menu_item_set_active (GTK_CHECK_MENU_ITEM
				  (lookup_widget
				   (GTK_WIDGET (window1),
				    "infinite_loop_mode1")),
				  ((bb->loops == 0) ? true : false));
  //end Menu checks init


  //start checkboxes init
  //TODO
//end checkboxes init


  //start sliders init
  //start manual beat frequency slider:     
  gtk_range_set_value (RangeVolumeNoise, bb->Volume_Noiz * 100);
  gtk_range_set_value (RangeVolumeTone,
		       bb->Volume_Tone * 100.0 / BB_VOLUMETONERANGE);
  gtk_range_set_value (RangeFreqBeat, bb->ManualFreqLOffset * 2);
  gtk_range_set_value (RangeFreqBase, bb->FreqBase);
  //end sliders init
}



//======================================
//called during inits mostly, since in GTK slider text keeps track of itself
void
bbl_UpdateGUI_ManualControlInfo ()
{
  sprintf (gnaural_tmpstr, LABEL_FLOATPRECISION, bb->FreqBase);
  gtk_label_set_text (LabelFreqBase, gnaural_tmpstr);
}


//======================================
//This is pretty minimal -- mostly about resetting NEWENTRY so GUI gets message to update
void
bbl_UpdateGUI_Info ()
{
  bbl_UpdateGUI_LoopInfo ();

  //now update GUI to reflect reset state :
  if (gnaural_pauseflag != 0)
    {
      gtk_label_set_text (LabelProgramStatus, "Program paused");
    }

  if (gnaural_writefilename != NULL)
    {
      strcat (gnaural_tmpstr, " (WRITING TO FILE)");
      gtk_label_set_text (LabelProgramStatus, gnaural_tmpstr);
    }

  bb->InfoFlag |= BB_NEWENTRY;
}


//======================================
void
bbl_mainCleanup ()
{
 bbl_SoundCleanup();
 
 if (bb != NULL)
    {
      bb->InfoFlag |= BB_COMPLETED;
      bb->WriteStop = TRUE;
     int itmp=0;
      while (gnaural_writefilename!=NULL) {
       g_usleep (1000);//wait for sound thread to die peacefully 
//NOTE: if user puts -sw filename.wav at cmdline, for example, this will indeed set 
//gnaural_filename (even though Gnaural exits after -s).  But breaking the loop
//in the manner below should be harmless.
       if (++itmp>3) 
        {
         DBGOUT("Forcibly freeing filename memory");
         delete gnaural_writefilename;
         gnaural_writefilename = NULL;
       }
      }
      DBGOUT ("Freeing memory for objects");
      delete bb;
     bb=NULL;
    }

  DBGOUT ("Done.");
}




//======================================
void
bbl_OnButton_Play ()
{
  if (gnaural_writefilename!=NULL) return;//disable Play button if we're writing an audio file.

  gnaural_pauseflag = ~gnaural_pauseflag;
  if (gnaural_pauseflag == 0)		// i.e., it was already paused:
    {
     //continue playing sound:
   if (gnaural_pa_stream!=NULL) Pa_StartStream( gnaural_pa_stream );
     
      //Following is  so I don't erase good stuff on screen until full UpdateLoopInforestart:
      if (((bb->InfoFlag) & BB_COMPLETED) != 0)
	{
	  bb->InfoFlag &= ~BB_COMPLETED;
	  bb->TotalSampleCountLooped = 0;
	}
      //SetDlgItemText(IDOK, "Pause");
      gtk_label_set_text_with_mnemonic (LabelPlay_Pause, "_Pause");
      gtk_image_set_from_stock (GTK_IMAGE
				(lookup_widget
				 (GTK_WIDGET (window1), "imagePlay_Pause")),
				"gtk-media-pause", GTK_ICON_SIZE_BUTTON);
    }
  else//e.g., it was not paused
    {
     //stop playing sound:
   if (gnaural_pa_stream!=NULL) Pa_StopStream( gnaural_pa_stream );
     
      //  SetDlgItemText(IDOK, "Start");
      gtk_label_set_text_with_mnemonic (LabelPlay_Pause, "_Play");
      gtk_label_set_text (LabelProgramStatus, "Paused");
      gtk_image_set_from_stock (GTK_IMAGE
				(lookup_widget
				 (GTK_WIDGET (window1), "imagePlay_Pause")),
				"gtk-media-play", GTK_ICON_SIZE_BUTTON);
    }

  bbl_UpdateGUI_Info ();
}



//======================================
void
bbl_UpdateGUI_LoopInfo ()
{
  if (bb->loops < 1)
    sprintf (gnaural_tmpstr, "Running loop %d of Endless Loop Mode",
	     1 + bb->loops - bb->loopcount);
  else
    sprintf (gnaural_tmpstr, "Running loop %d of %d", 1 + bb->loops - bb->loopcount,
	     bb->loops);

  gtk_label_set_text (LabelProgramStatus, gnaural_tmpstr);
}





//======================================
//all this does is updated the projected runtime; I should eventually change name
//to say "UpdateProjectedRuntime", but for simplicity in porting, I will keep it.
void
bbl_UpdateGUI_LoopModeInfo ()
{
//overall time schedule will run:
  if (bb->loops > 0)
    sprintf (gnaural_tmpstr, "Projected Runtime: %d min. %d sec.",
	     (int) (bb->loops * bb->ScheduleTime) / 60,
	     (int) (bb->loops * bb->ScheduleTime) % 60);
  else
    sprintf (gnaural_tmpstr, "*Inf. Loop Mode*");
  gtk_label_set_text (LabelOverallRuntime, gnaural_tmpstr);
}



//START writethreadstuff======================================
void *
bbl_writethreadWAV (void *ptr)
{
  DBGOUT ("Starting WAV write thread");
  bb->WriteWAVFile (gnaural_writefilename);
  if (gnaural_writefilename != NULL)
    delete gnaural_writefilename;
  gnaural_writefilename = NULL;
  DBGOUT ("WAV write thread exiting");
  //pthread_exit (0);
  g_thread_exit (0); //this should be updated to g_thread_join(GThread) when I get more time
}

//======================================
void
bbl_do_writethreadWAV (char *filename)
{
  if (gnaural_writefilename != NULL)
  {
   bbl_MessageDialogBox("You are already writing one");
    return;			//this means someone is already writing 
  }
  gnaural_writefilename = new char[strlen (filename) + 2];
  strcpy (gnaural_writefilename, filename);
  //Note: you must be in paused state to do writing!
  bbl_UpdateGUI_Info ();

  //create the sound engine thread (it shouldn't need GUI), it heeds only bb variables
//old gthread code (dead 20051126):      
//  if (pthread_create
//      (&gnaural_thread_WriteEngineWAV, NULL, bbl_writethreadWAV, (void *) NULL) != 0)
//    {
//      DBGOUT ("Couldn't create the WAV filewriting  thread.");
//    }
    
//new gthread code (new 20051126):      
 GError           *thread_err = NULL ;
 if( (gnaural_thread_WriteEngineWAV = g_thread_create((GThreadFunc)bbl_writethreadWAV, (void *)NULL, TRUE, &thread_err)) == NULL)
  {
     fprintf(stderr,"g_thread_create failed: %s!!\n", thread_err->message );
     g_error_free ( thread_err ) ;
  }

    
    
}
//END writethreadstuff======================================


//this is the PortAudio version of bbl_soundthread -- used for multi-platform compatibility. Use 
//ALSA version of Gnaural if you are only compiling this for Linux (PortAudio still uses OSS).
//======================================
int bbl_SoundInit()
{
  int size;
  unsigned int val;
  int dir;

//Start PortAudio vars:
#define GNAURAL_SAMPLE_RATE   (44100)
//#define GNAURAL_FRAMES_PER_BUFFER  (64)
#define GNAURAL_FRAMES_PER_BUFFER  ( 4096)
//#define GNAURAL_FRAMES_PER_BUFFER  (16384)
#define GNAURAL_OUTPUT_DEVICE Pa_GetDefaultOutputDeviceID()
    PaError err;
    int data;//this is leftover PA tutorial crap
//end PortAudio vars 
 
    DBGOUT ("Starting PortAudio sound");

    err = Pa_Initialize();
    if( err != paNoError ) {
      sprintf(gnaural_tmpstr, "PortAudio: %s", Pa_GetErrorText( err ) );
      ERROUT(gnaural_tmpstr);
      return EXIT_FAILURE;
    }
     if (gnaural_pa_SoundDevice==GNAURAL_USEDEFAULTSOUNDDEVICE)
     {
       gnaural_pa_SoundDevice=Pa_GetDefaultOutputDeviceID();
     }
    err = Pa_OpenStream(
              &gnaural_pa_stream,
              paNoDevice,
              0,         
              paInt16, // short -- use for inteleaved 32-bit stereo
              NULL,
              gnaural_pa_SoundDevice,
              2,          
              paInt16,// short -- use for inteleaved 32-bit stereo
              NULL,
              GNAURAL_SAMPLE_RATE,
              GNAURAL_FRAMES_PER_BUFFER,
              0,              // number of buffers, if zero then use default minimum
              paClipOff,      // we won't output out of range samples so don't bother clipping them
              patestCallback,
              &data );
    if( err != paNoError ) 
     {
       sprintf(gnaural_tmpstr, "PortAudio: %s", Pa_GetErrorText( err ) );
      ERROUT(gnaural_tmpstr);
       return EXIT_FAILURE;
    }
     return EXIT_SUCCESS;
   }//end bbl_SoundInit
   
   
//======================================
void bbl_SoundCleanup()   
   {    
    if (gnaural_pa_stream!=NULL) {
     DBGOUT("Stopping PortAudio sound");
     Pa_StopStream( gnaural_pa_stream );
     Pa_CloseStream( gnaural_pa_stream );
     Pa_Terminate();
     }
     gnaural_pa_stream=NULL;
    //all done with PortAudio
    }





//======================================
//this does the stuff that requires the GUI and bbl_mainInit() to be initialized first.
void bbl_InitGUI ()
{
  //initialize my speedier local handles for Glade's GUI scaffolding:
  LabelProgramStatus =
    GTK_LABEL (lookup_widget (GTK_WIDGET (window1), "label_ProgramStatus"));
 LabelFilename =
    GTK_LABEL (lookup_widget (GTK_WIDGET (window1), "label_Filename"));
  LabelOverallRuntime =
    GTK_LABEL (lookup_widget (GTK_WIDGET (window1), "label_OverallRuntime"));
  LabelOverallCurrentTime =
    GTK_LABEL (lookup_widget
	       (GTK_WIDGET (window1), "label_OverallCurrentTime"));
  LabelCurEntryNumber =
    GTK_LABEL (lookup_widget (GTK_WIDGET (window1), "label_CurEntryNumber"));
  LabelCurEntryDuration =
    GTK_LABEL (lookup_widget
	       (GTK_WIDGET (window1), "label_CurEntryDuration"));
  LabelCurEntryCountdown =
    GTK_LABEL (lookup_widget
	       (GTK_WIDGET (window1), "label_CurEntryCountdown"));
  LabelCurEntryHzRange =
    GTK_LABEL (lookup_widget (GTK_WIDGET (window1), "label_CurEntryHzRange"));
  LabelCurEntryLeftStartHz =
    GTK_LABEL (lookup_widget
	       (GTK_WIDGET (window1), "label_CurEntryLeftStartHz"));
  LabelCurEntryRightStartHz =
    GTK_LABEL (lookup_widget
	       (GTK_WIDGET (window1), "label_CurEntryRightStartHz"));
  LabelCurEntryRightEndHz =
    GTK_LABEL (lookup_widget
	       (GTK_WIDGET (window1), "label_CurEntryRightEndHz"));
  LabelCurEntryLeftEndHz =
    GTK_LABEL (lookup_widget
	       (GTK_WIDGET (window1), "label_CurEntryLeftEndHz"));
  LabelFreqBeat =
    GTK_LABEL (lookup_widget (GTK_WIDGET (window1), "label_FreqBeat"));
  LabelFreqRight =
    GTK_LABEL (lookup_widget (GTK_WIDGET (window1), "label_FreqRight"));
  LabelFreqBase =
    GTK_LABEL (lookup_widget (GTK_WIDGET (window1), "label_FreqBase"));
  LabelFreqLeft =
    GTK_LABEL (lookup_widget (GTK_WIDGET (window1), "label_FreqLeft"));
  LabelPlay_Pause =
    GTK_LABEL (lookup_widget (GTK_WIDGET (window1), "labelPlay_Pause"));
  ProgressBar_Overall =
    GTK_PROGRESS_BAR (lookup_widget
		      (GTK_WIDGET (window1), "progressbar_OverallProgress"));

  RangeVolumeTone =
    GTK_RANGE (lookup_widget (GTK_WIDGET (window1), "vscale_VolumeTone"));
  RangeVolumeNoise =
    GTK_RANGE (lookup_widget (GTK_WIDGET (window1), "vscale_VolumeNoise"));
  RangeFreqBeat =
    GTK_RANGE (lookup_widget (GTK_WIDGET (window1), "hscale_FreqBeat"));
  RangeFreqBase =
    GTK_RANGE (lookup_widget (GTK_WIDGET (window1), "hscale_FreqBase"));

 StatusBar_main = GTK_STATUSBAR (lookup_widget (GTK_WIDGET (window1), "statusbar1"));

  //InitGUI stuff:
  bbl_UpdateGUI_UserDataInfo ();
  bbl_UpdateGUI_Info ();
  bbl_UpdateGUI_ManualControlInfo ();	//just to fill in the zeroes
  bbl_UpdateGUI_LoopModeInfo ();
  
//Gray (disable) various menuitems:
 GtkWidget * menuitem;
 menuitem=lookup_widget (GTK_WIDGET (window1), "undo1");
 gtk_widget_set_sensitive ((GtkWidget *)menuitem, false);
 menuitem=lookup_widget (GTK_WIDGET (window1), "redo1");
 gtk_widget_set_sensitive ((GtkWidget *)menuitem, false);
 menuitem=lookup_widget (GTK_WIDGET (window1), "disable_screensaver1");
 gtk_widget_set_sensitive ((GtkWidget *)menuitem, false);
  

}


//======================================
void
bbl_mainInit ()
{
  //take care of command-line possibilites first:
  //trap Ctrl-C:
  if (signal (SIGINT, SIG_IGN) != SIG_IGN)
    signal (SIGINT, bbl_InterceptCtrl_C);

  //next need to set up paths, since bb (below) will need to know 
  //(among other things) where to access/create gnaural_schedule.txt file:
  bbl_SetupPathsAndFiles ();

  
  //Now that I have paths set, I can create the big object:
  DBGOUT("Allotting memory for objects");
  bb = new CBinauralBeat (szFile_gnaural_schedule_txt);
  if (bb == NULL)
    {
      DBGOUT ("Failed to create resources. Exiting.");
      exit (EXIT_FAILURE);
    }

  gnaural_FreqBase_old = bb->FreqBase;
  (bb->InfoFlag) = 0;
    
}


//======================================
void  bbl_UpdateGUI_Progressbar()
{
	  if (bb->ScheduleTime > 0 && bb->loops != 0)
	    {
	      float i = ((bb->TotalSampleCount +
			  bb->TotalSampleCountLooped) /
			 (float) (bb->ScheduleTime * 100));
	      if (i > 1)		i = 1;
	      else if (i < 0)		i = 0;
	      gtk_progress_bar_set_fraction (ProgressBar_Overall, i);
	    }
	  else
	    {
	      gtk_progress_bar_set_fraction (ProgressBar_Overall, 0);
	    } 
}



//======================================
//QUESTIONABLE BUG FIXES:
//-BUG: figure out why resetting bb->InfoFlag's caused update labels timer to quit.
//            Wow, this one was a bit as if the garage door opened every time I pulled
//            my hair. Bizarre. One of the wierdest bugs I've ever seen.
// SOLUTION: made BinauralBeat's InfoFlag an unsigned int (from int), made
//            explicit that the timer func returned TRUE, and rephrased bit setting line
//             back to what it had been when the bug first appeared (?!).
//            Hard to believe this really fixed it, but it no longer does the bad behavior. (20051020)
bool
bbl_UpdateGUI_Labels ()
{
//-----------------------------------------------------------
//START stuff that gets updated EVERY second:
//
// current time point within schedule:
  sprintf (gnaural_tmpstr, "Progress: %d min. %d sec.",
	   (bb->TotalSampleCount + bb->TotalSampleCountLooped) / 6000,
	   ((bb->TotalSampleCount + bb->TotalSampleCountLooped) / 100) % 60);
  gtk_label_set_text (LabelOverallCurrentTime, gnaural_tmpstr);

  //remaining time in entry:
  sprintf (gnaural_tmpstr, "Time left: %d sec.",
	   ((bb->Schedule[bb->ScheduleCount].AbsoluteEndTime_100 -
	     bb->TotalSampleCount) / 100));
  gtk_label_set_text (LabelCurEntryCountdown, gnaural_tmpstr);

  //Left frequency:
  sprintf (gnaural_tmpstr, LABEL_FLOATPRECISION, bb->FreqL);
  gtk_label_set_text (LabelFreqLeft, gnaural_tmpstr);

  //Right frequency:
  sprintf (gnaural_tmpstr, LABEL_FLOATPRECISION, bb->FreqR);
  gtk_label_set_text (LabelFreqRight, gnaural_tmpstr);

  //Beat frequency:
  sprintf (gnaural_tmpstr, LABEL_FLOATPRECISION, fabs(bb->FreqL - bb->FreqR)); //Added fabs() 20060115
  gtk_label_set_text (LabelFreqBeat, gnaural_tmpstr);

	  //Progressbar:
  bbl_UpdateGUI_Progressbar();

  //
  //END stuff that gets updated EVERY second:
  //-----------------------------------------------------------------------

  //-----------------------------------------------------------------------
//START check bb->InfoFlag
//now update things that BinauralBeat says are changed:
  if (bb->InfoFlag != 0)
    {

      //.........................
      //if schedule is done:
      if (((bb->InfoFlag) & BB_COMPLETED) != 0)
	{
//  bbl_OnButton_Stop ();
//        gtk_label_set_text (LabelProgramStatus, "Schedule Completed");
	  gnaural_pauseflag = 0;	//so bbl_OnButton_Play() thinks I was running
	  bbl_OnButton_Play ();	// I'm simulating user button push to pause
	  gtk_label_set_text (LabelProgramStatus, "Schedule Completed");
	  bb->loopcount = bb->loops;
	  gtk_progress_bar_set_fraction (ProgressBar_Overall, 1.0);

	}

      //.........................
      //if starting a new loop:
      if (((bb->InfoFlag) & BB_NEWLOOP) != 0)
	{
	  //reset the "new loop" bit of InfoFlag:
	  bb->InfoFlag &= ~BB_NEWLOOP;
	  bbl_UpdateGUI_LoopInfo ();
	}

      //.........................
      //START things that get updated every new entry:
      //if new entry
      if (((bb->InfoFlag) & BB_NEWENTRY) != 0)
	{
	  //reset the "new entry" bit of InfoFlag:
	  (bb->InfoFlag) &= (~BB_NEWENTRY);

	  //Progressbar:
//  bbl_UpdateGUI_Progressbar();
  
  //graph position indicator:
  if (ScheduleGUI_ProgressIndicatorFlag==TRUE)
  {
   GtkWidget * widget= lookup_widget (GTK_WIDGET (window1), "drawingarea1");
 //  ScheduleGUI_ConvertDurHzToXY(widget);
  	ScheduleGUI_DrawGraph(widget);
   gtk_widget_queue_draw_area (widget,0,0,widget->allocation.width, widget->allocation.height);
  }


	  //current entry number:
	  sprintf (gnaural_tmpstr, "%d of %d",
		   bb->ScheduleCount + 1, bb->ScheduleEntriesCount);
	  gtk_label_set_text (LabelCurEntryNumber, gnaural_tmpstr);

	  //total time in entry: 
	  sprintf (gnaural_tmpstr, "Duration: %g sec.",
		   bb->Schedule[bb->ScheduleCount].Duration);
	  gtk_label_set_text (LabelCurEntryDuration, gnaural_tmpstr);

//sprintf(gnaural_tmpstr,"%d",bb->Schedule[bb->ScheduleCount].AbsoluteEndTime_100/100);
//SetDlgItemText(lEndCount, gnaural_tmpstr);

	  sprintf (gnaural_tmpstr, "Beat Range: %g to %g Hz ",
		  fabs( bb->Schedule[bb->ScheduleCount].FreqLStart -
		   bb->Schedule[bb->ScheduleCount].FreqRStart),
		   fabs(bb->Schedule[bb->ScheduleCount].FreqLEnd -
		   bb->Schedule[bb->ScheduleCount].FreqREnd));
	  gtk_label_set_text (LabelCurEntryHzRange, gnaural_tmpstr);

//left
	  sprintf (gnaural_tmpstr, LABEL_FLOATPRECISION,
		   bb->Schedule[bb->ScheduleCount].FreqLStart);
	  gtk_label_set_text (LabelCurEntryLeftStartHz, gnaural_tmpstr);
	  sprintf (gnaural_tmpstr, LABEL_FLOATPRECISION,
		   bb->Schedule[bb->ScheduleCount].FreqLEnd);
	  gtk_label_set_text (LabelCurEntryLeftEndHz, gnaural_tmpstr);

//right
	  sprintf (gnaural_tmpstr, LABEL_FLOATPRECISION,
		   bb->Schedule[bb->ScheduleCount].FreqRStart);
	  gtk_label_set_text (LabelCurEntryRightStartHz, gnaural_tmpstr);
	  sprintf (gnaural_tmpstr, LABEL_FLOATPRECISION,
		   bb->Schedule[bb->ScheduleCount].FreqREnd);
	  gtk_label_set_text (LabelCurEntryRightEndHz, gnaural_tmpstr);
	}
      //END things that get updated every new entry


    }				//END check bb->InfoFlag
  //-------------------------------------------------------------
  return TRUE;			//returning FALSE quits the timer
}


//======================================
//this function only exists to support bbl_AboutDialogBox below
//#include <gnome.h>
static void
bbl_activate_url (GtkAboutDialog * about, const gchar * url, gpointer data)
{
#ifndef GNAURAL_WIN32
 //##########FOR LINUX##########
 //don't have anything yet for this
  //gnome_url_show (url, NULL);
  sprintf (gnaural_tmpstr, "mozilla %s &", url);
  system (gnaural_tmpstr);
// system("mozilla http://gnaural.sourceforge.net/ &");
#endif

 
#ifdef GNAURAL_WIN32
 //##########FOR WIN32##########
 ShellExecute(NULL,
  "open", 
  "http://gnaural.sourceforge.net", 
  "", 
  "c:\\", 
  SW_SHOWNORMAL);
 #endif
}




//======================================
void
bbl_AboutDialogBox ()
{
  static GtkWidget *about = NULL;

//  GdkPixbuf *logo = NULL;

  const gchar *authors[] = {
    "Bret Logan <gnaural@users.sourceforge.net>",
    NULL
  };

  const gchar *documenters[] = {
    "Bret Logan <gnaural@users.sourceforge.net>",
    NULL
  };

  const gchar *copyright = "Copyright \xc2\xa9 2003-2006 Bret Logan\n";
  
  const gchar *comments =
    _("A programmable audio generator intended as an aural aid to meditation and relaxation, implementing the binaural beat principle as described in Gerald Oster's Oct. 1973 Scientific American article \"Auditory Beats in the Brain.\"");
  
  const gchar *license =
"This program is free software; you can redistribute it and/or modify\n\
it under the terms of the GNU General Public License as published by\n\
the Free Software Foundation; either version 2 of the License, or\n\
(at your option) any later version.\n\n\
This program is distributed in the hope that it will be useful,\n\
but WITHOUT ANY WARRANTY; without even the implied warranty of\n\
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  For more\n\
details on the GNU General Public License, contact:\n\n\
\tFree Software Foundation, Inc.\n\
\t59 Temple Place - Suite 330\n\
\tBoston, MA 02111-1307, USA.";
  
  
  
  /* Translators: This is a special message that shouldn't be translated
     literally. It is used in the about box to give credits to
     the translators.
     Thus, you should translate it to your name and email address.
     You can also include other translators who have contributed to
     this translation; in that case, please write them on separate
     lines seperated by newlines (\n). */
  const gchar *translator_credits = _("translator-credits");


  if (about != NULL)
    {
      gtk_window_set_transient_for (GTK_WINDOW (about), GTK_WINDOW (window1));
      gtk_window_present (GTK_WINDOW (about));

      return;
    }

 /*
{
  gchar * tmpch="/usr/share/gnaural/pixmaps/gnaural-logo.png";
  if (0==bbl_TestPathOrFileExistence(tmpch))
    logo = gdk_pixbuf_new_from_file (tmpch, NULL);
}

    #ifndef GNAURAL_WIN32
    //############FOR LINUX###########
    //This is where Debian's deb maker ('dh_make' then 'dpkg-buildpackage -rfakeroot') puts the pixmaps, even though permissions cause problems:
    logo = gdk_pixbuf_new_from_file ("/usr/share/gnaural/pixmaps/gnaural-logo.png", NULL);
    //this is where "make install" puts it:
//   if (logo==NULL)  logo = gdk_pixbuf_new_from_file ("/usr/local/share/gnaural/pixmaps/gnaural-logo.png",NULL);
    //since this is kind of arbitrary, try a few more spots:
//   if (logo==NULL)  logo = gdk_pixbuf_new_from_file ("~/.gnaural/gnaural-logo.png", NULL);
//   if (logo==NULL)  logo = gdk_pixbuf_new_from_file ("pixmaps/gnaural-logo.png", NULL);
//   if (logo==NULL)  logo = gdk_pixbuf_new_from_file ("../pixmaps/gnaural-logo.png", NULL);
    #endif

    #ifdef GNAURAL_WIN32
    //############FOR WIN32###########
    //currently, nothing 20051201
//    logo = NULL;
//    logo = gnome_db_stock_get_icon_pixbuf (GTK_STOCK_DIALOG_INFO);
//    logo = gdk_pixbuf_new_from_file (GTK_STOCK_DIALOG_INFO, NULL);
//      gtk_window_set_default_icon_name (GTK_STOCK_DIALOG_INFO); //#define GTK_STOCK_DIALOG_INFO      "gtk-dialog-info"
//  gtk_window_set_icon_name (window1,GTK_STOCK_DIALOG_INFO); //#define GTK_STOCK_DIALOG_INFO      "gtk-dialog-info"
    #endif
*/
    
  gtk_about_dialog_set_url_hook (bbl_activate_url, NULL, NULL);

  about = GTK_WIDGET (g_object_new (GTK_TYPE_ABOUT_DIALOG,
				    "name", _("Gnaural"),
				    "version", VERSION,
				    "copyright", copyright,
				    "comments", comments,
				    "website",  "http://gnaural.sourceforge.net/",
				    "authors", authors, 
        "documenters", documenters, 
        "translator_credits",translator_credits, 
        "logo", NULL,  //this makes it use the default icon set up in main()
        "license", license,  
//        "wrap-license", FALSE,
        NULL)); 


  gtk_window_set_destroy_with_parent (GTK_WINDOW (about), TRUE);

  g_signal_connect (about, "response", G_CALLBACK (gtk_widget_destroy), NULL);
  g_signal_connect (about, "destroy", G_CALLBACK (gtk_widget_destroyed),
		    &about);

  gtk_window_set_transient_for (GTK_WINDOW (about), GTK_WINDOW (window1));
  gtk_window_present (GTK_WINDOW (about));

//  if (logo)    g_object_unref (logo);
}


//======================================
/*
GTK_MESSAGE_INFO	Informational message
GTK_MESSAGE_WARNING	Nonfatal warning message
GTK_MESSAGE_QUESTION	Question requiring a choice
GTK_MESSAGE_ERROR	Fatal error message

GTK_BUTTONS_NONE	no buttons at all
GTK_BUTTONS_OK	an OK button
GTK_BUTTONS_CLOSE	a Close button
GTK_BUTTONS_CANCEL	a Cancel button
GTK_BUTTONS_YES_NO	Yes and No buttons
GTK_BUTTONS_OK_CANCEL	OK and Cancel buttons

  GTK_RESPONSE_NONE = -1,//no response_id or the dialog was programmatically hidden or destroyed.
  GTK_RESPONSE_REJECT = -2, //user designated
  GTK_RESPONSE_ACCEPT = -3,//user designated
  GTK_RESPONSE_DELETE_EVENT = -4, //if dialog is deleted
  GTK_RESPONSE_OK     = -5,
  GTK_RESPONSE_CANCEL = -6,
  GTK_RESPONSE_CLOSE  = -7,
  GTK_RESPONSE_YES    = -8,
  GTK_RESPONSE_NO     = -9,
  GTK_RESPONSE_APPLY  = -10,
  GTK_RESPONSE_HELP   = -11
*/
//Basically, a way to get a quick "yes or no" from user, or just present some info:
bool
bbl_MessageDialogBox (char *msg, GtkMessageType messagetype,
		      GtkButtonsType buttonformation)
{
  GtkWidget *dialog = gtk_message_dialog_new (GTK_WINDOW (window1),
					      GTK_DIALOG_DESTROY_WITH_PARENT,
					      messagetype,
					      buttonformation,
					      msg);


//  "Error loading file '%s': %s",  gnaural_tmpstr, g_strerror (errno));
  gint res = gtk_dialog_run (GTK_DIALOG (dialog));
  gtk_widget_destroy (dialog);

  if (GTK_RESPONSE_OK == res
      || GTK_RESPONSE_YES == res
      || GTK_RESPONSE_APPLY == res || GTK_RESPONSE_ACCEPT == res)
    {
      return true;
    }
  else
    return false;
}
