/*
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
///////////////////////////////////////////////////////////////////////////////
/*
BinauralBeat.cpp: implementation of the CBinauralBeat class.
    Author: Bret Logan
    Start Date: Nov. 3, 2005
    Current Edit: Feb. 28, 2007
FUNCTION: 
    CBinauralBeat outputs stereo sound data iin 44.1khz PCM (.WAV-style) format, 
    coresponding to a Left-channel sine wave mixed with pink noise, and a Right-channel 
    sine wave mixed with pink noise. Why? Because we love you.
HOW TO USE:
    Create a CBinauralBeat object. That'll run InitBinauralBeat(), which in turn will
    check to see if you have have a gnaural_schedule.txt (the class' default name)
    file. If you do, it will be used, if you don't, a default internal one will used and
    also written to a text file calle gnauralGNAURAL_WIN32uture editing.
 If you just want to create a .WAV file and be done with it, call WriteWAVFile(char *szFilename). 
 WriteWAVFileToSTDOUT does basically the same thing, but to stdout -- so you can pipe the 
 output through another program (like LAME, for instance). Example: 
       gnaural | lame - MyMeditationFile.mp3
 But if you want to send the output directly to a sound server, pass a 
 buffer to UserSoundProc() --  a simple int array is OK, ready for Gnaural's 16-bit stereo 
 format -- left-channel short int/right-channel short int.
 UserSoundProc() acts like a "friend on the inside" that can contact  
 MainLoop (void *pSoundBuffer, long bufferLen). MainLoop does the filling,
 so you can in turn feed whatever sound server you use.
 
NOTES: 
 20070226 added a whole bunch of junk to allow BB to handle a XML schedule file
  format. Enough complaints about the old format's non-dot decimal-point bugs
  required that something be attempted; ultimately, it would have been nice to pull
  ALL file i/o code out of here, but the complexity and potential loss of time-proven
  stability finally convinced me the only practical way to solve this (esp. given the
  soon obsolescense of this file to Gnaural2's BinauralBeat.c) was to put even MORE 
  file i/o stuff in here. 
*/
///////////////////////////////////////////////////////////////////////////////
#include <ctype.h>
#include <time.h>
#include <glib.h>               //ONLY needed for XML support...
#include "ScheduleXML.h"
#include "BinauralBeat.h"

#define BB_DBGOUT(a)            //  fprintf(stderr,"BB: %s\n",a)
#define BB_DBGOUT_STR(a,b)      //  fprintf(stderr,"BB: %s %s\n",a, b);
#define BB_DBGOUT_INT(a,b)      //  fprintf(stderr,"BB: %s %d\n",a, b);
#define BB_DBGOUT_FLT(a,b)      //  fprintf(stderr,"BB: %s %g\n",a, b);
#define BB_DBGOUT_PNT(a,b)      //  fprintf(stderr,"BB: %s %p\n",a,b);
#define BB_ERROUT(a)   fprintf(stderr,"BB: #Error# %s\n",a)

//////////////////////////////////////////////////////////////////////
// Construction/Destruction
//////////////////////////////////////////////////////////////////////
//======================================
CBinauralBeat::CBinauralBeat (char *sched_filename)
{
 SchedBuff = DefaultSchedBuff;  //just so it has something to run if no file is found; SchedFilenameToSchedule() will fill SchedBuff otherwise
//  SchedFilename = DefaultSchedFilename;//default name for user's schedule file:
 SchedFilename = sched_filename;        //sched_filename defaults to DefaultSchedFilename in constructor if nothing is provided
 InitBinauralBeat ();
}

//======================================
void CBinauralBeat::InitBinauralBeat ()
{
 SeedRand (3676, 2676862);      //anything but 0 on either or twins seems OK
 me = this;
 Schedule = NULL;
 SetDefaultValues ();
 InfoFlag = 0;

 //these globals are needed only for XML file reading:
 BB_ParserXML_voicecount = 0;
 BB_ParserXML_curvoice = 0;
 BB_ParserXML_curentry = 0;

 //see if there is a file, if not use default schedule and write a default file:
 if (0 != SchedFilenameToSchedule ())
 {
//  SchedBuffToSchedule (SchedBuff);
  WriteDefaultScheduleToFile ();
 }
}                               //end InitBinauralBeat()

//======================================
void CBinauralBeat::SetDefaultValues ()
{
 StereoNoiz = 1;
 StereoSwap = 0;
 VolumeOverall_left = VolumeOverall_right = 1.0;
 FreqBase = 140.f;
 Volume_Tone = .85f;            //20070227 Volume_Tone range changed from 0 - BB_VOLUMETONERANGE to 0.0 - 1.0 
 Volume_Noiz = .2f;
 FileFlag = false;
 ManualFreqOffsetControl = false;
 WriteStop = false;
 ManualFreqLOffset = ManualFreqROffset = 0.f;
 loops = loopcount = 1;
 TotalSampleCount = TotalSampleCountLooped = 0;
}

//======================================
CBinauralBeat::~CBinauralBeat ()
{
 if (Schedule != NULL)
 {
  delete[]CBinauralBeat::Schedule;
 }
}

//======================================
//################################################
//START PSEUDORANDOM NUMBER GENERATOR SECTION
//Make a choice.
//=============================================
// The McGill Super-Duper Random Number Generator
// G. Marsaglia, K. Ananthanarayana, N. Paul
// Incorporating the Ziggurat method of sampling from decreasing
// or symmetric unimodal density functions.
// G. Marsaglia, W.W. Tsang
// Rewritten into C by E. Schneider
// [Date last modified: 05-Jul-1997]
//=============================================
#define ML_MULT 69069L
//Need to also define these in .h file:
//unsigned long mcgn, srgn;
//IMPORTANT NOTE:when using a fixed i2, for some reason Seed pairs
//for i1 like this:
// even
// even+1
//produce idential sequences when r2 returned (r1 >> 12).
//Practically, this means that 2 and 3
//produce one landscape; likewise 6 and 7, 100 and 101, etc.
//This is why i do the dopey "add 1" to i2
//ALSO, JUST DON'T USE 0 FOR i1 or i2. PLEASE
void CBinauralBeat::SeedRand (unsigned int i1, unsigned int i2)
//void rstart (long i1, long i2)
{
 if (i2 == (unsigned int) -1)
  i2 = i1 + 1;                  //yech
 mcgn = (unsigned long) ((i1 == 0L) ? 0L : i1 | 1L);
 srgn = (unsigned long) ((i2 == 0L) ? 0L : (i2 & 0x7FFL) | 1L);
}

//======================================
//returns int from -2^31 to +2^31; "PM" means "PlusMinus"
int CBinauralBeat::rand ()
{
 unsigned long r0, r1;
 r0 = (srgn >> 15);
 r1 = srgn ^ r0;
 r0 = (r1 << 17);
 srgn = r0 ^ r1;
 mcgn = ML_MULT * mcgn;
 r1 = mcgn ^ srgn;
 return r1;
}

/*
//THERE WAS NOTHING SONICALLY WRONG WITH THE
//FOLLOWING HOMEMADE PRNG FUNCTIONS
//And they don't have the problem with low or close seeds causing
//paterned output (unlike the one above). But they supposedly repeat sooner
//than the one above (haven't tested, though). Not sure if that makes them
//worse. I just like the idea of not repeating often, because 
//it is possible that the user can, with enough exposure, start to hear the
//repetition on very short cycle loops.
//
//returns a signed integer noise value:
int CBinauralBeat::rand() {
static int Seed=9182736;//this is just here for uncommenting-conveinence; real seedable one is in BinauralBeat.h
 //to make any of the following noise generators positive-
 //only returns, simply add &0x7FFFFFFF to the end to negate the 
 //sign.
 //all of these make perfectly acceptable audio noise (my criteria: "no discernable patterns"):
 //return (Seed=(Seed<<3)-Seed+11);
 //return Seed=(201*Seed+11);
 //return ++Seed*=3;
 //new formula, related to MTwister:
 ++Seed ^= (Seed >> 11);
 Seed ^= (Seed <<  7) & 0x9D2C5680U;
 return Seed ^= (Seed << 15) & 0xEFC60000U;
 
 //curiosities- these are amazing sequences, but
 //not very random sounding!:
 //pulsar:
 //return (Seed^=++Seed<<3);
 //machine shop:
 //return (Seed^=++Seed<<1);
}
*/
//END PSEUDORANDOM NUMBER GENERATOR SECTION
//################################################


//======================================
 CBinauralBeat * CBinauralBeat::me;


//======================================
void CBinauralBeat::UserSoundProc (void *pSoundBuffer, long bufferLenInBytes)
{
 //Saves me from indirecting every goddamn variable in the processing!
 //NOTE: Important to remember that bufferLenInBytes is literally the size of
 //the buffer you give UserSoundProc in bytes -- a common error is to send it
 //whatever the size of the buffer is in its native units (int, short, etc.)
 me->MainLoop (pSoundBuffer, bufferLenInBytes);
}

//======================================
//////////////////////////////////////////////////
//UserSoundProc(void *pSoundBuffer,long bufferLen)
//use the following to fill your sound buffers.
//it will automatically keep
//time, by using sample counts instead of a
//real-time clock. The
//basis for time is 0.01 seconds (1/100th of
//a second frequency updates)
void CBinauralBeat::MainLoop (void *pSoundBuffer, long bufferLen)
{
 //need these statics as bookmarks for where I am in the sound in each call:
 static BB_PRECISION_TYPE sinPosL = 0.f;
 static BB_PRECISION_TYPE sinPosR = 0.f;
 static int noizval1 = 1;
 static int noizval2 = 1;
 static int samplerezvar = 1;   //critical: initialize as 1
 static int oldj = -1;          //I think I have a bug concerning this variable
 BB_PRECISION_TYPE factor;
 BB_PRECISION_TYPE noiz1;
 BB_PRECISION_TYPE noiz2;
//20060331: next two were added to remove extra multiplications from speed loop:
 static BB_PRECISION_TYPE factoredFreqL = 0;
 static BB_PRECISION_TYPE factoredFreqR = 0;
// Convert params, assuming we create a 16bits, mono waveform.
 signed short *pSample = (signed short *) pSoundBuffer;
//Generally speaking, this is what I am doing:
//long    nbSample = bufferLen / (sizeof(int));
//But since I always know bufferlen in in chars, I just divide by four:
 long nbSample = bufferLen >> 2;

 //now deal with overall volume and stereo balance:
 int volTone_left =
  (int) (Volume_Tone * VolumeOverall_left * BB_VOLUMETONERANGE);
 int volTone_right =
  (int) (Volume_Tone * VolumeOverall_right * BB_VOLUMETONERANGE);
 BB_PRECISION_TYPE volNoise_left = (Volume_Noiz * VolumeOverall_left);
 BB_PRECISION_TYPE volNoise_right = (Volume_Noiz * VolumeOverall_right);
//-------------------------------------------
// START Fill sound buffer
//for every sample in pSoundBuffer I need to fill, do this:
 for (int i = 0; i < nbSample; i++)
 {
  //START 1/100th sec. updates
  //Philosophy: I update the frequency to match user's schedule directives
  //100 times a second, literally by doing this routine every 441 samples (i.e.,
  //that's once a second when you take 44.1k samples per second).
  //I also increment TotalSampleCount every 100th of a sec. in order to
  //use it as an index as to where we are in the schedule.
  if ((--samplerezvar) == 0)
  {
   samplerezvar = BB_INTEGRATION_REZ;   //BB_INTEGRATION_REZ should always equal 441
   if (!ManualFreqOffsetControl)
   {
    TotalSampleCount++;

    //START which entry loop
    //Now figure out which ScheduleCount I'm at:
    int j;
    for (j = 0; j < ScheduleEntriesCount; j++)
    {
     //asking "is total samples less than sched entry's endtime?"
     if (TotalSampleCount < Schedule[j].AbsoluteEndTime_100)
     {
      //answer: it is, therefore I know the ScheduleCount:
      ScheduleCount = j;

      //if this is not the old entry, let the world know one time:
      if (j != oldj)
      {
       InfoFlag |= BB_NEWENTRY;
       oldj = j;
      }

      //by dividing exact point in period by total period time, I have a factor
      //by which to calculate proper frequency:
      factor =
       (TotalSampleCount -
        Schedule[ScheduleCount].PrevAbsoluteEndTime_100) /
       (Schedule[ScheduleCount].Duration * 100.f);

      //Left Freq is equal to frequency spread from Left Start to Left End (next
      //schedule's Left Start) multiplied by above factor. Then add FreqBase.
      FreqL =
       (Schedule[ScheduleCount].FreqLSpread * factor) + FreqBase +
       Schedule[ScheduleCount].FreqLStart;
      factoredFreqL = FreqL * BB_SAMPLE_FACTOR;
      FreqR =
       (Schedule[ScheduleCount].FreqRSpread * factor) + FreqBase +
       Schedule[ScheduleCount].FreqRStart;
      factoredFreqR = FreqR * BB_SAMPLE_FACTOR;
      break;                    //exit the "figuring which entry" loop
     }
    }
//END which entry loop

    //following tests to see if Schedule was done:
    if (j >= ScheduleEntriesCount)
    {
     InfoFlag |= BB_NEWLOOP;
     TotalSampleCountLooped += TotalSampleCount;
     ScheduleCount = TotalSampleCount = 0;      //repeat is default for now
     if (--loopcount == 0)
     {                          //end this sucker
      InfoFlag |= BB_COMPLETED;
     }
    }
   }                            //end !ManualFreqOffsetControl
   else
   {                            //Start ManualFreqOffsetControl
    //it is manual control now:
    FreqL = ManualFreqLOffset + FreqBase;
    factoredFreqL = FreqL * BB_SAMPLE_FACTOR;   //SAMPLE_FACTOR=1 over sample rate
    FreqR = ManualFreqROffset + FreqBase;
    factoredFreqR = FreqR * BB_SAMPLE_FACTOR;
   }
  }
//END 1/100th sec. updates

//Now use the info taken from the 1/100th sec. update to actually make the sound:
//First, pink some noise:
  noiz1 = noizval1 = ((noizval1 * 31) + (rand () >> 15)) >> 5;
  if (0 == StereoNoiz)
  {
   noiz2 = noiz1;
  }
  else
  {
   noiz2 = noizval2 = ((noizval2 * 31) + (rand () >> 15)) >> 5;
  }
  noiz1 *= volNoise_left;
  noiz2 *= volNoise_right;

  //Second, do sine waves:
  sinPosL += factoredFreqL;
  if (sinPosL >= BB_TWO_PI)
  {
   sinPosL -= BB_TWO_PI;
  }
  sinPosR += factoredFreqR;
  if (sinPosR >= BB_TWO_PI)
  {
   sinPosR -= BB_TWO_PI;
  }

  //Finally, the big write:
  if (0 == StereoSwap)
  {
   *pSample++ = (signed short) (volTone_left * sin (sinPosL) + noiz1);
   *pSample++ = (signed short) (volTone_right * sin (sinPosR) + noiz2);
  }
  else
  {
   *pSample++ = (signed short) (volTone_right * sin (sinPosR) + noiz2);
   *pSample++ = (signed short) (volTone_left * sin (sinPosL) + noiz1);
  }
//PS for clarity, I'll add that this should be the equivalent of above:
// *pSample = (signed short) (Volume_Tone * sin (sinPosL) + noiz1); ++pSample;
// *pSample = (signed short) (Volume_Tone * sin (sinPosR) + noiz2); ++pSample;
 }
// END Fill sound buffer
//-------------------------------------------
}

//======================================
//default name of user's schedule; user can change it by pointing
//the actual variable used to open files (BinauralBeat::SchedFilename)
//to some other string:
char CBinauralBeat::DefaultSchedFilename[] = "schedule.gnaural";

//======================================
void CBinauralBeat::next ()
{
 TotalSampleCount = Schedule[ScheduleCount].AbsoluteEndTime_100;
 if (++ScheduleCount >= this->ScheduleEntriesCount)
 {
  ScheduleCount = 0;
 }
}

//======================================
void CBinauralBeat::previous ()
{
 /*
    if (ScheduleCount>0) {
    TotalSampleCount=Schedule[ScheduleCount].AbsoluteEndTime_100-Schedule[ScheduleCount-1].AbsoluteEndTime_100;
    }
    else Reset();
  */
 if ((--ScheduleCount) < 0)
 {
  Reset ();
 }
 else
  TotalSampleCount = Schedule[ScheduleCount].AbsoluteEndTime_100;
}

//======================================
//Allots Schedule AND sets ScheduleEntriesCount.
//Returns 0 on success.
int CBinauralBeat::Schedule_Allot (int entrycount)
{
 ScheduleEntriesCount = entrycount;
 if (Schedule != NULL)
 {
  delete[]Schedule;
 }
 Schedule = new CItinerary[ScheduleEntriesCount];
 if (Schedule == NULL)
 {
  return -1;
 }
 else
 {
  return 0;
 }
}

//======================================
//Precalcs all values implicit in provided numerical 
//schedule data; essential to do after any Schedule change.
void CBinauralBeat::Schedule_CalculateIntervals ()
{
 int listcount = -1;
 int previnlist, nextinlist;
 while ((++listcount) < ScheduleEntriesCount)
 {
  if ((nextinlist = listcount + 1) >= ScheduleEntriesCount)
  {
   nextinlist = 0;
  }
  Schedule[listcount].FreqLEnd = Schedule[nextinlist].FreqLStart;
  Schedule[listcount].FreqREnd = Schedule[nextinlist].FreqRStart;
  Schedule[listcount].FreqLSpread =
   Schedule[listcount].FreqLEnd - Schedule[listcount].FreqLStart;
  Schedule[listcount].FreqRSpread =
   Schedule[listcount].FreqREnd - Schedule[listcount].FreqRStart;
  if ((previnlist = listcount - 1) < 0)
  {
   Schedule[listcount].PrevAbsoluteEndTime_100 = 0;
  }
  else
  {
   Schedule[listcount].PrevAbsoluteEndTime_100 =
    Schedule[previnlist].AbsoluteEndTime_100;
  }
 }
}

//======================================
void CBinauralBeat::Mute (bool act)
{
 static BB_PRECISION_TYPE oldnoiz = Volume_Noiz;
 static BB_PRECISION_TYPE oldtone = Volume_Tone;
 if (act)
 {
  oldnoiz = Volume_Noiz;
  oldtone = Volume_Tone;
  Volume_Noiz = 0.f;
  Volume_Tone = 0.f;
 }
 else
 {
  Volume_Noiz = oldnoiz;
  Volume_Tone = oldtone;
 }
}

//======================================
void CBinauralBeat::WAVheader_fill ()
{
 wavh.ckID[0] = 'R';
 wavh.ckID[1] = 'I';
 wavh.ckID[2] = 'F';
 wavh.ckID[3] = 'F';
 wavh.ckSize = FileByteCount + sizeof (WAVheader) - 8;
 wavh.wave_ckID[0] = 'W';
 wavh.wave_ckID[1] = 'A';
 wavh.wave_ckID[2] = 'V';
 wavh.wave_ckID[3] = 'E';
 wavh.fmt_ckID[0] = 'f';
 wavh.fmt_ckID[1] = 'm';
 wavh.fmt_ckID[2] = 't';
 wavh.fmt_ckID[3] = ' ';
 wavh.fmt_ckSize = 16;
 wavh.formatTag = 1;
 wavh.nChannels = 2;
 wavh.nSamplesPerSec = 44100;
 wavh.nAvgBytesPerSec = 176400; //nAvgBytesPerSec= nSamplesPerSec* nBitsPerSample/8* nChannels
 wavh.nBlockAlign = 4;
 wavh.nBitsPerSample = 16;
 wavh.data_ckID[0] = 'd';
 wavh.data_ckID[1] = 'a';
 wavh.data_ckID[2] = 't';
 wavh.data_ckID[3] = 'a';
 wavh.data_ckSize = FileByteCount;      //arbitrary for now, fill in right later
}

//======================================
// I need to add two size parameters when I am done- in the simplified case of 44100
// 2 channel, I can basically sum the size of WAVheader in to total data size, then
// subtract 8 (god knows why - RIFF is insane)
bool CBinauralBeat::WriteWAVFile (char *szFilename)
{
 FILE *stream;
 if ((stream = fopen (szFilename, "wb")) == NULL)
 {
  return false;
 }
 WriteWAVFileToStream (stream);
//now that data has been written, write those two damn header values correctly this time:
 rewind (stream);
 wavh.data_ckSize = FileByteCount;
 wavh.ckSize = FileByteCount + sizeof (WAVheader) - 8;
 fwrite (&wavh, sizeof (WAVheader), 1, stream);
 fclose (stream);
}

//======================================
bool CBinauralBeat::WriteWAVFileToSTDOUT ()
{
//Apparently, Win32 sets stdout to text mode, while in Linux is binary. And yet,
//Linux is easy to change, while Win32 proprietary/wierd to change. I think to change
//Win32, do something like this: setmode(fileno(stdout), O_BINARY);
//A more general solution to make Linux happy:
//http://www.contrib.andrew.cmu.edu/~twm/stuff/binary-redirect.html
 /*
    //for POSIX compliant compilers, this is the approach:
    FILE *stream = fdopen (fileno(stdout), "wb");
    //   if (stream==NULL) return false;
    if (stream==NULL) stream=stdout; //give it a try anyway...
    WriteWAVFileToStream (stream);

    #ifdef WIN32
    _setmode (_fileno (stdout), O_BINARY);
    _setmode (_fileno (stdin), O_BINARY);
    #endif
  */
//NOTE: the above is too platform specific for this class; user should do that elsewhere.
 WriteWAVFileToStream (stdout);
}

//======================================
//it is up to the user to ensure that the stream is writable and in binary mode.
bool CBinauralBeat::WriteWAVFileToStream (FILE * stream)
{
//  FILE *stream;
//  stream = stdout;

//20051129 - I need to estimate final size of file for stream/piping use,
//and so something is in there in case user cancels mid-write:
 FileByteCount = (unsigned int) (loops * ScheduleTime * 176400);
 WAVheader_fill ();

 //create the header:
 fwrite (&wavh, sizeof (WAVheader), 1, stream);
 FileByteCount = 0;             //now reset FileByteCount of estimate in order to have it count real samples
 WriteStop = false;
 ManualFreqOffsetControl = false;
 signed short buff[4];
 Reset ();                      //make sure we start at absolute beginning
 while (!(InfoFlag & BB_COMPLETED) && !WriteStop)
 {
//  fputc((unsigned char)((val)&0xff), stream);
//  fputc((unsigned char)((val>>8)&0xff), stream);
//  fputc((unsigned char)(0), stream);
//  fputc((unsigned char)(0), stream);
  MainLoop (buff, 4);
  fwrite (buff, 4, 1, stream);
  FileByteCount += 4;
 }
// InfoFlag|=BB_COMPLETED;//just in case WriteStop was called
// WriteStop=false;
 return true;
}

//======================================
void CBinauralBeat::Reset ()
{
 TotalSampleCount = TotalSampleCountLooped = ScheduleCount = 0;
 InfoFlag &= ~BB_COMPLETED;
 loopcount = loops;
}

////////////////////////////////
//ALL SCHEDULE FILE READING TAKES PLACE BETWEEN FOLLOWING START AND END:
//START SCHEDULEFILE READ ROUTINES
//20070303 NOTE: replaced atof calls with g_ascii_strtod() for all reads, 
//since the docs say  strtod() is localed dependent, but that g_ascii_strtod() 
//should be used for config file sort of situations. Bottom line: 
//g_ascii_dtostr() ALWAYS (regardless of locale) writes a dot-decimal point.
//======================================
//This is what external code would call to read (and load) a schedule file. 
//Returns 0 on success, non-0 on failure.
//There are two types of files this opens. First it tries to open the Gnaural2
//style XML file; if that failes, it tries the old style.
int CBinauralBeat::SchedFilenameToSchedule ()
{
 if (0 != BB_XMLReadFile (SchedFilename))
 {
  BB_DBGOUT ("Not a valid Gnaural XML format file... trying old format");
  if (0 != SchedFilenameToSchedule_OriginalFileFormat ())
  {
   BB_ERROUT ("Invalid format or file doesn't exist...");
   return 1;                    //failure
  }
  else
  {
   BB_DBGOUT ("Success opening old format file");
   return 0;                    //success   
  }
 }
 BB_DBGOUT ("Success opening XML format file");
 return 0;                      //success
}

////////////////////////////////
//Reads and loads an original-style (comma delimited) BB schedule file.
//Returns 0 on success, non-0 on failure.
//Operation details of the old style parser: 
//Opens whatever is in SchedFilename, strips all content of spaces, commands, 
//and comments, and dumps the remains raw in to SchedBuff, then uses 
//SchedBuffToSchedule() to fill Schedule. Importantly, this does no checking
//to see what is or isn't a number; just deals with text.
//SchedBuffToSchedule() does all number determinations.
int CBinauralBeat::SchedFilenameToSchedule_OriginalFileFormat ()
{
 char *filename = SchedFilename;        //this line simply gives the global to a local var, in order to make this function easy to make independent
 FILE *stream;
 char *str;
 if ((filename == NULL) || ((stream = fopen (filename, "r")) == NULL))
 {                              //didn't find a schedule file, so return (could try to create one at this point with WriteScheduleToFile()
  //BB_BB_DBGOUT("FAILED");
  return 1;
 }
 FileFlag = true;
 char tmpchar;
 unsigned int i = 0;
 while (feof (stream) == 0)
 {
  tmpchar = fgetc (stream);
  if (tmpchar == '#')
  {
   while (feof (stream) == 0 && tmpchar != '\n')
    tmpchar = fgetc (stream);
  }
//  if (tmpchar>='0' && tmpchar <='9') i++;
//  else if (tmpchar==',' || tmpchar=='.') i++;
  if ((tmpchar >= '0' && tmpchar <= '9') ||
      tmpchar == ',' || tmpchar == '.' || tmpchar == '-')
  {
   i++;
  }
 }                              //end while

 str = new char[++i];
 rewind (stream);
 i = 0;
 while (feof (stream) == 0)
 {
  tmpchar = fgetc (stream);
  //deal with comments:
  if (tmpchar == '#')
  {                             //throw whole line away:
   while (feof (stream) == 0 && tmpchar != '\n')
    tmpchar = fgetc (stream);
  }
  //deal with command strings:
  else if (tmpchar == '[')
  {
   ParseCmd (stream);
  }
  //if it is a number, add it to the monster string:
  else if ((tmpchar >= '0' && tmpchar <= '9') || tmpchar == ','
           || tmpchar == '.' || tmpchar == '-')
  {
   str[i] = tmpchar;
   i++;
  }
 }

 fclose (stream);

 //put end on the string!!
 str[i] = '\0';
 SchedBuffToSchedule (str);
 delete[]str;
 return 0;
}                               //end SchedFilenameToSchedule()

//======================================
void CBinauralBeat::ParseCmd (FILE * stream)
{
 char strNum[128];
 char strCmd[128];
 char tmpchar;
 int inum = 0, icmd = 0;
 strNum[0] = strCmd[0] = '\0';
 while ((tmpchar = fgetc (stream)) != ']' && feof (stream) == 0)
 {
  //eat up any extra characters
  //while (tmpchar=='=' || ' ') tmpchar=fgetc( stream );
  //if it is a number, put it in the number string:
  if ((tmpchar >= '0' && tmpchar <= '9') || tmpchar == '.' || tmpchar == '-')
  {
   strNum[inum] = tmpchar;
   inum++;
  }

  //if it is a string, put it in Cmd string:
  else if ((tmpchar >= 'A' && tmpchar <= 'Z') ||
           (tmpchar >= 'a' && tmpchar <= 'z'))
  {
   strCmd[icmd] = toupper (tmpchar);
   icmd++;
  }
  else if (tmpchar == '#')
  {                             //user put a comment after a command!
   while (feof (stream) == 0 && tmpchar != '\n')
    tmpchar = fgetc (stream);
   return;
  }
 }                              //end of parsing line:

 strNum[inum] = '\0';
 strCmd[icmd] = '\0';
 if (!strcmp (strCmd, "BASEFREQ"))
 {
  FreqBase = g_ascii_strtod (strNum, NULL);
  if (FreqBase < 40)
  {
   FreqBase = 40;
  }
  else if (FreqBase > 1000)
  {
   FreqBase = 1000;
  }
 }

 if (!strcmp (strCmd, "TONEVOL"))
 {
  Volume_Tone = g_ascii_strtod (strNum, NULL);
  if (Volume_Tone < 0)
  {
   Volume_Tone = 0;
  }
  else if (Volume_Tone > 100)
  {
   Volume_Tone = 100;
  }
  Volume_Tone = (Volume_Tone * 0.01);
 }

 if (!strcmp (strCmd, "NOISEVOL"))
 {
  Volume_Noiz = g_ascii_strtod (strNum, NULL);
  if (Volume_Noiz < 0)
  {
   Volume_Noiz = 0;
  }
  else if (Volume_Noiz > 100)
  {
   Volume_Noiz = 100;
  }
  Volume_Noiz = (Volume_Noiz * 0.01);
 }

 if (!strcmp (strCmd, "LOOPS"))
 {
  loops = atoi (strNum);
  if (loops < 0)
  {
   loops = 0;
  }
  loopcount = loops;
 }

 if (!strcmp (strCmd, "STEREONOISE"))
 {
  if (atoi (strNum) != 0)
  {
   StereoNoiz = 1;
  }
  else
  {
   StereoNoiz = 0;
  }
 }
}

//======================================
//The philosophy here: take a string with commas separating the
//floats, hunt for commas. When one is found, start a new string
//beginning with next character. Take every following character up
//to (but not including) the next comma. Then store that float, dispose
//of the tempstring, and hunt some more until length of string is
//covered.
//NOTE: in porting this from Java, I now need to delete[] some stuff
//made in this function
//20070224 started working on Non-Dot Decimal Point issues
void CBinauralBeat::SchedBuffToSchedule (char *str)
{
 char tmpstr[256];
 BB_PRECISION_TYPE *tmpfloat = NULL;
 char substr[256];
 //first count how many floats are in the string:
 int floatcount = 1;
 unsigned long i;
 for (i = 0; i < strlen (str); i++)
 {
  if (str[i] == ',')
  {
   floatcount++;
  }                             // end if comma
 }                              //end for i
 //do it all again, now that I know how many floats:
 tmpfloat = new BB_PRECISION_TYPE[floatcount];
 floatcount = 0;
 int j = 0;
 for (i = 0; i < strlen (str); i++)
 {
  //first extract a number to a string:
  if ((str[i] >= '0' && str[i] <= '9') || str[i] == '.' || str[i] == '-')
   tmpstr[j++] = str[i];

  //if I found a comma, end the string:
  else if (str[i] == ',')
  {
   tmpstr[j] = '\0';
   strcpy (substr, tmpstr);
   tmpfloat[floatcount] = g_ascii_strtod (substr, NULL);
   j = 0;
   floatcount++;
  }                             // end if comma
 }                              //end for i
 if (j != 0)
 {                              //there should be one more float to get:
  tmpstr[j] = '\0';
  strcpy (substr, tmpstr);
  tmpfloat[floatcount] = g_ascii_strtod (substr, NULL);
  ++floatcount;
 }                              // end if j!=0

 //Allot new schedule and set ScheduleEntriesCount:
 Schedule_Allot (floatcount / 3);

 //now load key variables in to Schedule vars:
 int listcount = -1;
 int buffcount = 0;
 ScheduleTime = 0;
 while ((++listcount) < ScheduleEntriesCount)
 {
  Schedule[listcount].FreqLStart = tmpfloat[buffcount++];
  Schedule[listcount].FreqRStart = tmpfloat[buffcount++];
  Schedule[listcount].Duration = tmpfloat[buffcount++];
  ScheduleTime += Schedule[listcount].Duration;
  Schedule[listcount].AbsoluteEndTime_100 = (unsigned int) ScheduleTime *100;
 }
//now figure Schedule's FreqEnds and Spreads:
 Schedule_CalculateIntervals ();

//code to write a schedule to a file:
//  if (!FileFlag)
//  {
//   WriteScheduleToFile ();
//  }                             //end file writing routine
 delete[]tmpfloat;
}                               //end of extractfloats()

///////////////////////////////////////////////
//START XML SCHEDULE FILEFORMAT READING CODE 
///////////////////////////////////////////////
//this is only called by BB_XMLParser; don't call.
//point of this is to keep Event data separate, since it is the
//most numerous and thus should probably be done with Attributes
int CBinauralBeat::BB_XMLEventDataParser (const gchar * DataType,
                                          const gchar * Value)
{
 if (!strcmp (DataType, "beatfreq"))
 {
  if (BB_ParserXML_curvoice == 0)
  {
   Schedule[BB_ParserXML_curentry].FreqLStart =
    (g_ascii_strtod (Value, NULL) * 0.5);
   Schedule[BB_ParserXML_curentry].FreqRStart =
    -Schedule[BB_ParserXML_curentry].FreqLStart;
   BB_DBGOUT_FLT ("beatfreq:", g_ascii_strtod (Value, NULL));
   return 0;
  }
 }

 if (!strcmp (DataType, "duration"))
 {
  if (BB_ParserXML_curvoice == 0)
  {
   Schedule[BB_ParserXML_curentry].Duration = g_ascii_strtod (Value, NULL);
   ScheduleTime += Schedule[BB_ParserXML_curentry].Duration;
   Schedule[BB_ParserXML_curentry].AbsoluteEndTime_100 =
    (unsigned int) ScheduleTime *100;
   BB_DBGOUT_FLT ("duration:", Schedule[BB_ParserXML_curentry].Duration);
   return 0;
  }
 }

 //NOTE: A lot redundant setting these next few every time, so
 //only set stuff when 0 == BB_ParserXML_curentry
 if (0 == BB_ParserXML_curentry)
 {
  if (!strcmp (DataType, "basefreq"))
  {
   if (BB_ParserXML_curvoice == 0)
   {
    FreqBase = g_ascii_strtod (Value, NULL);
    BB_DBGOUT_FLT ("basefreq:", FreqBase);
    return 0;
   }

   //grab StereoNoiz state from Noise voice's singe entry's basefreq entry:
   if (BB_ParserXML_curvoice == 1)
   {
    StereoNoiz = atoi (Value);
    BB_DBGOUT_INT ("StereoNoiz:", StereoNoiz);
    return 0;
   }
  }

  if (!strcmp (DataType, "volume_left"))
  {
   if (BB_ParserXML_curvoice == 0)
   {
    Volume_Tone = g_ascii_strtod (Value, NULL);
    BB_DBGOUT_FLT ("volume_tone:", Volume_Tone);
    return 0;
   }
   if (BB_ParserXML_curvoice == 1)
   {
    Volume_Noiz = g_ascii_strtod (Value, NULL);
    BB_DBGOUT_FLT ("volume_noise:", Volume_Noiz);
    return 0;
   }
  }
 }
//==END OF EVENT PROPERTIES
//=========================
 return 1;
}

///////////////////////////////////////////////
//This is only called by ParserXML; don't call
//the only reason this exists is to avoid having to
//put me-> in front of every member variable in BB_XMLParser.
void CBinauralBeat::BB_XMLParser_callback (const gchar * CurrentElement,        //this must always be a valid string
                                           const gchar * Attribute,     //beware: this will equal NULL if there are none
                                           const gchar * Value) //beware: this will equal NULL to announce end of CurrentElement
{
 me->BB_XMLParser (CurrentElement, Attribute, Value);
}

///////////////////////////////////////////////
//this is only called by BB_XMLParser_callback; don't call
void CBinauralBeat::BB_XMLParser (const gchar * CurrentElement, //this must always be a valid string
                                  const gchar * Attribute,      //beware: this will equal NULL if there are none
                                  const gchar * Value)  //beware: this will equal NULL to announce end of CurrentElement
{
//Essential to do things in this order:
// 1) Check if this call represents the end of CurrentElement; if so, increment counter vars then return
// 2) See if there are Attributes with this CurrentElement; if so, assign them, then return;
// 3) If I got here, just slog through Elements and Values to assign to vars
//First see if this is the end of CurrentElement; if so, nothing to do:
 if (Value == NULL)
 {
  if (0 == strcmp (CurrentElement, "entry"))
  {
   BB_DBGOUT_INT ("==END Entry", BB_ParserXML_curentry);
   ++BB_ParserXML_curentry;
   return;
  }

  if (0 == strcmp (CurrentElement, "voice"))
  {
   BB_DBGOUT_INT ("====END Voice", BB_ParserXML_curvoice);
   ++BB_ParserXML_curvoice;
   BB_ParserXML_curentry = 0;
  }
  return;
 }

 /////////////
 //If we have a valid voicecount, we can process other things:
 if (BB_ParserXML_voicecount == 2)
 {
//Now deal with Attributes if there are any:
  if (Attribute != NULL)
  {
   BB_DBGOUT_STR (Attribute, Value);
   BB_XMLEventDataParser (Attribute, Value);
   return;
  }

  //loops will be next thing it will run in to:
  //20070302 BUGFIXED: loops wasn't picking up value.
  if (!strcmp (CurrentElement, "loops"))
  {
   loops = atoi (Value);
   BB_DBGOUT_INT ("Loops:", loops);
   //Zero means "infinite repeat mode":
   if (loops < 0)
   {
    loops = 0;
   }
   loopcount = loops;
   return;
  }

  //get overallvolume_left:
  if (!strcmp (CurrentElement, "overallvolume_left"))
  {
   VolumeOverall_left = g_ascii_strtod (Value, NULL);
   BB_DBGOUT_FLT ("overallvolume_left:", VolumeOverall_left);
   return;
  }

  //get overallvolume_right:
  if (!strcmp (CurrentElement, "overallvolume_right"))
  {
   VolumeOverall_right = g_ascii_strtod (Value, NULL);
   BB_DBGOUT_FLT ("overallvolume_right:", VolumeOverall_right);
   return;
  }

  //get StereoSwap:
  if (!strcmp (CurrentElement, "stereoswap"))
  {
   StereoSwap = atoi (Value);
   BB_DBGOUT_INT ("stereoswap:", StereoSwap);
   return;
  }

  //get entrycount:
  if (BB_ParserXML_curvoice == 0 &&
      0 == strcmp (CurrentElement, "entrycount"))
  {
   Schedule_Allot (atoi (Value));
   BB_DBGOUT_INT ("entrycount:", ScheduleEntriesCount);
   return;
  }
 }                              //end valid area for processing
 ////////////

 //This should go at the bottom, because voicecount is the startup 
 //flag (called once per file) allowing rest to be legally parsed:
 if (!strcmp (CurrentElement, "voicecount"))
 {
  BB_ParserXML_voicecount = atoi (Value);
  BB_DBGOUT_INT ("voicecount:", BB_ParserXML_voicecount);
  if (BB_ParserXML_voicecount != 2)
  {
   ParserXML_AbortFlag = 1;
   BB_ERROUT
    ("XML File Not Compatible With Gnaural: File has more than two voices");
  }
  return;
 }
}

//======================================
//This is all you call to try and open an XML style gnaural2 schedule file.
//Returns 0 on success, non-0 if something wasn't right... but 
//know that it may have already overwritten old Schedule anyway, so beware...
//Philosophy: Gnaural must always have two voices (BB and Noise),
//and they will always be expected to come in the XML file as the 
//BB voice first (able to have an unlimited number of entries), 
//and then the Noise voice second, having always only one entry. 
//Since Gnaural2 can have unlimited number of voices and entries, and
//Gnaural only 2, people opening Gnaural2 files is a problem
//These are some random Gnaural2 file format observations:
//- totaltime and totalentrycount values are useless for Gnaural
//- if voicecount is greater or less than 2, this couldn't be a Gnaural file
//- believing the entrycount value for the first voice IS valid
//- if more than one entrycount in second voice, this couldn't be a Gnaural file
//- there should only be parent 0 and parent 1, otherwise couldn't be a Gnaural file
//With that, the steps to reading the file are (in order):
//1) wait for voicecount, set BB_ParserXML_voicecount to flag it has been found; if != 2
//     (or at least less than 2) abort.
//1) if BB_ParserXML_voicecount was set, wait for loops, set loops
//3) look for voice 0's entrycount, then
//-    call Schedule_Allot (atoi(entrycount)); 
//-    start loading up resulting Schedule with entrydata, 
//     averaging the two vols and summing/subtracting half
//     the beatfreq from the basefreq
//4) if entrycount in voice 1 (second voice) != 1, throw alert, but just make up noise params
//- at next voice (Noise), just average the two vols and assign
int CBinauralBeat::BB_XMLReadFile (char *filename)
{
 if (filename == NULL)
 {
  return 1;
 }
 //to parse a Gnaural2 file:
 //- Give ParserXML SG's internal callback function
 //- Call ParserXML_parse_file_xml:
 ScheduleTime = 0;
 BB_ParserXML_voicecount = 0;
 BB_ParserXML_curvoice = 0;
 BB_ParserXML_curentry = 0;
 SetUserParserXMLHandler (BB_XMLParser_callback);
 BB_DBGOUT ("Starting with XML file parse:");
 if (0 == ParserXML_parse_file_xml (filename))
 {
  BB_DBGOUT ("XML file parsed successfully:");
  BB_DBGOUT_INT ("Total new Voice count: ", BB_ParserXML_voicecount);
  BB_DBGOUT_INT ("Total new Entry count: ", ScheduleEntriesCount);
  Schedule_CalculateIntervals ();
  return 0;
 }
 return 1;
}

//END XML SCHEDULE FORMATFILE CODE 
///////////////////////////////////////////////
//END SCHEDULEFILE READ ROUTINES
////////////////////////////////

////////////////////////////////
//ALL SCHEDULE FILE WRITING TAKES PLACE BETWEEN FOLLOWING START AND END:
//START SCHEDULEFILE WRITE ROUTINES
//======================================
//20070304 NOTE: Replaced all g_ascii_dtostr()'s with 
//g_ascii_formatd("%g"), because g_ascii_dtostr() had bizarre rounding 
//errors and represented numbers in an extremely ineficcient manner.
//20070303 NOTE: replaced all float writes with g_ascii_dtostr() 
//in order to tackle non-dot decimal point issues. g_ascii_dtostr()
//converts a gdouble to a string always using a dot for decimal point. 
//I then will use g_ascii_strtod() for all reads, since the docs say 
//strtod() is localed dependent, but that g_ascii_dtostr() should be 
//used for config file sort of situations. Bottom line: g_ascii_dtostr() 
//ALWAYS (regardless of locale) writing a dot-decimal point...
/*
void CBinauralBeat::WriteScheduleToFile ()
{
 WriteScheduleToFile (SchedFilename);
}
*/
void CBinauralBeat::WriteScheduleToFile_oldformat (char *file_name)
{
 FILE *stream;
 gchar strbuf[G_ASCII_DTOSTR_BUF_SIZE];
 const int strsize = sizeof (strbuf);
 if ((stream = fopen (file_name, "w")) == NULL)
 {
  return;
 }
 fprintf (stream, "# This is a Gnaural schedule file\n");
 fprintf (stream, "# Gnaural ignores lines starting with \"#\"\n\n");
 fprintf (stream, "# Base frequency for tones: (40-1000)\n");
 fprintf (stream, "[BASEFREQ=%s]\n\n",
          g_ascii_formatd (strbuf, strsize, "%g", FreqBase));
 fprintf (stream, "# Noise Volume: (0-100)\n");
 fprintf (stream, "[NOISEVOL=%s]\n\n",
          g_ascii_formatd (strbuf, strsize, "%g", Volume_Noiz * 100.0f));
 fprintf (stream, "# Tone Volume: (0-100)\n");
 fprintf (stream, "[TONEVOL=%s]\n\n",
          g_ascii_formatd (strbuf, strsize, "%g", Volume_Tone * 100.0f));
 fprintf (stream, "# Times to repeat schedule: (0 means infinite)\n");
 fprintf (stream, "[LOOPS=%d]\n\n", loops);
 fprintf (stream, "# Use Stereo Noise 1=true, 0=false\n");
 fprintf (stream, "[STEREONOISE=%d]\n\n", StereoNoiz ? 1 : 0);
 fprintf (stream, "# Runtime: %d min. %d sec. (%s sec.), %d Entries\n",
          (int) ScheduleTime / 60,
          (int) ScheduleTime % 60,
          g_ascii_formatd (strbuf, strsize, "%g", ScheduleTime),
          ScheduleEntriesCount);
 fprintf (stream, "#FreqLStart\tFreqRStart\tDuration\n");
 int listcount = -1;
 while ((++listcount) < (ScheduleEntriesCount - 1))
 {
  fprintf (stream, "%s,\t\t",
           g_ascii_formatd (strbuf, strsize, "%g",
                            Schedule[listcount].FreqLStart));
  fprintf (stream, "%s,\t\t",
           g_ascii_formatd (strbuf, strsize, "%g",
                            Schedule[listcount].FreqRStart));
  fprintf (stream, "%s,\n",
           g_ascii_formatd (strbuf, strsize, "%g",
                            Schedule[listcount].Duration));
 }
 fprintf (stream, "%s,\t\t",
          g_ascii_formatd (strbuf, strsize, "%g",
                           Schedule[listcount].FreqLStart));
 fprintf (stream, "%s,\t\t",
          g_ascii_formatd (strbuf, strsize, "%g",
                           Schedule[listcount].FreqRStart));
 fprintf (stream, "%s\n",
          g_ascii_formatd (strbuf, strsize, "%g",
                           Schedule[listcount].Duration));
 fclose (stream);
 //DEBUG LINE 20070223:
 //WriteScheduleToFile_XML ("debug.gnaural");
}

//======================================
void CBinauralBeat::WriteDefaultScheduleToFile ()
{
 SetDefaultValues ();
 SchedBuffToSchedule (DefaultSchedBuff);        //Used to call WriteScheduleToFile() at end
 BB_DBGOUT ("Writing default file to:");
 BB_DBGOUT (SchedFilename);
 WriteScheduleToFile (SchedFilename);

}

//======================================
void CBinauralBeat::WriteScheduleToFile (char *file_name)
{
#define BB_XMLFILEVERSION "1.20070301"
#ifndef VERSION
#define VERSION "OLDGNAURAL1"
#endif
#define BB_VOICETYPE_BINAURALBEAT   0
#define BB_VOICETYPE_PINKNOISE      1
#define BB_SELECTED                 1
#define BB_UNSELECTED               0
//First count how many DPs to copy. One reason to do this first is to be sure
//there really are selected DPs before throwing away the old buffer:
//prepare file access:
 FILE *stream;
 if ((stream = fopen (file_name, "w")) == NULL)
 {
//    SG_ERROUT ("Failed to open file for writing!");
  return;
 }

 gchar strbuf[G_ASCII_DTOSTR_BUF_SIZE];
 const int strsize = sizeof (strbuf);
 fprintf (stream, "<!-- See http://gnaural.sourceforge.net -->\n");
 fprintf (stream, "<schedule>\n");
 fprintf (stream, "<gnauralfile_version>%s</gnauralfile_version>\n", BB_XMLFILEVERSION);        //always keep this first
 fprintf (stream, "<gnaural_version>%s</gnaural_version>\n", VERSION);
 time_t curtime;
 struct tm *loctime;
 curtime = time (NULL);
 loctime = localtime (&curtime);
 fprintf (stream, "<date>%s</date>\n", asctime (loctime));
 fprintf (stream, "<title>YeOldGnauralFile</title>\n");
 fprintf (stream,
          "<schedule_description>OldGnaural1Schedule</schedule_description>\n");
 fprintf (stream, "<author>BinauralBeat.cpp</author>\n");
 fprintf (stream, "<totaltime>%s</totaltime>\n",
          g_ascii_formatd (strbuf, strsize, "%g", ScheduleTime));
 fprintf (stream, "<voicecount>2</voicecount>\n");
 fprintf (stream, "<totalentrycount>%d</totalentrycount>\n", ScheduleEntriesCount + 1); //both voices added together
 fprintf (stream, "<loops>%d</loops>\n", loops);
 fprintf (stream, "<overallvolume_left>%s</overallvolume_left>\n",
          g_ascii_formatd (strbuf, strsize, "%g", VolumeOverall_left));
 fprintf (stream, "<overallvolume_right>%s</overallvolume_right>\n",
          g_ascii_formatd (strbuf, strsize, "%g", VolumeOverall_right));
 fprintf (stream, "<stereoswap>%d</stereoswap>\n", StereoSwap);

//Setup first voice (BB tones):
 fprintf (stream, "<voice>\n");
 fprintf (stream, "<description>BinauralBeat Tone Voice</description>\n");
 fprintf (stream, "<id>%d</id>\n", 0);
 fprintf (stream, "<type>%d</type>\n", BB_VOICETYPE_BINAURALBEAT);
 fprintf (stream, "<voice_state>%d</voice_state>\n", BB_SELECTED);
 fprintf (stream, "<entrycount>%d</entrycount>\n", ScheduleEntriesCount);
 fprintf (stream, "<entries>\n");

//write the BB Tone entries:
 int listcount = -1;
 while ((++listcount) < ScheduleEntriesCount)
 {
  BB_PRECISION_TYPE tmpFreqBase =
   ((FreqBase + Schedule[listcount].FreqLStart) +
    (FreqBase + Schedule[listcount].FreqRStart)) * 0.5;
  BB_PRECISION_TYPE tmpFreqBeat =
   fabs (Schedule[listcount].FreqLStart - Schedule[listcount].FreqRStart);
  BB_PRECISION_TYPE tmpVol = Volume_Tone;
  fprintf (stream, "<entry parent=\"0\" duration=\"%s\" ",
           g_ascii_formatd (strbuf, strsize, "%g",
                            Schedule[listcount].Duration));
  fprintf (stream, "volume_left=\"%s\" ",
           g_ascii_formatd (strbuf, strsize, "%g", tmpVol));
  fprintf (stream, "volume_right=\"%s\" ",
           g_ascii_formatd (strbuf, strsize, "%g", tmpVol));
  fprintf (stream, "beatfreq=\"%s\" ",
           g_ascii_formatd (strbuf, strsize, "%g", tmpFreqBeat));
  fprintf (stream, "basefreq=\"%s\" state=\"%d\"/>\n",
           g_ascii_formatd (strbuf, strsize, "%g", tmpFreqBase), BB_SELECTED);
 }
 //finish that voice:
 fprintf (stream, "</entries>\n");
 fprintf (stream, "</voice>\n");

 //start Noise voice:
 //NOTE: StereoNoiz state is carried in "basefreq"
 fprintf (stream, "<voice>\n");
 fprintf (stream, "<description>Pink Noise Voice</description>\n");
 fprintf (stream, "<id>%d</id>\n", 1);
 fprintf (stream, "<type>%d</type>\n", BB_VOICETYPE_PINKNOISE);
 fprintf (stream, "<voice_state>%d</voice_state>\n", BB_UNSELECTED);
 fprintf (stream, "<entrycount>1</entrycount>\n");
 fprintf (stream, "<entries>\n");
//write the single Noise entry: 
 fprintf (stream, "<entry parent=\"1\" duration=\"%s\" ",
          g_ascii_formatd (strbuf, strsize, "%g", ScheduleTime));
 fprintf (stream, "volume_left=\"%s\" ",
          g_ascii_formatd (strbuf, strsize, "%g", Volume_Noiz));
 fprintf (stream,
          "volume_right=\"%s\" beatfreq=\"0\" basefreq=\"%d\" state=\"%d\"/>\n",
          g_ascii_formatd (strbuf, strsize, "%g", Volume_Noiz), StereoNoiz,
          BB_UNSELECTED);
 //finish that voice:
 fprintf (stream, "</entries>\n");
 fprintf (stream, "</voice>\n");

 //finish the schedule: 
 fprintf (stream, "</schedule>\n");
 fclose (stream);
//  SG_BB_DBGOUT_INT("Wrote DataPoints:", DP_count);
}

//END SCHEDULEFILE WRITE ROUTINES
////////////////////////////////

/* *INDENT-OFF* */ 
//======================================
//this works great with 110Hz basefreq and 85% tone 20% noiz
char CBinauralBeat::DefaultSchedBuff[] = 
"0,0,9,\
6,-6,45,\
4,-4,60,\
3,-3,60,\
2.5,-2.5,120,\
2.15,-2.15,180,\
2,-2,180,\
1.95,-1.95,6,\
3.5,-3.5,6,\
1.95,-1.95,360,\
2.1,-2.1,6,\
3.5,-3.5,6,\
1.95,-1.95,180,\
2,-2,180,\
1.95,-1.95,6,\
3.5,-3.5,6,\
1.95,-1.95,340,\
2.1,-2.1,6,\
3.5,-3.5,6,\
2,-2,180,\
2.1,-2.1,180,\
1.9,-1.9,6,\
3.5,-3.5,6,\
1.95,-1.95,400,\
2.1,-2.1,6,\
3.5,-3.5,6,\
2.1,-2.1,180,\
1.95,-1.95,180,\
2,-2,6,\
3.5,-3.5,6,\
2,-2,300,\
1.9,-1.9,6,\
3.5,-3.5,6,\
1.95,-1.95,180,\
2.05,-2.05,180,\
1.95,-1.95,6,\
3.5,-3.5,6,\
1.95,-1.95,360,\
2.05,-2.05,6,\
3.5,-3.5,6,\
1.95,-1.95,180,\
1.8,-1.8,180,\
2,-2,6,\
3.5,-3.5,6,\
2.15,-2.15,64";
/* *INDENT-ON* */
