/* GNE - Game Networking Engine, a portable multithreaded networking library.
 * Copyright (C) 2001-2006 Jason Winnebeck 
 * Project website: http://www.gillius.org/gne/
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
 */

#include "gneintern.h"
#include <gnelib/Connection.h>
#include <gnelib/ConnectionStats.h>
#include <gnelib/ConnectionListener.h>
#include <gnelib/Buffer.h>
#include <gnelib/Packet.h>
#include <gnelib/ExitPacket.h>
#include <gnelib/PacketParser.h>
#include <gnelib/ConnectionEventGenerator.h>
#include <gnelib/Error.h>
#include <gnelib/Errors.h>
#include <gnelib/SocketPair.h>
#include <gnelib/Address.h>
#include <gnelib/GNE.h>
#include <gnelib/EventThread.h>
#include <gnelib/Lock.h>

namespace GNE {

Connection::Connection()
: state( NeedsInitialization ), timeout_copy( 0 ) {
}

void Connection::disconnectAll() {
  Thread::requestAllShutdown( Thread::CONNECTION );
}

Connection::~Connection() {
  if ( state == ReadyToConnect ) //if only ClientConnection::open
    disconnect();

  assert( state == NeedsInitialization || state == Disconnected );
}

ConnectionListener::sptr Connection::getListener() const {
  LockMutex lock( sync );

  if ( eventThread )
    return eventThread->getListener();
  else
    return ConnectionListener::sptr();
}

void Connection::setListener(const ConnectionListener::sptr& listener) {
  LockMutex lock( sync );

  if ( eventThread )
    eventThread->setListener( listener );
}

int Connection::getTimeout() {
  LockMutex lock( sync );

  if ( eventThread )
    return eventThread->getTimeout();
  else
    return timeout_copy;
}

void Connection::setTimeout(int ms) {
  LockMutex lock( sync );

  if ( eventThread )
    eventThread->setTimeout(ms);
  timeout_copy = ms;
}

PacketStream& Connection::stream() {
  assert(ps);
  return *ps;
}

ConnectionStats Connection::getStats(int reliable) const {
  LockMutex lock( sync );
  return sockets.getStats(reliable);
}

Address Connection::getLocalAddress(bool reliable) const {
  LockMutex lock( sync );
  return Address(sockets.getLocalAddress(reliable));
}

Address Connection::getRemoteAddress(bool reliable) const {
  LockMutex lock( sync );
  return Address(sockets.getRemoteAddress(reliable));
}

Connection::State Connection::getState() const {
  return state;
}

bool Connection::isConnected() const {
  return state == Connected;
}

void Connection::disconnect() {
  LockMutex lock( sync );

  unreg(true, true);

  if ( state != Disconnected && state != Disconnecting ) {
    state = Disconnecting;

    gnedbgo2(2, "disconnecting r: %i, u: %i", sockets.r, sockets.u);

    if ( ps && ps->hasStarted() ) {
      ps->shutDown(); //PacketStream will try to send the required ExitPacket.

      //we have to release sync to prevent deadlock in case PacketStream decides
      //to lock sync (from processError).
      sync.release();
      ps->join(); //we have to join to wait for the ExitPacket to go out.
      sync.acquire();
    }
  }

  //Shutdown the EventThread.
  if ( eventThread ) {
    eventThread->onDisconnect();
    eventThread.reset(); //Kill the cycle we participate in
  }

  //Even if the PS or ET aren't running, the low-level stuff in the connection
  //threads should get an error when we disconnect the actual sockets.
  sockets.disconnect();

  state = Disconnected;
}

void Connection::disconnectSendAll(int waitTime) {
  LockMutex lock( sync );
  if (isConnected())
    ps->waitToSendAll(waitTime);
  disconnect();
}

void Connection::setThisPointer( const wptr& weakThis ) {
  assert( this_.expired() );
  this_ = weakThis;
  eventThread = EventThread::create( weakThis.lock() );
}

void Connection::addHeader(Buffer& raw) {
  raw << (gbyte)'G' << (gbyte)'N' << (gbyte)'E';
}

void Connection::addVersions(Buffer& raw) {
  GNEProtocolVersionNumber us = GNE::getGNEProtocolVersion();
  //Write the GNE version numbers.
  raw << us.version << us.subVersion << us.build;

  //Write the whole game name buffer
  raw.writeRaw((const gbyte*)GNE::getGameName(), GNE::MAX_GAME_NAME_LEN + 1);

  //Write the user version
  raw << GNE::getUserVersion();
}

void Connection::checkHeader(Buffer& raw,
                             ProtocolViolation::ViolationType t) {
  gbyte headerG, headerN, headerE;
  raw >> headerG >> headerN >> headerE;

  if (headerG != 'G' || headerN != 'N' || headerE != 'E')
    throw ProtocolViolation(t);
}

void Connection::checkVersions(Buffer& raw) {
  //Get the version numbers
  GNEProtocolVersionNumber them;
  raw >> them.version >> them.subVersion >> them.build;

  //Read the game name
  gbyte rawName[GNE::MAX_GAME_NAME_LEN + 1];
  raw.readRaw(rawName, GNE::MAX_GAME_NAME_LEN + 1);
  //And convert it to a string, making sure it is of the proper length and
  //NULL-terminated.
  char gameName[GNE::MAX_GAME_NAME_LEN + 1];
  memset( gameName, 0, GNE::MAX_GAME_NAME_LEN + 1 );
  strncpy(gameName, (const char*)rawName, GNE::MAX_GAME_NAME_LEN);

  //Read the user version number
  guint32 themUser;
  raw >> themUser;

  //This will throw an Error of the versions are wrong.
  GNE::checkVersions(them, gameName, themUser);
}

void Connection::onReceive() {
  LockMutex lock( sync );

  if( eventThread )
    eventThread->onReceive();
}

void Connection::finishedInit() {
  assert( state == NeedsInitialization );
  state = ReadyToConnect;
}

void Connection::startConnecting() {
  assert( state == ReadyToConnect );
  state = Connecting;
}

void Connection::startThreads() {
  LockMutex lock( sync );

  assert( state == Connecting );
  ps->start();
  eventThread->start();
}

void Connection::finishedConnecting() {
  LockMutex lock( sync );

  assert( state != NeedsInitialization );
  assert( state != ReadyToConnect );
  if ( state == Connecting )
    state = Connected;
}

void Connection::onReceive(bool reliable) {
  Buffer buf;
  int temp = 0;

  //We have to assert that the connection is still active, since we can be
  //disconnected at any time.
  {
    LockMutex lock( sync );
    if ( state == Connected || state == Connecting )
      temp = sockets.rawRead(reliable, buf);
    else
      return; //ignore the event.
  }
  if (temp == NL_INVALID) {
    NLenum error = nlGetError();
    if (error == NL_MESSAGE_END) {
      //in HawkNL 1.4b4 and later, this means that the connection was
      //closed on the network-level because the client disconnected or
      //has dropped.  If we got an ExitPacket, this will be ignored.
      processError(Error::ConnectionDropped);
    } else {
      //This is some other bad error that we need to report
      processError(LowLevelError(Error::Read));
    }
  } else if (temp == 0) {
    //In HawkNL 1.4b3 and earlier, this _USED_ to mean that...
    //This means the connection was closed on the network-level because
    //remote computer has purposely closed or has dropped.
    //We should never get this now, because HawkNL traps this message, but
    //for completeness we check for it anyways.
    processError(Error::ConnectionDropped);

  } else {
    //Stream read success
    //parse the packets and add them to the PacketStream
    try {
      Packet* next = NULL;
      while ((next = PacketParser::parseNextPacket(buf)) != NULL) {
        //We want to intercept ExitPackets, else we just add it.
        if (next->getType() == ExitPacket::ID) {
          //All further errors will be ignored after we call onExit, due to
          //contract of EventThread.
          {
            LockMutex lock( sync ); //protect on eventThread
            if( eventThread )       //have we not disconnected?
              eventThread->onExit();
          }

          PacketParser::destroyPacket( next );

        } else
          ps->addIncomingPacket(next);
      }

      //Notify that packets were received.
      onReceive();

    } catch ( Error& err ) {
      //if PacketParser fails or readPacket fails.
      processError( err );
    }
  }
}

void Connection::processError(const Error& error) {
  switch(error.getCode()) {

  case Error::UnknownPacket:
    {
      LockMutex lock( sync ); //protect on eventThread
      if( eventThread )       //have we not disconnected?
        eventThread->onError(error);
    }
    break;

  default:
    {
      LockMutex lock( sync ); //lock early since unreg needs it anyway.

      //if we don't unreg we generate endless error messages because
      //we need to unregister immediately.  So we do that.
      unreg(true, true);

      if( eventThread )       //have we not disconnected?
        eventThread->onFailure(error);
    }
    break;
  }
}

Connection::Listener::Listener(const Connection::sptr& listener, bool isReliable) 
: conn(listener), reliable(isReliable) {
}

Connection::Listener::~Listener() {
}

void Connection::Listener::onReceive() {
  conn->onReceive(reliable);
}

void Connection::reg(bool reliable, bool unreliable) {
  LockMutex lock( sync );

  if ( reliable && sockets.r != NL_INVALID ) {
    eGen->reg( sockets.r, Listener::sptr( new Listener( this_.lock(), true ) ) );
    gnedbgo1(3, "Registered reliable socket %i", sockets.r);
  }
  if ( unreliable && sockets.u != NL_INVALID ) {
    eGen->reg( sockets.u, Listener::sptr( new Listener( this_.lock(), false ) ) );
    gnedbgo1(3, "Registered unreliable socket %i", sockets.u);
  }
}

void Connection::unreg(bool reliable, bool unreliable) {
  LockMutex lock( sync );

  if ( reliable && sockets.r != NL_INVALID ) {
    eGen->unreg(sockets.r);
    gnedbgo1(3, "Unregistered reliable socket %i", sockets.r);
  }
  if ( unreliable && sockets.u != NL_INVALID ) {
    eGen->unreg(sockets.u);
    gnedbgo1(3, "Unregistered unreliable socket %i", sockets.u);
  }
}

} //Namespace GNE
