/* filesystem.c
   Interface to the posix (well...) file system
   Copyright (C) 2001 Linus Walleij

This file is part of the GNOMAD package.

GNOMAD is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation; either version 2, or (at your option)
any later version.

You should have received a copy of the GNU General Public License
along with GNOMAD; see the file COPYING.  If not, write to
the Free Software Foundation, 59 Temple Place - Suite 330,
Boston, MA 02111-1307, USA. 

*/

#include "common.h"
#include "metadata.h"
#include "filesystem.h"
#include "filenaming.h"
#include "prefs.h"
#include "util.h"
#include "mp3file.h"
#include "id3read.h"
#include "wmaread.h"
#include "wavfile.h"
#include "xfer.h"
#if !GTK_CHECK_VERSION(2,4,0)
#include <libgnomeui/libgnomeui.h>
#endif

/* Extra includes used by this file only */

#include <unistd.h>
#include <pwd.h>
#include <dirent.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>

/* Globally imported variables */
extern GtkWidget *main_window;

/* Local variables */
static gint preffd;

/* Filecopy routine loosely based on
 * http://www.snippets.org/snippets/portable/WB_FCOPY+C.php3 */
gboolean copy_file(gchar *source, gchar *dest)
{
  gint f1, f2;
  gboolean retval = FALSE;
  
  f1 = (gint) open(source, O_RDONLY, 0);
  if (f1 < 0)
    return FALSE;

  f2 = (gint) creat(dest, (mode_t) CREATE_FILEMODE);
  if (f2 >= 0) {
    guint bufsiz;
    
    for (bufsiz = 0x8000; bufsiz >= 128; bufsiz >>= 1) {
      register gchar *buffer;
      
      buffer = (gchar *) g_malloc(bufsiz);
      if (buffer) {
	while (1) {
	  register guint n;
	  
	  n = read(f1,buffer,bufsiz);
	  if (n == -1)
	    break;
	  if (n == 0) {
	    retval = TRUE;
	    break;
	  }
	  if (n != write(f2,buffer,n))
	    break;
	}
	g_free(buffer);
	break;
      }
    }
    if (retval) {
      close(f1);
      close(f2);
    } else {
      close(f1);
      close(f2);
      remove(dest); // Delete partial file
    }
    return retval;
  }
  close(f1);
  return FALSE;
}

/* Unicode conversion wrappers that enables
 * forcing the local charset to something disregarding
 * the locale setting and such stuff. */
gchar *filename_toutf8(const gchar *instr)
{
  gchar *tmp;

  if (get_prefs_force_8859()) {
    tmp = g_convert(instr,-1,"UTF-8","ISO-8859-1",NULL,NULL,NULL);
  } else {
    tmp = g_filename_to_utf8(instr,-1,NULL,NULL,NULL);
    /* If this conversion fails, fall back to ISO 8859-1 */
    if (tmp == NULL) {
      /* Something is wrong with the locales */
      gchar *lang = getenv("LANG");
      g_print("Seems like you're using a filesystem which still contains ISO 8859-1 characters\n");
      g_print("though your locale is \"%s\". The right thing to do is to convert\n", lang);
      g_print("all file- and directorynames in your filesystem to UTF-8 format.\n");
      g_print("Otherwise, consider checking \"force to 8859-1\" option for a workaround.\n\n");
      tmp = g_convert(instr,-1,"UTF-8","ISO-8859-1",NULL,NULL,NULL);
    }
  }
  return tmp;
}

gchar *filename_fromutf8(const gchar *instr)
{
  gchar *tmp;
  
  if (get_prefs_force_8859()) {
    tmp = g_convert(instr,-1,"ISO-8859-1","UTF-8",NULL,NULL,NULL);
  } else {
    tmp = g_filename_from_utf8(instr,-1,NULL,NULL,NULL);
  }
  return tmp;
}

gboolean is_directory(gchar *dir) {
  struct stat st;
  gchar *tmp;

  tmp = filename_fromutf8(dir);
  stat(tmp, &st);
  g_free(tmp);
  return S_ISDIR(st.st_mode);
}

static GList *get_directory_listing(gchar* base_path, gboolean anyfile)
{
  GList *filelist;
  DIR * DirPointer;
  struct dirent * Entry;
  gchar *tmppath;

  /* Returns a listing of applicable files */
  /* returned GSList contains file / size pairs. */
  /* Directories: no full path, just dirname; size 0 */
  filelist = NULL;
  tmppath = filename_fromutf8(base_path);
  DirPointer = opendir(tmppath);
  if (! DirPointer){
    /* g_print("Unable to open directory %s.\n", tmppath); */
    return NULL;
  }
  while ((Entry = readdir(DirPointer)) != NULL)
    {
      struct stat EntryStatus;
      gchar cTempPath[512];
      gchar *cTempFilename;
      int nTempSize;
      int bIsDir;
      int bIsFile;

      strncpy(cTempPath, tmppath, 512);
      strcat(cTempPath, "/");
      strcat(cTempPath, Entry->d_name);
      if (stat(cTempPath, &EntryStatus)) {
        /* g_print("Failed to read file status for %s.\n", cTempPath); */
        continue;
      }
      nTempSize = (int) EntryStatus.st_size;
      bIsDir = S_ISDIR(EntryStatus.st_mode);
      bIsFile = S_ISREG(EntryStatus.st_mode);
      if (bIsDir) {
	/* We don't want this "dir" */
	if (!strcmp(Entry->d_name,".")) {
	  continue;
	}
	/* If the prefs are selected not to display the hidden
	 * directories, we hide them. */
	if (strlen(Entry->d_name) >= 2) {
	  if (Entry->d_name[0] == '.' &&
	      Entry->d_name[1] != '.' &&
	      !get_prefs_display_dotdirs())
	    continue;
	}
      } else if (bIsFile) {
	static char cTempCodec[4];
	int i;
	
	if (strlen(Entry->d_name) < 4)
	  continue;
	strcpy(cTempCodec, Entry->d_name+strlen(Entry->d_name)-3);
	g_strdown(cTempCodec);
	/* Only display valid files */
	if (!anyfile &&
	    g_ascii_strcasecmp(cTempCodec,"mp3") &&
	    g_ascii_strcasecmp(cTempCodec,"wav") &&
	    g_ascii_strcasecmp(cTempCodec,"wma"))
	  continue;
      }
      /* only directories and valid files come here, the
       * list is NOT in UTF-8, that conversion will be
       * done by the caller of this routine. */
      if (bIsDir) {
	/* For directories, set size to 0 */
	nTempSize = 0;
      }
      cTempFilename = g_strdup(cTempPath);
      filelist = g_list_append(filelist, cTempFilename);
      filelist = g_list_append(filelist, GUINT_TO_POINTER(nTempSize));
    }
  closedir(DirPointer);
  g_free(tmppath);
  return filelist;
}

static GList *get_metadata_for_dirlist(GList *inlist, 
				       listtype_t listtype, 
				       dirfill_thread_arg_t *args)
{
  GList *metalist = NULL;
  GList *filelist;
  guint nofiles, currentfile;
  gboolean use_dialog = FALSE;
  gboolean cancelled = FALSE;

  /* Count the number of files */
  if (args != NULL) {
    use_dialog = TRUE;
    filelist = inlist;
    nofiles = 0;
    while (filelist) {
      nofiles ++;
      filelist = filelist->next;
      filelist = filelist->next;
    }
    currentfile = 0;
  }
  
  filelist = inlist;
  while (filelist && !cancelled) {
    metadata_t *meta;
    gchar *path;
    gchar *data;
    gchar *tmp;
    guint size;
    gint i;
    
    meta = new_metadata_t();
    /* Full path -> ID column */
    path = filelist->data;
    meta->path = filename_toutf8(path);
    if (meta->path == NULL) {
      g_print("%s created a NULL value in conversion to UTF-8!\n", path);
    }
    /* If filename is ISO-8859-1 convert it by force! */
    filelist = filelist->next;
    meta->size = GPOINTER_TO_UINT(filelist->data);
    tmp = g_path_get_basename(path);
    meta->filename = filename_toutf8(tmp);
    g_free(tmp);
    if (use_dialog) {
      gdk_threads_enter();
      tmp = g_strdup_printf("File %d of %d", currentfile, nofiles);
      /* g_print("File: %s\n", meta->filename); */
      currentfile ++;
      gtk_label_set_text(GTK_LABEL(args->label), tmp);
      gdk_threads_leave();
    }
    if (listtype == HDDATA_LIST) {
      /* For data files, provide basic data only 
       * nothing additional is needed.
       */
    } else {
      /* For music files, fill in row by using libid3tag and friends */
      if (meta->size != 0) {
	/* Find codec first */
	static gchar codec[4];
	
	strcpy(codec, meta->path+strlen(meta->path)-3);
	g_strup(codec);
	meta->codec = g_strdup(codec);
	if (!strcmp(meta->codec,"MP3")) {
	  /* Gets the ID3 tag if any */
	  get_tag_for_mp3file(meta);
	  /* Scans the mp3 frames to calculate play length */
	  if ((meta->length == NULL) && (meta->path != NULL)) {
	    // g_print("Calling length_from_file()...\n");
	    gchar *tmppath = filename_fromutf8(meta->path);
	    meta->length = length_from_file(tmppath, meta->size);
	    g_free(tmppath);
	  }
	}
	if (!strcmp(meta->codec,"WAV")) {
	  /* FIXME: learn how to read metadata from WAV files */
	  get_tag_for_wavfile(meta);
	  if (meta->length == NULL) {
	    meta->length = strdup("0:01");
	  }
	}
	if (!strcmp(meta->codec,"WMA")) {
	  get_tag_for_wmafile(meta);
	}
	/* 
	 * Gather additional information from the path and file name
	 * if this preference is set. 
	 */
	if (get_prefs_detect_metadata()) {
	  info_from_path(meta, TRUE);
	} else {
	  info_from_path(meta, FALSE);
	}
	
	/* At last pad any empty fields to be <unknown>
	 * (the way playcenter does it) */
	
	if (meta->artist == NULL)
	  meta->artist = g_strdup("<Unknown>");
	if (meta->title == NULL)
	  meta->title = g_strdup("<Unknown>");
	if (meta->album == NULL)
	  meta->album = g_strdup("<Unknown>");
	if (meta->codec == NULL)
	  meta->codec = g_strdup("<Unknown>");
	if (meta->genre == NULL)
	  meta->genre = g_strdup("<Unknown>");
	if (meta->length == NULL)
	  meta->length = g_strdup("<Unknown>");
	if (meta->filename == NULL) {
	  tmp = g_path_get_basename(path);
	  meta->filename = filename_toutf8(tmp);
	  g_free(tmp);
	}
      } else {
	if (meta->path == NULL) {
	  meta->artist = g_strdup("<Unknown>");
	} else {
	  /* For directories: copy dirname to Artist field */
	  gchar **tmpv;
	  gint i;
	  
	  tmpv = g_strsplit(meta->path, G_DIR_SEPARATOR_S, 0);
	  /* Find last component */
	  for (i = 0; tmpv[i] != NULL; i++) {};
	  if (i >= 1)
	    meta->artist = g_strdup(tmpv[i-1]);
	  g_strfreev(tmpv);
	}
      }
    }
    /* Add this to the metadata list */
    metalist = g_list_append(metalist, meta);
    /* Update progress bar */
    if (use_dialog) {
      gfloat fraction = ((gdouble) currentfile / (gdouble) nofiles);
      gdk_threads_enter();
      gtk_progress_bar_set_fraction(GTK_PROGRESS_BAR(args->progress), fraction);
      gdk_threads_leave();
      cancelled = args->cancelled;
    }
    filelist = filelist->next;
  }
  return metalist;
}

/**
 * @param args if this and the following parameter is null the call is assumed to be 
 *        non-threaded, else it is updated and handled as if called from a thread.
 *        Else it is assumed to be called from a single-threaded context and will not
 *        display update information.
 */
GList *get_metadata_dir(listtype_t listtype, 
			gchar *path, 
			dirfill_thread_arg_t *args)
{
  GList *metalist;
  GList *filelist;
  GList *tmp;

  /* If it is for the data view, get for ALL files */
  if (args != NULL) {
    gdk_threads_enter();
    gtk_label_set_text(GTK_LABEL(args->label), _("Getting file list from filesystem"));
    gdk_threads_leave();
  }
  filelist = get_directory_listing(path, (listtype == HDDATA_LIST));
  /* This is the heavy part */
  metalist = get_metadata_for_dirlist(filelist, listtype, args);
  /* Free the filelist, not used any more */
  tmp = filelist;
  while (tmp) {
    g_free(tmp->data); /* Free all paths */
    tmp = tmp->next;
    tmp = tmp->next;
  }
  g_list_free(filelist);
  return metalist;
}

gpointer dirfill_thread(gpointer thread_args)
{
  dirfill_thread_arg_t *args = (dirfill_thread_arg_t *) thread_args;
  GList *metalist;
  GList *tmp;

  /* This is time consuming */
  metalist = get_metadata_dir(args->listtype, args->path, args);

  /* This will interfere with other threads. */
  gdk_threads_enter();
  gtk_label_set_text(GTK_LABEL(args->label), _("Adding metadata to view"));
  recreate_list_store(args->listtype);
  tmp = metalist;
  while (tmp) {
    metadata_t *meta;
    
    meta = (metadata_t *) tmp->data;
    add_metadata_to_model(meta, args->listtype);
    tmp = tmp->next;
    destroy_metadata_t(meta);
  }
  view_and_sort_list_store(args->listtype);
  gtk_widget_destroy(args->dialog);
  gdk_threads_leave();
  g_list_free(metalist);
  g_free(args->path);
  g_free(args);
}

/**
 * This callback function will cancel the ongoing
 * fill-in of metadata for a certain thread.
 */
#if GTK_CHECK_VERSION(2,4,0)
void swapped_cancel_fillin_callback(gpointer *data)
{
  dirfill_thread_arg_t *args = (dirfill_thread_arg_t *) data;  

  g_print("Cancelled scan of %s!\n", args->path);
  args->cancelled = TRUE;
}
#else
void cancel_fillin_callback(GtkWidget *widget, gpointer *data)
{
  dirfill_thread_arg_t *args = (dirfill_thread_arg_t *) data;  

  g_print("Cancelled scan of %s!\n", args->path);
  args->cancelled = TRUE;
}
#endif

/**
 * This will fire up a separate thread which retrieves
 * the metadata for a certain host-side directory. Both
 * tracks and common data files.
 * @param listtype the list pane to retrieve. Must be one
 *        of the harddisk-refering panes.
 * @param path the path to retrieve metadata from.
 */
void fill_in_dir(listtype_t listtype, gchar *path)
{
  dirfill_thread_arg_t *dirfill_thread_args;
  GtkWidget *label1;
  GtkWidget *dialog, *scanlabel, *metadata_progress_bar, *separator;
  dirfill_thread_args = (dirfill_thread_arg_t *) g_malloc(sizeof(dirfill_thread_arg_t));

  // Create a dialog...
#if GTK_CHECK_VERSION(2,4,0)
  dialog = gtk_message_dialog_new (GTK_WINDOW(main_window),
				   GTK_DIALOG_MODAL | GTK_DIALOG_DESTROY_WITH_PARENT,
				   GTK_MESSAGE_INFO,
				   GTK_BUTTONS_CLOSE,
				   (listtype == HDDATA_LIST) ? _("Getting file metadata") : _("Getting track metadata"));
  gtk_window_set_title (GTK_WINDOW (dialog), (listtype == HDDATA_LIST) ? _("Getting file metadata") : _("Getting track metadata"));
  gtk_box_set_spacing (GTK_BOX (GTK_DIALOG (dialog)->vbox), 5);
  gtk_box_set_homogeneous (GTK_BOX (GTK_DIALOG (dialog)->action_area), TRUE);
  //gtk_window_set_position (GTK_WINDOW (dialog), GTK_WIN_POS_MOUSE);
  g_signal_connect_swapped(GTK_OBJECT(dialog),
		   "delete_event",
		   G_CALLBACK(swapped_cancel_fillin_callback),
		   (gpointer) dirfill_thread_args);
  g_signal_connect_swapped(GTK_OBJECT(dialog), 
		   "response",
		   G_CALLBACK(swapped_cancel_fillin_callback), 
		   (gpointer) dirfill_thread_args);
  label1 = gtk_label_new(path);
  scanlabel = gtk_label_new("");
  metadata_progress_bar = gtk_progress_bar_new();
  separator = gtk_hseparator_new ();
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), label1, TRUE, TRUE, 0);
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), scanlabel, TRUE, TRUE, 0);
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), separator, TRUE, TRUE, 0);
  gtk_box_pack_start(GTK_BOX(GTK_DIALOG(dialog)->vbox), metadata_progress_bar, TRUE, TRUE, 0);
  gtk_progress_bar_update(GTK_PROGRESS_BAR(metadata_progress_bar), (gfloat) 0);
  gtk_widget_show_all(dialog);
#else
  dialog = gnome_dialog_new((listtype == HDDATA_LIST) ? _("Getting file metadata") : _("Getting track metadata"),
			    GNOME_STOCK_BUTTON_CANCEL,
			    NULL);
  gnome_dialog_button_connect(GNOME_DIALOG(dialog),
			      0,
			      G_CALLBACK(cancel_fillin_callback),
			      (gpointer) dirfill_thread_args);
  g_signal_connect(GTK_OBJECT(dialog),
		   "delete_event",
		   G_CALLBACK(cancel_fillin_callback),
		   (gpointer) dirfill_thread_args);
  label1 = gtk_label_new((listtype == HDDATA_LIST) ? _("Getting file metadata") : _("Getting track metadata"));
  gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(dialog)->vbox), label1, TRUE, TRUE, 0);
  label1 = gtk_label_new(path);
  gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(dialog)->vbox), label1, TRUE, TRUE, 0);
  scanlabel = gtk_label_new("");
  gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(dialog)->vbox), scanlabel, TRUE, TRUE, 0);
  separator = gtk_hseparator_new ();
  gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(dialog)->vbox), separator, TRUE, TRUE, 0);
  metadata_progress_bar = gtk_progress_bar_new();
  gtk_box_pack_start(GTK_BOX(GNOME_DIALOG(dialog)->vbox), metadata_progress_bar, TRUE, TRUE, 0);
  gtk_progress_bar_update(GTK_PROGRESS_BAR(metadata_progress_bar), (gfloat) 0);
  gtk_widget_show_all(dialog);
#endif
  
  dirfill_thread_args->dialog = dialog;
  dirfill_thread_args->label = scanlabel;
  dirfill_thread_args->progress = metadata_progress_bar;
  dirfill_thread_args->listtype = listtype;
  dirfill_thread_args->path = g_strdup(path);
  dirfill_thread_args->cancelled = FALSE;
    
  // TODO: check no reentrance while this thread is running...
  g_thread_create(dirfill_thread,(gpointer) dirfill_thread_args, FALSE, NULL);
}

gchar *get_current_dir()
{
  gchar *tmp, *path;

  /* Retrieve the current working directory somehow */
  tmp = g_get_current_dir();
  path = filename_toutf8(tmp);
  g_free(tmp);
  return path;
}

void change_directory(gchar *dir)
{
  gchar *tmp;

  tmp = filename_fromutf8(dir);
  chdir(tmp);
  g_free(tmp);
}

void delete_files(GList *filelist)
{
  GList *tmplist = g_list_first(filelist);
  gchar *tmp;

  while (tmplist != NULL) {
    metadata_t *meta = (metadata_t *) tmplist->data;
    /* g_print("Deleting: %s\n", meta->path); */
    tmp = filename_fromutf8(meta->path);
    unlink(tmp);
    g_free(tmp);
    tmplist = tmplist->next;
  }
}

/* Copy an MP3 file and remove the tag */
gboolean clone_and_strip_id3(gchar *source, gchar *dest)
{
  gchar *tmpsource, *tmpdest;

  tmpsource = filename_fromutf8(source);
  tmpdest = filename_fromutf8(dest);
  if (copy_file(tmpsource, tmpdest)) {
    /* Next strip the tag */
    /* g_print("Stripping tag from %s\n", dest); */
    remove_tag_from_mp3file(tmpdest);
    g_free(tmpsource);
    g_free(tmpdest);
    return TRUE;
  }
  g_free(tmpsource);
  g_free(tmpdest);
  return FALSE;
}

/* Get a file descriptor for the preference file */
gint get_prefsfile_fd(gboolean write)
{
  static gchar filename[] = "." PACKAGE "rc";
  static gchar *path = NULL;

  if (path == NULL) {
    g_strdown(filename);
    path = g_strdup(g_get_home_dir());
    path = stringcat(path, G_DIR_SEPARATOR_S);
    path = stringcat(path, filename);
  }
  if (write) {
    preffd = (gint) open(path, O_WRONLY, (mode_t) CREATE_FILEMODE);
    if (preffd < 0)
      preffd = (gint) creat(path, (mode_t) CREATE_FILEMODE);
  }
  else
    preffd = (gint) open(path, O_RDONLY, 0);
  if (preffd < 0 &&
      write)
    create_error_dialog(_("Could not write preference file"));
  return preffd;
}

/* Close a preference file */
void close_prefsfile(void)
{
  close(preffd);
  preffd = 0;
}

/* Close a preference file */
void rewrite_prefsfile(gchar *prefs)
{
  get_prefsfile_fd(TRUE);
  write(preffd, prefs, strlen(prefs));
  close_prefsfile();
}

/* Function from gftp lib/misc.c */
char *expand_path (char *src)
{
  char *str, *pos, *endpos, *prevpos, *newstr, *tempstr, tempchar;
  struct passwd *pw;

  pw = NULL;

  str = filename_fromutf8(src);

  if (*str == '~')
    {
      if (*(str + 1) == '/' || (pos = strchr (str, '/')) == NULL)
	pw = getpwuid (geteuid ());
      else
	{
	  *pos = '\0';
	  pw = getpwnam (str + 1);
	  *pos = '/';
	}
    }

  endpos = str;
  newstr = NULL;
  while ((pos = strchr (endpos, '/')) != NULL)
    {
      pos++;
      while (*pos == '/')
	pos++;
      if ((endpos = strchr (pos, '/')) == NULL)
	endpos = strchr (pos, '\0');
      tempchar = *endpos;
      *endpos = '\0';
      if (strcmp (pos, "..") == 0)
	{
	  *(pos - 1) = '\0';
	  if (newstr != NULL && (prevpos = strrchr (newstr, '/')) != NULL)
	    *prevpos = '\0';
	}
      else if (strcmp (pos, ".") != 0)
	{
	  if (newstr == NULL)
	    {
	      newstr = g_malloc (strlen (pos - 1) + 1);
	      strcpy (newstr, pos - 1);
	    }
	  else
	    {
	      tempstr = g_strconcat (newstr, pos - 1, NULL);
	      g_free (newstr);
	      newstr = tempstr;
	    }
	}
      *endpos = tempchar;
      if (*endpos == '\0')
	break;
      endpos = pos + 1;
    }

  if (newstr == NULL || *newstr == '\0')
    {
      if (newstr != NULL)
	g_free (newstr);
      newstr = g_malloc (2);
      *newstr = '/';
      *(newstr + 1) = '\0';
    }

  if (newstr != NULL)
    {
      g_free (str);
      str = newstr;
    }

  if (pw != NULL)
    {
      if ((pos = strchr (str, '/')) == NULL)
	{
	  newstr = g_malloc (strlen (pw->pw_dir) + 1);
	  strcpy (newstr, pw->pw_dir);
	}
      else
	newstr = g_strconcat (pw->pw_dir, pos, NULL);
      if (str)
	g_free (str);
      return (newstr);
    }
  else
    {
      newstr = g_malloc (strlen (str) + 1);
      strcpy (newstr, str);
    }
  g_free (str);
  str = newstr;
  newstr = filename_toutf8(str);
  g_free(str);
  return (newstr);
}

/* Test if a file is writeable or not */
gboolean test_writeable(gchar *path)
{
  FILE *file;

  /* Test to know if we can write into the file */
  if ((file = fopen(path,"r+"))==NULL ) {
    return FALSE;
  }
  fclose(file);
  return TRUE;
}

gboolean create_directory(gchar *path)
{
  if (mkdir(path, CREATE_DIRMODE) == -1) {
    return FALSE;
  } else {
    return TRUE;
  }
}

void create_dirs(gchar *filename) {
  struct stat st;
  gchar **tmpv;
  gchar *path;
  gint i, dirs;

  tmpv = g_strsplit(filename, G_DIR_SEPARATOR_S, 0);
  for (dirs=0; tmpv[dirs]; ++dirs);
  if (--dirs < 1)
    goto out;
  
  path = g_strdup(tmpv[0]);
  for (i=0;;) {
    gchar *tmp;
    gint ret = 0;

    ret = stat(path, &st);
    if (ret < 0 || !S_ISDIR(st.st_mode))
      create_directory(path);
    if (++i >= dirs)
      break;
    tmp = path;
    path = g_strjoin(G_DIR_SEPARATOR_S, path, tmpv[i], NULL);
    g_free(tmp);
  }
  g_free(path);
out:
  g_strfreev(tmpv);
}

void export_playlistfile(gchar *filename, gchar *formatstring, GList *playlist, gboolean m3ufile, gboolean plsfile)
{
  gchar *tmpname = filename_toutf8(filename);
  gint fd;
  gchar m3uheader[] = "#EXTM3U\n";
  gchar plsheader[] = "[playlist]\n";
  GList *tmp;
  guint i;

  fd = (gint) open(tmpname, O_WRONLY, (mode_t) CREATE_FILEMODE);
  if (fd < 0)
    fd = (gint) creat(tmpname, (mode_t) CREATE_FILEMODE);
  if (fd < 0) {
    create_error_dialog(_("Could not write exported playlist"));
    return;
  }

  if (m3ufile) {
    write(fd, m3uheader, strlen(m3uheader));
  } else if (plsfile) {
    write(fd, plsheader, strlen(plsheader));
  }

  tmp = g_list_first(playlist);
  i = 1;
  while (tmp != NULL) {
    gchar *song;
    gchar *row;
    metadata_t *meta;

    meta = (metadata_t *) tmp->data;
    tmp = g_list_next(tmp);
    song = compose_metadata_string(meta, formatstring, FALSE);
    if (m3ufile || plsfile) {
      guint tmplen = mmss_to_seconds(meta->length);
      gchar *tmpfname = compose_filename(meta);
      if (m3ufile) {
	row = g_strdup_printf("#EXTINF:%u,%s\n%s\n", tmplen, song, tmpfname);
      } else {
	row = g_strdup_printf("File%u=%s\nTitle%u=%s\nLength%u=%u\n", i, tmpfname,
			      i, song, i, tmplen);
      }
      g_free(tmpfname);
    } else {
      row = g_strdup_printf("%d. %s\n", i, song);
    }
    write(fd, row, strlen(row));
    g_free(song);
    g_free(row);
    i++;
  }
  
  if (plsfile) {
    gchar *footer = g_strdup_printf("NumberOfEntries=%u\nVersion=2\n", i-1);
    write(fd, footer, strlen(footer));
    g_free(footer);
  }

  close(fd);
}
