/* 
 * Copyright (C) 2000 JP Rosevear
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA
 */

#include <gnome.h>
#include <glade/glade.h>
#include "makros.h"
#include "prefs.h"
#include "dialogs.h"

#define GLADE_FILE "prefs.glade"

enum { BUTTON_OK = 0, BUTTON_APPLY, BUTTON_CLOSE, BUTTON_HELP };

typedef struct _PrefsState
{
	GladeXML *gui;
	
	GtkWidget *pb;

	GtkWidget *dark_color;
	GtkWidget *light_color;
	GtkWidget *piece_dir;

	GtkWidget *beep;
	GtkWidget *promote_to_queen;
	GtkWidget *autoflag;
	GtkWidget *sa_notation;
	GtkWidget *co_notation;

	GList *servers;
	GtkWidget *server_dlg;
	GtkWidget *server_list;
	GtkWidget *server_name;
	GtkWidget *server_port;
	GtkWidget *server_username;
	GtkWidget *server_password;
	GtkWidget *server_connect;

	GtkWidget *telnet_program;
	
	GList *programs;
	GtkWidget *program_dlg;
	GtkWidget *program_list;
	GtkWidget *program_name;
	GtkWidget *program_loc;
} PrefsState;

static PrefsState *state = NULL;

static void
dp_changed (PrefsState *dlg)
{
	gnome_property_box_changed (GNOME_PROPERTY_BOX (dlg->pb));
}

static void
cb_color_changed (GnomeColorPicker *gcp, guint r, guint g, 
		  guint b, guint a, gpointer user_data) 
{
	dp_changed ((PrefsState *) user_data);
}

static void
cb_toggle_changed (GtkToggleButton *toggle, gpointer user_data) 
{
	dp_changed ((PrefsState *) user_data);
}

static void
cb_entry_changed (GtkWidget *widget, gpointer user_data)
{
	dp_changed ((PrefsState *) user_data);
}

static void
run_program_dialog (PrefsState *dlg, gint row) 
{
	GtkCList *clist = GTK_CLIST (dlg->program_list);
	PrefsProgram *pp = NULL;
	char *text2;
	gint btn;

	if (row >= 0) {
		pp = gtk_clist_get_row_data (clist, row);
		
		gtk_entry_set_text (GTK_ENTRY (dlg->program_name), pp->name);
		gtk_entry_set_text (GTK_ENTRY (dlg->program_loc), pp->location);
	} else {
		gtk_entry_set_text (GTK_ENTRY (dlg->program_name), "Program");
		gtk_entry_set_text (GTK_ENTRY (dlg->program_loc), "/usr/bin/chess");
		
		pp = g_new0 (PrefsProgram, 1);
	}

	
	btn = gnome_dialog_run_and_close (GNOME_DIALOG (dlg->program_dlg));
	if (btn == 0) {
		pp->name = gtk_editable_get_chars (GTK_EDITABLE (dlg->program_name), 0, -1);
  		pp->location = gtk_editable_get_chars (GTK_EDITABLE (dlg->program_loc), 0, -1);
		if (row >= 0) {
			gtk_clist_set_text (clist, row, 0, pp->name);
			gtk_clist_set_text (clist, row, 1, pp->location);
		} else {
			char *text[255] = {"", ""};
			gint row;
			
			text[0] = pp->name;
			text[1] = pp->location;
			row = gtk_clist_append (clist, text);
			gtk_clist_set_row_data (clist, row, pp);
		}
 		dp_changed (dlg);
	}
}

static void
cb_add_program (GtkWidget *widget, gpointer user_data)
{
	PrefsState *dlg = user_data;

	run_program_dialog (dlg, -1);
}

static void
cb_mod_program (GtkWidget *widget, gpointer user_data)
{
	PrefsState *dlg = user_data;
	gint row;
	
	if (GTK_CLIST (dlg->program_list)->selection)
		row = (gint) (GTK_CLIST (dlg->program_list)->selection->data);
	else
		return;
	
	run_program_dialog (dlg, row);
}

static void
cb_del_program (GtkWidget *widget, gpointer user_data)
{
	PrefsState *dlg = user_data;
	gint row;

	if (GTK_CLIST (dlg->program_list)->selection)
		row = (gint) (GTK_CLIST (dlg->program_list)->selection->data);
	else
		return;
	
	gtk_clist_remove (GTK_CLIST (dlg->program_list), row);
	dp_changed (dlg);
}

static void
run_server_dialog (PrefsState *dlg, gint row) 
{
	GtkCList *clist = GTK_CLIST (dlg->server_list);
	PrefsServer *ps = NULL;
	char *text2;
	gint btn;

	if (row >= 0) {
		ps = gtk_clist_get_row_data (clist, row);
		
		gtk_entry_set_text (GTK_ENTRY (dlg->server_name), ps->server);
		gtk_entry_set_text (GTK_ENTRY (dlg->server_port), ps->port);
		gtk_entry_set_text (GTK_ENTRY (dlg->server_username), ps->user);
		gtk_entry_set_text (GTK_ENTRY (dlg->server_password), ps->password);
		gtk_entry_set_text (GTK_ENTRY (dlg->server_connect), ps->connect);
	} else {
		gtk_entry_set_text (GTK_ENTRY (dlg->server_name), "chess.server.com");
		gtk_entry_set_text (GTK_ENTRY (dlg->server_port), "5000");
		gtk_entry_set_text (GTK_ENTRY (dlg->server_username), g_get_user_name ());
		gtk_entry_set_text (GTK_ENTRY (dlg->server_password), "");
		gtk_entry_set_text (GTK_ENTRY (dlg->server_connect), prefs_get_telnetprogram ());
		
		ps = g_new0 (PrefsServer, 1);
	}

	
	btn = gnome_dialog_run_and_close (GNOME_DIALOG (dlg->server_dlg));
	if (btn == 0) {
		ps->server = gtk_editable_get_chars (GTK_EDITABLE (dlg->server_name), 0, -1);
  		ps->port = gtk_editable_get_chars (GTK_EDITABLE (dlg->server_port), 0, -1);
  		ps->user = gtk_editable_get_chars (GTK_EDITABLE (dlg->server_username), 0, -1);
  		ps->password = gtk_editable_get_chars (GTK_EDITABLE (dlg->server_password), 0, -1);
  		ps->connect = gtk_editable_get_chars (GTK_EDITABLE (dlg->server_connect), 0, -1);
		if (row >= 0) {
			gtk_clist_set_text (clist, row, 0, ps->user);
			gtk_clist_set_text (clist, row, 1, ps->server);
			gtk_clist_set_text (clist, row, 2, ps->port);
		} else {
			char *text[255] = {"", "", ""};
			gint row;
			
			text[0] = ps->user;
			text[1] = ps->server;
			text[2] = ps->port;
			row = gtk_clist_append (clist, text);
			gtk_clist_set_row_data (clist, row, ps);
		}
 		dp_changed (dlg);
	}
}

static void
cb_add_server (GtkWidget *widget, gpointer user_data)
{
	PrefsState *dlg = user_data;

	run_server_dialog (dlg, -1);
}

static void
cb_mod_server (GtkWidget *widget, gpointer user_data)
{
	PrefsState *dlg = user_data;
	gint row;
	
	if (GTK_CLIST (dlg->server_list)->selection)
		row = (gint) (GTK_CLIST (dlg->server_list)->selection->data);
	else
		return;
	
	run_server_dialog (dlg, row);
}

static void
cb_del_server (GtkWidget *widget, gpointer user_data)
{
	PrefsState *dlg = user_data;
	gint row;

	if (GTK_CLIST (dlg->server_list)->selection)
		row = (gint) (GTK_CLIST (dlg->server_list)->selection->data);
	else
		return;
	
	gtk_clist_remove (GTK_CLIST (dlg->server_list), row);
	dp_changed (dlg);
}

static void
cb_apply (GnomePropertyBox *pb, gint page_num, gpointer user_data)
{
	PrefsState *dlg = user_data;
	GList *programs = NULL, *servers = NULL;
	guint8 r, g, b;
	gboolean act;
	gchar *str;
	int i;
	
	if (page_num != -1)
		return;

	gnome_color_picker_get_i8 (GNOME_COLOR_PICKER (dlg->light_color), &r, &g, &b, NULL);
	prefs_set_light_color (r, g, b);
	gnome_color_picker_get_i8 (GNOME_COLOR_PICKER (dlg->dark_color), &r, &g, &b, NULL);
	prefs_set_dark_color (r, g, b);

	str = gtk_editable_get_chars (GTK_EDITABLE (dlg->piece_dir), 0, -1);
	prefs_set_piecedir (str);
	g_free (str);

	act = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (dlg->beep));
	prefs_set_beeponmove (act);
	act = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (dlg->promote_to_queen));
	prefs_set_promotetoqueen (act);
	act = gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (dlg->autoflag));
	prefs_set_autoflag (act);

	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (dlg->sa_notation)))
		prefs_set_notation (SAN);
	else
		prefs_set_notation (COORD);

	for (i = 1; i <= GTK_CLIST (dlg->program_list)->rows; i++) {
		PrefsProgram *pp;

		pp = gtk_clist_get_row_data (GTK_CLIST (dlg->program_list), i - 1);
		programs = g_list_append (programs, pp);
	}
	prefs_set_programs (programs);

	for (i = 1; i <= GTK_CLIST (dlg->server_list)->rows; i++) {
		PrefsServer *ps;

		ps = gtk_clist_get_row_data (GTK_CLIST (dlg->server_list), i - 1);
		servers = g_list_append (servers, ps);
	}
 	prefs_set_servers (servers);

	str = gtk_editable_get_chars (GTK_EDITABLE (dlg->telnet_program), 0, -1);
	prefs_set_telnetprogram (str);
	g_free (str);

	prefs_sync ();
}

static void
dialog_prefs_destroy (GtkObject *object, gpointer user_data)
{
	prefs_free_server_list (state->servers);
	prefs_free_program_list (state->programs);
	
	gtk_object_unref (GTK_OBJECT (state->gui));

	g_free (state);
	state = NULL;
}

static void
dialog_prefs_impl (PrefsState *dlg)
{
	GtkWidget *btn;
	GList *programs, *servers, *l;
	guint8 r, g, b;
	gint not;

	dlg->pb = glade_xml_get_widget (dlg->gui, "dialog");

	gtk_signal_connect (GTK_OBJECT (dlg->pb), "destroy",
			    GTK_SIGNAL_FUNC (dialog_prefs_destroy), dlg);

	/* Board and Pieces Page */
	dlg->light_color = glade_xml_get_widget (dlg->gui, "light_color");
	dlg->dark_color = glade_xml_get_widget (dlg->gui, "dark_color");
	dlg->piece_dir = glade_xml_get_widget (dlg->gui, "piece_dir");

	prefs_get_light_color (&r, &g, &b);
	gnome_color_picker_set_i8 (GNOME_COLOR_PICKER (dlg->light_color), r, g, b, 0);
	prefs_get_dark_color (&r, &g, &b);
	gnome_color_picker_set_i8 (GNOME_COLOR_PICKER (dlg->dark_color), r, g, b, 0);
	
	gtk_entry_set_text (GTK_ENTRY (dlg->piece_dir),
			    prefs_get_piecedir ());

	gtk_signal_connect (GTK_OBJECT (dlg->light_color), "color-set",
			    GTK_SIGNAL_FUNC (cb_color_changed), dlg);
	gtk_signal_connect (GTK_OBJECT (dlg->dark_color), "color-set",
			    GTK_SIGNAL_FUNC (cb_color_changed), dlg);
	gtk_signal_connect (GTK_OBJECT (dlg->piece_dir), "changed",
			    GTK_SIGNAL_FUNC (cb_entry_changed), dlg);

	/* Playing Page */
	dlg->beep = glade_xml_get_widget (dlg->gui, "beep");
	dlg->promote_to_queen = glade_xml_get_widget (dlg->gui, "promote_to_queen");
	dlg->autoflag = glade_xml_get_widget (dlg->gui, "autoflag");

	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (dlg->beep),
				      prefs_get_beeponmove ());
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (dlg->promote_to_queen),
				      prefs_get_promotetoqueen ());
	gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (dlg->autoflag),
				      prefs_get_autoflag ());
	

	gtk_signal_connect (GTK_OBJECT (dlg->beep), "toggled",
			    GTK_SIGNAL_FUNC (cb_toggle_changed), dlg);
	gtk_signal_connect (GTK_OBJECT (dlg->promote_to_queen), "toggled",
			    GTK_SIGNAL_FUNC (cb_toggle_changed), dlg);
	gtk_signal_connect (GTK_OBJECT (dlg->autoflag), "toggled",
			    GTK_SIGNAL_FUNC (cb_toggle_changed), dlg);

	dlg->sa_notation = glade_xml_get_widget (dlg->gui, "sa_notation");
	dlg->co_notation = glade_xml_get_widget (dlg->gui, "co_notation");
	not = prefs_get_notation ();
	if (not == SAN)
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (dlg->sa_notation),
					      TRUE);
	else
		gtk_toggle_button_set_active (GTK_TOGGLE_BUTTON (dlg->co_notation),
					      TRUE);

	gtk_signal_connect (GTK_OBJECT (dlg->sa_notation), "toggled",
			    GTK_SIGNAL_FUNC (cb_toggle_changed), dlg);
	gtk_signal_connect (GTK_OBJECT (dlg->co_notation), "toggled",
			    GTK_SIGNAL_FUNC (cb_toggle_changed), dlg);

	/* Chess Servers Page */
	dlg->server_dlg = glade_xml_get_widget (dlg->gui, "dialog_server");
	dlg->server_list = glade_xml_get_widget (dlg->gui, "server_list");
	dlg->server_name = glade_xml_get_widget (dlg->gui, "server_name");
	dlg->server_port = glade_xml_get_widget (dlg->gui, "server_port");
	dlg->server_username = glade_xml_get_widget (dlg->gui, "server_username");
	dlg->server_password = glade_xml_get_widget (dlg->gui, "server_password");
	dlg->server_connect = glade_xml_get_widget (dlg->gui, "server_connect");

	servers = prefs_copy_server_list (prefs_get_servers ());
	for (l = servers; l != NULL; l = l->next) {
		PrefsServer *ps = l->data;		
		char *text[255] = {"", "", ""};
		gint row;
		
		text[0] = g_strdup (ps->user);
		text[1] = g_strdup (ps->server);
		text[2] = g_strdup (ps->port);
		
		row = gtk_clist_append (GTK_CLIST (dlg->server_list), text);
		gtk_clist_set_row_data (GTK_CLIST (dlg->server_list), row, ps);
	}
	dlg->servers = servers;
	
	btn = glade_xml_get_widget (dlg->gui, "add_server");
	gtk_signal_connect (GTK_OBJECT (btn), "clicked",
			    GTK_SIGNAL_FUNC (cb_add_server), dlg);

	btn = glade_xml_get_widget (dlg->gui, "modify_server");
	gtk_signal_connect (GTK_OBJECT (btn), "clicked",
			    GTK_SIGNAL_FUNC (cb_mod_server), dlg);

	btn = glade_xml_get_widget (dlg->gui, "delete_server");
	gtk_signal_connect (GTK_OBJECT (btn), "clicked",
			    GTK_SIGNAL_FUNC (cb_del_server), dlg);

	/* Telnet Program */
	dlg->telnet_program = glade_xml_get_widget (dlg->gui, "telnet_program");
	gtk_entry_set_text (GTK_ENTRY (dlg->telnet_program), 
			    prefs_get_telnetprogram ()); 

	gtk_signal_connect (GTK_OBJECT (dlg->telnet_program), "changed",
			    GTK_SIGNAL_FUNC (cb_entry_changed), dlg);

	/* Chess Programs Page */
	dlg->program_dlg = glade_xml_get_widget (dlg->gui, "dialog_program");
	dlg->program_list = glade_xml_get_widget (dlg->gui, "program_list");
	dlg->program_name = glade_xml_get_widget (dlg->gui, "program_name");
	dlg->program_loc = glade_xml_get_widget (dlg->gui, "program_location");
	
	programs = prefs_copy_program_list (prefs_get_programs ());
	for (l = programs; l != NULL; l = l->next) {
		PrefsProgram *pp = l->data;		
		char *text[255] = {"", ""};
		gint row;
		
		text[0] = g_strdup (pp->name);
		text[1] = g_strdup (pp->location);

		row = gtk_clist_append (GTK_CLIST (dlg->program_list), text);
		gtk_clist_set_row_data (GTK_CLIST (dlg->program_list), row, pp);
	}
	dlg->programs = programs;
	
	btn = glade_xml_get_widget (dlg->gui, "add_program");
	gtk_signal_connect (GTK_OBJECT (btn), "clicked",
			    GTK_SIGNAL_FUNC (cb_add_program), dlg);

	btn = glade_xml_get_widget (dlg->gui, "modify_program");
	gtk_signal_connect (GTK_OBJECT (btn), "clicked",
			    GTK_SIGNAL_FUNC (cb_mod_program), dlg);

	btn = glade_xml_get_widget (dlg->gui, "delete_program");
	gtk_signal_connect (GTK_OBJECT (btn), "clicked",
			    GTK_SIGNAL_FUNC (cb_del_program), dlg);

	/* Dialog stuff */
	gnome_dialog_set_default (GNOME_DIALOG (dlg->pb), BUTTON_CLOSE);
	gtk_window_set_policy (GTK_WINDOW (dlg->pb), FALSE, TRUE, FALSE);

	gtk_signal_connect (GTK_OBJECT (dlg->pb), "apply",
			    GTK_SIGNAL_FUNC (cb_apply), dlg);

	gtk_widget_show_all (dlg->pb);
}

/*
 * Wrapper around dialog_prefs_impl
 * To libglade'ify it
 */
void
dialog_prefs (void)
{
	if (state != NULL) {
		gtk_widget_show_all (state->pb);
		return;
	}
	
	state = g_new0 (PrefsState, 1);

	state->gui = glade_xml_new (GNOMECHESS_GLADEDIR "/" GLADE_FILE , NULL);
	if (!state->gui) {
		printf ("Could not find " GLADE_FILE "\n");
		return;
	}

	dialog_prefs_impl (state);
}
