/* -*- Mode: C; tab-width: 8; indent-tabs-mode: nil; c-basic-offset: 8 -*- */
/* gdm-properties-list.c
 *
 * Copyright (C) 2007 David Zeuthen
 *
 * This program is free software; you can redistribute it and/or 
 * modify it under the terms of the GNU General Public License as 
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307
 * USA
 */

#include <config.h>
#include <glib/gi18n.h>
#include "gdm-properties-list.h"

enum
{
        KEY_STRING_COLUMN,
        TYPE_STRING_COLUMN,
        VALUE_STRING_COLUMN,
        N_COLUMNS
};

GtkTreeView *
gdm_properties_list_new (void)
{
        GtkCellRenderer *renderer;
        GtkTreeViewColumn *column;
        GtkTreeView *tree_view;
        GtkListStore *store;

        store = gtk_list_store_new (N_COLUMNS,
                                    G_TYPE_STRING,
                                    G_TYPE_STRING,
                                    G_TYPE_STRING);

        gtk_tree_sortable_set_sort_column_id (GTK_TREE_SORTABLE (store), KEY_STRING_COLUMN, GTK_SORT_ASCENDING);

        tree_view = GTK_TREE_VIEW (gtk_tree_view_new_with_model (GTK_TREE_MODEL (store)));

        column = gtk_tree_view_column_new ();
        gtk_tree_view_column_set_title (column, _("Key"));
        renderer = gtk_cell_renderer_text_new ();
        gtk_tree_view_column_pack_start (column, renderer, TRUE);
        gtk_tree_view_column_set_attributes (column, renderer, "text", KEY_STRING_COLUMN, NULL);
        gtk_tree_view_append_column (tree_view, column);

        column = gtk_tree_view_column_new ();
        gtk_tree_view_column_set_title (column, _("Type"));
        renderer = gtk_cell_renderer_text_new ();
        gtk_tree_view_column_pack_start (column, renderer, TRUE);
        gtk_tree_view_column_set_attributes (column, renderer, "text", TYPE_STRING_COLUMN, NULL);
        gtk_tree_view_append_column (tree_view, column);

        column = gtk_tree_view_column_new ();
        gtk_tree_view_column_set_title (column, _("Value"));
        renderer = gtk_cell_renderer_text_new ();
        g_object_set (renderer, "editable", TRUE, NULL);
        gtk_tree_view_column_pack_start (column, renderer, TRUE);
        gtk_tree_view_column_set_attributes (column, renderer, "text", VALUE_STRING_COLUMN, NULL);
        gtk_tree_view_append_column (tree_view, column);

        gtk_tree_view_set_headers_visible (tree_view, TRUE);

        return tree_view;
}


typedef struct {
        const char *key;
        gboolean found;
        GtkTreeIter iter;
} FIBKData;

static gboolean 
find_iter_by_key_foreach (GtkTreeModel *model,
                          GtkTreePath *path,
                          GtkTreeIter *iter,
                          gpointer data)
{
        gboolean ret;
        char *key = NULL;
        FIBKData *fibk_data = (FIBKData *) data;

        ret = FALSE;
        gtk_tree_model_get (model, iter, KEY_STRING_COLUMN, &key, -1);
        if (g_ascii_strcasecmp (key, fibk_data->key) == 0) {
                fibk_data->found = TRUE;
                fibk_data->iter = *iter;
                ret = TRUE;
        }
        if (key != NULL)
                g_free (key);

        return ret;
}


static gboolean
find_iter_by_key (GtkListStore *store, const char *key, GtkTreeIter *iter)
{
        FIBKData fibk_data;
        gboolean ret;

        fibk_data.key = key;
        fibk_data.found = FALSE;
        gtk_tree_model_foreach (GTK_TREE_MODEL (store), find_iter_by_key_foreach, &fibk_data);
        if (fibk_data.found) {
                *iter = fibk_data.iter;
                ret = TRUE;
        } else {
                ret = FALSE;
        }

        return ret;
}

static void
show_for_device (GtkTreeView *properties_list_view, GdmDevice *device, const char *only_for_key)
{
        GtkListStore *store;
        LibHalPropertySet *properties;
        LibHalPropertySetIterator it;

        store = GTK_LIST_STORE (gtk_tree_view_get_model (properties_list_view));

        if (only_for_key == NULL) {
                gtk_list_store_clear (store);
        } else {
                GtkTreeIter iter;
                if (find_iter_by_key (store, only_for_key, &iter)) {
                        gtk_list_store_remove (store, &iter);
                }                   
        }

        if (device == NULL)
                goto out;

        properties = gdm_device_get_properties (device);
        if (properties == NULL)
                goto out;

        libhal_psi_init (&it, properties);
        while (libhal_psi_has_more (&it)) {
                GtkTreeIter iter;
                char *key;
                char *type;
                char *value = NULL;

                key = libhal_psi_get_key (&it);

                if (only_for_key != NULL && g_ascii_strcasecmp (key, only_for_key) != 0) {
                        libhal_psi_next (&it);
                        continue;
                }

                switch (libhal_psi_get_type (&it)) {
                case LIBHAL_PROPERTY_TYPE_STRING:
                        type = _("string");
                        value = g_strdup (libhal_psi_get_string (&it));
                        break;
                case LIBHAL_PROPERTY_TYPE_INT32:
                        type = _("int32");
                        value = g_strdup_printf ("%d (0x%0x)", libhal_psi_get_int (&it), libhal_psi_get_int (&it));
                        break;
                case LIBHAL_PROPERTY_TYPE_UINT64:
                        type = _("uint64");
                        value = g_strdup_printf ("%lld (0x%0llx)", libhal_psi_get_uint64 (&it), libhal_psi_get_uint64 (&it));
                        break;
                case LIBHAL_PROPERTY_TYPE_DOUBLE:
                        type = _("double");
                        value = g_strdup_printf ("%g", libhal_psi_get_double (&it));
                        break;
                case LIBHAL_PROPERTY_TYPE_BOOLEAN:
                        type = _("bool");
                        value = g_strdup (libhal_psi_get_bool (&it) ? _("True") : _("False"));
                        break;
                case LIBHAL_PROPERTY_TYPE_STRLIST:
                        type = _("string[]");
                        value = g_strjoinv (", ", libhal_psi_get_strlist (&it));
                        break;
                default:
                        type = _("unknown");
                        value = NULL;
                        break;
                }

                if (value == NULL)
                        value = g_strdup ("");

                gtk_list_store_append (store, &iter);
                gtk_list_store_set (store, &iter,
                                    KEY_STRING_COLUMN, key,
                                    TYPE_STRING_COLUMN, type,
                                    VALUE_STRING_COLUMN, value,
                                    -1);

                g_free (value);

                libhal_psi_next (&it);
        }

out:
        gtk_tree_view_columns_autosize (properties_list_view);
}

static GdmDevice *now_showing_device = NULL;
static gulong now_showing_handler_id;

static void
hal_property_changed (GdmDevice *device, const char *key, GtkTreeView *properties_list_view)
{
        show_for_device (properties_list_view, device, key);
}

void
gdm_properties_list_show_for_device (GtkTreeView *properties_list_view, GdmDevice *device)
{

        if (now_showing_device != NULL) {
                g_signal_handler_disconnect (now_showing_device, now_showing_handler_id);
        }
        now_showing_device = device;
        if (device != NULL) {
                now_showing_handler_id = g_signal_connect (device, 
                                                           "hal_property_changed", 
                                                           (GCallback) hal_property_changed, 
                                                           properties_list_view);
        }
        show_for_device (properties_list_view, device, NULL);
}

