//  
//  Copyright (C) 2009 GNOME Do
// 
//  This program is free software: you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation, either version 3 of the License, or
//  (at your option) any later version.
// 
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
// 
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
// 

using System;
using System.Collections.Generic;

using Cairo;
using Gtk;

using Do.Interface.CairoUtils;

using Docky.Core;
using Docky.Interface;
using Docky.Interface.Painters;
using Do.Platform;
using Docky.Utilities;

namespace WeatherDocklet
{
	/// <summary>
	/// A painter that displays information about weather forecasts.
	/// </summary>
	public class WeatherPainter : AbstractIntegratedPainter
	{
		/// <value>
		/// The color to draw most text in.
		/// </value>
		static readonly Cairo.Color colorTitle = new Cairo.Color (0.627, 0.627, 0.627, 1);
		
		/// <value>
		/// The color to draw high temperatures in.
		/// </value>
		static readonly Cairo.Color colorHigh = new Cairo.Color (0.945, 0.431, 0.431, 1);
		
		/// <value>
		/// The color to draw low temperatures in.
		/// </value>
		static readonly Cairo.Color colorLow = new Cairo.Color (0.427, 0.714, 0.945, 1);
		
		/// <value>
		/// Indicates the current page the painter should show.
		/// </value>
		int Page { get; set; }
		
		/// <value>
		/// The number of pages for this painter.
		/// </value>
		const int pages = 3;
		
		/// <value>
		/// Buffers for each page of the painter.
		/// </value>
		Surface[] buffers = new Surface [pages];
		
		/// <value>
		/// A flag to indicate if the painter needs repainted.
		/// </value>
		public bool Dirty { get; set; }
		
		protected override bool NeedsRepaint { get { return Dirty; } }
		
		/// <summary>
		/// Creates a new weather painter object.
		/// </summary>
		/// <param name="docklet">
		/// A <see cref="WeatherDocklet"/> that owns this painter.
		/// </param>
		public WeatherPainter (WeatherDocklet docklet) : base (docklet as AbstractDockItem)
		{
			Dirty = true;
			
			ShowRequested += HandleOnShowRequested;
		}
		
		protected override int PainterWidth {
			get {
				return 2 * WeatherController.Weather.ForecastDays * (DockServices.DrawingService.CurrentDockHeight);
			}
		}
		
		#region IDockPainter implementation 
		
		protected override void PaintArea (Cairo.Context cr, Gdk.Rectangle paintArea)
		{
			if (buffers [Page] == null) {
				buffers [Page] = cr.Target.CreateSimilar (cr.Target.Content, paintArea.Width, paintArea.Height);
				using (Cairo.Context context = new Cairo.Context (buffers [Page])) {
					switch (Page)
					{
						default:
						case 0:
							DrawOverview (context, paintArea.Height);
							break;
						
						case 1:
							DrawTemps (context, paintArea.Height);
							break;
						
						case 2:
							DrawConditions (context, paintArea.Height);
							break;
					}
				}
			}
			
			cr.Rectangle (paintArea.X, paintArea.Y, paintArea.Width, paintArea.Height);
			cr.Clip ();
			
			buffers [Page].Show (cr, paintArea.X, paintArea.Y);
			cr.ResetClip ();

			Dirty = false;
		}
		
		#endregion
		
		/// <summary>
		/// Paints an overview of the forecast including high/low temps and a condition icon.
		/// </summary>
		/// <param name="cr">
		/// A <see cref="Cairo.Context"/> to do the painting.
		/// </param>
		/// <param name="size">
		/// A <see cref="System.Int32"/> indicating the height of the paint area.
		/// </param>
		void DrawOverview (Cairo.Context cr, int size)
		{
			int xOffset = 0;
			
			TextRenderContext textContext = new TextRenderContext (cr, string.Empty, size);
			textContext.Alignment = Pango.Alignment.Center;
			textContext.FontSize = (int) (size / 6);
			
			for (int day = 0; day < WeatherController.Weather.ForecastDays; day++)
			{
				cr.Color = colorTitle;
				textContext.Text = string.Format ("<b>{0}</b>", WeatherController.Weather.Forecasts [day].dow);
				textContext.LeftCenteredPoint = new Gdk.Point (xOffset, size * 2 / 9);
				DockServices.DrawingService.TextPathAtPoint (textContext);
				cr.Fill ();
				
				cr.Color = colorHigh;
				textContext.Text = string.Format ("<b>{0}{1}</b>", WeatherController.Weather.Forecasts [day].high, WeatherUnits.TempUnit);
				textContext.LeftCenteredPoint = new Gdk.Point (xOffset, size * 5 / 9);
				DockServices.DrawingService.TextPathAtPoint (textContext);
				cr.Fill ();
				
				cr.Color = colorLow;
				textContext.Text = string.Format ("<b>{0}{1}</b>", WeatherController.Weather.Forecasts [day].low, WeatherUnits.TempUnit);
				textContext.LeftCenteredPoint = new Gdk.Point (xOffset, size * 8 / 9);
				DockServices.DrawingService.TextPathAtPoint (textContext);
				cr.Fill ();
				
				WeatherDocklet.RenderIconOntoContext (cr, WeatherController.Weather.Forecasts [day].image,
				                               xOffset + size,
				                               size);
				
				xOffset += 2 * (size);
			}
		}
		
		/// <summary>
		/// Paints the forecast temperatures as a chart.
		/// </summary>
		/// <param name="cr">
		/// A <see cref="Cairo.Context"/> to do the painting.
		/// </param>
		/// <param name="size">
		/// A <see cref="System.Int32"/> indicating the height of the paint area.
		/// </param>
		void DrawTemps (Cairo.Context cr, int size)
		{
			int max = -1000, min = 1000;
			
			for (int day = 0; day < WeatherController.Weather.ForecastDays; day++)
			{
				if (WeatherController.Weather.Forecasts [day].high > max)
					max = WeatherController.Weather.Forecasts [day].high;
				if (WeatherController.Weather.Forecasts [day].low > max)
					max = WeatherController.Weather.Forecasts [day].low;
				if (WeatherController.Weather.Forecasts [day].high < min)
					min = WeatherController.Weather.Forecasts [day].high;
				if (WeatherController.Weather.Forecasts [day].low < min)
					min = WeatherController.Weather.Forecasts [day].low;
		    }
			
			if (max <= min)
				return;
			
			TextRenderContext textContext = new TextRenderContext (cr, string.Empty, size);
			textContext.Alignment = Pango.Alignment.Center;
			textContext.FontSize = (int) (size / 6);
			
			// high/low temp
			cr.Color = colorHigh;
			textContext.Text = string.Format ("<b>{0}{1}</b>", max, WeatherUnits.TempUnit);
			textContext.LeftCenteredPoint = new Gdk.Point (PainterWidth - size, size / 6);
			DockServices.DrawingService.TextPathAtPoint (textContext);
			cr.Fill ();
			
			cr.Color = colorLow;
			textContext.Text = string.Format ("<b>{0}{1}</b>", min, WeatherUnits.TempUnit);
			textContext.LeftCenteredPoint = new Gdk.Point (PainterWidth - size, size * 6 / 9);
			DockServices.DrawingService.TextPathAtPoint (textContext);
			cr.Fill ();
			
			// day names
			textContext = new TextRenderContext (cr, string.Empty, 2 * size);
			textContext.Alignment = Pango.Alignment.Center;
			textContext.FontSize = (int) (size / 6);
			
			cr.Color = colorTitle;
			for (int day = 0; day < WeatherController.Weather.ForecastDays; day++)
			{
				textContext.Text = string.Format ("<b>{0}</b>", WeatherController.Weather.Forecasts [day].dow);
				textContext.LeftCenteredPoint = new Gdk.Point (day * size * 2 - size / 2, size * 8 / 9);
				DockServices.DrawingService.TextPathAtPoint (textContext);
			}
			cr.Fill ();
			
			cr.LineWidth = 3;
			double height = ((double) size * 2 / 3 - 5) / (max - min);
			
			// high temp graph
			cr.Color = colorHigh;
			cr.MoveTo (size / 2, 5 + height * (max - WeatherController.Weather.Forecasts [0].high));
			for (int day = 1; day < WeatherController.Weather.ForecastDays; day++)
				cr.LineTo (day * size * 2 + size / 2, 5 + height * (max - WeatherController.Weather.Forecasts [day].high));
			cr.Stroke ();
			
			// low temp graph
			cr.Color = colorLow;
			cr.MoveTo (size / 2, 5 + height * (max - WeatherController.Weather.Forecasts [0].low));
			for (int day = 1; day < WeatherController.Weather.ForecastDays; day++)
				cr.LineTo (day * size * 2 + size / 2, 5 + height * (max - WeatherController.Weather.Forecasts [day].low));
			cr.Stroke ();
			
			// high temp points
			for (int day = 0; day < WeatherController.Weather.ForecastDays; day++)
				DrawDataPoint (cr, size, height, max, day, WeatherController.Weather.Forecasts [day].high);
			
			// low temp points
			for (int day = 0; day < WeatherController.Weather.ForecastDays; day++)
				DrawDataPoint (cr, size, height, max, day, WeatherController.Weather.Forecasts [day].low);
		}
		
		void DrawDataPoint (Cairo.Context cr, int cellWidth, double height, int max, int day, int temp)
		{
			cr.Color = new Cairo.Color (0, 0, 0, 0.4);
			cr.Arc (day * cellWidth * 2 + cellWidth / 2 + 2, 7 + height * (max - temp), 3, 0, 2 * Math.PI);
			cr.Fill ();
			
			cr.Color = colorTitle;
			cr.Arc (day * cellWidth * 2 + cellWidth / 2, 5 + height * (max - temp), 3, 0, 2 * Math.PI);
			cr.Fill ();
		}
		
		/// <summary>
		/// Paints the forecast conditions.
		/// </summary>
		/// <param name="cr">
		/// A <see cref="Cairo.Context"/> to do the painting.
		/// </param>
		/// <param name="size">
		/// A <see cref="System.Int32"/> indicating the height of the paint area.
		/// </param>
		void DrawConditions (Cairo.Context cr, int size)
		{
			TextRenderContext textContext = new TextRenderContext (cr, string.Empty, 2 * size);
			textContext.Alignment = Pango.Alignment.Center;
			textContext.EllipsizeMode = Pango.EllipsizeMode.None;
			
			Cairo.Color colorWhite = new Cairo.Color (1, 1, 1, 0.9);
			
			for (int day = 0; day < WeatherController.Weather.ForecastDays; day++)
			{
				cr.Color = colorTitle;
				textContext.Text = string.Format ("<b>{0}</b>", WeatherController.Weather.Forecasts [day].dow);
				textContext.LeftCenteredPoint = new Gdk.Point (2 * size * day, size / 6);
				textContext.FontSize = (int) (size / 6);
				DockServices.DrawingService.TextPathAtPoint (textContext);
				cr.Fill ();

				cr.Color = colorWhite;
				textContext.Text = string.Format ("<b>{0}</b>", WeatherController.Weather.Forecasts [day].condition);
				textContext.LeftCenteredPoint = new Gdk.Point (2 * size * day, size *4 / 6);
				textContext.FontSize = (int) (size / 7);
				DockServices.DrawingService.TextPathAtPoint (textContext);
				cr.Fill ();
			}
		}
		
		/// <summary>
		/// Moves to the next page in the painter, making it refresh.
		/// </summary>
		void NextPage ()
		{
			if (Page < pages - 1)
				Page++;
			else
				Page = 0;
			
			Dirty = true;
			OnPaintNeeded (new PaintNeededArgs ());
		}
		
		/// <summary>
		/// Moves to the previous page in the painter, making it refresh.
		/// </summary>
		void PreviousPage ()
		{
			if (Page > 0)
				Page--;
			else
				Page = pages - 1;
			
			Dirty = true;
			OnPaintNeeded (new PaintNeededArgs ());
		}
		
		protected override bool ReceiveClick (Gdk.Rectangle paintArea, Gdk.Point cursor)
		{
			NextPage ();
			
			return false;
		}
		
		public override void Scrolled (Gdk.ScrollDirection direction)
		{
			if (direction == Gdk.ScrollDirection.Down)
				NextPage ();
			else
				PreviousPage ();
		}
		
		/// <summary>
		/// Resets the painter to page 1 and then displays the painter.
		/// </summary>
		public void Summon ()
		{
			OnShowRequested ();
		}
		
		void HandleOnShowRequested (object o, EventArgs e)
		{
			if (Page != 0)
				Dirty = true;
			Page = 0;
		}
		
		/// <summary>
		/// Called when new weather data arrives, to purge the buffers and redraw.
		/// </summary>
		public void WeatherChanged ()
		{
			for (int i = 0; i < buffers.Length; i++)
				if (buffers [i] != null)
				{
					buffers [i].Destroy ();
					buffers [i] = null;
				}
			
			Dirty = true;
			OnPaintNeeded (new PaintNeededArgs ());
		}
		
		public override void Dispose ()
		{
			base.Dispose ();
		}
	}
}
