/**
 * gnome-gmail-notifier: the gnome gmail notifier.
 * Copyright (C) 2007 Bradley A. Worley.
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License,
 * or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the
 * 
 *     Free Software Foundation, Inc.
 *     59 Temple Place, Suite 330
 *     Boston, MA 02111-1307 USA
 **/

/*
 * include our application header.
 */
#include <main.h>

/*
 * type definitions.
 */
typedef struct _GgnAccount GgnAccount;

/*
 * account information struct.
 */
struct _GgnAccount {
    /* account-specific information. */
    gchar* name;
    gchar* user;
    gchar* pass;
    gboolean enab;
};

/*
 * private object definition.
 */
struct _GgnPreferencesPrivate {
    /* our xml parser. */
    GgnXmlParser* parser;
    
    /* our accounts. */
    gint count;
    GgnAccount* cur;
    GPtrArray* accounts;
    
    /* our general settings. */
    gint rate;
    gboolean use_msgs;
    gboolean use_errs;
    gboolean use_snds;
    gchar* snd_file;
    gboolean first_run;
    
    /* read-in status. */
    gboolean read;
};

/*
 * forward function definitions.
 */
static void ggn_preferences_init (GgnPreferences* self);
static void ggn_preferences_class_init (GgnPreferencesClass* klass);
static void ggn_preferences_finalize (GObject* obj);

/*
 * define the gobject type and its basic functions.
 */
G_DEFINE_TYPE (GgnPreferences, ggn_preferences, G_TYPE_OBJECT);

/*
 * define the signals used.
 */
enum {
    ACCOUNTS_MODIFIED,
    LAST_SIGNAL
};
static guint signals[LAST_SIGNAL] = { 0 };

/*
 * ggn_preferences_keyring_attribs:
 *
 * This function removes repeated code by generating the attributes array
 * for any given username in the default keyring. If this function were not
 * available, each function would have to recreate the array itself.
 *
 * Return value: a GnomeKeyringAttributeList.
 */
GnomeKeyringAttributeList* ggn_preferences_keyring_attribs (gchar* user) {
    /* initialize the attributes list. */
    GnomeKeyringAttribute attrib;
    GnomeKeyringAttributeList* keyattribs;
    keyattribs = gnome_keyring_attribute_list_new ();
    
    /* set the "user" attribute. */
    attrib.name = g_strdup ("user");
    attrib.type = GNOME_KEYRING_ATTRIBUTE_TYPE_STRING;
    attrib.value.string = g_strdup (user);
    g_array_append_val (keyattribs, attrib);
    
    /* set the "server" attribute. */
    attrib.name = g_strdup ("server");
    attrib.type = GNOME_KEYRING_ATTRIBUTE_TYPE_STRING;
    attrib.value.string = g_strdup (GGN_ATOM_FEED_SERVER);
    g_array_append_val (keyattribs, attrib);
    
    /* set the "object" attribute. */
    attrib.name = g_strdup ("object");
    attrib.type = GNOME_KEYRING_ATTRIBUTE_TYPE_STRING;
    attrib.value.string = g_strdup (GGN_ATOM_FEED_OBJECT);
    g_array_append_val (keyattribs, attrib);
    
    /* set the "protocol" attribute. */
    attrib.name = g_strdup ("protocol");
    attrib.type = GNOME_KEYRING_ATTRIBUTE_TYPE_STRING;
    attrib.value.string = g_strdup (GGN_ATOM_FEED_PROTOCOL);
    g_array_append_val (keyattribs, attrib);
    
    /* return the attributes array. */
    return keyattribs;
}

/*
 * ggn_preferences_keyring_create:
 *
 * This function uses gnome-keyring to create a new password entry in the
 * default keyring, updating a current entry if one exists already.
 *
 * Return value: void.
 */
void ggn_preferences_keyring_create (gchar* user, gchar* pass) {
    /* define some default values for saving to gnome-keyring. */
    const gchar* desc = g_strdup_printf ("Gmail password for %s", user);
    const gchar* secret = g_strdup (pass);
    
    /* define some variables for saving the passphrase. */
    guint32 id;
    GnomeKeyringResult result;
    GnomeKeyringAttributeList* keyattribs = ggn_preferences_keyring_attribs (user);
    
    /* synchronously write to the keyring. */
    result = gnome_keyring_item_create_sync (NULL, GNOME_KEYRING_ITEM_NETWORK_PASSWORD,
                                             desc, keyattribs, secret, TRUE, &id);
    
    /* free the attributes list. */
    gnome_keyring_attribute_list_free (keyattribs);
}

/*
 * ggn_preferences_keyring_find:
 *
 * This function uses gnome-keyring to find an already created password
 * entry in the default keyring, which may or may not exist.
 *
 * Return value: the passphrase.
 */
gchar* ggn_preferences_keyring_find (gchar* user) {
    /* define a passphrase to be returned. */
    gchar* phrase = NULL;
    
    /* define a list of keyring search results. */
    GList* found;
    
    /* define attributes by which to search for a key. */
    GnomeKeyringFound* item;
    GnomeKeyringResult result;
    GnomeKeyringAttributeList* keyattribs = ggn_preferences_keyring_attribs (user);
    
    /* search for the password. */
    result = gnome_keyring_find_items_sync (GNOME_KEYRING_ITEM_NETWORK_PASSWORD,
                                            keyattribs, &found);
    
    /* did we find a passphrase? */
    if (result == GNOME_KEYRING_RESULT_OK) {
        /* read the passphrase from the _first_ found item. */
        item = g_list_nth_data (found, 0);
        phrase = g_strdup (item->secret);
    }
    
    /* free the variables used for the search. */
    gnome_keyring_found_list_free (found);
    gnome_keyring_attribute_list_free (keyattribs);
    
    /* return the passphrase. */
    return phrase;
}

/*
 * ggn_preferences_keyring_delete:
 *
 * This function uses gnome-keyring to delete an already created password
 * entry in the default keyring, which may or may not exist.
 *
 * Return value: void.
 */
void ggn_preferences_keyring_delete (gchar* user) {
    /* define a list of keyring search results. */
    GList* found;
    
    /* define attributes by which to search for a key. */
    GnomeKeyringFound* item;
    GnomeKeyringResult result;
    GnomeKeyringAttributeList* keyattribs = ggn_preferences_keyring_attribs (user);
    
    /* search for the password. */
    result = gnome_keyring_find_items_sync (GNOME_KEYRING_ITEM_NETWORK_PASSWORD,
                                            keyattribs, &found);
    
    /* did we find a passphrase? */
    if (result == GNOME_KEYRING_RESULT_OK) {
        /* read the passphrase from the _first_ found item. */
        item = g_list_nth_data (found, 0);
        
        /* delete this item. */
        gnome_keyring_item_delete_sync (NULL, item->item_id);
    }
    
    /* free the variables used for the search. */
    gnome_keyring_found_list_free (found);
    gnome_keyring_attribute_list_free (keyattribs);
}

/*
 * ggn_preferences_xml_begin:
 *
 * This function handles all tag openings in the parsed XML string.
 *
 * Return value: void.
 */
void ggn_preferences_xml_begin (GgnXmlParser* parser,
                                gchar* xpath, gchar* name,
                                GHashTable* attributes,
                                gpointer data) {
    /* cast the preferences file. */
    GgnPreferences* prefs = GGN_PREFERENCES (data);
    
    /* see what the path is. */
    if (g_utf8_collate (xpath, GGN_PREFS_XPATH_ACCOUNT) == 0) {
        /* make a new message object. */
        prefs->priv->cur = g_new0 (GgnAccount, 1);
        
        /* increment the count. */
        prefs->priv->count++;
        
        /* get the account name. */
        if (attributes != NULL) {
            /* use the values in xml. */
            prefs->priv->cur->name = g_strdup (g_hash_table_lookup (attributes, "name"));
            prefs->priv->cur->enab = ggn_strtob (g_hash_table_lookup (attributes, "enabled"));
        }
        else {
            /* use a default values. */
            prefs->priv->cur->name = g_strdup (_("Unnamed Account"));
            prefs->priv->cur->enab = TRUE;
        }
    }
}

/*
 * ggn_preferences_xml_end:
 *
 * This function handles all tag closings in the parsed XML string.
 *
 * Return value: void.
 */
void ggn_preferences_xml_end (GgnXmlParser* parser,
                              gchar* xpath,
                              gchar* name,
                              gpointer data) {
    /* cast the preferences file. */
    GgnPreferences* prefs = GGN_PREFERENCES (data);
    
    /* see what the path is. */
    if (g_utf8_collate (xpath, GGN_PREFS_XPATH_ACCOUNT) == 0) {
        /* move our current message into the array. */
        g_ptr_array_add (prefs->priv->accounts, prefs->priv->cur);
    }
}

/*
 * ggn_preferences_xml_text:
 *
 * This function handles all text content in the parsed XML string.
 *
 * Return value: void.
 */
void ggn_preferences_xml_text (GgnXmlParser* parser,
                               gchar* xpath,
                               gchar* name,
                               gchar* value,
                               gpointer data) {
    /* cast the preferences file. */
    GgnPreferences* prefs = GGN_PREFERENCES (data);
    
    /* see what the path is. */
    if (g_utf8_collate (xpath, GGN_PREFS_XPATH_USERNAME) == 0) {
        /* read our value. */
        prefs->priv->cur->user = g_strdup (value);
        
        /* try to get the passphrase from gnome-keyring now. */
        prefs->priv->cur->pass = ggn_preferences_keyring_find (value);
    }
    else if (g_utf8_collate (xpath, GGN_PREFS_XPATH_RATE) == 0) {
        /* read our value. */
        prefs->priv->rate = (gint) g_ascii_strtoll (value, NULL, 10);
    }
    else if (g_utf8_collate (xpath, GGN_PREFS_XPATH_MESSAGES) == 0) {
        /* read our value. */
        prefs->priv->use_msgs = ggn_strtob (value);
    }
    else if (g_utf8_collate (xpath, GGN_PREFS_XPATH_ERRORS) == 0) {
        /* read our value. */
        prefs->priv->use_errs = ggn_strtob (value);
    }
    else if (g_utf8_collate (xpath, GGN_PREFS_XPATH_SOUNDS) == 0) {
        /* read our value. */
        prefs->priv->use_snds = ggn_strtob (value);
    }
    else if (g_utf8_collate (xpath, GGN_PREFS_XPATH_SOUND) == 0) {
        /* read our value. */
        prefs->priv->snd_file = g_strdup (value);
    }
    else if (g_utf8_collate (xpath, GGN_PREFS_XPATH_FIRSTRUN) == 0) {
        /* read our value. */
        prefs->priv->first_run = ggn_strtob (value);
    }
}

/*
 * ggn_preferences_default_accounts_modified_cb:
 *
 * This is the default "accounts_modified" callback function manager.
 *
 * Return value: void.
 */
static void ggn_preferences_default_accounts_modified_cb (GgnPreferences* prefs) {
    /* exit the function. */
    return;
}

/*
 * ggn_preferences_init:
 *
 * This function is used by the gobject library to
 * generate a new instance of our object.
 */
static void ggn_preferences_init (GgnPreferences* self) {
    /* set up the private data structure. */
    self->priv = g_new0 (GgnPreferencesPrivate, 1);
    
    /* setup simplest objects. */
    self->priv->count = 0;
    self->priv->rate = 0;
    self->priv->use_msgs = FALSE;
    self->priv->use_errs = FALSE;
    self->priv->use_snds = FALSE;
    self->priv->snd_file = NULL;
    self->priv->first_run = FALSE;
    self->priv->read = FALSE;
    
    /* setup the account array. */
    self->priv->accounts = g_ptr_array_new ();
    
    /* setup the xml parser. */
    self->priv->parser = ggn_xml_parser_new ();
    
    /* link the begin_element callback into our parser. */
    g_signal_connect (G_OBJECT (self->priv->parser),
                      "begin_element",
                      G_CALLBACK (ggn_preferences_xml_begin),
                      self);
    
    /* link the end_element callback into our parser. */
    g_signal_connect (G_OBJECT (self->priv->parser),
                      "end_element",
                      G_CALLBACK (ggn_preferences_xml_end),
                      self);
    
    /* link the text callback into our parser. */
    g_signal_connect (G_OBJECT (self->priv->parser),
                      "text",
                      G_CALLBACK (ggn_preferences_xml_text),
                      self);
}

/*
 * ggn_preferences_class_init:
 *
 * This function is used by the gobject library to
 * generate a new class object of our object.
 */
static void ggn_preferences_class_init (GgnPreferencesClass* klass) {
    /* setup a gobject class. */
    GObjectClass* gobj_class = G_OBJECT_CLASS (klass);
    
    /* set the locations of our destruction function. */
    gobj_class->finalize = ggn_preferences_finalize;
    
    /* setup the default signal handlers. */
    klass->accounts_modified = ggn_preferences_default_accounts_modified_cb;
    
    /*
     * GgnPreferences::accounts_modified:
     *
     * Emitted when any data pertaining to accounts
     * has been changed within the XML preferences file.
     */
    signals[ACCOUNTS_MODIFIED] = g_signal_new ("accounts_modified",
                                               G_OBJECT_CLASS_TYPE (gobj_class),
                                               G_SIGNAL_RUN_FIRST,
                                               G_STRUCT_OFFSET (GgnPreferencesClass, accounts_modified),
                                               NULL, NULL,
                                               ggn_marshal_VOID__VOID,
                                               G_TYPE_NONE, 0);
}

/*
 * ggn_preferences_finalize:
 *
 * This function is used by the gobject library to cleanly finish
 * the destruction process started by the dispose function.
 */
static void ggn_preferences_finalize (GObject* obj) {
    /* make a reference to ourself. */
    GgnPreferences* self = GGN_PREFERENCES (obj);
    
    /* free the string values. */
    g_free (self->priv->snd_file);
    
    /* reset the integer objects. */
    self->priv->count = 0;
    self->priv->rate = 0;
    
    /* reset the boolean objects. */
    self->priv->use_msgs = FALSE;
    self->priv->use_errs = FALSE;
    self->priv->use_snds = FALSE;
    self->priv->first_run = FALSE;
    
    /* free the parser. */
    ggn_xml_parser_free (self->priv->parser);
    g_ptr_array_free (self->priv->accounts, FALSE);
    
    /* destroy the private object. */
    g_free (self->priv);
    self->priv = NULL;
    
    /* chain up to the parent class. */
    G_OBJECT_CLASS (ggn_preferences_parent_class)->finalize (obj);
}

/*
 * ggn_preferences_new:
 *
 * Creates a new GgnPreferences with default values, which are
 * used to read preferences files.
 *
 * Return value: the new xml parser.
 */
GgnPreferences* ggn_preferences_new (void) {
    /* make a newly created gobject. */
    GgnPreferences* prefs = g_object_new (GGN_TYPE_PREFERENCES, NULL);
    
    /* return the new object. */
    return prefs;
}

/*
 * ggn_preferences_free:
 *
 * Frees the given preferences object by decreasing its reference count.
 *
 * Return value: void.
 */
void ggn_preferences_free (GgnPreferences* prefs) {
    /* unreference the object. */
    while (G_IS_OBJECT (prefs)) {
        /* unreference this object. */
        g_object_unref (G_OBJECT (prefs));
    }
}

/*
 * ggn_preferences_ensure:
 *
 * This function ensures that the preferences file exists in
 * a controlled environment.
 *
 * Return value: Filename of the Preferences file.
 */
gchar* ggn_preferences_ensure (void) {
    /* determine the directory name. */
    gchar* dirname = g_strdup_printf ("%s/.gnome2/%s",
                                      g_get_home_dir (),
                                      g_get_prgname ());
    
    /* make sure the directory exists. */
    if (g_file_test (dirname, G_FILE_TEST_IS_DIR) == FALSE) {
        /* create the directory. */
        g_mkdir (dirname, S_IRWXU);
    }
    
    /* determine the file name. */
    gchar* filename = g_strdup_printf ("%s/preferences", dirname);
    
    /* make sure the file exists. */
    if (g_file_test (filename, G_FILE_TEST_IS_REGULAR) == FALSE) {
        /* create the file contents. */
        gchar* content = g_strdup_printf (GGN_PREFERENCES_WFORMAT,
                                          _("My Gmail Inbox"),
                                          ggn_btostr (TRUE), "",
                                          10, ggn_btostr (TRUE),
                                          ggn_btostr (FALSE),
                                          ggn_btostr (TRUE),
                                          "/usr/share/sounds/pop.wav",
                                          ggn_btostr (TRUE));
        
        /* dump the file's contents. */
        g_file_set_contents (filename, content, -1, NULL);
        
        /* free the content string. */
        g_free (content);
    }
    
    /* free the directory string. */
    g_free (dirname);
    
    /* return the filename. */
    return filename;
}

/*
 * ggn_preferences_read:
 *
 * This function loads the notifier's preferences XML file and parses
 * it. If the preferences file, which by default is saved to the file
 * ~/.gnome2/gnome-gmail-notifier/preferences, does not exist, then a
 * default file (and if neccessary, directory tree) is created.
 *
 * Return value: success.
 */
gboolean ggn_preferences_read (GgnPreferences* prefs) {
    /* don't reread this. */
    if (prefs->priv->read == TRUE) {
        /* bail. */
        return TRUE;
    }
    
    /* make sure the preferences file exists. */
    gchar* filename = ggn_preferences_ensure ();
    
    /* parse the xml file. */
    ggn_xml_parser_load_file (prefs->priv->parser, filename);
    
    /* free the filename. */
    g_free (filename);
    
    /* set the read variable. */
    prefs->priv->read = TRUE;
    
    /* exit. */
    return TRUE;
}

/*
 * ggn_preferences_write:
 *
 * This function saves the current settings to the preferences XML file.
 * If the preferences file, which by default is saved to the file
 * ~/.gnome2/gnome-gmail-notifier/preferences, does not exist, then a
 * default file (and if neccessary, directory tree) is created.
 *
 * Return value: success.
 */
gboolean ggn_preferences_write (GgnPreferences* prefs) {
    /* make sure the prefs file was read. */
    if (prefs->priv->read == FALSE) {
        /* read in the file. */
        ggn_preferences_read (prefs);
    }
    
    /* determine the filename. */
    gchar* filename = g_strdup_printf ("%s/.gnome2/%s/preferences",
                                       g_get_home_dir (),
                                       g_get_prgname ());
    
    /* create the file contents. */
    GString* gstr = g_string_new ("<?xml version=\"1.0\" encoding=\"utf-8\"?>\n");
    g_string_append_printf (gstr, "<preferences>\n");
    
    /* output the accounts. */
    gint num;
    for (num = 0; num < ggn_preferences_get_accounts (prefs); num++) {
        /* print the account info to the file. */
        g_string_append_printf (gstr, "    <account name=\"%s\" enabled=\"%s\">\n",
                                ggn_preferences_get_account_name (prefs, num),
                                ggn_btostr (ggn_preferences_get_account_enab (prefs, num)));
        g_string_append_printf (gstr, "        <username>%s</username>\n",
                                ggn_preferences_get_account_username (prefs, num));
        g_string_append_printf (gstr, "    </account>\n");
    }
    
    /* finish the file contents. */
    g_string_append_printf (gstr, "    <application>\n");
    g_string_append_printf (gstr, "        <checkrate>%d</checkrate>\n",
                            ggn_preferences_get_rate (prefs));
    g_string_append_printf (gstr, "        <notification>\n");
    g_string_append_printf (gstr, "            <messages>%s</messages>\n",
                            ggn_btostr (ggn_preferences_get_notify_msgs (prefs)));
    g_string_append_printf (gstr, "            <errors>%s</errors>\n",
                            ggn_btostr (ggn_preferences_get_notify_errs (prefs)));
    g_string_append_printf (gstr, "            <sounds>%s</sounds>\n",
                            ggn_btostr (ggn_preferences_get_notify_snds (prefs)));
    g_string_append_printf (gstr, "            <sound>%s</sound>\n",
                            ggn_preferences_get_soundfile (prefs));
    g_string_append_printf (gstr, "        </notification>\n");
    g_string_append_printf (gstr, "        <firstrun>%s</firstrun>\n",
                            ggn_btostr (ggn_preferences_get_firstrun (prefs)));
    g_string_append_printf (gstr, "    </application>\n");
    g_string_append_printf (gstr, "</preferences>\n");
    
    /* dump the file's contents. */
    g_file_set_contents (filename, gstr->str, gstr->len, NULL);
    
    /* free the content strings. */
    g_string_free (gstr, FALSE);
    g_free (filename);
    
    /* exit. */
    return TRUE;
}

/*
 * ggn_preferences_get_accounts:
 *
 * Returns the number of accounts listed in the preferences file.
 *
 * Return value: Accounts count.
 */
gint ggn_preferences_get_accounts (GgnPreferences* prefs) {
    /* return verbatim. */
    return prefs->priv->count;
}

/*
 * ggn_preferences_get_account_enab:
 *
 * Returns whether or not an account is enabled.
 *
 * Return value: Account enabled boolean.
 */
gboolean ggn_preferences_get_account_enab (GgnPreferences* prefs, gint num) {
    /* exit if we have a bounds error. */
    if ((num > prefs->priv->count - 1) || (num < 0)) {
        /* out-of-bounds. */
        return FALSE;
    }
    
    /* get the account. */
    GgnAccount* account = g_ptr_array_index (prefs->priv->accounts, num);
    
    /* return the value, if non-null. */
    return account->enab;
}

/*
 * ggn_preferences_get_account_name:
 *
 * Returns the name of an account given by an integer index value.
 * If an error occured or the index is out of bounds, the function
 * will return NULL.
 *
 * Return value: Account name string.
 */
gchar* ggn_preferences_get_account_name (GgnPreferences* prefs, gint num) {
    /* exit if we have a bounds error. */
    if ((num > prefs->priv->count - 1) || (num < 0)) {
        /* out-of-bounds. */
        return NULL;
    }
    
    /* get the account. */
    GgnAccount* account = g_ptr_array_index (prefs->priv->accounts, num);
    
    /* return the value, if non-null. */
    if (account->name != NULL) {
        /* return verbatim. */
        return g_strdup (account->name);
    }
    else {
        /* return a default. */
        return g_strdup ("");
    }
}

/*
 * ggn_preferences_get_account_username:
 *
 * Returns the username of an account given by an integer index value.
 * If an error occured or the index is out of bounds, the function
 * will return NULL.
 *
 * Return value: Account user string.
 */
gchar* ggn_preferences_get_account_username (GgnPreferences* prefs, gint num) {
    /* exit if we have a bounds error. */
    if ((num > prefs->priv->count - 1) || (num < 0)) {
        /* out-of-bounds. */
        return g_strdup ("");
    }
    
    /* get the account. */
    GgnAccount* account = g_ptr_array_index (prefs->priv->accounts, num);
    
    /* return the value, if non-null. */
    if (account->user != NULL) {
        /* return verbatim. */
        return g_strdup (account->user);
    }
    else {
        /* return a default. */
        return g_strdup ("");
    }
}

/*
 * ggn_preferences_get_account_password:
 *
 * Returns the password of an account given by an integer index value.
 * If an error occured or the index is out of bounds, the function
 * will return NULL.
 *
 * Return value: Account password string.
 */
gchar* ggn_preferences_get_account_password (GgnPreferences* prefs, gint num) {
    /* exit if we have a bounds error. */
    if ((num > prefs->priv->count - 1) || (num < 0)) {
        /* out-of-bounds. */
        return NULL;
    }
    
    /* get the account. */
    GgnAccount* account = g_ptr_array_index (prefs->priv->accounts, num);
    
    /* return the value, if non-null. */
    if (account->pass != NULL) {
        /* return verbatim. */
        return g_strdup (account->pass);
    }
    else {
        /* return a default. */
        return g_strdup ("");
    }
}

/*
 * ggn_preferences_get_proxy:
 *
 * Returns the proxy server URI in use by the application for
 * its connection to the Gmail server.
 *
 * Return value: Proxy URI string.
 */
gchar* ggn_preferences_get_proxy (GgnPreferences* prefs) {
    /* define a uri string. */
    gchar* uri = NULL;
    
    /* grab the proxy enabled value. */
    gboolean enabled = gconf_client_get_bool (gconf_client_get_default (),
                                              GGN_GCONF_PROXY_USE,
                                              NULL);
    
    /* grab the proxy insecure host value. */
    gchar* host = gconf_client_get_string (gconf_client_get_default (),
                                           GGN_GCONF_PROXY_HOST,
                                           NULL);
    
    /* grab the proxy insecure port value. */
    gint port = gconf_client_get_int (gconf_client_get_default (),
                                      GGN_GCONF_PROXY_PORT,
                                      NULL);
    
    /* grab the proxy secure host value. */
    gchar* ssl_host = gconf_client_get_string (gconf_client_get_default (),
                                               GGN_GCONF_PROXY_SSL_HOST,
                                               NULL);
    
    /* grab the proxy secure port value. */
    gint ssl_port = gconf_client_get_int (gconf_client_get_default (),
                                          GGN_GCONF_PROXY_SSL_PORT,
                                          NULL);
    
    /* see if the proxy is enabled. */
    if (enabled == TRUE) {
        /* make a qualified string. */
        if ((strlen (ssl_host) > 0) && (ssl_port != 0)) {
            /* use the secure host and port. */
            uri = g_strdup_printf ("http://%s:%d", ssl_host, ssl_port);
        }
        else if ((strlen (host) > 0) && (port != 0)) {
            /* use the regular host and port. */
            uri = g_strdup_printf ("http://%s:%d", host, port);
        }
        else {
            /* return a default. */
            uri = g_strdup ("");
        }
    }
    else {
        /* return a default. */
        uri = g_strdup ("");
    }
    
    /* free the strings we got from gconf. */
    g_free (host);
    g_free (ssl_host);
    
    /* return the final value. */
    return uri;
}

/*
 * ggn_preferences_get_rate:
 *
 * Returns the frequency (in minutes) that the application will
 * check the user's inbox(es) for new messages.
 *
 * Return value: Update frequency.
 */
gint ggn_preferences_get_rate (GgnPreferences* prefs) {
    /* return verbatim. */
    return prefs->priv->rate;
}

/*
 * ggn_preferences_get_notify_msgs:
 *
 * Returns whether or not the application will notify the user
 * when they have new messages.
 *
 * Return value: Boolean.
 */
gboolean ggn_preferences_get_notify_msgs (GgnPreferences* prefs) {
    /* return verbatim. */
    return prefs->priv->use_msgs;
}

/*
 * ggn_preferences_get_notify_errs:
 *
 * Returns whether or not the application will notify the user
 * when an error occurs while accessing their Gmail inbox(es).
 *
 * Return value: Boolean.
 */
gboolean ggn_preferences_get_notify_errs (GgnPreferences* prefs) {
    /* return verbatim. */
    return prefs->priv->use_errs;
}

/*
 * ggn_preferences_get_notify_snds:
 *
 * Returns whether or not the application will use a sound file
 * to notify the user when they have new messages.
 *
 * Return value: Boolean.
 */
gboolean ggn_preferences_get_notify_snds (GgnPreferences* prefs) {
    /* return verbatim. */
    return prefs->priv->use_snds;
}

/*
 * ggn_preferences_get_soundfile:
 *
 * Returns the filename of the *.WAV sound file which will be
 * played when the user recieves new mail, if the user selected
 * to be notified with sounds.
 *
 * Return value: Sound filename string.
 */
gchar* ggn_preferences_get_soundfile (GgnPreferences* prefs) {
    /* return the value, if non-null. */
    if (prefs->priv->snd_file != NULL) {
        /* return verbatim. */
        return g_strdup (prefs->priv->snd_file);
    }
    else {
        /* return a default. */
        return g_strdup ("");
    }
}

/*
 * ggn_preferences_get_firstrun:
 *
 * Returns whether or not this function is running for the
 * first time ever or not. This value will only be TRUE when
 * the user installs their FIRST notifier version.
 *
 * Return value: Boolean.
 */
gboolean ggn_preferences_get_firstrun (GgnPreferences* prefs) {
    /* return verbatim. */
    return prefs->priv->first_run;
}

/*
 * ggn_preferences_set_account_enab:
 *
 * Sets the activity of an account by a provided index.
 *
 * Return value: void.
 */
void ggn_preferences_set_account_enab (GgnPreferences* prefs,
                                       gint num,
                                       gboolean value) {
    /* exit if we have a bounds error. */
    if ((num > prefs->priv->count - 1) || (num < 0)) {
        /* out-of-bounds. */
        return;
    }
    
    /* get the account. */
    GgnAccount* account = g_ptr_array_index (prefs->priv->accounts, num);
    
    /* set the new value. */
    account->enab = value;
    
    /* emit the "accounts_modified" signal. */
    g_signal_emit (prefs, signals[ACCOUNTS_MODIFIED], 0);
}

/*
 * ggn_preferences_set_account_name:
 *
 * Sets the name of an account at the given index, by pulling its
 * account struct pointer from the pointer array and changing that
 * data.
 *
 * Return value: void.
 */
void ggn_preferences_set_account_name (GgnPreferences* prefs,
                                       gint num,
                                       gchar* value) {
    /* exit if we have a bounds error. */
    if ((num > prefs->priv->count - 1) || (num < 0)) {
        /* out-of-bounds. */
        return;
    }
    
    /* get the account. */
    GgnAccount* account = g_ptr_array_index (prefs->priv->accounts, num);
    
    /* do we free the original value? */
    if (account->name != NULL) {
        /* free the string. */
        g_free (account->name);
    }
    
    /* set the new value. */
    account->name = g_strdup (value);
    
    /* emit the "accounts_modified" signal. */
    g_signal_emit (prefs, signals[ACCOUNTS_MODIFIED], 0);
}

/*
 * ggn_preferences_set_account_username:
 *
 * Sets the user name of an account at the given index, by pulling its
 * account struct pointer from the pointer array and changing that
 * data.
 *
 * Return value: void.
 */
void ggn_preferences_set_account_username (GgnPreferences* prefs,
                                           gint num,
                                           gchar* value) {
    /* exit if we have a bounds error. */
    if ((num > prefs->priv->count - 1) || (num < 0)) {
        /* out-of-bounds. */
        return;
    }
    
    /* get the account. */
    GgnAccount* account = g_ptr_array_index (prefs->priv->accounts, num);
    
    /* do we free the original value? */
    if (account->user != NULL) {
        /* free the string. */
        g_free (account->user);
    }
    
    /* set the new value. */
    account->user = g_strdup (value);
    
    /* emit the "accounts_modified" signal. */
    g_signal_emit (prefs, signals[ACCOUNTS_MODIFIED], 0);
}

/*
 * ggn_preferences_set_account_password:
 *
 * Sets the password of an account at the given index, by pulling its
 * account struct pointer from the pointer array and changing that
 * data.
 *
 * Return value: void.
 */
void ggn_preferences_set_account_password (GgnPreferences* prefs,
                                           gint num,
                                           gchar* value) {
    /* exit if we have a bounds error. */
    if ((num > prefs->priv->count - 1) || (num < 0)) {
        /* out-of-bounds. */
        return;
    }
    
    /* get the account. */
    GgnAccount* account = g_ptr_array_index (prefs->priv->accounts, num);
    
    /* do we free the original value? */
    if (account->pass != NULL) {
        /* free the string. */
        g_free (account->pass);
    }
    
    /* set the new value. */
    account->pass = g_strdup (value);
    
    /* set the passphrase in gnome-keyring too, where it counts. */
    ggn_preferences_keyring_create (account->user, value);
    
    /* emit the "accounts_modified" signal. */
    g_signal_emit (prefs, signals[ACCOUNTS_MODIFIED], 0);
}

/*
 * ggn_preferences_set_rate:
 *
 * Sets the update frequency (in minutes) of the application
 * to be saved the next time the preferences XML file is
 * written to disk. (or... ramdisk?) :-P
 *
 * Return value: void.
 */
void ggn_preferences_set_rate (GgnPreferences* prefs, gint value) {
    /* who doesn't love integers? */
    prefs->priv->rate = value;
}

/*
 * ggn_preferences_set_notify_msgs:
 *
 * Sets whether or not to notify on new messages.
 *
 * Return value: void.
 */
void ggn_preferences_set_notify_msgs (GgnPreferences* prefs, gboolean value) {
    /* set the value. */
    prefs->priv->use_msgs = value;
}

/*
 * ggn_preferences_set_notify_errs:
 *
 * Sets whether or not to notify on errors.
 *
 * Return value: void.
 */
void ggn_preferences_set_notify_errs (GgnPreferences* prefs, gboolean value) {
    /* set the value. */
    prefs->priv->use_errs = value;
}

/*
 * ggn_preferences_set_notify_snds:
 *
 * Sets whether or not to notify with a sound file.
 *
 * Return value: void.
 */
void ggn_preferences_set_notify_snds (GgnPreferences* prefs, gboolean value) {
    /* set the value. */
    prefs->priv->use_snds = value;
}

/*
 * ggn_preferences_set_soundfile:
 *
 * Sets the filename of the *.WAV sound file, which is played when
 * the user has more than zero messages in his inbox. (Not necessarily
 * new, however.)
 *
 * Return value: void.
 */
void ggn_preferences_set_soundfile (GgnPreferences* prefs, gchar* value) {
    /* do we free the original value? */
    if (prefs->priv->snd_file != NULL) {
        /* free the string. */
        g_free (prefs->priv->snd_file);
    }
    
    /* set the new value. */
    prefs->priv->snd_file = g_strdup (value);
}

/*
 * ggn_preferences_set_firstrun:
 *
 * Sets the first-run behavior when the application loads up. This is
 * pretty much not even here. Ignore it. It is... INVISIBLE!
 *
 * Return value: void.
 */
void ggn_preferences_set_firstrun (GgnPreferences* prefs, gboolean value) {
    /* set the value. */
    prefs->priv->first_run = value;
}

/*
 * ggn_preferences_get_account_by_name:
 *
 * Returns a given account's ID after looking it up by name. If
 * no match occurs, then the function returns -1.
 *
 * Return value: Account ID.
 */
gint ggn_preferences_get_account_by_name (GgnPreferences* prefs,
                                          gchar* value) {
    /* declare helping variables. */
    gint num;
    gchar* name;
    
    /* loop through the accounts. */
    for (num = 0; num < ggn_preferences_get_accounts (prefs); num++) {
        /* get the account name. */
        name = ggn_preferences_get_account_name (prefs, num);
        
        /* check if our name matches. */
        if (g_utf8_collate (name, value) == 0) {
            /* free the string and return. */
            g_free (name);
            return num;
        }
        
        /* free the account name. */
        g_free (name);
    }
    
    /* our check failed. */
    return -1;
}

/*
 * ggn_preferences_add_account:
 *
 * Adds an account with the provided descriptive name, user name,
 * and password string.
 *
 * Return value: void.
 */
void ggn_preferences_add_account (GgnPreferences* prefs,
                                  gchar* name,
                                  gchar* user,
                                  gchar* pass) {
    /* create a new account. */
    GgnAccount* account = g_new0 (GgnAccount, 1);
    
    /* set the values. */
    account->name = g_strdup (name);
    account->user = g_strdup (user);
    account->pass = g_strdup (pass);
    
    /* also save the passphrase to gnome-keyring. */
    /* it's probably safe to not call this here, as the only place
     * this function is called from defines the username as "".
     * ggn_preferences_keyring_create (user, pass);
     */
    
    /* add the account. */
    g_ptr_array_add (prefs->priv->accounts, account);
    prefs->priv->count++;
    
    /* emit the "accounts_modified" signal. */
    g_signal_emit (prefs, signals[ACCOUNTS_MODIFIED], 0);
}

/*
 * ggn_preferences_remove_account:
 *
 * Removes a given account by looking it up by its ID.
 *
 * Return value: void.
 */
void ggn_preferences_remove_account (GgnPreferences* prefs,
                                     gint num) {
    /* exit if we have a bounds error. */
    if ((num > prefs->priv->count - 1) || (num < 0)) {
        /* out-of-bounds. */
        return;
    }
    
    /* remove the passphrase from gnome-keyring. */
    GgnAccount* account = g_ptr_array_index (prefs->priv->accounts, num);
    ggn_preferences_keyring_delete (account->user);
    
    /* remove it. */
    g_ptr_array_remove_index (prefs->priv->accounts, num);
    prefs->priv->count--;
    
    /* emit the "accounts_modified" signal. */
    g_signal_emit (prefs, signals[ACCOUNTS_MODIFIED], 0);
}

/*
 * ggn_preferences_remove_account_by_name:
 *
 * Removes a given account by looking it up by its
 * descriptive name.
 *
 * Return value: void.
 */
void ggn_preferences_remove_account_by_name (GgnPreferences* prefs,
                                             gchar* name) {
    /* try and find the account. */
    gint index = ggn_preferences_get_account_by_name (prefs, name);
    
    /* if we found it, remove it. */
    if (index >= 0) {
        /* remove the passphrase from gnome-keyring. */
        GgnAccount* account = g_ptr_array_index (prefs->priv->accounts, index);
        ggn_preferences_keyring_delete (account->user);
        
        /* remove it. */
        g_ptr_array_remove_index (prefs->priv->accounts, index);
        prefs->priv->count--;
    }
    
    /* emit the "accounts_modified" signal. */
    g_signal_emit (prefs, signals[ACCOUNTS_MODIFIED], 0);
}
