/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * Copyright (C) 2005 Imendio AB
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License as
 * published by the Free Software Foundation; either version 2 of the
 * License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 *
 * You should have received a copy of the GNU General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA 02111-1307, USA.
 */

/*
 *  lb-module-recent.c
 */

#include "config.h"

#include <string.h>

#include <gdk-pixbuf/gdk-pixbuf.h>

#include "lb-item-file.h"
#include "lb-module-recent.h"
#include "lb-recent-list.h"
#include "lb-recent-item.h"
#include "lb-utils.h"


static void    lb_module_recent_class_init   (LbModuleRecentClass *klass);
static void    lb_module_recent_init         (LbModuleRecent      *recent);

static void    lb_module_recent_finalize     (GObject             *object);

static GList * lb_module_recent_query        (LbModule            *module,
					      const gchar         *match);


static LbModuleClass *parent_class = NULL;


GType
lb_module_recent_get_type (void)
{
  static GType type = 0;

  if (! type) {
	  static const GTypeInfo info = {
		  sizeof (LbModuleRecentClass),
		  (GBaseInitFunc) NULL,
		  (GBaseFinalizeFunc) NULL,
		  (GClassInitFunc) lb_module_recent_class_init,
		  NULL,           /* class_finalize */
		  NULL,           /* class_data     */
		  sizeof (LbModuleRecent),
		  0,              /* n_preallocs    */
		  (GInstanceInitFunc) lb_module_recent_init,
	  };

	  type = g_type_register_static (LB_TYPE_MODULE,
					 "LbModuleRecent",
					 &info, 0);
    }

  return type;
}

static void
lb_module_recent_class_init (LbModuleRecentClass *klass)
{
	GObjectClass  *object_class = G_OBJECT_CLASS (klass);
	LbModuleClass *module_class = LB_MODULE_CLASS (klass);

	parent_class = g_type_class_peek_parent (klass);

	object_class->finalize     = lb_module_recent_finalize;
	module_class->query        = lb_module_recent_query;
}

static void
lb_module_recent_init (LbModuleRecent *recent)
{
}

static void
lb_module_recent_finalize (GObject *object)
{
	LbModuleRecent *recent = LB_MODULE_RECENT (object);

	if (recent->recent_items) {
		g_list_foreach (recent->recent_items,
				(GFunc) lb_recent_item_unref, NULL);
		g_list_free (recent->recent_items);
	}

	G_OBJECT_CLASS (parent_class)->finalize (object);
}

static GList *
lb_module_recent_query (LbModule    *module,
			const gchar *match)
{
	LbModuleRecent *recent = LB_MODULE_RECENT (module);
	GList          *list;
	GList          *result = NULL;

	if (recent->recent_items) {
		if (recent->mtime != lb_recent_list_get_mtime ()) {
			g_list_foreach (recent->recent_items,
					(GFunc) lb_recent_item_unref, NULL);
			g_list_free (recent->recent_items);
			recent->recent_items = NULL;
			recent->mtime        = 0;
		}
	}

	if (! recent->recent_items) {
		recent->recent_items = lb_recent_list_get ();
		recent->mtime        = lb_recent_list_get_mtime ();
	}

	for (list = recent->recent_items; list; list = list->next) {
		LbRecentItem *item = list->data;
		const gchar  *uri;
		gchar        *basename;

		uri      = lb_recent_item_get_uri (item);
		basename = g_path_get_basename (uri);

		if (lb_string_has_substring (basename, match)) {
			LbItem      *new;
			const gchar *mime_type;
			gchar       *icon_name;
			GdkPixbuf   *pixbuf = NULL;

			mime_type = lb_recent_item_get_mime_type (item);

			icon_name = lb_get_icon_name_for_uri (uri, NULL);

			if (icon_name) {
				pixbuf = lb_get_pixbuf_from_icon_name (icon_name);
				g_free (icon_name);
			}

			new = g_object_new (LB_TYPE_ITEM_FILE,
					    "name",      basename,
					    "uri",       uri,
					    "mime-type", mime_type,
					    "pixbuf",    pixbuf,
					    NULL);

			if (pixbuf) {
				g_object_unref (pixbuf);
			}

			result = g_list_prepend (result, new);
		}
	}

	return g_list_reverse (result);
}
