//  Gnomoradio - roboradio/audio/ogg.cc
//  Copyright (C) 2003  Jim Garrison
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version)
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

#include "roboradio/audio/ogg.h"
#include <ao/ao.h>

#include <sys/time.h>
#include <sys/types.h>
#include <sys/select.h>
#include <unistd.h>

#include <string>
#include <algorithm>

#include <iostream>

#if USE_VORBIS

using namespace std;
using namespace Glib;

Roboradio::Audio::Ogg::Ogg (const ustring &file)
	: filename(file),
	  thread(0)
{
	static bool initialized = false;
	if (!initialized) {
		ao_initialize();
		initialized = true;
	}

	done_dispatch.connect(mem_fun(*this, &Ogg::send_done));
	position_dispatch.connect(mem_fun(*this, &Ogg::send_position_changed));
}

Roboradio::Audio::Ogg::~Ogg ()
{
	if (thread)
		stop();
}

void Roboradio::Audio::Ogg::play ()
{
	if (thread)
		return;

	_pause = false;
	_stop = false;
	_position = 0;
	_seek = -1;

	if (!thread_supported())
		thread_init();
	thread = Thread::create(sigc::mem_fun(*this, &Ogg::thread_function), true);
}

void Roboradio::Audio::Ogg::stop ()
{
	if (!thread)
		return;

	mutex.lock();
	_stop = true;
	mutex.unlock();
	thread->join();
	thread = 0;
}

void Roboradio::Audio::Ogg::pause ()
{
	Mutex::Lock lock(mutex);
	_pause = true;
}

void Roboradio::Audio::Ogg::unpause ()
{
	Mutex::Lock lock(mutex);
	_pause = false;
}

void Roboradio::Audio::Ogg::seek (unsigned int pos)
{
	Mutex::Lock lock(mutex);
	_seek = pos;
}

int Roboradio::Audio::Ogg::get_position ()
{
	Mutex::Lock lock(mutex);
	return _position;
}

map<ustring,ustring> Roboradio::Audio::Ogg::get_info (unsigned int &length)
{
	length = 0;
	map<ustring,ustring> info;

	FILE *file = fopen(filename.c_str(), "rb");
	if (file) {
		OggVorbis_File vf;
		if (ov_open(file, &vf, 0, 0) != 0) {
			fclose(file);
		} else {
			length = (unsigned int) ov_time_total(&vf, -1);
			vorbis_comment *c = ov_comment(&vf, -1);
			if (c) {
				for (int i = 0; i < c->comments; ++i) {
					try {
						ustring comment(string(c->user_comments[i]));
						if (comment.validate()) {
							ustring::size_type eq = comment.find('=');
							if (eq != ustring::npos) {
								string key(comment.substr(0, eq));
								// FIXME: this could be done better
								transform(key.begin(), key.end(), key.begin(), ::tolower);
								info.insert(make_pair(key, comment.substr(eq + 1)));
							}
						}
					} catch (...) {
					}
				}
			}
			ov_clear(&vf);
		}
	}

	map<ustring,ustring>::iterator p = info.find("title");
	if (p == info.end()
	    || p->second == "") {
		info.erase("title");
		ustring title(filename);
		
		// remove path
		ustring::size_type s = title.rfind('/');
		if (s != ustring::npos)
			title = title.substr(s + 1);
		
		// remove extension
		s = title.rfind(".ogg");
		if (s == ustring::npos)
			s = title.rfind(".OGG");
		if (s == ustring::npos)
			s = title.rfind(".Ogg");
		if (s != ustring::npos)
			title = title.substr(0, s);

		info.insert(make_pair(ustring("title"), title));
	}

	return info;
}

void Roboradio::Audio::Ogg::thread_function ()
{
	FILE *file = fopen(filename.c_str(), "rb");

	if (!file) {
		mutex.lock();
		_position = -1;
		mutex.unlock();
		done_dispatch();
		cerr << "Audio::Ogg: file could not be opened" << endl;
		return;
	}

	OggVorbis_File vf;
	if (ov_open(file, &vf, 0, 0) != 0) {
		fclose(file);
		mutex.lock();
		_position = -1;
		mutex.unlock();
		done_dispatch();
		cerr << "Audio::Ogg: file is not in ogg format" << endl;
		return;
	}

	const unsigned int buf_size = (4 * 1024);
	char buffer[buf_size];

	int device_id = ao_default_driver_id();
	ao_sample_format output_fmt;
	output_fmt.bits = 16;
	output_fmt.rate = 44100;
	output_fmt.channels = 2;
	output_fmt.byte_format = AO_FMT_LITTLE;
	ao_device *device = 0;
	if (device_id >= 0)
		device = ao_open_live(device_id, &output_fmt, 0);
	else
		cerr << "Audio::Ogg: libao cannot find default audio device" << endl;

	if (device) {
		int bitstream = 0;
		for (;;) {
			bool currently_paused;
			bool stopped = false;
			{
				Mutex::Lock lock(mutex);
				if (_seek >= 0) {
					ov_time_seek(&vf, double(_seek));
					_seek = -1;
				}
				currently_paused = _pause;
			}
			
			if (currently_paused) {
				ao_close(device);
				device = 0;

				for (;;) {
					struct timeval tv;
					tv.tv_sec = 0;
					tv.tv_usec = 100000;
					select(0, 0, 0, 0, &tv);
					Mutex::Lock lock(mutex);
					stopped = _stop;
					if (!_pause || _stop)
						break;
				}
				if (stopped)
					break;

				device = ao_open_live(device_id, &output_fmt, 0);
				if (!device)
					break;
			}

			long r = ov_read(&vf, buffer, buf_size, 0, 2, 1, &bitstream);
			if (r <= 0)
				break;
			else {
				if (!ao_play(device, buffer, r))
					break;

				unsigned int current_pos = (unsigned int) ov_time_tell(&vf);

				Mutex::Lock lock(mutex);
				if (r <= 0 || _stop)
					break;
				if (current_pos != _position) {
					_position = current_pos;
					position_dispatch();
				}
			}
		}
		if (device)
			ao_close(device);
	} else
		cerr << "Audio::Ogg: libao failed to open audio device" << endl;

	mutex.lock();
	bool s = _stop;
	mutex.unlock();

	if (!s)
		done_dispatch();

	ov_clear(&vf);
}

void Roboradio::Audio::Ogg::send_position_changed ()
{
	signal_position_changed(get_position());
}

void Roboradio::Audio::Ogg::send_done ()
{
	thread->join();
	thread = 0;
	signal_done();
}

#endif
