/* -*- Mode: C; tab-width: 8; indent-tabs-mode: t; c-basic-offset: 8 -*- */
/*
 * GnoWavCut -- a GNOME/GTK+ based RIFF PCM Wave file splitter
 * Copyright (C) 2000 Yoichi Imai <yoichi@silver-forest.com>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */
#include "config.h"

#include <gnome.h>
#include <limits.h>
#include <stdlib.h>
/* #include <pthread.h> */

#include "main.h"

void utils_msgbox_error(const gchar *msg)
{
	gtk_widget_show(gnome_message_box_new(msg, GNOME_MESSAGE_BOX_ERROR, GNOME_STOCK_BUTTON_OK,
					      NULL));
}
void utils_msgbox_info(const gchar *msg)
{
	gtk_widget_show(gnome_message_box_new(msg, GNOME_MESSAGE_BOX_INFO, GNOME_STOCK_BUTTON_OK,
					      NULL));
}
void utils_msgbox_warning(const gchar *msg)
{
	gtk_widget_show(gnome_message_box_new(msg, GNOME_MESSAGE_BOX_WARNING, GNOME_STOCK_BUTTON_OK,
					      NULL));
}
gchar *utils_get_time_from_seconds(int in_sec)
{
	int sec;
	int min;
	gchar *time;

	min = in_sec / 60;
	sec = in_sec % 60;

	time = g_strdup_printf("%.3d:%.2d", min, sec);

	return time;
}
gchar *utils_get_mini_time_from_cur(WaveInfo *wave_info, guint32 cur)
{
	guint32 i, j;
	int micro;
	int sec;
	int min;
	gchar *time;

	i = (cur - WAVE_HEADER_SIZE) / wave_info->size_per_sec;
	j = (cur - WAVE_HEADER_SIZE) % wave_info->size_per_sec;

	min = i / 60;
	sec = i % 60;

	micro = j * 1000 / wave_info->size_per_sec;

	time = g_strdup_printf("%.3d:%.2d.%.3d", min, sec, micro);

	return time;
}

gboolean utils_get_cur_from_mini_time(WaveInfo *wave_info, gchar *time_in, guint32 *cur_out)
{
	guint32 cur;
	gchar *s_sec;
	gchar *s_msec;
	gchar *endptr;
	gchar *time;
	long l_min;
	long l_sec;
	long l_msec;

	if(cur_out != NULL)
		*cur_out = 0;

	g_return_val_if_fail(time_in != NULL, FALSE);

	time = g_strdup(time_in);

	l_min = strtol(time, &endptr, 10);
	if (*endptr != ':') {
		g_warning(_("get_cur_from_mini_time error, \":\" is not found."));
		g_free(time);
		return FALSE;
	}
	if(l_min > INT_MAX / 61 / wave_info->size_per_sec || l_min < 0) {
		g_warning(_("get_cur_from_mini_time error, min is too large or too small.\n"));
		g_free(time);
		return FALSE;
	}

	s_sec = strstr(time, ":") + 1;
	l_sec = strtol(s_sec, &endptr, 10);
	if (l_sec >= 60 || l_sec < 0) {
		g_warning(_("get_cur_from_mini_time error, sec >= 60 or sec < 0.\n"));
		g_free(time);
		return FALSE;
	}

	l_msec = 0;
	if ((s_msec = strstr(s_sec, ".")) != NULL) {
		s_msec++;
		l_msec = strtol(s_msec, &endptr, 10);
		if (l_msec > 1000 || l_msec < 0) {
			g_warning(_("get_cur_from_mini_time error, msec > 1000 or l_msec < 0.\n"));
			g_free(time);
			return FALSE;
		}
	}

	cur = l_min * 60 * wave_info->size_per_sec +
		l_sec * wave_info->size_per_sec;

	if (l_msec > 0)
		cur += ((float) l_msec / 1000) * wave_info->size_per_sec;
  
  
	cur -= cur % wave_info->size_per_sample;

	if(cur > wave_info->data_size) {
		g_warning(_("get_cur_from_mini_time error, return value > loaded wave size.\n"));
		g_free(time);
		return FALSE;
	}
	cur += WAVE_HEADER_SIZE;
    
	if(cur_out != NULL)
		*cur_out = cur; 

	g_free(time);
	return TRUE;
}
void utils_mini_control_set_sensitive(GnoWavCut *gnowavcut, gboolean bool)
{
	gtk_widget_set_sensitive(gnowavcut->buttons->b_mini_rew_long, bool);
	gtk_widget_set_sensitive(gnowavcut->buttons->b_mini_rew, bool);
	gtk_widget_set_sensitive(gnowavcut->buttons->b_mini_ff_long, bool);
	gtk_widget_set_sensitive(gnowavcut->buttons->b_mini_ff, bool);
	gtk_widget_set_sensitive(gnowavcut->buttons->b_mini_play, bool);
	gtk_widget_set_sensitive(gnowavcut->buttons->b_mini_insert, bool);
}
void utils_gnowavcut_if_playing(GnoWavCut *gnowavcut, gboolean bool)
{
	if(bool == FALSE) {
		if(gnowavcut->input_tag != -1) {
			gdk_input_remove(gnowavcut->input_tag);
			gnowavcut->input_tag = -1;
		}
		if(gnowavcut->dsp_fd != -1) {
			close(gnowavcut->dsp_fd);
			gnowavcut->dsp_fd = -1;
		}
		if(gnowavcut->wave_fd != -1) {
			close(gnowavcut->wave_fd);
			gnowavcut->wave_fd = -1;
		}
		gnowavcut->mini_play_size = 0;
		gtk_label_set_text(GTK_LABEL(gnowavcut->label_middle), "000:00");
		gtk_adjustment_set_value(GTK_ADJUSTMENT(gnowavcut->hadj), 0.0);
		gnowavcut->now_mini_playing = FALSE;

		gtk_adjustment_set_value(GTK_ADJUSTMENT(gnowavcut->volume_adj), 0.0);
		utils_mini_control_set_sensitive(gnowavcut, FALSE);
	}
	if (bool == TRUE)
		gtk_widget_set_sensitive(gnowavcut->buttons->button_play, FALSE);
	else
		gtk_widget_set_sensitive(gnowavcut->buttons->button_play, TRUE); 
  
	gtk_widget_set_sensitive(gnowavcut->hscale, bool);
	gtk_widget_set_sensitive(gnowavcut->buttons->button_rew, bool);
	gtk_widget_set_sensitive(gnowavcut->buttons->button_stop, bool);
	gtk_widget_set_sensitive(gnowavcut->buttons->button_pause, bool);
	gtk_widget_set_sensitive(gnowavcut->buttons->button_ff, bool);
  
	gtk_label_set_text(GTK_LABEL(gnowavcut->label_mini), "");
	gnowavcut->now_playing = bool;
}

/* '  a  b  c   ' -> 'a b c' */
/* return value must be freed. */
gchar *utils_remove_duplicated_spaces(const gchar *str)
{
	gchar *p, *ret;
	gchar *first, *last;
	p = g_strdup(str);
	ret = g_strdup(g_strstrip(p));
	g_free(p);

	ret = p;
	while(*p != '\0') {
		if(*p == ' ') {
			first = p;
			while(*p != '\0' && *p == ' ') {
				last = p;
				p++;
			}
			if(*p == '\0') break;
			if(last - first > 0) {
				memmove(first + 1, last + 1, strlen(last + 1) + 1);
				p = first + 1;
			}
		}
		p++;
	}
	return ret;
}
gchar **utils_get_command_array(const gchar *command_line, gchar *filename) {
	gchar *command;
	gchar **array;
	gchar **tmp;
	gchar *p;
	gchar *buf;

	buf = g_strdup(filename);

	if(!( (p = strchr(buf, '%')) && *(p + 1) == 's' && 
	   !strchr(p + 2, '%') ))
		utils_msgbox_error(_("Illegal command"));

	command = utils_remove_duplicated_spaces(buf);
	g_free(buf);
	array = g_strsplit(command, " ", 0);
	g_free(command);

	tmp = array;
	while(*tmp != NULL) {
		if(strstr(*tmp, "%s")) {
			buf = *tmp;
			*tmp = g_strdup_printf(*tmp, filename);
			g_free(buf);
		}
		tmp++;
	}
	return array;
}
