# GNU Enterprise Application Server - Instance Object
#
# Copyright 2001-2004 Free Software Foundation
#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# $Id: geasInstance.py 5527 2004-03-26 10:50:44Z johannes $

import types
import string
import mx.DateTime
import mx.DateTime.ISO
from gnue.appserver.language import Object, Session
from gnue.common.logic.language import getLanguageAdapter

# =============================================================================
# Exceptions
# =============================================================================

class DbValueError (gException):
  def __init__ (self, propertyName, value):
    message = u_("Database returned invalid value '%(value)s' for property "
                 "'%(property)s'") \
              % {"value"   : repr (value),
                 "property": propertyName}
    gException.__init__ (self, message)

class PropertyValueError (gException):
  def __init__ (self, propertyName, value):
    message = u_("Invalid value '%(value)s' for property '%(property)s'") \
              % {"value": repr (value),
                 "property": propertyName}
    gException.__init__ (self, message)

class ParameterValueError (gException):
  def __init__ (self, parameterName, value):
    message = u_("Invalid value '%(value)s' for parameter '%(parameter)s'") \
              % {"value"    : repr (value),
                 "parameter": parameterName}
    gException.__init__ (self, message)


# =============================================================================
# Instance class
# =============================================================================

class geasInstance:

  # ---------------------------------------------------------------------------
  # Initalize
  # ---------------------------------------------------------------------------

  def __init__ (self, session, record, classdef):
    self.__session = session
    self.__record = record
    self.__classdef = classdef

  # ---------------------------------------------------------------------------
  # Convert a value into the give type if possible
  # ---------------------------------------------------------------------------

  def __convert (self, value, propertydef, exception):

    if value is None:
      # "None" is always valid, independent of data type
      return None

    elif propertydef.dbType == "string":
      # String property: we expect the database to deliver a unicode string
      if isinstance (value, types.UnicodeType):
        return value
      elif isinstance (value, types.StringType):
        return unicode (value)
      else:
        try:
          return unicode (str (value))
        except:
          raise exception (propertydef.fullName, value)

    elif propertydef.dbType == "number":
      # Number property: Must be something that can be converted to an int or
      # float
      try:
        if propertydef.gnue_scale:
          # ... with fractional part
          return float (value)
        else:
          # ... without fractional part
          return int (value)
      except ValueError:
        raise exception (propertydef.fullName, value)
      
    elif propertydef.dbType == "date":
      # Date property: Must be an mx.DateTime or at least parseable as such
      if isinstance (value, types.UnicodeType):
        value = mx.DateTime.ISO.ParseDateTime (value.encode ('utf-8') + \
                                               ' 00:00:00')
      elif isinstance (value, types.StringType):
        value = mx.DateTime.ISO.ParseDateTime (value + ' 00:00:00')

      if isinstance (value, mx.DateTime.DateTimeType):
        return mx.DateTime.DateTime (value.year, value.month, value.day,
                                     0, 0, 0)
      else:
        raise exception (propertydef.fullName, value)

    elif propertydef.dbType == "time":
      # Time property: Must be an mx.DateTime or at least parseable as such
      if isinstance (value, types.UnicodeType):
        value = mx.DateTime.ISO.ParseDateTime ('0001-01-01 ' + \
                                               value.encode ('utf-8'))
      elif isinstance (value, types.StringType):
        value = mx.DateTime.ISO.ParseDateTime ('0001-01-01 ' + value)

      if isinstance (value, mx.DateTime.DateTimeType) or \
         isinstance (value, mx.DateTime.DateTimeDeltaType):
        return mx.DateTime.DateTime (1, 1, 1,
                                     value.hour, value.minute, value.second)
      else:
        raise exception (propertydef.fullName, value)

    elif propertydef.dbType == "datetime":
      # Date/Time property: Must be an mx.DateTime or at least parseable as
      # such
      if isinstance (value, types.UnicodeType):
        return mx.DateTime.ISO.ParseDateTime (value.encode ('utf-8'))
      elif isinstance (value, types.StringType):
        return mx.DateTime.ISO.ParseDateTime (value)

      if isinstance (value, mx.DateTime.DateTimeType):
        return value
      else:
        raise exception (propertydef.fullName, value)

    elif propertydef.dbType == "boolean":
      # Boolean property: Must be something meaningful
      if value in [0, "0", "f", "F", "false", "FALSE", "n", "N", "no", "NO",
                   False]:
        return False
      elif value in [1, "1", "t", "T", "true", "TRUE", "y", "Y", "yes", "YES",
                     True]:
        return True
      else:
        raise exception (propertydef.fullName, value)

    else:
      # Unknown property type
      raise exception (propertydef.fullName, value)

  # ---------------------------------------------------------------------------
  # Get the value of a property
  # ---------------------------------------------------------------------------

  def __getValue (self, propertyname):

    propertydef = self.__classdef.properties [propertyname]

    value = self.__record.getField (propertydef.column)

    return self.__convert (value, propertydef, DbValueError)

  # ---------------------------------------------------------------------------
  # Get the values of a list of properties
  # ---------------------------------------------------------------------------

  def get (self, propertylist):

    return [self.__getValue (aProperty) for aProperty in propertylist]

  # ---------------------------------------------------------------------------
  # Set the value of a property
  # ---------------------------------------------------------------------------

  def __putValue (self, propertyname, value):

    propertydef = self.__classdef.properties [propertyname]

    __value = self.__convert (value, propertydef, PropertyValueError)

    self.__record.putField (propertydef.column, __value)

  # ---------------------------------------------------------------------------
  # Set the values of a list of properties
  # ---------------------------------------------------------------------------

  def put (self, propertylist, valuelist):

    for i in range (0, len (propertylist)):
      self.__putValue (propertylist [i], valuelist [i])

  # ---------------------------------------------------------------------------
  # Call a procedure
  # ---------------------------------------------------------------------------

  def call (self, procedurename, params):

    # TODO: This should run in a separate process so that a segfaulting
    # procedure doesn't kill appserver.
    # (needs to be implemented as an option in gnue-common)

    # Create a session object which with the actual session id
    sess = Session.Session (self.__session.sm, self.__session.id)

    # Create an object representing the current business object
    obj = Object.Object (sess, self.__classdef.fullName, 
                         self.__getValue (u'gnue_id'))

    # fetch the procedure definition
    proceduredef = self.__classdef.procedures [procedurename]

    # check the parameters
    if params is not None:
      for parameter in params.keys ():
        paramDef = proceduredef.parameters [parameter]
        params [parameter] = self.__convert (params [parameter], paramDef,
                                             ParameterValueError)
    else:
      params = {}

    engine = getLanguageAdapter (proceduredef.gnue_language)

    cx = engine.createNewContext ()

    # describe the context
    cx.shortname   = '%s.%s' % (self.__classdef.fullName, procedurename)
    cx.description = proceduredef.gnue_comment

    # the object itself
    cx.bindObject ('self', obj, Object.Object)

    # language interface, session functions
    cx.bindFunction ('find',       sess.find)
    cx.bindFunction ('setcontext', sess.setcontext)
    cx.bindFunction ('new',        sess.new)

    # direct access to RPC API func.
    cx.bindFunction ('direct_request', self.__session.request)
    cx.bindFunction ('direct_fetch',   self.__session.fetch)
    cx.bindFunction ('direct_load',    self.__session.load)
    cx.bindFunction ('direct_store',   self.__session.store)
    cx.bindFunction ('direct_call',    self.__session.call)

    # set context for the procedure
    sess.setcontext (proceduredef.module.gnue_name)

    try:
      method = cx.buildFunction (proceduredef.gnue_name, proceduredef.gnue_code,
                               params)
      result = method (**params)

    except Exception, msg:

      print _("Error occured during method invokation: %s") % msg
      result = None

    # Save changes made by the procedure
    # obj.store ()

    # return value of '1' to make XMLRPC happy
    return result


  # ---------------------------------------------------------------------------
  # Validate all properties of an instance
  # ---------------------------------------------------------------------------

  def validate (self):
    """
    This function checks all properties marked as 'not nullable' to have a
    value other than None. If a none value is encountered a PropertyValueError
    is raised.
    """
    for prop in self.__classdef.properties.values ():
      if prop.gnue_nullable is not None and not prop.gnue_nullable:
        value = self.__record.getField (prop.column)
        if value is None:
          raise PropertyValueError (prop.fullName, None)
    
