# GNU Enterprise Application Server - Language interface: Object
#
# Copyright 2003-2004 Free Software Foundation
#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it 
# and/or modify it under the terms of the GNU General Public 
# License as published by the Free Software Foundation; either 
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be 
# useful, but WITHOUT ANY WARRANTY; without even the implied 
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public 
# License along with program; see the file COPYING. If not, 
# write to the Free Software Foundation, Inc., 59 Temple Place 
# - Suite 330, Boston, MA 02111-1307, USA.
#
# $Id: Object.py 5515 2004-03-25 23:23:46Z reinhard $

import string

from Procedure import Procedure


# ===========================================================================
# Exceptions
# ===========================================================================

class MemberNotFoundError (gException):
  def __init__ (self, classname, member):
    msg = u_("Class '%(classname)s' has no member '%(member)s'") % \
          {"classname": classname, "member": member}
    gException.__init__ (self, msg)


# ===========================================================================
# Helper methods
# ===========================================================================

# ===========================================================================
# Implements a single business object
# ===========================================================================
class Object:

  # -------------------------------------------------------------------------
  # Constructor
  # -------------------------------------------------------------------------
  def __init__ (self, session, classname, objectId = None):
    # we won't trigger __setattr__ here ...
    self.__dict__['objectId']         = objectId
    self.__dict__['_Object__session'] = session
    self.__dict__['_Object__class']   = classname

    if self.objectId is None:
      # Create new object
      sm   = self.__session.getSessionManager ()
      sid  = self.__session.getSessionId ()

      res = sm.store (sid, self.__class, [None], [], [[]])
      self.objectId = res [0]


  # -------------------------------------------------------------------------
  # Return an attribute's value; if not available in cache load from server
  # -------------------------------------------------------------------------
  def __getattr__ (self, attr):
    name = self.__session.qualify (attr)
    if name == "gnue_id":
      return self.objectId

    if self.__attrIsMethod (name):
      return Procedure (self.__session, self.objectId, self.__class, name)

    elif self.__attrIsValid (name):
      sm   = self.__session.getSessionManager ()
      sid  = self.__session.getSessionId ()

      try:
        res = sm.load (sid, self.__class, [self.objectId, ''], [name])
        (value, datatype) = (res [0][0], res [1][0])

      except:
        # TODO: use a more specific exception 
        return None

      # Convert reference fields to object references
      if string.split (datatype, '(', 1) [0] in \
         ['id', 'string', 'number', 'date', 'time', 'datetime', 'boolean']:
        return value
      else:
        return Object (self.__session, datatype, value)

    else:
      raise MemberNotFoundError (self.__class, name)


  # -------------------------------------------------------------------------
  # Set an attribute's value
  # -------------------------------------------------------------------------
  def __setattr__ (self, attr, value):

    if self.__dict__.has_key (attr):
      self.__dict__ [attr] = value

    elif self.__attrIsValid (attr):
      name = self.__session.qualify (attr)
      sm   = self.__session.getSessionManager ()
      sid  = self.__session.getSessionId ()

      if isinstance (value, Object):
        __value = value.objectId
      else:
        __value = value

      sm.store (sid, self.__class, [self.objectId], [name], [[__value]])

    else:
      raise AttributeError, attr


  # -------------------------------------------------------------------------
  # Return the Object-ID as string representation
  # -------------------------------------------------------------------------
  def __str__ (self):
    if self.objectId:
      return self.objectId
    else:
      return ''

  # -------------------------------------------------------------------------
  # Return the Object as string representation
  # -------------------------------------------------------------------------
  def __repr__ (self):
    if self.objectId:
      return '<%s instance with id %s>' % (self.__class, self.objectId)
    else:
      return '<deleted %s instance>' % self.__class


  # -------------------------------------------------------------------------
  # Make object['property'] possible
  # -------------------------------------------------------------------------
  def __getitem__ (self, item):
    return self.__getattr__ (item)
  def __setitem__ (self, item, value):
    self.__setattr__ (item, value)


  # -------------------------------------------------------------------------
  # Check if 'attr' is a valid property of this class
  # -------------------------------------------------------------------------
  def __attrIsValid (self, attr):
    name = self.__session.qualify (attr)
    cdef = self.__session.getSessionManager ().classes [self.__class]
    return cdef.properties.has_key (name)

  # -------------------------------------------------------------------------
  # Check if 'attr' is a valid procedure of this class
  # -------------------------------------------------------------------------
  def __attrIsMethod (self, attr):
    name = self.__session.qualify (attr)
    cdef = self.__session.getSessionManager ().classes [self.__class]
    return cdef.procedures.has_key (name)


  # -------------------------------------------------------------------------
  # Delete instance and clear object-id
  # -------------------------------------------------------------------------
  def delete (self):
    self.__session.getSessionManager ().delete (
               self.__session.getSessionId (), self.__class, [self.objectId])
    self.objectId = None
