# GNU Enterprise Application Server - Class Repository: Base classes
#
# Copyright 2003-2005 Free Software Foundation
#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it 
# and/or modify it under the terms of the GNU General Public 
# License as published by the Free Software Foundation; either 
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be 
# useful, but WITHOUT ANY WARRANTY; without even the implied 
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public 
# License along with program; see the file COPYING. If not, 
# write to the Free Software Foundation, Inc., 59 Temple Place 
# - Suite 330, Boston, MA 02111-1307, USA.
#
# $Id: Base.py 7104 2005-03-07 16:49:25Z johannes $

import types
import helpers
from gnue import appserver
from gnue.common.apps import errors

# =============================================================================
# Exceptions
# =============================================================================

class ClassRepositoryError (errors.ApplicationError):
  pass

class ItemNotFoundError (ClassRepositoryError):
  """
  This is the base exception class for all 'item not found' errors
  """
  def __init__ (self, key):
    ClassRepositoryError.__init__ (self, u_("Item '%s' not found.") % key)


class ItemValidationError (helpers.ValidationError):
  def __init__ (self, item, message):
    msg = u_("'%(item)s': %(error)s") \
          % {"item" : item,
             "error": message }
    helpers.ValidationError.__init__ (self, msg)


# =============================================================================
# Class-wrapper for dictionaries
# =============================================================================

class BaseDictionary:
  """
  This class implements the common behaviour of class repository dictionaries.
  """

  # ---------------------------------------------------------------------------
  # Construct an instance 
  # ---------------------------------------------------------------------------

  def __init__ (self, session, tablename):
    self.table    = tablename
    self._session = session
    self._items   = {}


  # ---------------------------------------------------------------------------
  # Reload the dictionary from the database by merging new records in
  # ---------------------------------------------------------------------------

  def reload (self, idAsKey = False):
    """
    This function fetches a list of all available items of the dictionaries
    table and merges them into the objectlist.
    """
    gEnter (6)

    rlist = self._session.find (self.table, self._getReloadCondition (),
                                            self._getSortorder (),
                                            self._getColumns ())
    self.mergeIntoList (rlist, idAsKey)

    gLeave (6)


  # ---------------------------------------------------------------------------
  # Merge a given object list with the dictionaries list
  # ---------------------------------------------------------------------------

  def mergeIntoList (self, addList, idAsKey = False):
    """
    Merge all language interface objects from @addList into the dictionaries
    objectlist. If an object is new in the list, use _getNewItem () to build a
    new instance for the object.
    """
    gEnter (6)

    for obj in addList:
      found = 0

      for item in self._items.values ():
        if item.gnue_id == obj.gnue_id:
          found = 1
          break

      if not found:
        newItem = self._getNewItem (obj)

        # idAsKey is set to TRUE if we'd like to create a global dictionary,
        # which means holding properties of all classes for example. This is
        # neccessary for booting the initial class repository.
        key = idAsKey and newItem.gnue_id or newItem.fullName
        self [key] = newItem

    gLeave (6)


  # ---------------------------------------------------------------------------
  # Return the conditions to be used on reload ()
  # ---------------------------------------------------------------------------

  def _getReloadCondition (self):
    """
    Descendants override this method to specify a condition for reloading all
    elements of the dictionary.
    """
    return gLeave (6, [])


  # ---------------------------------------------------------------------------
  # Return the list of columns to be used on reload ()
  # ---------------------------------------------------------------------------

  def _getColumns (self):
    """
    Specify a list of fields to be fetched on a reload ().
    """
    return gLeave (6, [u'gnue_name'])


  # ---------------------------------------------------------------------------
  # Return the sortorder to be used on reload ()
  # ---------------------------------------------------------------------------

  def _getSortorder (self):
    """
    Specify a list of fields used as sort-order.
    """
    return gLeave (6, [u'gnue_id'])


  # ---------------------------------------------------------------------------
  # Get a condition to fetch a single item for the dictionary
  # ---------------------------------------------------------------------------

  def _getSingleCondition (self, key):
    """
    Descendants override this method to specify a condition for fetching a
    single item given by @key.
    """
    raise ClassRepositoryError, \
        u_("%s: _getSingleCondition() not implemented!") % self.table


  # ---------------------------------------------------------------------------
  # Use _getNewItem in derived classes to instanciate new dictionary-items
  # ---------------------------------------------------------------------------

  def _getNewItem (self, obj):
    """
    Descendants override this method to create an apropriate instance out of
    @obj, which is a language interface object.
    """
    raise errors.SystemError, \
        u_("%s: _getNewItem() not implemented!") % self.table


  # ---------------------------------------------------------------------------
  # Return all keys 
  # ---------------------------------------------------------------------------

  def keys (self):
    """
    This function returns all keys of the dictionary.
    """
    keys = self._items.keys ()
    keys.sort ()
    return keys


  # ---------------------------------------------------------------------------
  # Return all values
  # ---------------------------------------------------------------------------

  def values (self):
    """
    This function returns all values of the dictionary.
    """
    keys = self.keys ()
    return map (self._items.get, keys)


  # ---------------------------------------------------------------------------
  # Return all key-value-pairs
  # ---------------------------------------------------------------------------

  def items (self):
    """
    This function returns all items of the dictionary.
    """
    return self._items.items ()


  # ---------------------------------------------------------------------------
  # Check for a specified key
  # ---------------------------------------------------------------------------

  def has_key (self, key):
    """
    This function returns True, if the key @key is available in the dictionary.
    """
    return self._items.has_key (key.lower ())


  # ---------------------------------------------------------------------------
  # Virtual: Create an exception 'item not found error'
  # ---------------------------------------------------------------------------
  def _itemNotFoundError (self, key):
    """
    A descendant of this class can override this function to return an
    appropriate exception.
    """
    return ItemNotFoundError (key)


  # ---------------------------------------------------------------------------
  # Virtual: Create an exception for a validation error
  # ---------------------------------------------------------------------------

  def _validationError (self, item, verr):
    return ItemValidationError (item.fullName, verr.message)


  # ---------------------------------------------------------------------------
  # Return the number of items in the dictionary
  # ---------------------------------------------------------------------------
  def __len__ (self):
    """
    Return the number of items in the dictionary
    """
    return len (self._items)


  # ---------------------------------------------------------------------------
  # Access to the underlying dictionary
  # ---------------------------------------------------------------------------

  def __getitem__ (self, key):
    """
    This function provides access to the underlying dictionary. If a key isn't
    found in the dictionary, it will be looked up via language interface. If
    it is not found in the datasource, an _itemNotFoundError will be raised.
    """

    result = None

    if self.has_key (key):
      result = self._items [key.lower ()]
    else:
      list = self._session.find (self.table, self._getSingleCondition (key),
                                             self._getSortorder (),
                                             self._getColumns ())
      if len (list) == 1:
        newItem = self._getNewItem (list [0])
        newItem.complete ()
        try:
          newItem.validate ()

        except helpers.ValidationError, verr:
          raise self._validationError (newItem, verr)

        self._items [key.lower ()] = newItem
        result = newItem

    if result is None:
      raise self._itemNotFoundError (key)

    return result


  # ---------------------------------------------------------------------------
  # Remove a key from the internal dictionary
  # ---------------------------------------------------------------------------

  def __delitem__ (self, key):
    del self._items [key.lower ()]


  # ---------------------------------------------------------------------------
  # Add an item to the internal dictionary
  # ---------------------------------------------------------------------------

  def __setitem__ (self, key, value):
    self._items [key.lower ()] = value




# =============================================================================
# This class maps all dictionary-keys to properties
# =============================================================================
class BaseObject:
  """
  This class implements the basic dictionary element class.
  """

  # ---------------------------------------------------------------------------
  # Constructor
  # ---------------------------------------------------------------------------

  def __init__ (self, session, classname, aObject = None, pDefs = None):

    self.classname    = classname
    self._session     = session

    self.__object     = aObject
    self.__predefined = {}

    if pDefs is not None:
      for key in pDefs.keys ():
        if key in ["gnue_length", "gnue_scale"]:
          if pDefs [key]:
            self.__predefined [key] = int (pDefs [key])
            self.__dict__ [key] = int (pDefs [key])
          else:
            self.__dict__ [key] = None
            self.__predefined [key] = None

        elif key in ["gnue_nullable"]:
          if pDefs [key] == "FALSE":
            self.__dict__ [key] = False
            self.__predefined [key] = False
          else:
            self.__dict__ [key] = True
            self.__predefined [key] = True
        else:
          self.__dict__ [key] = pDefs [key]
          self.__predefined [key] = pDefs [key]



  # ---------------------------------------------------------------------------
  # Return an attributes value
  # ---------------------------------------------------------------------------

  def __getattr__ (self, attr):
    """
    If an attribute is not a 'predefined' value it is delegated to the language
    interface object of the instance.
    """

    if self.__predefined.has_key (attr):
      result = self.__predefined [attr]
    else:
      result = getattr (self._getObject (), attr)

    return result


  # ---------------------------------------------------------------------------
  # Get a language interface object
  # ---------------------------------------------------------------------------

  def _getObject (self):
    """
    This function returns the corresponding language interface object of this
    instance. It will be created the first time this function is called.
    """

    if self.__object is None:
      self.__object = self._session._get (self.classname, self.gnue_id)

    result = self.__object

    return result


  # ---------------------------------------------------------------------------
  # Validate an instance
  # ---------------------------------------------------------------------------

  def validate (self):
    """
    Descendants can override this function to perform validations on a newly
    created instance. Note: this function won't be called on instances created
    by a reload ().
    """
    pass


  # ---------------------------------------------------------------------------
  # Update object references
  # ---------------------------------------------------------------------------

  def replaceReferences (self, aObject):
    """
    This function replaces all references to @aObject in it's predefined
    dictionary.
    """
    gEnter (6)

    for (key, value) in self.__predefined.items ():
      if isinstance (value, types.UnicodeType) and value == aObject.objectId:
        self.__dict__ [key]     = aObject
        self.__predefined [key] = aObject

    gLeave (6)


  # ---------------------------------------------------------------------------
  # Complete an instance
  # ---------------------------------------------------------------------------

  def complete (self):
    """
    Descendants can override this function to perform tasks to make an instance
    complete. This function will be called after a new instance has been
    created on a request to the dictionary (__getitem__).
    """
    pass
