# GNU Enterprise Application Server - Class Repository: Property
#
# Copyright 2003-2005 Free Software Foundation
#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it 
# and/or modify it under the terms of the GNU General Public 
# License as published by the Free Software Foundation; either 
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be 
# useful, but WITHOUT ANY WARRANTY; without even the implied 
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public 
# License along with program; see the file COPYING. If not, 
# write to the Free Software Foundation, Inc., 59 Temple Place 
# - Suite 330, Boston, MA 02111-1307, USA.
#
# $Id: Property.py 7104 2005-03-07 16:49:25Z johannes $

from Base import *
from Namespace import createName, splitName
import helpers


# =============================================================================
# Exceptions
# =============================================================================

class PropertyNotFoundError (ClassRepositoryError):
  def __init__ (self, classname, key):
    msg = u_("Class '%(classname)s' has no property '%(property)s'") % \
             { "classname": classname, "property": key }
    ClassRepositoryError.__init__ (self, msg)

class PropertyValidationError (helpers.ValidationError):
  def __init__ (self, classname, propertyname, message):
    msg = u_("Property '%(property)s': %(error)s") \
          % {"property": "%s.%s" % (classname, propertyname),
             "error"   : message}
    helpers.ValidationError.__init__ (self, msg)


# =============================================================================
# Dictionary with all properties of a given class
# =============================================================================
class PropertyDict (BaseDictionary):

  # ---------------------------------------------------------------------------
  # Construct a Property-Dictionary for class aClass
  # ---------------------------------------------------------------------------
  def __init__ (self, session, aClass, predefs = None, modules = None):
    BaseDictionary.__init__ (self, session, 'gnue_property')

    self.__class   = aClass
    if aClass is not None:
      self.__module  = self.__class.module
      self.__modules = modules or self.__module.modules
    else:
      self.__module  = None
      self.__modules = modules

    # predefs is a list of dictionaries, describing the properties
    if predefs is not None:
      for pDef in predefs:
        pMod = self.__modules.find (pDef ['gnue_module'])
        prop = Property (session, pMod, None, pDef)

        self._items [prop.fullName.lower ()] = prop


  # ---------------------------------------------------------------------------
  # Create a new instance of a dictionary item
  # ---------------------------------------------------------------------------
  def _getNewItem (self, aObject):
    pMod = self.__modules.find (aObject.gnue_module.objectId)
    return Property (self._session, pMod, aObject,
                     {"gnue_id": aObject.objectId})


  # ---------------------------------------------------------------------------
  # Restrict a reload () to the classes properties 
  # ---------------------------------------------------------------------------
  def _getReloadCondition (self):
    if self.__class is not None:
      result = ['eq', ['field', u'gnue_class'], ['const', self.__class.gnue_id]]
    else:
      result = []
    return gLeave (6, result)


  # ---------------------------------------------------------------------------
  # Return a condition to match a given Property
  # ---------------------------------------------------------------------------
  def _getSingleCondition (self, key):

    (module, name) = splitName (key)

    return ['and', self._getReloadCondition (),
                   ['eq', ['field', u'gnue_module.gnue_name'],
                          ['const', module]],
                   ['eq', ['field', u'gnue_name'], ['const', name]]]


  # ---------------------------------------------------------------------------
  # Return a list of columns to be prepared by a find ()
  # ---------------------------------------------------------------------------
  def _getColumns (self):
    result = [u"gnue_module", u"gnue_class", u"gnue_name", u"gnue_type",
            u"gnue_length", u"gnue_scale", u"gnue_comment", u"gnue_nullable"]
    return gLeave (6, result)


  # ---------------------------------------------------------------------------
  # Create a key-not-found-exception
  # ---------------------------------------------------------------------------

  def _itemNotFoundError (self, key):
    return PropertyNotFoundError (self.__class.fullName, key)


  # ---------------------------------------------------------------------------
  # Create a validation exception
  # ---------------------------------------------------------------------------

  def _validationError (self, item, verr):
    return PropertyValidationError (self.__class.fullName, item.fullName,
                                    verr.message)

  # ---------------------------------------------------------------------------
  # update property dictionary with calculated properties from procedure dict.
  # ---------------------------------------------------------------------------

  def loadCalculated (self, procDict):
    """
    This function merges all 'calculated fields' from the given procedure
    dictionary into the property dictionary
    """

    for proc in procDict.values ():
      if proc.isCalculated:
        newProp = CalculatedProperty (proc)
        self._items [newProp.fullName.lower ()] = newProp


# =============================================================================
# Base class for properties
# =============================================================================

class BaseProperty:

  _BASE_TYPES = ['boolean', 'date', 'datetime', 'number', 'string', 'time']
  _ID_TYPE    = "string"
  _ID_LENGTH  = 32
  _ID_SCALE   = 0

  # ---------------------------------------------------------------------------
  # Create the proper type information
  # ---------------------------------------------------------------------------

  def _updateTypeInfo (self):
    """
    This function updates the properties 'fullType', 'dbType', 'dbLength' and
    'dbScale' based on 'gnue_type', 'gnue_length' and 'gnue_scale'
    """
    if self.gnue_type in ["string", "number"]:
      if self.gnue_type == "number" and self.gnue_length and self.gnue_scale:
        self.fullType = "%s(%d.%d)" % (self.gnue_type, self.gnue_length,
                                                       self.gnue_scale)
      elif self.gnue_length:
        self.fullType = "%s(%d)" % (self.gnue_type, self.gnue_length)
      else:
        self.fullType = self.gnue_type
    else:
      self.fullType = self.gnue_type

    # build database specific type information
    if self.gnue_type in self._BASE_TYPES:
      self.dbType   = self.gnue_type
      self.dbLength = self.gnue_length
      self.dbScale  = self.gnue_scale

    else:
      self.dbType   = self._ID_TYPE
      self.dbLength = self._ID_LENGTH
      self.dbScale  = self._ID_SCALE

    if self.dbType in ["string", "number"]:
      if self.dbType == "number" and self.dbLength and self.dbScale:
        self.dbFullType = "%s(%d.%d)" % (self.dbType, self.dbLength,
                                                      self.dbScale)
      elif self.dbLength:
        self.dbFullType = "%s(%d)" % (self.dbType, self.dbLength)
      else:
        self.dbFullType = self.dbType
    else:
      self.dbFullType = self.dbType


# =============================================================================
# A Business Object Property
# =============================================================================
class Property (BaseProperty, BaseObject):

  # ---------------------------------------------------------------------------
  # Construct a new property from module, class and business object
  # ---------------------------------------------------------------------------
  def __init__ (self, session, module, object, pDefs = None):
    BaseObject.__init__ (self, session, 'gnue_property', object, pDefs)

    self.module   = module
    self.fullName = createName (self.module.gnue_name, self.gnue_name)
    self.column   = self.fullName

    self.isValidated     = False
    self.isReference     = False
    self.referencedClass = None
    self.isCalculated    = False
    self.procedure       = None

    # build appserver specific type information
    self._updateTypeInfo ()


  # ---------------------------------------------------------------------------
  # Validate a given property definition
  # ---------------------------------------------------------------------------
  def validate (self):
    """
    This function verifies the property type and sets the reference properties.
    If a property definition is a reference-property @referencedClass holds the
    referenced class definition.
    """

    res = helpers.verifyType (self.gnue_type, self.gnue_length,
                              self.gnue_scale, self.module.modules)
    self.isReference     = res is not None
    self.referencedClass = res
    self.isValidated     = True


# =============================================================================
# Class for calculated properties
# =============================================================================

class CalculatedProperty (BaseProperty):

  # ---------------------------------------------------------------------------
  # Constructor
  # ---------------------------------------------------------------------------

  def __init__ (self, aProcedure):
    self.classname = '<calculated>'
    self._session  = aProcedure._session

    self.module    = aProcedure.module
    self.gnue_name = aProcedure.calcName
    self.fullName  = aProcedure.calcFullName
    self.column    = None

    self.isValidated     = True
    self.isReference     = False
    self.referencedClass = None
    self.isCalculated    = True
    self.procedure       = aProcedure

    self.gnue_type     = aProcedure.gnue_type
    self.gnue_length   = aProcedure.gnue_length
    self.gnue_scale    = aProcedure.gnue_scale
    self.gnue_comment  = aProcedure.gnue_comment
    self.gnue_module   = aProcedure.gnue_module
    self.gnue_nullable = aProcedure.gnue_nullable

    self._updateTypeInfo ()


  # ---------------------------------------------------------------------------
  # Validate a calculated property
  # ---------------------------------------------------------------------------
  def validate (self):
    pass
