# GNU Enterprise Application Server - Gnue Schema Definition Generator
#
# Copyright 2001-2005 Free Software Foundation
#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# $Id: readgcd.py 6998 2005-02-11 11:06:42Z johannes $

import sys
import os
import string

from gnue.common.apps import i18n, errors
from gnue.common.apps.GClientApp import *
from gnue.common.utils.FileUtils import openResource
from gnue.common.datasources import GDataSource
from gnue.common.utils.uuid import UUID

from gnue.appserver import VERSION
from gnue.appserver.gcd import GCParser
from gnue.appserver.classrep import Namespace, helpers
from gnue.appserver import geasConfiguration


# =============================================================================
# Exceptions
# =============================================================================


class ReferenceLookupError (errors.SystemError):
  def __init__ (self, key, value):
    msg = u_("Cannot find '%(key)s' named '%(value)s in lookup dictionary") \
          % {'key': key, 'value': value}
    errors.SystemError.__init__ (self, msg)

class StartupError (errors.UserError):
  pass

class Error (errors.ApplicationError):
  def __init__ (self, message, files = None):
    errors.ApplicationError.__init__ (self, message)
    text = []

    if files:
      text.append (u_("In file(s) '%s':") % string.join (files, ', '))

    text.append (message)
    self.detail = string.join (text, os.linesep)


class ModuleNotFoundError (Error):
  def __init__ (self, module, files = None):
    msg = u_("Module '%s' not found in class repository") % module
    Error.__init__ (self, msg, files)

class ClassNotFoundError (Error):
  def __init__ (self, classname, files = None):
    msg = u_("Class '%s' not found in class repository") % classname
    Error.__init__ (self, msg, files)

class FilterNotFoundError (Error):
  def __init__ (self, classname, filtername, files = None):
    msg = u_("Filter '%(filter)s' of class '%(class)s' not found in class "
             "repository") \
          % {'filter': filtername, 'class' : classname}
    Error.__init__ (self, msg, files)

class FilterChangeError (Error):
  def __init__ (self, name, files = None):
    msg = u_("Changing filter of existing class '%s' is not allowed") % name
    Error.__init__ (self, msg, files)

class NullableError (Error):
  def __init__ (self, pName, files = None):
    msg = u_("Property '%s' cannot be added to existing class with "
             "'NOT NULL' constraint") % pName
    Error.__init__ (self, msg, files)

class DuplicateClassError (Error):
  def __init__ (self, classname, file1, file2):
    msg = u_("Class '%s' is defined multiple times") % classname
    Error.__init__ (self, msg, [file1, file2])



# =============================================================================
# Update a database schema from GNUe Class Definitions and update classrep.
# =============================================================================

class gcdClient (GClientApp):

  NAME    = "readgcd"
  VERSION = VERSION
  COMMAND = "readgcd"
  USAGE   = "%s %s" % (GClientApp.USAGE, " [OPTIONS] file")
  SUMMARY = _(
"""Create or update a database schema from a GNUe Class Definition (gcd) file
and maintain data for all gnue_* classes""")

  # ---------------------------------------------------------------------------
  # Constructor
  # ---------------------------------------------------------------------------

  def __init__ (self, connections = None):

    self.addCommandOption ('connection', 'c', argument='connectionname',
        help = _("Use the connection <connectionname> for creating the schema"))

    self.addCommandOption ('output','o', argument='filename',
        help = _("Also send the code for creating the schema to this file."))

    self.addCommandOption ('file-only', 'f', default = False,
        help = _("If this flag is set, only code is sent to the output file "
                 "and the schema is not created automatically."))

    self.addCommandOption ('username', 'u', argument="user",
        help = _("Set the username for the database. If the database is to be "
                 "created, this username will be it's owner."))

    self.addCommandOption ('password', 'p', argument="password",
        help = _("Set the password for the database."))

    ConfigOptions = geasConfiguration.ConfigOptions

    GClientApp.__init__ (self, connections, 'appserver', ConfigOptions)

    if self.OPTIONS ['connection']:
      self._connection = self.OPTIONS ['connection']
    else:
      # Compatibility for 'database' setting in gnue.conf
      dbName = gConfig ('database')
      coName = gConfig ('connection')

      if dbName and coName == 'gnue':
        self._connection = dbName
      else:
        self._connection = coName



  # ---------------------------------------------------------------------------
  # Main program
  # ---------------------------------------------------------------------------

  def run (self):
    """
    Create a new instance of a gcdReader and pass all options to it. Before
    starting the process username and password will be set for the connection.
    """

    reader = gcdReader (self.connections,
                        self._connection,
                        [unicode (a, i18n.encoding) for a in self.ARGUMENTS],
                        self.OPTIONS ['output'],
                        self.OPTIONS ['file-only'])

    self._prepareConnection ()

    reader.run ()


  # ---------------------------------------------------------------------------
  # Prepare the connection
  # ---------------------------------------------------------------------------

  def _prepareConnection (self):
    """
    This function makes sure the connection will have the proper username and
    password set.
    """

    connection = self.connections.getConnection (self._connection)

    if not connection.parameters.has_key ('username'):
      connection.parameters ['username'] = 'gnue'
    if not connection.parameters.has_key ('password'):
      connection.parameters ['password'] = 'gnue'

    if self.OPTIONS ['username'] is not None:
      connection.parameters ['username'] = self.OPTIONS ['username']

    if self.OPTIONS ['password'] is not None:
      connection.parameters ['password'] = self.OPTIONS ['password']


# =============================================================================
# This class implements an integrator for GNUe Class Definition files
# =============================================================================

class gcdReader:

  # ---------------------------------------------------------------------------
  # Constructor
  # ---------------------------------------------------------------------------

  def __init__ (self, connections, database, files, dump = None, fonly = False):
    """
    Create a new instance of a gcd reader

    @param connections: GConnections instance to be used
    @param database: name of the connection to use (in connection.conf)
    @param files: sequence of filenames to integerate
    @param dump: If not None, the SQL-code of the schema-update will be stored
        into a file with this name
    @param fonly: If True, only the SQL-dump will be done, no change of the
        backend will happen.
    """

    self._connections = connections
    self._database    = database
    self._filenames   = files
    self._files       = []
    self._dump        = dump
    self._fileonly    = fonly
    self._uuidType    = gConfig ('uuidtype', section = 'appserver').lower ()

    # ResultSets
    self.moduleRS = None
    self.classRS  = None
    self.propRS   = None
    self.procRS   = None
    self.parmRS   = None

    # Lookup dictionaries
    self.__lpModule   = {}        # gnue_id -> name, name -> gnue_id
    self.__lpClass    = {}        # fqName -> gnue_id
    self.__lpFilter   = {}        # fqName -> gnue_id
    self.__lpProperty = {}        # class-id.module-id_name -> gnue_id
    self.__lpProcedure= {}        # class-name.module-name_name -> gnue_id
    self.__duplos     = {}        # fqName

    # Data from gcd files
    self.schema     = {}
    self.modules    = {}
    self.classes    = {}
    self.properties = {}
    self.procedures = {}
    self.parameters = {}

    if not len (self._filenames):
      raise StartupError, u_("No input file specified.")

    if not self._database:
      raise StartupError, u_("No connection specified.")

    if self._fileonly and self._dump is None:
      raise StartupError, \
          u_("Output to file only requested, but no filename specified.")

    try:
      for filename in files:
        self._files.append (openResource (filename))

    except IOError, err:
      for item in self._files:
        item.close ()

      raise StartupError, u_("Unable to open input file: %s") % \
          errors.getException () [2]


  # ---------------------------------------------------------------------------
  # Process the given GCD-files
  # ---------------------------------------------------------------------------

  def run (self):
    """
    This is the main function of the whole process. It loads all schema
    definitions and then logs into the connection to perform all actions
    requested.
    """

    self.__getModuleLookup ()
    self.__getClassLookup ()
    self.__getPropertyLookup ()
    self.__getProcedureLookup ()

    print o (u_("Loading gcd files ..."))

    for item in xrange (len (self._files)):
      try:
        self.__currentFile = self._filenames [item]

        schema = GCParser.loadFile (self._files [item])
        schema.walk (self.__iterateObjects)

      finally:
        self._files [item].close ()

    self.__integrityCheck ()

    self.executeAndGenerateCode ()
    self.updateRepository ()


  # ---------------------------------------------------------------------------
  # Execute and generate the code
  # ---------------------------------------------------------------------------

  def executeAndGenerateCode (self):
    """
    This function logs into the given connection and calls it for an update of
    it's schema according to the loaded table definitions. Additionally the
    schema creation code is generated by this call, which will be stored in the
    given output file (if requested by options).
    """

    connection = self._connections.getConnection (self._database)
    self._connections.loginToConnection (connection)

    print _("Updating schema ...")
    code = connection.updateSchema (self.schema.values (), self._fileonly)

    if self._dump is not None:
      dest = open (self._dump, 'w')

      for item in code:
        for line in item:
          dest.write (line + os.linesep)

      dest.close ()


  # ---------------------------------------------------------------------------
  # Update the class repository
  # ---------------------------------------------------------------------------

  def updateRepository (self):
    """
    This function updates all parts of the class repository.
    """

    print _("Updating class repository ...")

    commit  = self.__updateModules ()
    commit |= self.__updateClasses ()
    commit |= self.__updateProperties ()
    commit |= self.__updateProcedures ()
    commit |= self.__updateParameter ()

    if commit:
      self._connections.commitAll ()


  # ---------------------------------------------------------------------------
  # Iterate over all top level elements
  # ---------------------------------------------------------------------------

  def __iterateObjects (self, sObject):
    """
    This function iterates over all objects of a GCD tree and processes the
    instances.

    @param sObject: current GCD object to be processed
    """

    if sObject._type == 'GCModule':
      self.__currentModule = sObject
      self.__addModuleToRepository (sObject)

    elif sObject._type == 'GCClass':
      self.__currentClass = sObject
      self.__addClassToSchema (sObject)
      self.__addClassToRepository (sObject)

    elif sObject._type == 'GCProperty':
      self.__addPropertyToSchema (sObject)
      self.__addPropertyToRepository (sObject)

    elif sObject._type == 'GCProcedure':
      self.__addProcedureToRepository (sObject)

    elif sObject._type == 'GCParameter':
      self.__addParameterToRepository (sObject)

    elif sObject._type == 'GCIndex':
      self.__addIndexToSchema (sObject)


  
  # ---------------------------------------------------------------------------
  # Create a lookup dictionary for modules
  # ---------------------------------------------------------------------------

  def __getModuleLookup (self):
    """
    This function creates a lookup dictionary for modules, where it maps names
    to ids and vice versa
    """

    self.__lpModule = {}

    if self.moduleRS is None:
      self.moduleRS = self.__openSource ('gnue_module',
                   ['gnue_id', 'gnue_name', 'gnue_comment']).createResultSet ()

    rec = self.moduleRS.firstRecord ()
    while rec is not None:
      gid  = rec.getField ('gnue_id')
      name = rec.getField ('gnue_name')

      self.__lpModule [gid]  = name
      self.__lpModule [name.lower ()] = gid

      rec  = self.moduleRS.nextRecord ()



  # ---------------------------------------------------------------------------
  # Create a lookup dictionary for classes
  # ---------------------------------------------------------------------------

  def __getClassLookup (self):
    """
    This function creates a lookup dictionary for classes and filters. The
    class lookup maps names to ids and vice versa. If a class has a filter
    defined it will be set into the filter dictionary.
    """

    self.__lpClass  = {}
    self.__lpFilter = {}

    if self.classRS is None:
      self.classRS = self.__openSource ('gnue_class', ['gnue_id', 'gnue_name',
          'gnue_module', 'gnue_comment', 'gnue_filter']).createResultSet ()

    rec = self.classRS.firstRecord ()
    while rec is not None:
      gid  = rec.getField ('gnue_id')
      name = rec.getField ('gnue_name')
      mid  = rec.getField ('gnue_module')
      fid  = rec.getField ('gnue_filter')

      fqName = Namespace.createName (self.__lpModule [mid], name)

      self.__lpClass [gid]    = fqName
      self.__lpClass [fqName.lower ()] = gid

      if fid is not None:
        self.__lpFilter [fqName.lower ()] = fid

      rec  = self.classRS.nextRecord ()



  # ---------------------------------------------------------------------------
  # Build a lookup-dictionary for properties
  # ---------------------------------------------------------------------------

  def __getPropertyLookup (self):
    """
    This function creates a lookup dictionary for properties where the key is
    constructed from "classid.moduleid_propertyname".
    """

    self.__lpProperty = {}

    if self.propRS is None:
      self.propRS = self.__openSource ('gnue_property', ['gnue_id',
          'gnue_module', 'gnue_class', 'gnue_name', 'gnue_type', 'gnue_length',
          'gnue_scale', 'gnue_nullable', 'gnue_comment']).createResultSet ()

    rec = self.propRS.firstRecord ()
    while rec is not None:
      mid  = rec.getField ('gnue_module')
      cid  = rec.getField ('gnue_class')
      name = rec.getField ('gnue_name')
      key  = "%s.%s_%s" % (cid, mid, name.lower ())

      self.__lpProperty [key] = rec.getField ('gnue_id')

      rec  = self.propRS.nextRecord ()


  # ---------------------------------------------------------------------------
  # Build a lookup-dictionary for procedures
  # ---------------------------------------------------------------------------

  def __getProcedureLookup (self):
    """
    This function creates a procedure lookup dictionary where the key is built
    from "classname.modulename_procedurename".
    """

    self.__lpProcedure = {}

    if self.procRS is None:
      self.procRS = self.__openSource ('gnue_procedure', ['gnue_id',
        'gnue_module', 'gnue_class', 'gnue_name', 'gnue_type', 'gnue_length',
        'gnue_scale', 'gnue_nullable', 'gnue_comment', 'gnue_code',
        'gnue_language']).createResultSet ()

    rec = self.procRS.firstRecord ()
    while rec is not None:
      mName = self.__lpModule.get (rec.getField ('gnue_module'))
      cName = self.__lpClass.get (rec.getField ('gnue_class'))
      pName = rec.getField ('gnue_name')
      key   = "%s.%s_%s" % (cName, mName, pName)

      self.__lpProcedure [key.lower ()] = rec.getField ('gnue_id')

      rec  = self.procRS.nextRecord ()


  # ---------------------------------------------------------------------------
  # Add a module to the repository dictionary
  # ---------------------------------------------------------------------------

  def __addModuleToRepository (self, item):
    """
    This function adds a given module to the dictionary used for repository
    update later.

    @param item: GCModule instance of the module to be added
    """

    mkey = item.name.lower ()
    mDef = self.__fetchTags (item, ['name', 'comment'])
    mDef ['gnue_id'] = self.__lpModule.get (mkey)

    if not self.modules.has_key (mkey):
      self.modules [mkey] = {}

    self.modules [mkey] = mDef


  # ---------------------------------------------------------------------------
  # Add a class to the schema definition
  # ---------------------------------------------------------------------------

  def __addClassToSchema (self, item):
    """
    This function adds a class to the schema definition used for backend schema
    generation later on. If the same class should be added twice a
    'DuplicateClassError' will be raised.

    @param item: GCClass instance to be added to schema definition
    """

    ckey = item.fullName.lower ()
    if not self.schema.has_key (ckey):
      self.schema [ckey] = {'name': item.fullName, 'fields': []}

    if item.action == 'create':
      self.schema [ckey] ['primarykey'] = { \
          'name'  : 'pk_%s' % item.fullName,
          'fields': ['gnue_id']}

      if self.__duplos.has_key (ckey):
        raise DuplicateClassError, \
            (item.fullName, self.__currentFile, self.__duplos [ckey])

      self.__duplos [ckey] = self.__currentFile


  # ---------------------------------------------------------------------------
  # Add an index to the schema definition
  # ---------------------------------------------------------------------------

  def __addIndexToSchema (self, item):
    """
    This function adds an index to the schema definition

    @param item: GCIndex instance with the index to be added
    """

    classDef = self.schema [self.__currentClass.fullName.lower ()]
    if not classDef.has_key ('indices'):
      classDef ['indices'] = []

    classDef ['indices'].append ({'name'  : item.fullName,
                                  'unique': item.unique,
                                  'fields': item.fieldNames})


  # ---------------------------------------------------------------------------
  # Add a class instance to the repository dictionary
  # ---------------------------------------------------------------------------

  def __addClassToRepository (self, item):
    """
    This function adds a class to the class dictionary used for repository
    update later on.

    @param item: GCClass instance of the class to add
    """

    key  = item.fullName.lower ()
    cDef = self.__fetchTags (item, ['module', 'name', 'comment', 'filter'])
    cDef ['gnue_id'] = self.__lpClass.get (key)

    if not self.classes.has_key (key):
      self.classes [key] = {'_files': []}

    self.classes [key].update (cDef)
    self.classes [key]['_files'].append (self.__currentFile)


  # ---------------------------------------------------------------------------
  # Add a property to the schema definition
  # ---------------------------------------------------------------------------

  def __addPropertyToSchema (self, item):
    """
    This function adds another property to the schema definition used for
    backend schema generation.

    @param item: GCProperty instance to add
    """

    classDef = self.schema [item._parent.fullName.lower ()]

    pDef = {'name'    : item.fullName,
            'type'    : item.datatype,
            'nullable': item.nullable}

    if item.length: pDef ['length']    = item.length
    if item.scale:  pDef ['precision'] = item.scale

    classDef ['fields'].append (pDef)

    # If the property is a reference to another class, add a constraint to the
    # class definition
    if item.isReference:
      if not classDef.has_key ('constraints'):
        classDef ['constraints'] = []

      classDef ['constraints'].append ( \
          {'name'     : "fk_%s_%s" % (item._parent.fullName, item.fullName),
           'fields'   : [item.fullName],
           'reftable' : item.type,
           'reffields': ['gnue_id']})


  # ---------------------------------------------------------------------------
  # Add a property to the repository dictionary
  # ---------------------------------------------------------------------------

  def __addPropertyToRepository (self, item):
    """
    This function adds a property to the property dictionary used for
    repository update later on.

    @param item: GCProperty instance to add
    """

    fqName = "%s.%s" % (item._parent.fullName, item.fullName)
    pkey   = "%s.%s_%s" % (self.__lpClass.get (item._parent.fullName.lower ()),
                         self.__lpModule.get (item.module.lower ()), item.name)

    pDef = {'gnue_id'   : self.__lpProperty.get (pkey.lower ()),
            'gnue_class': item._parent.fullName,
            'gnue_type' : item.datatype}

    pDef.update (self.__fetchTags (item,
                 ['module', 'name', 'length', 'scale', 'nullable', 'comment']))
    
    if item.isReference:
      pDef.update ({'gnue_type': item.type, 'gnue_length': None})

    elif item.fullName.lower () == 'gnue_id':
      pDef.update ({'gnue_type': 'id', 'gnue_length': None})

    self.properties [fqName.lower ()] = pDef


  # ---------------------------------------------------------------------------
  # Add a procedure to the repository dictionary 
  # ---------------------------------------------------------------------------

  def __addProcedureToRepository (self, item):
    """
    This function adds a procedure to the dictionary used for repository update
    later on.

    @param item: GCProcedure instance to add
    """

    fqName = "%s.%s" % (item._parent.fullName, item.fullName)
    pDef = {'gnue_id'   : None,
            'gnue_class': item._parent.fullName,
            'gnue_type' : item.datatype,
            'gnue_code' : item.getChildrenAsContent ()}

    pDef.update (self.__fetchTags (item, ['module', 'name', 'length', 'scale',
                                          'nullable', 'comment', 'language']))

    self.procedures [fqName.lower ()] = pDef


  # ---------------------------------------------------------------------------
  # Add a parameter to the repository dictionary
  # ---------------------------------------------------------------------------

  def __addParameterToRepository (self, item):
    """
    This function adds a parameter to the dictionary used for repository update
    later on.

    @param item: GCParameter instance to add
    """

    pName  = "%s.%s" % (self.__currentClass.fullName, item._parent.fullName)
    fqName = "%s.%s" % (pName, item.name)

    par = {'gnue_procedure': pName, 'gnue_type': item.datatype}
    par.update (self.__fetchTags (item, ['name', 'length', 'scale', 'comment']))

    self.parameters [fqName.lower ()] = par


  # ---------------------------------------------------------------------------
  # Verify the built dictionaries
  # ---------------------------------------------------------------------------

  def __integrityCheck (self):
    """
    This function does some integrity checking on the dictionaries built from
    all the gcd files.
    """

    for (name, item) in self.classes.items ():
      # Make sure the module referenced is a valid module, either an existing
      # or a new one
      mname = item ['gnue_module'].lower ()
      if not self.__lpModule.get (mname) and not self.modules.get (mname):
        raise ModuleNotFoundError, (item ['gnue_module'], item ['_files'])

      # Make sure the filter class will be available if one is defined
      if item.get ('gnue_filter') is not None:
        fname = item ['gnue_filter']
        if not self.__lpClass.get (fname) and not self.classes.get (fname):
          raise FilterNotFoundError, (name, fname, item ['_files'])

      # For an existing class we need to make sure the filter-attribute won't
      # get changed, because we cannot add a not NULL field to an existing class
      if item ['gnue_id'] is not None:
        iFilter = item.get ('gnue_filter') and \
                  self.__lpClass [item ['gnue_filter'].lower ()]

        if iFilter != self.__lpFilter.get (name):
          raise FilterChangeError, (name, item ['_files'])


    # Integrity checks on properties
    for (name, item) in self.properties.items ():
      classDef = self.classes [item ['gnue_class'].lower ()]

      # If a property is a reference, make sure the referenced class exists
      if not item ['gnue_type'] in helpers.BASE_TYPES:
        refname = item ['gnue_type'].lower ()
        if not self.__lpClass.get (refname) and not self.classes.get (refname):
          raise ClassNotFoundError, (item ['gnue_type'], classDef ['_files'])

      # For an existing class make sure to *not* add NOT NULL properties 
      if classDef ['gnue_id'] is not None:
        if item ['gnue_id'] is None and not item ['gnue_nullable']:
          raise NullableError, (name, classDef ['_files'])


    # remove obsolete keys from dictionaries
    for item in self.classes.values ():
      del item ['_files']


  # ===========================================================================

  # ---------------------------------------------------------------------------
  # Update/add modules to the class repository
  # ---------------------------------------------------------------------------

  def __updateModules (self):
    """
    This function updates all modules listed in 'gnue_modules' and populates a
    lookup-dictionary for *all* modules available.

    @return: True if a commit is needed, False otherwise
    """

    # Update the result set and the lookup dictionary
    stat = self.__processResultSet (self.moduleRS, self.modules, ['gnue_name'])
    self.__getModuleLookup ()

    print o (u_("  Modules   : %(ins)3d inserted, %(upd)3d updated, %(kept)3d "
                "unchanged.") \
             % {'ins': stat [0], 'upd': stat [1], 'kept': stat [2]})

    return (stat [0] + stat [1]) > 0
  

  # ---------------------------------------------------------------------------
  # Update/add classes to the class repository
  # ---------------------------------------------------------------------------

  def __updateClasses (self):
    """
    This function updates all classes listed in the table 'gnue_class' and it
    updates the lookup dictionary for all classes available.

    @return: True if a commit is needed, False otherwise
    """

    # First make sure we have valid 'gnue_module' entries for all classes
    self.__replaceReferences (self.classes, self.__lpModule, 'gnue_module')

    # Then have a look at the filters defined on the classes
    filterStack = []

    for (fullName, item) in self.classes.items ():
      if item.has_key ('gnue_filter'):
        filterName = item ['gnue_filter']
        filterId   = self.__lpClass.get (filterName)

        # If the filter-class is available we can use it
        if filterId is not None:
          item ['gnue_filter'] = filterId
        else:
          # since the filter-id will be created later, we add it to the queue
          # and remove the attribute from the class.
          filterStack.append ((item, filterName, fullName))
          del item ['gnue_filter']

    # Merge all changes into the current result set
    cond = ['gnue_module', 'gnue_name']
    stat = self.__processResultSet (self.classRS, self.classes, cond)
    needCommit = (stat [0] + stat [1]) > 0

    # Update the lookup dictionary for classes
    self.__getClassLookup ()

    # Process all classes with a 'new' filter class
    filterChanged = False

    if filterStack:
      for (item, filterName, fullName) in filterStack:
        item ['gnue_filter'] = self.__lpClass.get (filterName)

      fst = self.__processResultSet (self.classRS, self.classes, cond)
      filterChanged = fst [1] > 0

    print o (u_("  Classes   : %(ins)3d inserted, %(upd)3d updated, %(kept)3d "
                "unchanged.") \
          % {'ins': stat [0], 'upd': stat [1], 'kept': stat [2]})

    return needCommit or filterChanged


  # ---------------------------------------------------------------------------
  # Update/add Properties to the class repository
  # ---------------------------------------------------------------------------

  def __updateProperties (self):
    """
    This function updates all properties listed in 'gnue_property'.

    @return: True if a commit is needed, False otherwise
    """

    # Make sure all properties have a valid 'gnue_module' and 'gnue_class'
    self.__replaceReferences (self.properties, self.__lpModule, 'gnue_module')
    self.__replaceReferences (self.properties, self.__lpClass, 'gnue_class')

    # Load and update all properties
    cond = ['gnue_module', 'gnue_class', 'gnue_name']
    stat = self.__processResultSet (self.propRS, self.properties, cond)

    print o (u_("  Properties: %(ins)3d inserted, %(upd)3d updated, %(kept)3d "
                "unchanged.") \
             % {'ins': stat [0], 'upd': stat [1], 'kept': stat [2]})

    return (stat [0] + stat [1]) > 0


  # ---------------------------------------------------------------------------
  # Update/add Procedures to the class repository
  # ---------------------------------------------------------------------------

  def __updateProcedures (self):
    """
    This function updates all procedures listed in 'gnue_procedure' and updates
    the lookup dictionary for them.

    @return: True if a commit is needed, False otherwise
    """

    # Make sure all procedures have a valid 'gnue_module' and 'gnue_class'
    self.__replaceReferences (self.procedures, self.__lpModule, 'gnue_module')
    self.__replaceReferences (self.procedures, self.__lpClass, 'gnue_class')

    # Load and update all procedures
    cond = ['gnue_module', 'gnue_class', 'gnue_name']
    stat = self.__processResultSet (self.procRS, self.procedures, cond)
    self.__getProcedureLookup ()

    print o (u_("  Procedures: %(ins)3d inserted, %(upd)3d updated, %(kept)3d "
                "unchanged.") \
              % {'ins': stat [0], 'upd': stat [1], 'kept': stat [2]})

    return (stat [0] + stat [1]) > 0


  # ---------------------------------------------------------------------------
  # Update/add parameters to the class repository
  # ---------------------------------------------------------------------------

  def __updateParameter (self):
    """
    This function updates all parameters listed in 'gnue_parameter'.

    @return: True if a commit is needed, False otherwise
    """

    # Make sure all parameters have a valid 'gnue_procedure'
    self.__replaceReferences (self.parameters, self.__lpProcedure,
        'gnue_procedure')

    self.parmRS = self.__openSource ('gnue_parameter', ['gnue_id',
        'gnue_procedure', 'gnue_name', 'gnue_type', 'gnue_scale',
        'gnue_length', 'gnue_comment']).createResultSet ()

    # Load and update all parameters
    cond = ['gnue_procedure', 'gnue_name']
    stat = self.__processResultSet (self.parmRS, self.parameters, cond)

    print o (u_("  Parameters: %(ins)3d inserted, %(upd)3d updated, %(kept)3d "
                "unchanged.") \
              % {'ins': stat [0], 'upd': stat [1], 'kept': stat [2]})

    return (stat [0] + stat [1]) > 0


  # ---------------------------------------------------------------------------
  # Update references in a dictionary
  # ---------------------------------------------------------------------------

  def __replaceReferences (self, dictionary, lookup, dname):
    """
    This function replaces all references in the 'dictionary' using a lookup
    dictionary. If an item is not available in the lookup dictionary a
    'ReferenceLookupError' will be raised (this would point us to a bug).

    @param dictionary: the dictionary to update references in
    @param lookup: the dictionary to search for the replacement values
    @param dname: the key to use in 'dictionary' and 'lookup'
    """

    for item in dictionary.values ():
      if lookup.has_key (item [dname].lower ()):
        item [dname] = lookup [item [dname].lower ()]
      else:
        raise ReferenceLookupError, (dname, item [dname])

  # ===========================================================================

  # ---------------------------------------------------------------------------
  # Create a new datasource for a given class
  # ---------------------------------------------------------------------------

  def __openSource (self, classname, fieldList):
    """
    This function creates a new datasource for the given classname with the
    given fieldlist. The primary key is always 'gnue_id'

    @param classname: name of the table to create a datasource for
    @param fieldList: sequence of fieldnames to use

    @return: datasource instance
    """

    return GDataSource.DataSourceWrapper (self._connections,
                                          fieldList,
                                          {'name'      : "dts_%s" % classname,
                                           'database'  : self._database,
                                           'table'     : classname,
                                           'primarykey': 'gnue_id'},
                                          unicodeMode = True)


  # ---------------------------------------------------------------------------
  # Get a dictionary with all keys listed in tags and values from sObject
  # ---------------------------------------------------------------------------

  def __fetchTags (self, sObject, tags):
    """
    This function creates a dictionary with all attributes from sObject listed
    in tags, where the keys are constructed by 'gnue_%s' % attributename.

    @param sObject: Schema object to retriev attributes from
    @param tags: list of all attributes to retrieve

    @return: dictionary with the attribute names as keys and their values
    """
    res = {}
    for item in tags:
      if hasattr (sObject, item):
        res ["gnue_%s" % item] = getattr (sObject, item)

    return res


  # ---------------------------------------------------------------------------
  # Update a given resultset using a dictionary of records
  # ---------------------------------------------------------------------------

  def __processResultSet (self, resultSet, items, condition):
    """
    This function iterates over the given resultset and updates all records
    listed in the given dictionary. All items of the dictionary not listed in
    the resultset will be inserted as new records.

    @param resultSet: the resultset to update
    @param items: dictionary of records to update the resultset with
    @param condition: sequence of key-fields used for comparison.
    @return: triple with number of (inserted, modified, unchanged) records
    """

    stat = [0, 0, 0]
    needPost = False

    # First we build a mapping using the condition as key for all items
    mapping = {}
    for item in items.values ():
      mapping [self.__getKey (item, condition)] = item

    # Now run over all existing records and update the records if necessary
    rec = resultSet.firstRecord ()
    while rec is not None:
      key = self.__getKey (rec, condition)
      if mapping.has_key (key):
        mapping [key]['gnue_id'] = rec.getField ('gnue_id')

        post = self.doUpdate (resultSet, mapping [key])
        needPost |= post
        stat [1 + [1, 0][post]] += 1
        # Remove the record from the mapping
        del mapping [key]

      rec = resultSet.nextRecord ()

    # If there are keys left in the mapping, they must be new records
    for item in mapping.values ():
      item ['gnue_id'] = self.__generateId ()
      resultSet.insertRecord ()
      self.doUpdate (resultSet, item)
      needPost = True
      stat [0] += 1

    if needPost:
      resultSet.post ()

    return stat


  # ---------------------------------------------------------------------------
  # Get a tuple of fields from a record
  # ---------------------------------------------------------------------------

  def __getKey (self, record, fields):
    """
    This function creates a tuple of fields from a record. If a field value has
    a 'lower' function the value will be lowered. This ensures to get a
    case-insensitive key.

    @param record: dict-like record to retrieve values from
    @param fields: sequence of fieldnames to retrieve

    @return: n-tuple with the (lowered) values of all fields
    """

    result = []
    for field in fields:
      value = record [field]
      if hasattr (value, 'lower'):
        value = value.lower ()

      result.append (value)

    return tuple (result)


  # ---------------------------------------------------------------------------
  # Perform an update on the given resultset using a given data dictionary
  # ---------------------------------------------------------------------------

  def doUpdate (self, resultSet, data):
    """
    This function sets all fields in the current record of the resultset based
    on the key/values given by the data dictionary. It returns True, if a field
    value has been changed, otherwise False.

    @param resultSet: resultset with the current record to be updated
    @param data: dictionary with keys and values used for updates

    @return: True if a field has been changed, False if no field has been
        changed.
    """
    doPost = False

    for key in data:
      if resultSet.current.getField (key) != data [key]:
        resultSet.current.setField (key, data [key])
        doPost = True

    return doPost


  # ---------------------------------------------------------------------------
  # Generate a new object id
  # ---------------------------------------------------------------------------

  def __generateId (self):
    """
    This function generates a new gnue_id like it is done by appserver. Once
    this algorithm should be replace by a better one.
    """

    if self._uuidType == 'time':
      return UUID.generateTimeBased ()
    else:
      return UUID.generateRandom ()



if __name__ == "__main__":
  gcdClient ().run ()
