# GNU Enterprise Application Server - Filter Support
#
# Copyright 2001-2005 Free Software Foundation
#
# This file is part of GNU Enterprise
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# $Id: geasFilter.py 6901 2005-01-14 08:34:49Z johannes $

from language import Session
from classrep import Namespace
from gnue.common.apps import errors


# =============================================================================
# Exceptions
# =============================================================================

class CircularFilterError (errors.ApplicationError):
  def __init__ (self):
    msg = u_("The filters have circular references")
    errors.ApplicationError.__init__ (self, msg)


# =============================================================================
# This class encapsulates handling of filters
# =============================================================================

class geasFilter:

  # ---------------------------------------------------------------------------
  # Constructor
  # ---------------------------------------------------------------------------

  def __init__ (self, sm):
    self.__sm      = sm
    self.__session = Session.InternalSession (sm)
    self.__filters = self.__loadFilters ()
    self.__labels  = self.__fetchLabels ()
    self.__cLabels = self.__fetchClassLabels ()


  # ---------------------------------------------------------------------------
  # Get a structure describing the defined filters, their labels and values
  # ---------------------------------------------------------------------------

  def getFilter (self, language):
    """
    This function creates a structure holding all defined filters, their labels
    in a requested language and all their possible values. The result is a
    sequence of tuples, where a tuple is constructed as follows:

      (filter-class, labels, master, data)

    filter-class: tuple with the name of the filter class and it's label
    labels: a sequence of tuples describing the labels (label, search, field)
    master: name of the master-filter or None if no such master is defined
    data: dictionary describing the allowed values of the filter's fields

    @return: sequence of tuples as described above.
    """

    result = []

    for (fId, filterDef) in self.__labels:
      # Create an ordered sequence of all search/info-properties as stated in
      # the GNUe Language Definition of the filter class
      fields = []

      for (propertyId, propertyDef) in filterDef.items ():
        field = self.__getLabel (self.__getLanguages (language), propertyDef)
        fields.append ((field ['order'], field ['label'], field))

      # order sequence by search-value or info-value
      fields.sort ()

      classdef = self.__sm.classes.find (fId)
      master   = classdef.gnue_filter and classdef.gnue_filter or None
      if master is not None:
        master = self.__sm.classes.find (master.gnue_id).fullName

      labels = [(f [1], f [2]['search'], f [2]['name']) for f in fields]
      names  = [f [2]['name'] for f in fields]

      if self.__cLabels.has_key (fId):
        classLabel = self.__getLabel (self.__getLanguages (language),
                                      self.__cLabels [fId])
      else:
        classLabel = labels [0][0]

      data = self.__getData (fId, master, names)
      result.append (((classdef.fullName, classLabel), labels, master, data))

    return result



  # ---------------------------------------------------------------------------
  # Load a sorted list of all filters defined in the class repository
  # ---------------------------------------------------------------------------

  def __loadFilters (self):
    """
    This function creates a sequence with all filter-Ids defined in the class
    repository. The sequence is given in such an order which respects internal
    filter dependancies. 

    @return: sequence with gnue_id's of all filter classes found in the class
        repository.
    """

    self.__fDict = {}

    # First get all filters defined in class repository
    for klass in self.__sm.classes.values ():
      if klass.gnue_filter:
        self.__addFilter (klass.gnue_filter)

    result = []
    add = self.__getIndependantFilters ()

    while len (add):
      result.extend (add)
      add = self.__getIndependantFilters ()

    return result
    

  # ---------------------------------------------------------------------------
  # Add a filter class to the filter dictionary
  # ---------------------------------------------------------------------------

  def __addFilter (self, filterClass):
    """
    This function adds a filter to the filter dicionary. If the filter has
    another master itself this function starts a recursion with that master.

    @param filterClass: Class instance of the filter class to be added
    """

    filterId = filterClass.gnue_id
    if not self.__fDict.has_key (filterId):
      self.__fDict [filterId] = []

    if filterClass.gnue_filter:
      refId = filterClass.gnue_filter.gnue_id
      if not refId in self.__fDict [filterId]:
        self.__fDict [filterId].append (refId)

      self.__addFilter (filterClass.gnue_filter)


  # ---------------------------------------------------------------------------
  # Get all items from the filter dictionary which have no dependancies
  # ---------------------------------------------------------------------------

  def __getIndependantFilters (self):
    """
    This function returns a sequence of all filter-Ids without any dependancies
    to other filters. If no such filter is found but there are still filters in
    the dictionary a CircularFilterError will be raised.

    @return: sequence holding the gnue_id's of all filters without any
        reference to another filter.
    """

    result = []

    for (filterId, references) in self.__fDict.items ():

      # if a filter has no other references, we can add it to the result and
      # remove all references in other filters to this filter.
      if not len (references):
        result.append (filterId)

        for others in self.__fDict.values ():
          if filterId in others:
            others.remove (filterId)

        del self.__fDict [filterId]

    # If no filter was independant, but there are still filters in the
    # dictionary, there must be a circular reference
    if not len (result) and len (self.__fDict.keys ()):
      raise CircularFilterError

    return result


  # ---------------------------------------------------------------------------
  # Create a sequence with all labels per filter (in all languages)
  # ---------------------------------------------------------------------------

  def __fetchLabels (self):
    """
    This function creates a sequence of all filters and all their labels in all
    languages defined in the class repository. Each element of the sequence is
    a tuple with the gnue_id of the filter class and a dictionary describing
    the filter labels. This dicionary is constructed as follows:
    {property-gnue_id: {language: {'order':?, 'name':?, 'label':?, 'search':?}}}
    
    @return: sequence of tuples describing the filters and their labels
    """

    result = []

    for filterId in self.__filters:
      cond = ['and', ['or', ['notnull', ['field', 'gnue_search']],
                            ['notnull', ['field', 'gnue_info']]],
                     ['eq', ['field', 'gnue_property.gnue_class'],
                            ['const', filterId]]]

      labels = self.__session.find ('gnue_label', cond, [], ['gnue_language',
          'gnue_label', 'gnue_search', 'gnue_info', 'gnue_id', 'gnue_property',
          'gnue_property.gnue_class'])

      entry = {}

      for label in labels:
        prop  = label.gnue_property
        klass = self.__sm.classes.find (prop.gnue_class.gnue_id)

        fullName = None
        for p in klass.properties.values ():
          if p.gnue_id == prop.gnue_id:
            fullName = p.fullName
            break

        if not entry.has_key (prop.gnue_id):
          entry [prop.gnue_id] = {}
          
        entry [prop.gnue_id][label.gnue_language] = \
            {'order' : label.gnue_search or label.gnue_info,
             'label' : label.gnue_label,
             'name'  : fullName,
             'search': label.gnue_search is not None}

      if not len (labels):
        filterClass = self.__sm.classes.find (filterId)
        prop = filterClass.properties ['gnue_id']
        entry [prop.gnue_id] = {'C': {'order' : 0,
                                      'label' : filterClass.fullName,
                                      'name'  : 'gnue_id',
                                      'search': True}}

      result.append ((filterId, entry))

    return result


  # ---------------------------------------------------------------------------
  # Get a dictionary with all class labels of the filter classes
  # ---------------------------------------------------------------------------

  def __fetchClassLabels (self):
    """
    This function creates a dictionary of class-labels for all filters
    currently in use. This dictionary has the filter-id's as keys and another
    dictionary - with language = label - as values.

    @return: dictionary {filter-id: {language: label}, {language: label} ..}
    """

    result = {}

    for filterId in self.__filters:
      filterClass = self.__sm.classes.find (filterId)
      prop = filterClass.properties ['gnue_id']
      cond = ['eq', ['field', 'gnue_property'], ['const', prop.gnue_id]]

      labels = self.__session.find ('gnue_label', cond, [],
                                    ['gnue_language', 'gnue_label'])
      if labels:
        result [filterId] = {}

        for item in labels:
          result [filterId] [item.gnue_language] = item.gnue_label

    return result


  # ---------------------------------------------------------------------------
  # Create a sequence of languages in descending order
  # ---------------------------------------------------------------------------

  def __getLanguages (self, language):
    """
    This function creates a sequence of languages to be used for fetching
    labels out of a given language. Where the starts with the most specific
    language. The result contains at least the language 'C'.

    @param language: Language to create a sequence out of, i.e. 'de_AT'
    @return: sequence of languages to try, i.e. ['de_AT', 'de', 'C']
    """

    result = []

    if '_' in language:
      result = [language.split ('_') [0]]

    if not language in result:
      result.insert (0, language)

    if not 'C' in result:
      result.append ('C')

    return result



  # ---------------------------------------------------------------------------
  # Get the first usable label for a given language
  # ---------------------------------------------------------------------------

  def __getLabel (self, languages, items):
    """
    This function returns a label from the given dictionary according to the
    order given in the languages-sequence.

    @param languages: sequence with languages to fetch a label for
    @param items: dictionary with labels, where the language is the key
    @return: dictionary describing the label
    """

    for lang in languages:
      if items.get (lang) is not None:
        return items.get (lang)

    # if no item was found for the requested languages try an english one. If
    # no such item is defined we use the first element defined
    return items.get ('en', items.values () [0])


  # ---------------------------------------------------------------------------
  # Get a dictionary with all valid data for a given filter
  # ---------------------------------------------------------------------------

  def __getData (self, filterId, master, fields):
    """
    This function creates a dictionary with all allowed values for a given
    filter. If the filter has a master assigned, the dictionary's keys will be
    all the master's values (gnue_id), where the values of the dictionary are
    all fields of the filter plus a gnue_id.

    @param filterId: gnue_id of the filter class
    @param master: name of the master-filter or None
    @param fields: sequence with all fieldnames to be fetched

    @return: dictionary with the possible filter values. This dictionary has
        one key per entry in the master-filter and a sequence with all fields
        as data values. Each entry in this sequence is a dictionary with
        the filter's field-names and -values.
    """

    result    = {}
    classname = self.__sm.classes.find (filterId).fullName

    # if we have a master-filter add the master-field to the query as well as a
    # 'select-all' condition to prevent geasSession.request () of inserting a
    # filter-value
    if master is not None:
      masterField = "%s.gnue_id" % master
      condition   = ['like', ['field', master], ['const', '%']]
      fields.append (masterField)

    else:
      masterField = None
      condition   = []

    data = self.__session.find (classname, condition, [], fields)

    for row in data:
      # use the master-id as dictionary key or None if no master available
      key = master is not None and row [master].gnue_id or None

      if not result.has_key (key):
        result [key] = []

      add = {'gnue_id': row.gnue_id}
      for field in fields:
        if field != masterField:
          add [field] = row [field]

      result [key].append (add)

    return result
