#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# Copyright 2000-2005 Free Software Foundation
#
# FILE:
# DBSIG2/DataObject.py
#
# DESCRIPTION:
"""
Generic implementation of dbdriver using Python DB-SIG v2
specification.
"""
#
# NOTES:
# The classes below are meant to be extended
#
# HISTORY:
#

__all__ = ['DataObject','DataObject_SQL','DataObject_Object']

from gnue.common.datasources import GConditions, Exceptions
from gnue.common.datasources.drivers.Base import DataObject as BaseDataObject
from gnue.common.apps import errors
import string
import types
import mx.DateTime
from string import join, replace

from ResultSet import ResultSet

######################################################################
#
#
#
class DataObject(BaseDataObject):

  _escapeSingleQuote = "'"
  _resultSetClass = ResultSet
  _DatabaseError = None

  # The date/time format used in insert/select statements
  # (based on format used for time.strftime())
  _dateTimeFormat = "'%c'"
  _timeFormat = "'%X'"

  # If a DB driver supports a unique identifier for rows,
  # list it here.  _primaryIdField is the field name (lower case)
  # that would appear in the recordset (note that this can be
  # a system generated format). If a primary id is supported,
  # _primaryIdFormat is the WHERE clause to be used. It will have
  # the string  % (fieldvalue) format applied to it.
  #
  # See Oracle drivers for example
  _primaryIdField = None         # Internal recordset field name (lowercase!!!)
  _primaryIdSelect = ""  # Select clause
  _primaryIdFormat = "__gnue__ = '%s'"         # Where clause format

  conditionElements = {
       'add':             (2, 999, '(%s)',                   '+'       ),
       'sub':             (2, 999, '(%s)',                   '-'       ),
       'mul':             (2, 999, '(%s)',                   '*'       ),
       'div':             (2, 999, '(%s)',                   '/'       ),
       'and':             (1, 999, '(%s)',                   ' AND '   ),
       'or':              (2, 999, '(%s)',                   ' OR '    ),
       'not':             (1,   1, '(NOT %s)',               None      ),
       'negate':          (1,   1, '-%s',                    None      ),
       'null':            (1,   1, '(%s IS NULL)',           None      ),
       'notnull':         (1,   1, '(%s IS NOT NULL)',       None      ),
       'eq':              (2,   2, '(%s = %s)',              None      ),
       'ne':              (2,   2, '(%s != %s)',             None      ),
       'gt':              (2,   2, '(%s > %s)',              None      ),
       'ge':              (2,   2, '(%s >= %s)',             None      ),
       'lt':              (2,   2, '(%s < %s)',              None      ),
       'le':              (2,   2, '(%s <= %s)',             None      ),
       'like':            (2,   2, '%s LIKE %s',             None      ),
       'notlike':         (2,   2, '%s NOT LIKE %s',         None      ),
       'between':         (3,   3, '%s BETWEEN %s AND %s',   None      ),
       'notbetween':      (3,   3, '(%s NOT BETWEEN %s AND %s)', None  ),
       # These two are hacks... these are not really valid tags
       # Used when the 2nd value of EQ or NE is NULL.
       '__iseq':          (2,   2, '(%s IS %s)',             None      ),
       '__isne':          (2,   2, '(%s IS NOT %s)',         None      ),
       'upper':           (1,   1, 'UPPER(%s)',              None      ),
       'lower':           (1,   1, 'LOWER(%s)',              None      )}


  def __init__(self, *args, **params):
    BaseDataObject.__init__(self, *args, **params)

    gDebug (8, "DB-SIG database driver backend initializing")


    # TODO: uh, hmm..
    self._strictQueryCount = True   # strictQueryCount

    self.distinct = False

    # If possible, this will contain the field names used in the last select
    # in the same order used in the SELECT statement.
    self._fieldOrder = []

    # Internal flag to avoid consistently doing the same check
    # If this is set to 1 initially, then the
    self._primaryIdChecked = False  # Internal flag



  def _toSqlString(self, value):
    if isinstance (value, mx.DateTime.DateTimeType):
      if (value.year, value.month, value.day) == (1, 1, 1):
        return value.strftime (self._timeFormat)
      else:
        return value.strftime (self._dateTimeFormat)
    else:
      if value == None:
        return "NULL"

      elif type(value) == types.IntType:
        return "'%d'" % value

      elif type(value) == types.LongType:
        return "'%d'" % value

      elif type(value) == types.FloatType:
        if value==int(value):
          return "'%d'" % value
        else:
          return "'" + str (value) + "'"

      elif type(value) == types.StringType:

        if self._unicodeMode:
          gDebug (1, 'WARNING: non-unicode passed to the dbdriver (%s)' % value)

        return "'%s'" % replace(value,
                                       "'",
                                       "%s'" % self._escapeSingleQuote)

      elif type(value) == types.UnicodeType:
        return "'%s'" % replace(value,
                                       "'",
                                       "%s'" % self._escapeSingleQuote)

      # TODO: make this test faster, possibly move type check to GTypecast
      elif hasattr(types,'BooleanType') and type(value) == types.BooleanType:
        if value:
          return 'TRUE'
        else:
          return 'FALSE'

      else:
        err = u_("Object of unknown type (%s) passed to DBSIG2 based dbdriver.") % type(value)
        # FIXME: raise an error instead of just printing a warning, after some transition time
        #raise Exceptions.UnknownDataType, err
        gDebug (1, err)
        return "'%s'" % replace(str(value),
                                       "'",
                                       "%s'" % self._escapeSingleQuote)


  # Used by drivers with a unique id (like rowid) (see Oracle for example)
  def _checkForPrimaryId(self):
    self._primaryIdChecked = True


  def _createResultSet(self, conditions={}, readOnly=False,
                       masterRecordSet=None,sql=""):

    # Used by drivers with a unique id (like rowid)
    if not self._primaryIdChecked: self._checkForPrimaryId()

    try:
      query = self._buildQuery(conditions, additionalSQL = sql)
      cursor = self._connection.makecursor(query)
      cursor.arraysize = self.cache

      # pull a record count
      if self._strictQueryCount:
        recordCount = cursor.rowcount
        #disable the count query and see if anyone screams.
        #recordCount = self._getQueryCount(conditions,sql)
        #  ARGH!!!! Oh, the agony... the agony....

    except self._DatabaseError:
      raise Exceptions.ConnectionError, errors.getException () [2]

    rs = self._resultSetClass(self, cursor=cursor, masterRecordSet=masterRecordSet,
       fieldOrder=self._fieldOrder)
    if self._strictQueryCount:
      rs._recordCount = recordCount
    if readOnly:
      rs._readonly = readOnly

    return rs

  def _getQueryCount(self,conditions={},sql=""):
    query = self._buildQueryCount (conditions, additionalSQL = sql)
    return self._connection.sql1 (query)


  # Used to convert a condition tree to an sql where clause
  def _conditionToSQL (self, condition):
    if condition == {} or condition == None:
      return ""
    elif type(condition) == types.DictType:
      cond = GConditions.buildConditionFromDict(condition)
    else:
      cond = condition

    if not len(cond._children):
      return ""
    elif len(cond._children) > 1:
      chillun = cond._children[:]
      cond._children = []
      _and = GConditions.GCand(cond)
      _and._children = chillun

    where = " WHERE (%s)" % (self.__conditionToSQL (cond._children[0]))
    gDebug (8, where)
    return where

  #
  # Used internally by _conditionToSQL
  #
  # This code recursively travels down a condition tree replacing the objects
  # with a strings representation
  def __conditionToSQL (self, element):
    if type(element) != types.InstanceType:
      return "%s" % element
    else:
      # Note that we strip the GC from the object types and lowercase the rest
      otype = string.lower(element._type[2:])
      #print "Otype: ",otype
      if otype == 'cfield':
        return "%s" % element.name
      elif otype == 'cconst':
        if element.value == None:
          return "NULL"
        elif element.type == 'number':
          return "%s" % element.value
        else:
          return self._toSqlString(element.value)
      elif otype == 'cparam':
        v = element.getValue()
        return (v == None and "NULL") or ("'%s'" % v)
      elif self.conditionElements.has_key(otype):
        result=[]
        for i in range(0, len(element._children)):
          result.append(self.__conditionToSQL(element._children[i]))
        if len(result) == 2 and \
           otype in ('eq','ne') and \
           result[1] == 'NULL':
           otype = "__is%s" % otype

        if len(result) < self.conditionElements [otype][0]:
          tmsg = u_('Condition element "%(element)s" expects at least '
                    '%(expected)s arguments; found %(found)s') \
                 % {'element' : otype,
                    'expected': self.conditionElements[otype][0],
                    'found'   : len (result)}
          raise GConditions.ConditionError, tmsg

        if len(result) > self.conditionElements [otype][1]:
          tmsg = u_('Condition element "%(element)s" expects at most '
                    '%(expected)s arguments; found %(found)s') \
                 % {'element' : otype,
                    'expected': self.conditionElements[otype][1],
                    'found'   : len (result)}
          raise GConditions.ConditionError, tmsg

        if self.conditionElements[otype][3] == None:
          return self.conditionElements[otype][2] % tuple(result)
        else:
          return self.conditionElements[otype][2] % \
            (join(result, self.conditionElements[otype][3]))
      else:
        tmsg = u_('Condition clause "%s" is not supported by this db driver.') % otype
        raise GConditions.ConditionNotSupported, tmsg



######################################################################
#
#
#
class DataObject_Object(DataObject):

  def _buildQuery(self, conditions={}, forDetail=None, additionalSQL=""):
    gDebug (8, 'DBSIG2-DO::Implicit Fields: %s' % self._fieldReferences)
    if self.distinct:
      distinct = "distinct "
    else:
      distinct = ""

    if self._primaryIdSelect:
      pis = "%s," % self._primaryIdSelect
    else:
      pis = ""

    whereClause = self._conditionToSQL(conditions)
    if additionalSQL:
      if len(whereClause):
        whereClause += ' and %s' % (additionalSQL)
      else:
        whereClause = ' WHERE %s' % (additionalSQL)

    if forDetail:
      q = "(%s) in (SELECT %s FROM %s%s)" % \
                     (join(self._masterfields,","),
                      join(self._detailfields,","),
                      self.table, whereClause)
    elif self._fieldReferences:
      self._fieldOrder = fields = self._fieldReferences.keys()
      q = "SELECT %s%s%s FROM %s%s" % \
           (distinct, pis, join(fields,","), self.table,
            whereClause)
      if pis:
        self._fieldOrder.insert(0,None)
    else:
      self._fieldOrder = []
      self._primaryIdSelect = None
      q = "SELECT %s* FROM %s%s" % (distinct, self.table,
                        whereClause)

    if hasattr (self, 'sorting') and not forDetail and self.sorting:
      order = []
      for item in self.sorting:
        field      = item ['name']
        descending = item.get ('descending') or False
        ignorecase = item.get ('ignorecase') or False

        fmt = ignorecase and "LOWER(%s)%s" or "%s%s"
        order.append (fmt % (field, descending and ' desc' or ''))

      q = "%s ORDER BY %s " % (q, string.join (order, ", "))

    gDebug (8, q)

    return q

  def _buildQueryCount(self, conditions={}, additionalSQL=""):
    whereClause = self._conditionToSQL(conditions)
    if additionalSQL:
      if len(whereClause):
        whereClause += ' and %s' % (additionalSQL)
      else:
        whereClause = ' WHERE %s' % (additionalSQL)


    q = "SELECT count(*) FROM %s%s" % (self.table, whereClause)

    gDebug (8, q)

    return q



######################################################################
#
#
#
class DataObject_SQL(DataObject):

  # Set by GDataSource
  # (GDataSource.GSql instance)
  _rawSQL = None

  def _buildQuery(self, conditions={}, forDetail=None, additionalSQL=""):
    try:
      sql= self.__sql
    except AttributeError:
      sql = self.__sql = self._rawSQL.getChildrenAsContent()
    return sql


