#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or(at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# Copyright 2000-2005 Free Software Foundation
#
# FILE:
# Introspection.py
#
# DESCRIPTION:
#
# NOTES:
#
# $Id: Introspection.py 6851 2005-01-03 20:59:28Z jcater $

__all__ = ['Introspection']

import string

from gnue.common.datasources import GIntrospection

# =============================================================================
# Exceptions
# =============================================================================

# -----------------------------------------------------------------------------
# Base class for all Introspection exceptions
# -----------------------------------------------------------------------------
class IntrospectionError (gException):
  pass

# -----------------------------------------------------------------------------
# Requested module does not exist in class repository
# -----------------------------------------------------------------------------
class ModuleNotFoundError (IntrospectionError):
  def __init__ (self, module):
    msg = u_("Module '%s' not found.") % module
    gException.__init__ (self, msg)



# =============================================================================
# Introspection support for application server
# =============================================================================
class Introspection (GIntrospection.Introspection):

  # list of the types of Schema objects this driver provides
  types = [('object',_('Business Object Class'),1)]


  # -------------------------------------------------------------------------
  # Get a list of schema objects for name (and type)
  # -------------------------------------------------------------------------
  def find (self, name = None, type = None):
    self._modules = self._getModuleDictionary ()
    self._classes = {}
    result        = []

    if name is None:
      if not type in ['object', None]:
        return []

      cond = []
    else:
      cond = self._getNameCondition (name)

    fields = ["gnue_module", "gnue_name"]
    sess = self._connection._sm
    sessId = self._connection._sess_id
    listId = sess.request (sessId, 'gnue_class', cond, fields, fields)

    for rec in sess.fetch (sessId, listId, 0, sess.count (sessId, listId)):
      modname = self._modules [rec [1]]
      self._classes [rec [0]] = "%s_%s" % (modname, rec [2])


    for classid in self._classes.keys ():
      attrs = {"id"     : string.lower (self._classes [classid]),
               "name"   : self._classes [classid],
               "type"   : "object",
               "gnue_id": classid}
      
      result.append (GIntrospection.Schema (attrs,
                                       getChildSchema = self._getChildSchema))

    if len (result):
      return result

    else:
      return None



  # -------------------------------------------------------------------------
  # Create a dictionary with all available modules
  # -------------------------------------------------------------------------

  def _getModuleDictionary (self):
    """
    This function creates a dictionary of all modules available, where the
    gnue_id acts as key.
    """
    res    = {}
    sess   = self._connection._sm
    sessId = self._connection._sess_id
    listId = sess.request (sessId, 'gnue_module', [], ['gnue_id'], 
                               ['gnue_name'])
    for rec in sess.fetch (sessId, listId, 0, sess.count (sessId, listId)):
      res [rec [0]] = rec [1]
      
    return res


  # -------------------------------------------------------------------------
  # Create a condition dictionary for a given classname
  # -------------------------------------------------------------------------

  def _getNameCondition (self, name):
    """
    This function creates a condition dictionary which matches the given name.
    """
    (module, classname) = string.split (name, '_')

    if not module in self._modules.values ():
      raise ModuleNotFoundError, module

    # Find the Module-Id
    for moduleId in self._modules.keys ():
      if self._modules [moduleId] == module:
        break

    return ['and', ['eq', ['field', 'gnue_module'], ['const', moduleId]],
                   ['eq', ['field', 'gnue_name'], ['const', classname]]]
                  



  # -------------------------------------------------------------------------
  # Get a list of all properties owned by a given class
  # -------------------------------------------------------------------------
  def _getChildSchema (self, parent):
    result = []
    self._modules = self._getModuleDictionary ()

    sort   = ['gnue_module', 'gnue_name']
    fields = sort + ['gnue_type', 'gnue_length', 'gnue_scale']

    sess   = self._connection._sm
    sessId = self._connection._sess_id
    listId = sess.request (sessId, 'gnue_property',
                    ['eq', ['field', 'gnue_class'], ['const', parent.gnue_id]],
                           sort, fields)

    for rec in sess.fetch (sessId, listId, 0, sess.count (sessId, listId)):
      name = "%s_%s" % (self._modules [rec [1]], rec [2])
      nativetype = rec [3]

      if name == 'gnue_id' or '_' in nativetype:
        length = 32
      else:
        length = rec [4]

      if nativetype == 'number':
        datatype = 'number'
      elif nativetype in ['date', 'datetime']:
        datatype = 'date'
      else:
        datatype = 'text'

      attrs = {'id'        : "%s.%s" % (parent.id, name.lower ()),
               'name'      : name,
               'datatype'  : datatype,
               'precision' : rec [5] or 0,
               'nativetype': nativetype,
               'length'    : length,
               'required'  : 0}

      result.append (GIntrospection.Schema (attrs))
    
    # 
    # add calculated properties
    #
    listId = sess.request (sessId, 'gnue_procedure',
        ['and', ['eq', ['field', 'gnue_class'], ['const', parent.gnue_id]],
                ['like', ['field', 'gnue_name'], ['const', 'get%']],
                ['notnull', ['field', 'gnue_type']]],
                           sort, fields)

    for rec in sess.fetch (sessId, listId, 0, sess.count (sessId, listId)):
      pList = sess.request (sessId, 'gnue_parameter',
                ['eq', ['field', 'gnue_procedure'], ['const', rec [0]]],
                [], ['gnue_id'])

      if sess.count (sessId, pList):
        continue

      name = "%s_%s" % (self._modules [rec [1]], rec [2] [3:])
      nativetype = rec [3]

      if '_' in nativetype:
        length = 32
      else:
        length = rec [4]

      if nativetype == 'number':
        datatype = 'number'
      elif nativetype in ['date', 'datetime']:
        datatype = 'date'
      else:
        datatype = 'text'

      attrs = {'id'        : "%s.%s" % (parent.id, name.lower ()),
               'name'      : name,
               'datatype'  : datatype,
               'precision' : rec [5],
               'nativetype': nativetype,
               'length'    : length,
               'required'  : 0}

      result.append (GIntrospection.Schema (attrs))

    return result
