# GNU Enterprise Common - SQLite DB Driver - Schema creation
#
# Copyright 2001-2005 Free Software Foundation
#
# This file is part of GNU Enterprise
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# $Id: Creation.py 6851 2005-01-03 20:59:28Z jcater $

import os
import re
import string

from gnue.common.apps import errors
from gnue.common.datasources.drivers.DBSIG2.Schema.Creation import \
    Creation as Base

# =============================================================================
# Excpetions
# =============================================================================

class MissingTableError (errors.AdminError):
  def __init__ (self, table):
    msg = u_("Cannot find table '%s' anymore") % table
    errors.AdminError.__init__ (self, msg)

# =============================================================================
# Class implementing schema creation for SQLite
# =============================================================================

class Creation (Base.Creation):

  MAX_NAME_LENGTH = 31
  ALTER_MULTIPLE  = False

  _CMD = re.compile ('(.*?)\((.*)\)(.*)')


  # ---------------------------------------------------------------------------
  # Create a new database
  # ---------------------------------------------------------------------------

  def createDatabase (self):
    """
    """
    dbname   = self.connection.parameters.get ('dbname', None)
    self.connection.manager.loginToConnection (self.connection)


  # ---------------------------------------------------------------------------
  # Create a primary key
  # ---------------------------------------------------------------------------

  def createPrimaryKey (self, tableName, keyDefinition, codeOnly = False):
    """
    This function creates a primary key for the given table using the primary
    key definition.

    @param tableName: name of the table for which a key should be created
    @param keyDefinition: a dictionary of the primary key to be created 
    @param codeOnly: if TRUE no operation takes place, but only the code will
        be returned.
    @return: a tuple of sequences (prologue, body, epliogue) containing the
        code to perform the action.
    """
    res = ([], [], [])
    fields  = string.join (keyDefinition ['fields'], ", ")
    keyName = self._shortenName (keyDefinition ['name'])
    code = u"PRIMARY KEY (%s)" % fields
    res [1].append (code)

    return res


  # ---------------------------------------------------------------------------
  # Create a constraint
  # ---------------------------------------------------------------------------

  def createConstraint (self, tableName, constraintDef, codeOnly = False):
    """
    This function returns an empty code-tuple, since SQLite does not support
    foreign key constraints.

    @param tableName: name of the table for which an index should be created
    @param constraintDef: a dictionary of the constraint to be created 
    @param codeOnly: if TRUE no operation takes place, but only the code will
        be returned.
    @return: an tuple of empty sequences.
    """
    print "WARNING: SQLite doesn't support FOREIGN KEY constraints"
    return ([], [], [])


  # ---------------------------------------------------------------------------
  # Modify a table
  # ---------------------------------------------------------------------------

  def modifyTable (self, tableDefinition, codeOnly = False):
    """
    This function modifies a table according to the given definition.

    @param tableDefinition: a dictionary of the table to be modified
    @param codeOnly: if TRUE no operation takes place, but only the code will
        be returned.
    @return: a tuple of sequences (prologue, body, epliogue) containing the
        code to perform the action.
    """

    res = ([], [], [])
    body = res [1]

    table     = tableDefinition ['name']
    hasFields = tableDefinition.has_key ('fields') and \
        len (tableDefinition ['fields'])

    # Remove old indices
    if not hasFields and tableDefinition.has_key ('old_indices'):
      for ixName in tableDefinition ['old_indices']:
        self.mergeTuple (res, self.dropIndex (table, ixName, True))

    if hasFields:
      tSchema = self.introspector.find (name = tableDefinition ['name'])
      if tSchema is None:
        raise MissingTableError, tableDefinition ['name']

      parts = self._CMD.match (tSchema [0].sql)
      if not parts:
        raise gException, u_("Cannot split SQL command: '%s'") % tSchema [0].sql

      fields = [f.name for f in tSchema [0].fields ()]
      nfields = ["NULL" for f in tableDefinition ['fields']]
      nfields.extend (fields)

      body.append (u"CREATE TEMPORARY TABLE t1_backup (%s)" \
                   % string.join (fields, ","))
      body.append (u"INSERT INTO t1_backup SELECT %(fields)s FROM %(table)s" \
                   % {'fields': string.join (fields, ","),
                      'table' : tableDefinition ['name']})
      body.append (u"DROP TABLE %s" % tableDefinition ['name'])

      fCode = self.createFields (tableDefinition ['name'],
                                 tableDefinition ['fields'])
      self.mergeTuple (res, (fCode [0], [], fCode [2]))

      oldSQL = parts.groups ()
      newBody = [string.join (fCode [1], ", ")]
      if len (oldSQL [1]):
        newBody.append (oldSQL [1])

      cmd = u"%s (%s)%s" % (oldSQL [0], string.join (newBody, ", "), oldSQL [2])

      body.append (cmd)
      body.append (u"INSERT INTO %(table)s SELECT %(fields)s FROM t1_backup" \
                   % {'table' : tableDefinition ['name'],
                      'fields': string.join (nfields, ",")})
      body.append (u"DROP TABLE t1_backup")

      # add all 'unchanged' indices
      if tSchema [0].indices is not None:
        for (indexName, index) in tSchema [0].indices.items ():
          addIndices = tableDefinition.get ('indices')
          if addIndices is None or indexName in addIndices:
            continue

          index ['name'] = indexName
          self.mergeTuple (res, self.createIndex (table, index, True))


    # Create all requested indices
    if tableDefinition.has_key ('indices'):
      for ixDef in tableDefinition ['indices']:
        self.mergeTuple (res, self.createIndex (table, ixDef, True))

    if not codeOnly:
      self._executeCodeTuple (res)

    return res


  # ---------------------------------------------------------------------------
  # Handle special defaults
  # ---------------------------------------------------------------------------

  def _defaultwith (self, code, tableName, fieldDef, forAlter):
    """
    This function creates a sequence for 'serials' and sets the default for
    'timestamps' to 'now ()'

    @param code: code-tuple to merge the result in
    @param tableName: name of the table
    @param fieldDef: dictionary describing the field with the default
    @param forAlter: TRUE if the definition is used in a table modification
    """
    if fieldDef ['defaultwith'] == 'serial':
      print "WARNING: SQLite has no serials"

    elif fieldDef ['defaultwith'] == 'timestamp':
      print "WARNING: SQLite has no default timestamp"


  # ---------------------------------------------------------------------------
  # A key is an integer
  # ---------------------------------------------------------------------------

  def key (self, fieldDefinition):
    """
    Native datatype for a 'key'-field is 'integer'

    @param fieldDefinition: dictionary describing the field
    @return: string with the native datatype 'integer'
    """
    return "integer"


  # ---------------------------------------------------------------------------
  # Create an apropriate type for a number
  # ---------------------------------------------------------------------------

  def number (self, fieldDefinition):
    """
    This function returns an apropriate type for a number according to the
    given length and precision.

    @param fieldDefinition: dictionary describing the field
    @return: string with the native datatype
    """
    scale  = 0
    length = 0

    if fieldDefinition.has_key ('precision'):
      scale = fieldDefinition ['precision']
    if fieldDefinition.has_key ('length'):
      length = fieldDefinition ['length']

    if scale == 0:
      return "integer"
    else:
      return "numeric (%s,%s)" % (length, scale)


  # ---------------------------------------------------------------------------
  # Native datatype for boolean is boolean
  # ---------------------------------------------------------------------------

  def boolean (self, fieldDefinition):
    """
    This function returns the native data type for a boolean. Since SQLite is
    typeless by default we use an 'integer'. This is needed so a pysqlite
    returns 0 or 1 for boolean values.

    @param fieldDefinition: dictionary describing the field
    @return: 'integer'
    """
    return "integer"


