# GNU Enterprise Common Library - i18n support
#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# Copyright 2001-2004 Free Software Foundation
#
# $Id: i18n.py 5637 2004-04-05 11:17:10Z reinhard $
"""
Internationalization support
"""
from types import *

import exceptions
import gettext
import locale
import os.path
import string
import sys

from gnue import paths

# -----------------------------------------------------------------------------
# Global variables
# -----------------------------------------------------------------------------

__modules = {}                          # Modules by filename
__catalogs = {}                         # Message catalogs by domain

language = None
encoding = None
enc_policy = "replace"                  # policy to use if an unicode character

# -----------------------------------------------------------------------------
# Find a module from filename
# -----------------------------------------------------------------------------

def __find_module (filename):
  if __modules.has_key (filename):
    return __modules [filename]
  for mod in sys.modules.values ():
    if hasattr (mod, '__file__'):
      # mod.__file__ can be .pyc if the module is already compiled
      f = mod.__file__
      if f [-1:] == 'c':
        f = f [:-1]
      if os.path.normcase (f) == os.path.normcase (filename):
        __modules [filename] = mod
        return mod

# -----------------------------------------------------------------------------
# Find the correct translation catalog
# -----------------------------------------------------------------------------

def __find_catalog ():

  # find out the filename of the calling function
  caller_file = (sys._getframe (2)).f_code.co_filename

  # find out the module name
  caller_module = __find_module (caller_file)

  if caller_module is None:
    return None

  # make 'gnue-common' from 'gnue.common.foo.bar'
  x = string.replace (caller_module.__name__, '.', '-', 1)
  i = string.find (x, '.')
  domain = x [:i]

  if __catalogs.has_key (domain):
    return __catalogs [domain]
  else:
    try:
      if os.name == 'posix':
        catalog = gettext.translation (domain, paths.data + '/share/locale')
      else:
        catalog = gettext.translation (domain, paths.data + '/share/locale',
                                       [language])
    except:
      catalog = None

    __catalogs [domain] = catalog

    return catalog

# -----------------------------------------------------------------------------
# Translate a message and return unicode
# -----------------------------------------------------------------------------

def utranslate (message):
  """
  Translates a message and returns a unicode string.  This function is
  available as the builtin function "u_()".
  """
  catalog = __find_catalog ()

  if catalog is None:
    return message

  return catalog.ugettext (message)

# -----------------------------------------------------------------------------
# Translate a message and return local encoding
# -----------------------------------------------------------------------------

def translate (message):
  """
  Translates a message and returns an 8 bit string, encoded with @encoding (the
  current locale's encoding).  This function is available as the builtin
  function "_()".
  """
  catalog = __find_catalog ()

  if catalog is None:
    return message

  return (catalog.ugettext (message)).encode (encoding, enc_policy)

# -----------------------------------------------------------------------------
# Convert Unicode to String, let everything else untouched. This is o().
# -----------------------------------------------------------------------------

def outconv (message):
  """
  Encodes a message to @encoding (the current locale's encoding).  This
  function is available as the builtin function "o()".
  """
  if isinstance (message, UnicodeType):
    return message.encode (encoding, enc_policy)
  else:
    return message

# -----------------------------------------------------------------------------
# New basic exception class. Python's standard exception class cannot handle
# unicode messages.
# -----------------------------------------------------------------------------

class gException (Exception):
  """
  The same as the builtin python Exception, but can handle messages that are
  unicode strings.  This exception is available as the builtin class
  "gException".  All other user-defined exceptions should be derived from this
  class.
  """
  def __init__ (self, message):
    self.message = message
    exceptions.Exception.__init__ (self, o(message))

# -----------------------------------------------------------------------------
# Module initialization
# -----------------------------------------------------------------------------

# Initialize locale. This has been reported to fail on Mac.
try:
  locale.setlocale (locale.LC_ALL, '')
except:
  pass

# This will give us the user's language and encoding even on non-POSIX systems
(language, encoding) = locale.getdefaultlocale ()

# Make sure encoding is not None
if not encoding:
  encoding = 'ascii'

# Now define the new builtin stuff
import __builtin__  
__builtin__.__dict__['u_'] = utranslate
__builtin__.__dict__['_'] = translate
__builtin__.__dict__['o'] = outconv
__builtin__.__dict__['gException'] = gException
