#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# Copyright 2002-2004 Free Software Foundation
#
# $Id: interbase.py 5614 2004-04-02 10:53:54Z johannes $
#

from gnue.common.schema.scripter.processors.SQL import SQLProcessor

name        = "Interbase"
description = "Interbase (5.x/6.x)/Firebird (1.x)"


# =============================================================================
# GSD processor for Interbase
# =============================================================================
class Processor (SQLProcessor):

  COMMENT_BEGIN      = "/*"
  COMMENT_END        = "*/"
  COMMENT_SINGLELINE = False
  MAX_NAME_LENGTH    = 31


  # ---------------------------------------------------------------------------
  # Add a GSField instance to the table definition
  # ---------------------------------------------------------------------------
  def _processField (self, tableDef, gsField, isLast):
    field = "  %s" % self._qualify (gsField)
    phase = tableDef.getPhase (0)

    if gsField.defaultwith == "serial":
      gen = self._getSequenceName (tableDef.name, gsField)
      phase.header.append ("CREATE GENERATOR %s%s" % (gen, self.END_COMMAND))

      self.__addTrigger (tableDef, gsField, gen)

    elif gsField.defaultwith == "timestamp":
      field += " DEFAULT 'NOW'"

    elif hasattr (gsField, "default") and gsField.default is not None:
      field += " DEFAULT %s" % gsField.default

    if not gsField.nullable:
      field += " NOT NULL"

    if not isLast:
      field += ", "

    phase.body.append (field)


  # ---------------------------------------------------------------------------
  # Add a generator trigger to the table definition
  # ---------------------------------------------------------------------------
  def __addTrigger (self, tableDef, gsField, gen):
    epi = tableDef.getPhase (0).epilogue
    epi.append ("")
    epi.append ("SET TERM ^ ;")
    epi.append ("CREATE TRIGGER trg_%s FOR %s" % (gsField.name, tableDef.name))
    epi.append ("  ACTIVE BEFORE INSERT POSITION 0 AS")
    epi.append ("  BEGIN")
    epi.append ("    NEW.%s = GEN_ID (%s,1);" % (gsField.name, gen))
    epi.append ("  END ^")
    epi.append ("SET TERM ; ^")


  # ---------------------------------------------------------------------------
  # On start of schema dump, define boolean domain
  # ---------------------------------------------------------------------------

  def startSchema (self):
    """
    Defines the 'boolean' domain.
    """
    self._writeText ("\n")
    self._writeText (self.comment ("Define the domain for boolean values"))
    self._writeText ("CREATE DOMAIN boolean AS "
                     "smallint CHECK (value IN (0,1));\n")


  # ===========================================================================
  # Datatype translation
  # ===========================================================================

  # ---------------------------------------------------------------------------
  # Keys are allways 'integer'
  # ---------------------------------------------------------------------------

  def key (self, gsField):
    """
    A key is of type 'integer'.
    """
    return "integer"


  # ---------------------------------------------------------------------------
  # String becomes either 'varchar' or 'blob'
  # ---------------------------------------------------------------------------

  def string (self, gsField):
    """
    If the length of the string is missing or less than 2000 'varchar' is
    returned, otherwise 'blob'.
    """
    if hasattr (gsField, "length") and gsField.length <= 2000:
      return "varchar (%s)" % gsField.length

    else:
      return "blob"


  # ---------------------------------------------------------------------------
  # translate a number according to it's precision and length
  # ---------------------------------------------------------------------------

  def number (self, gsField):
    """
    A number is translated into a 'smallint', 'integer' or 'numeric' according
    to it's precision and length.
    """
    if gsField.precision == 0:
      if gsField.length <= 4:
        return "smallint"

      elif gsField.length <= 9:
        return "integer"

      else:
        return "numeric (%s,0)" % gsField.length

    else:
      return "numeric (%s,%s)" % (gsField.length + gsField.precision, 
                                  gsField.precision)


  # ---------------------------------------------------------------------------
  # boolean becomes a smallint
  # ---------------------------------------------------------------------------

  def boolean (self, gsField):
    """
    We use the 'boolean' domain that we create.
    """
    return "boolean"


  # ---------------------------------------------------------------------------
  # Datetime maps to date
  # ---------------------------------------------------------------------------

  def datetime (self, gsField):
    """
    Interbase has no datetime, but uses 'timestamp' instead.
    """
    return "timestamp"
