#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# Copyright 2002-2004 Free Software Foundation
#
# $Id: mysql.py 5665 2004-04-07 11:04:32Z reinhard $
#

from gnue.common.schema.scripter.processors.SQL import SQLProcessor

name        = "MySQL"
description = "MySQL (3.x/4.x)"


# =============================================================================
# GSD processor for PostgreSQL
# =============================================================================
class Processor (SQLProcessor):

  MAX_NAME_LENGTH = 64
  ESCAPECHAR      = '\\'


  # ---------------------------------------------------------------------------
  # Process a GSField instance
  # ---------------------------------------------------------------------------
  
  def _processField (self, tableDef, gsField, isLast):
    field = "  %s" % self._qualify (gsField)
    
    if gsField.defaultwith == "serial":
      field += " auto_increment"

    elif gsField.defaultwith == "timestamp":
      if gsField.type != "timestamp":
        gsField.type = "timestamp"
        field = "  %s" % self._qualify (gsField)

        print _("WARNING: changing column type of '%s.%s' to 'timestamp'") % \
                (tableDef.name, gsField.name)

    elif hasattr (gsField, "default") and gsField.default is not None:
      field += " DEFAULT %s" % gsField.default

    if not gsField.nullable:
      field += " NOT NULL"

    if not isLast:
      field += ","

    tableDef.getPhase (0).body.append (field)



  # ===========================================================================
  # Datatype translation
  # ===========================================================================

  # ---------------------------------------------------------------------------
  # Keys are unsinged integers
  # ---------------------------------------------------------------------------

  def key (self, gsField):
    """
    A key is an unsigned integer.
    """
    return "int unsigned"


  # ---------------------------------------------------------------------------
  # String
  # ---------------------------------------------------------------------------

  def string (self, gsField):
    """
    If the string has no length or exceeds 255 characters the datatype 'text'
    will be used, otherwise the result is 'varchar'.
    """
    if hasattr (gsField, "length") and gsField.length <= 255:
      return "varchar (%s)" % gsField.length
    else:
      return "text"


  # ---------------------------------------------------------------------------
  # Translate a number according to it's precision and length
  # ---------------------------------------------------------------------------

  def number (self, gsField):
    """
    Number translates to 'smallint', 'int', 'bigint' or 'decimal' according to
    it's precision and length.
    """
    if gsField.precision == 0:
      if gsField.length <= 4:
        return "smallint"

      elif gsField.length <= 9:
        return "int"

      elif gsField.length <= 18:
        return "bigint"

      else:
        return "decimal (%s,0)" % gsField.length
    else:
      return "decimal (%s,%s)" % (gsField.length, gsField.precision)


  # ---------------------------------------------------------------------------
  # MySQL has no native boolean data type
  # ---------------------------------------------------------------------------

  def boolean (self, gsField):
    """
    In MySQL a boolean maps to a tinyint (1)
    """
    return "tinyint (1) unsigned"


  # ---------------------------------------------------------------------------
  # MySQL has a timestamp, which is needed for 'defaultwith timestamp' columns
  # ---------------------------------------------------------------------------

  def timestamp (self, gsField):
    """
    MySQL is allowed to use timestamp for fields with a default-timestamp
    value. 
    """
    if gsField.defaultwith == 'timestamp':
      return "timestamp"
    else:
      return SQLProcessor.timestamp (self, gsField)
