#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# Copyright 2001-2004 Free Software Foundation
#
# $Id: oracle.py 5614 2004-04-02 10:53:54Z johannes $
#

from gnue.common.schema.scripter.processors.SQL import SQLProcessor

name        = "Oracle"
description = "Oracle (7.x/8i/9i+)"


# =============================================================================
# GSD processor for PostgreSQL
# =============================================================================
class Processor (SQLProcessor):

  COMMENT_BEGIN   = "REM ** "
  MAX_NAME_LENGTH = 31
  _PK_PRECISION   = 10
  ESCAPECHAR      = '\\'

  # ---------------------------------------------------------------------------
  # add a GSField instance to the table definition
  # ---------------------------------------------------------------------------
  def _processField (self, tableDef, gsField, isLast):
    field = "  %s" % self._qualify (gsField)
    phase = tableDef.getPhase (0)

    # Add a 'serial' as default value
    if gsField.defaultwith == "serial":
      seq = self._getSequenceName (tableDef.name, gsField)

      phase.prologue.append ("CREATE SEQUENCE %s MAXVALUE %s NOCYCLE%s" % \
                            (seq, "9" * self._PK_PRECISION, self.END_COMMAND))
      trig = []
      trig.append ("   IF :new.%s IS NULL THEN" % gsField.name)
      trig.append ("      SELECT %s.nextval INTO :new.%s FROM dual;" % \
                   (seq, gsField.name))
      trig.append ("   END IF;")

      self.__addTrigger (tableDef, gsField, trig)

    # add a timestamp as default value
    elif gsField.defaultwith == "timestamp":
      if gsField.type == "date":
        sysdate = "TRUNC (sysdate)"
      else:
        sysdate = "sysdate"

      trig = [] 
      trig.append ("   IF :new.%s IS NULL THEN" % gsField.name)
      trig.append ("      :new.%s := %s;" % (gsField.name, sysdate))
      trig.append ("   END IF;")

      self.__addTrigger (tableDef, gsField, trig)

    # add a given 'constant' as default value 
    elif hasattr (gsField, "default") and gsField.default is not None:
      field += " DEFAULT '%s'" % gsField.default

    if not gsField.nullable:
      field += " NOT NULL"

    if not isLast:
      field += ","

    phase.body.append (field)


  # ---------------------------------------------------------------------------
  # Add a trigger for defaults to the table definition
  # ---------------------------------------------------------------------------
  def __addTrigger (self, tableDef, gsField, body):
    epi = tableDef.getPhase (0).epilogue

    epi.append ("")
    epi.append ("CREATE OR REPLACE TRIGGER t__%s_%s_pre" % (tableDef.name,
               gsField.name))
    epi.append ("  BEFORE INSERT ON %s" % tableDef.name)
    epi.append ("  FOR EACH ROW WHEN (new.%s IS NULL)" % gsField.name)
    epi.append ("  BEGIN")
    epi.extend (body)
    epi.append ("  END;")
    epi.append ("/")


  # ===========================================================================
  # Datatype translation
  # ===========================================================================

  # ---------------------------------------------------------------------------
  # String becomes 'varchar2'
  # ---------------------------------------------------------------------------

  def string (self, gsField):
    """
    If the string has no length or exceeds 2000 characters, we use the
    datatype 'long', otherwise 'varchar2' will be used.
    """
    if hasattr (gsField, 'length') and gsField.length <= 2000:
      return "varchar2 (%s)" % gsField.length
    else:
      return "blob"


  # ---------------------------------------------------------------------------
  # time becomes 'date'
  # ---------------------------------------------------------------------------

  def time (self, gsField):
    """
    Time translates to 'date'
    """
    return "date"


  # ---------------------------------------------------------------------------
  # datetime becomes 'date'
  # ---------------------------------------------------------------------------

  def datetime (self, gsField):
    """
    Datetime translates to 'date'
    """
    return "date"


  # ---------------------------------------------------------------------------
  # A key is a number of _PK_PRECISION
  # ---------------------------------------------------------------------------

  def key (self, gsField):
    """
    A key is a number of a given precision (see _PK_PRECISION)
    """
    return "number (%s)" % self._PK_PRECISION


  # ---------------------------------------------------------------------------
  # Oracle doesn't seem to have booleans, so we're using number (1)
  # ---------------------------------------------------------------------------

  def boolean (self, gsField):
    """
    Oracle has no native boolean type. So a boolean translates to a number (1)
    with a CHECK constraint, which allows only 0, 1 or NULLs.
    """
    if gsField.nullable:
      return "number (1) CHECK (%s IS NULL OR %s IN (0,1))" % \
             (gsField.name, gsField.name)
    else:
      return "number (1) CHECK (%s IN (0,1))" % gsField.name

  # ---------------------------------------------------------------------------
  # Translate a number according to it's precision and length
  # ---------------------------------------------------------------------------

  def number (self, gsField):
    """
    A number will be kept as 'number' with the given precision.
    """
    if gsField.precision == 0:
      return "number (%s)" % gsField.length
    else:
      return "number (%s,%s)" % (gsField.length + gsField.precision, 
                                 gsField.precision)

