#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# Copyright 2002-2004 Free Software Foundation
#
# $Id: postgresql.py 5614 2004-04-02 10:53:54Z johannes $
#

from gnue.common.schema.scripter.processors.SQL import SQLProcessor
from string import join

name        = "PostgreSQL"
description = "PostgreSQL (6.x/7.x)"

# =============================================================================
# GSD processor for PostgreSQL
# =============================================================================

class Processor (SQLProcessor):


  MAX_NAME_LENGTH = 31
  ESCAPECHAR      = '\\'
  ALTER_MULTIPLE  = False


  # ---------------------------------------------------------------------------
  # Process a GSField instance
  # ---------------------------------------------------------------------------

  def _processField (self, tableDef, gsField, isLast):
    field = "  %s" % self._qualify (gsField)

    # build a default value for this field
    if gsField.defaultwith == "serial":
      seq = self._getSequenceName (tableDef.name, gsField)

      tableDef.getPhase (0).header.append ("CREATE SEQUENCE %s;" % seq)
      field += " DEFAULT nextval ('%s')" % seq

    elif gsField.defaultwith == "timestamp":
      field += " DEFAULT now()"

    elif hasattr (gsField, "default") and len (gsField.default):
      field += " DEFAULT %s" % gsField.default

    if not gsField.nullable:
      field += " NOT NULL"

    if not isLast:
      field += ", "

    tableDef.getPhase (0).body.append (field)


  # ===========================================================================
  # Datatype translation
  # ===========================================================================

  # ---------------------------------------------------------------------------
  # A key field is of type 'int8'
  # ---------------------------------------------------------------------------

  def key (self, gsField):
    """
    In PostgreSQL we use 'int8' as key type
    """
    return "int8"


  # ---------------------------------------------------------------------------
  # A number needs special treatment
  # ---------------------------------------------------------------------------
  def number (self, gsField):
    """
    Number translates to smallint, integer, bigint or numeric according to it's
    precision and length.
    """
    if gsField.precision == 0:
      if gsField.length <= 4:
        return "smallint"

      elif gsField.length <= 9:
        return "integer"

      elif gsField.length <= 18:
        return "bigint"

      else:
        return "numeric (%s,0)" % gsField.length
    else:
      return "numeric (%s,%s)" % (gsField.length + gsField.precision,
                                  gsField.precision)

  # ---------------------------------------------------------------------------
  # Keep boolean as 'boolean'
  # ---------------------------------------------------------------------------

  def boolean (self, gsField):
    """
    PostgreSQL has a native type boolean.
    """
    return "boolean"


  # ---------------------------------------------------------------------------
  # Change 'datetime' to 'timestamp'
  # ---------------------------------------------------------------------------

  def datetime (self, gsField):
    """
    PostgreSQL uses a 'timestamp' for date with time values
    """
    return "timestamp without time zone"


  # ---------------------------------------------------------------------------
  # Set the client encoding as requested by the scripter
  # ---------------------------------------------------------------------------

  def client_encoding (self, encoding):
    """
    PostgreSQL sets the client encoding with a '\encoding' command.
    """
    SQLProcessor.client_encoding (self, encoding)
    self._writeText ("\\encoding %s\n" % self.encoding)


  # ===========================================================================
  # Data transformation services
  # ===========================================================================

  # ---------------------------------------------------------------------------
  # booleans are represented by "FALSE" and "TRUE"
  # ---------------------------------------------------------------------------

  def dts_boolean (self, gsValue):
    """
    Use 1 for TRUE and 0 for FALSE
    """
    if gsValue.value:
      return "TRUE"
    else:
      return "FALSE"
