#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# Copyright 2001-2005 Free Software Foundation
#
# FILE:
# masks.py
#
# DESCRIPTION:
"""
Parse mask definitions into tokens that the Mask class can use.

Tokens are ......
"""
# NOTES:
#
from gnue.common.external.plex import *
import string
from Errors import *

digit = Any(string.digits)

class BaseToken:
  """
  Basic parser class. Not used directly,
  but inherited by the other defined tokens
  Literal, Token, etc.
  """
  numeric=False
  date=False
  text=False
  literal=False
  token=False

  def __init__(self, t1, t2=None, *args):
    if t2:
      self.token = t2
    else:
      self.token = t1

# -----------------------------------------------------------------------------
# Standard token classes
# -----------------------------------------------------------------------------
class Token(BaseToken):
  """
  Class typically used to create normal tokens as 
  opposed to special tokens like literal.
  
  It sets the standard options so that each individual 
  token class doesn't need to.
  """
  force_lower = False
  force_upper = False
  token=True

class NumberToken(Token):
  """
  Numeric token (#9-,.)
  """
  numeric = True

class DateToken(Token):
  """
  Date token (MDYyHIS:/)
  """
  date=True

class TextToken(Token):
  """
  Text token
  """
  text = True

class TokenSet(Token):
  """
  Token defined by user with [] notation.
  Can behave like a  NumberToken or TextToken,
  depending on contents of [].
  """
  def __init__(self, token, *args):
    # TODO: Expand the set
    # Are we all-numeric?
    self.numeric = True
    self.token = token
    for t in token:
      if not t in string.digits:
        self.numeric = False
    if not self.numeric:
      self.text = True

# -----------------------------------------------------------------------------
# Special token classes
# -----------------------------------------------------------------------------
class Literal(BaseToken):
  """
  A literal string that the developer wants in the string.
  Note that for our purposes, the basic "literals" aren't
  really Literals(), but special cases of Token classes.
  So all literals represented by this class are denoted
  with \ or "" syntaxes.
  """
  literal=True


class RightToLeft(BaseToken):
  """
  Temporary token class used to note the
  position of ! modifiers
  """
  numeric = True

class CaseModifier:
  """
  Temporary token class used to record > and <
  markers. These cause the modified token to have
  either force_upper or force_lower set, so the
  other classes won't ever see CaseModifier
  instances.
  """
  pass

class Repeater:
  """
  Temporary token class used to record {#}
  markers. These are replaced with the actual
  represented tokens before being passed out
  of MaskParser (i.e., 0{3} would be returned
  as 000, so the other classes won't ever see
  Repeater instances.
  """
  def __init__(self, count):
    self.count = count

##
##
##
# =============================================================================
# Input mask parser
# =============================================================================
class InputMaskParser(Scanner):
  """
  Takes a file handle containing an input mask and creates a 
  list of Tokens which define the input mask.
  """
  def getType(self):
    """
    Returns the apparent type of this mask.
    
    @rtype: string
    @return: The value 'text', 'numeric', or 'date'
    """
    return type

  def getTokens(self):
    """
    Returns a list of the tokens after parsing the input mask.
    
    @rtype: list
    @return: The list of tokens
    """
    return self.tokens[:]

  # ===========================================================================
  # Private stuff
  # ===========================================================================
  
  # ---------------------------------------------------------------------------
  # Lexicon action functions
  # ---------------------------------------------------------------------------
  def _check_single(self, text):
    """
    Function to add single instance tokens to the input mask.
    
    A single instance token is something that can appear only once
    in an input mask.
    """
    if text in self.__singles:
      raise Errors.UnrecognizedInput(self, 'Mask can only have one "%s" token' %text)
    self.__singles.append(text)
    if text == '!':
      self.produce (RightToLeft(text))
    elif text in '.+,':
      self.produce(NumberToken(text))
    else:
      self.produce(TextToken(text))

  def _literal(self, text):
    """
    A text literal that should appear as is in the mask
    """
    print "literal %s" % text
    self.produce(Literal(text))

  def _literal_2nd(self, text):
    """
    Closes the literal string
    """
    print "literal 2nd %s" % text    
    return self.literal(text[1:])

  def _escape(self, text):
    """
    An escaped character such as \$ to display a $ 
    """
    print "escape %s" % text
    self.begin('')
    self.produce(Literal(text))

  def _repeater(self, text):  
    self.produce(Repeater(int(text)))

  def _begin_set(self, text):
    self.begin('set')
    self._set = ""

  def _add_set(self, text):
    self._set += text

  def _add_set_2nd(self, text):
    return self.add_set(text[1:])

  def _end_set(self, text):
    self.begin('')
    self.produce(TokenSet(self._set))

  # Basic lexicon used by both input and output masks
  _lexicon = [
      State('escape',  [
          (AnyChar,        _escape),
        ]),

      State('quoted',  [
          (Str("\\")+Str("'"),  _literal_2nd),
          (Str("'"),       Begin('')),
          (AnyChar,        _literal)
        ]),

      State('quoted2',  [
          (Str("\\")+Str('"'),  _literal_2nd),
          (Str('"'),       Begin('')),
          (AnyChar,        _literal)
        ]),

      State('repeater',  [
          (Str('}'),       Begin('')),
          (Rep1(digit),    _repeater)
        ]),

      State('set',  [
          (Str("\\")+Any('[]'),  _add_set_2nd),
          (Str(']'),       _end_set),
          (AnyChar,        _add_set)
        ]),

      (Str('\\'),          Begin('escape')),
      (Str("'"),           Begin('quoted')),
      (Str('"'),           Begin('quoted2')),
      (Str('{'),           Begin('repeater')),
      (Str('['),           _begin_set),
      (Str(' '),           Literal),
      (Any('+.,'),        _check_single),
      (Any('_?AaLlCc'),    TextToken),
      (Any('MDYyHISPp:/'), DateToken),
      (Any('#0'),         NumberToken),
      (Any('<>'),          CaseModifier)
  ]

  # Lexicon used by input masks
  _extra_lexicon = [
        (Any('!'),        _check_single),
  ]

  def __process(self, token):
    """
    Adds the standard tokens to the list of tokens
    generated for this input mask.  Deals with special
    tokens.
    """
    if isinstance(token,Repeater):
      for i in range(0, token.count-1):
        self.__process(self.__last)
    elif isinstance(token, CaseModifier):
      self.__modify.append(token)
    else:
      if self.__modify and isinstance(token, TextToken):
        mod = self.__modify.pop(0)
        if mod.token == '<':
          token.force_upper = True
        elif mod.token == '>':
          token.force_lower = True
      self.tokens.append(token)

    self.__last = token

  def __init__(self, file, name, numeric=False, date=False):
    """
    @type file: input stream
    @param file: The text to be used as the mask
    @type name: string
    @param name: The name of the input mask(TODO: ?)
    @type numeric: boolean
    @param numeric: Is this a numeric input mask
    @type date: boolean
    @param date: Is this a numeric input mask
    """
    self.__singles = []
    self.tokens = []
    self.__last = None
    self.__modify = []
    # -------------------------------------------------------------------------
    # Read the input mask and convert into instances of Token classes
    # -------------------------------------------------------------------------
    try:
      Scanner.__init__(self, Lexicon(self._lexicon + self._extra_lexicon), file, name)

      while True:
        token, extra = self.read()
        if token is None:
          break
        self.__process(token)
        
    except Errors.PlexError, msg:
      raise MaskDefinitionError, msg

    if self.__modify:
      print "WARNING: Modifier found at end of mask."

    # -------------------------------------------------------------------------
    # Set appropriate flags
    # -------------------------------------------------------------------------

    # If any two of these are non-zero, then the
    # mask is a text mask, not date or numeric.
    num_markers = 0
    date_markers = 0
    text_markers = 0
    rtl_pos = -1

    i = 0
    for token in self.tokens:
      if isinstance(token,RightToLeft):
        rtl_pos = i
      if not isinstance(token, Literal):
        if token.numeric:
          num_markers += 1
        elif token.date:
          date_markers += 1
        else:
          text_markers += 1
      i += 1

    if not (num_markers or date_markers or text_markers):
      raise MaskDefinitionError, 'Mask has no character tokens'

    if numeric and (date_markers or text_markers):
      raise MaskDefinitionError, 'Numeric mask has non-numeric tokens'

    if date and (num_markers or text_markers):
      raise MaskDefinitionError, 'Date/Time mask has non-date tokens'

    # Check for "!" in non-numeric mask
    if rtl_pos >= 0:
      self.tokens.pop(rtl_pos)
    else:
      rtl_pos = 0

    self.rtl_pos = rtl_pos
    
    # Set the type of parser
    if (num_markers and date_markers) or text_markers:
      self.type = 'text'
    elif num_markers:
      self.type = 'numeric'
    else:
      self.type = 'date'
