#
# Copyright 2001-2004 Free Software Foundation
#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# FILE:
# MRUManager.py
#
# DESCRIPTION:
# Handles the MRU/Open Recent for Designer
#
# NOTES:
#

import os
from gnue.common.apps import RuntimeSettings
from wxPython import wx

class MRUManager: 
  def __init__(self, app): 
    self._menus = []
    self._locations = []
    self.menulist = []
    self._app = app
    self.mruMenuMap = {}

    # TODO: Should this be user configurable?
    self._maxMRUs = 9

    # Set ourselves up to be saved in the runtime settings file
    # Each instance will register for us
    self.runtime_section = "MRUList"

    # Load the saved MRU list
    for i in range(self._maxMRUs):
      location = RuntimeSettings.get(self.runtime_section, "%s" % i, None)
      if location is None:
        break
      self._locations.append (location)

    self._refreshMenuList()


  def addLocation(self, location): 
    absloc = os.path.normcase(os.path.abspath(location))
    try:
      self._locations.remove(absloc)
    except ValueError:
      pass
    ## for i in range(len(self._locations)): 
##       if self._locations[i] == absloc: 
##         self._locations.pop(i)
##         break

    self._locations.insert(0, absloc)

    if len(self._locations) >= self._maxMRUs: 
      self._locations = self._locations[:self._maxMRUs]
 
    self._refreshMenuList()


  def _refreshMenuList(self):
    # This figures out the common prefix of all the forms and reduces appropriately.
    # i.e., if you have /usr/home/me/form1.gfd, /usr/home/you/form2.gfd then
    # this will reduce to me/form1.gfd and you/form2.gfd.. This is to keep the
    # MRU menu list from being too wide!

    commonpos=len(os.path.commonprefix(
       map(lambda x: os.path.join(os.path.split(x)[0],''),self._locations)))

    if commonpos > 2:
      self.menulist = map(lambda x, pos=commonpos: x[pos:], self._locations)
    else:
      self.menulist = self._locations[:]

    self.refreshMenus()


  def addMenu(self, menu, instance):
    self._menus.append(menu)
    menu.__instance = instance
    self.refreshMenu(menu)

  def removeMenu(self, menu, instance):
    self._menus.remove(menu)


  def refreshMenus(self):
    map(self.refreshMenu, self._menus)


  def refreshMenu(self, menu):

    # Rid ourselves of old entries
    for i in menu.GetMenuItems():
      menu.Delete(i.GetId())

    i = 0
    for location in self.menulist:
      tid = wx.wxNewId()
      menu.Append ( tid, '&%s %s' % (i+1, location),
                    _('Open "%s" in a new window') % self._locations[i] )

      wx.EVT_MENU(menu.__instance, tid, self._app.OnOpenRecent)
      self.mruMenuMap[tid] = self._locations[i]
      i += 1


  #
  #  Used by RuntimeSettings
  #
  def saveRuntimeSettings(self):
    results = {}

    for i in range(len(self._locations)):
      results["%s" % i] = self._locations[i]

    return ( self.runtime_section, results )



