#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# Copyright 2001-2004 Free Software Foundation
#
# FILE:
# TemplateParser.py
#
# DESCRIPTION:
# Parses & runs a template
#
# NOTES:

import string
from wxPython.wx import *
from gnue.designer.base import TemplateBase
from gnue.common.utils.FileUtils import dyn_import
from gnue.common.utils import TextUtils
from gnue.common.events import Event

class TemplateParser:
  def __init__(self, instance, rootObject, parent,
               templateInformation, currentObject=None):
    self.parent = parent
    self.instance = instance
    self.templateInformation = templateInformation
    self.rootObject = rootObject
    self.currentObject = currentObject

    self.elements = instance.incubator.elements


  def run(self, **params):

    self.template = self.templateInformation['BaseClass'](self)

    completed = 0

    try:
      # Clear out any old variables
      self.template.variables = {}
      self.template.Start(self.rootObject,
                          self.instance.buildWizardCurrentDict(),
                          **params)

      # If this is simply a template and not a wizard,
      # generate the results and get out of Dodge.
      if self.templateInformation['Behavior'] == TemplateBase.TEMPLATE:
        return self.template.Finalize()

      self.wizard = wxDialog(self.parent, -1, self.templateInformation['Name'],
         style=wxDEFAULT_DIALOG_STYLE|wxDIALOG_MODAL)

      self.panel = wxPanel(self.wizard, -1, wxPoint(0,0), wxSize(400,400))
      self.wizard.SetClientSize(wxSize(400,400))

      self.wizardPage = WizardPage(self, self.panel)

      self.prevButton = wxButton(self.panel, -1, _('< Back'))
      self.nextButton = wxButton(self.panel, -1, _('Continue >'))
      self.cancelButton = wxButton(self.panel, -1, _('Cancel'))

      self.nextButton.SetDefault()

      self.prevButton.SetSize(self.nextButton.GetSize())
      self.cancelButton.SetSize(self.nextButton.GetSize())

      y = self.panel.GetClientSize().y - 9 - self.nextButton.GetSize().y
      dx = self.nextButton.GetSize().x + 6
      x = self.panel.GetClientSize().x - 9 - self.nextButton.GetSize().x

      self.cancelButton.SetPosition( (x, y) )
      self.nextButton.SetPosition( (x - dx, y) )
      self.prevButton.SetPosition( (x - dx*2, y) )

      EVT_BUTTON(self.wizard,self.prevButton.GetId(), self.wizardPage.OnPrevStep)
      EVT_BUTTON(self.wizard,self.nextButton.GetId(), self.wizardPage.OnNextStep)
      EVT_BUTTON(self.wizard,self.cancelButton.GetId(), self.wizardPage.OnCancel)

      EVT_CLOSE(self.wizard, self.wizardPage.OnCancel)

      self.title = wxStaticText(self.panel, -1, _("Wizard Header"), pos=wxPoint(10,10))
      font = self.title.GetFont()
      font.SetPointSize(int(self.title.GetFont().GetPointSize()*1.5))
      font.SetStyle(wxITALIC)
#      font.SetWeight(wxBOLD)
      self.title.SetForegroundColour(wxColour(255,255,255))
      self.title.SetFont(font)
      self.title2 = wxStaticText(self.panel, -1, _("Wizard Header"), pos=wxPoint(11,11))
      self.title2.SetForegroundColour(wxColour(0,0,102))
      self.title2.SetFont(font)

      self.wizard.Fit()

      self.wizardPage.SetPosition((20, 20 + self.title.GetSize().y))
      w,h = self.wizard.GetClientSizeTuple()
      w = w - 50
      h = h - 56 - self.title.GetSize().y - self.nextButton.GetSize().y
      self.wizardPage.SetSize((w,h))

      r = self.panel.GetBackgroundColour().Red()
      g = self.panel.GetBackgroundColour().Green()
      b = self.panel.GetBackgroundColour().Blue()

      self.wizardPage.SetBackgroundColour(
          wxColour(
            (r <= 223 and r or 223) + 32,
            (g <= 223 and g or 223) + 32,
            (b <= 223 and b or 223) + 32) )

      self.wizardPage.setStep(self.template.FIRST_STEP)
      completed = self.wizard.ShowModal()
      if completed:
        completed = self.template.Finalize()

    except TemplateBase.InsufficientInformation, msg:
      # Give feedback if the wizard raised an exception
      if str(msg):
        msg = "\n\n" + TextUtils.lineWrap(str(msg),40)
      else:
        msg = ""

      dlg = wxMessageDialog(NULL,
              _("Unable to perform the requested action.") + msg,
              _("Wizard Error"), style=wxOK|wxICON_WARNING)
      dlg.ShowModal()
      dlg.Destroy()

    # Cleanup...
    try:      self.wizardPage.Destroy()
    except:   pass
    try:      self.wizard.Destroy()
    except:   pass

    return completed

  def cancel(self):
    self.wizard.EndModal(0)


class WizardPage(wxPanel):
  def __init__(self, parser, parent):
    wxPanel.__init__(self, parent, -1, pos=wxPoint(0,0), size=wxSize(400,250), style=wxSIMPLE_BORDER)
    self.parser = parser
    self.parent = parent
    self.panel = None

  def GetPrev(self, parent):
    if self.prevStep != None:
      return self
    else:
      return None

  def GetNext(self, parent):
    if self.nextStep != None:
      self.setStep(self.stepInfo['next'])
      return self
    else:
      return None

  def setStep(self, step):
    self.step = step
    self.stepInfo = self.parser.template.GetStep(step)
    self.nextStep = self.stepInfo['next']
    self.prevStep = self.stepInfo['prev']
    self.buildPage()

  # User clicked the "Next >" (or "Finish") button
  def OnNextStep(self, event):

    validation = []

    # Save the variables from current step
    for o in self.editorMappings.keys():
      value = o.get()
      if o.source.required and (value == None or not o.source.set and
               o.source.typecast == TemplateBase.text and not len(value)):
        validation.append('A required value is missing for "%s"' % (
            hasattr(o.source,'label') and o.source.label or o.source.variable))

        ## TODO: Add the various other validation checks

      self.parser.template.variables[self.editorMappings[o].variable] = value


    # Ask the template instance if it validates or not
    if not validation:
      validation = self.parser.template.ValidateStep(self.step)

    # Whoop! The user forgot something... inform them
    if validation:
      # Give feedback
      dlg = wxMessageDialog(self.parser.wizard,
              _("Please correct the following mistakes before continuing:\n\n - ") + \
              string.join(validation,'\n - '),
              _("Wizard Error"), style=wxOK|wxICON_WARNING)
      dlg.ShowModal()
      dlg.Destroy()

      return

    # Life is good and we have our info.... High-tail it out of here
    else:
      if self.nextStep == None:
        self.parser.wizard.EndModal(1)
      else:
        self.setStep(self.nextStep)
        event.Skip()

  # User clicked the "< Prev" button
  def OnPrevStep(self, event):
    # Save the variables from current step
    for o in self.editorMappings.keys():
      self.parser.template.variables[self.editorMappings[o].variable] = o.get()

    self.setStep(self.prevStep)

  # User clicked the "Cancel" button
  def OnCancel(self, event):
    self.parser.wizard.EndModal(0)

  # Build all the widgets and position them
  def buildPage(self):
    for child in self.GetChildren():
      child.Destroy()

    xMargin = 6
    xMarginInput = xMargin * 3
    yMargin = 6

    xSpacing = 10
    ySpacing = 6
    nextY = yMargin

    if self.nextStep == None:
      self.parser.nextButton.SetLabel(_('Finished'))
    else:
      self.parser.nextButton.SetLabel(_('Continue >'))

    if self.prevStep == None:
      self.parser.prevButton.Enable(0)
    else:
      self.parser.prevButton.Enable(1)

    self.parser.title.SetLabel(self.stepInfo['title'])
    self.parser.title2.SetLabel(self.stepInfo['title'])

    xMargin = xMargin
    nextY = yMargin

    self.editorMappings = {}
    self.textctrlList = []

    for object in self.stepInfo['content']:

      if isinstance(object, TemplateBase.WizardText):

        width = self.GetSize().x - xMargin * 2

        o = WrappedStaticText(self, -1, object.text, width,
                                pos=wxPoint(xMargin, nextY))

        nextY = nextY + ySpacing + o.GetSize().y

      elif isinstance(object, TemplateBase.WizardInput):

        x = xMarginInput
        if object.label != None:
          o = wxStaticText(self, -1, object.label, pos=wxPoint(x, nextY+3))
          x = x + xSpacing + o.GetSize().x

        size = (self.GetClientSize().x - x - xMarginInput,-1)

        if object.set != None and len(object.set):
          if object.maxSelections != 1 and object.orderable:
            x = xMarginInput
            nextY = nextY + ySpacing + o.GetSize().y
            size = (self.GetClientSize().x - x - xMarginInput,-1)
            o = SortableListField(object, self, pos=wxPoint(x, nextY), size=size)
          elif  object.lines > 1 or object.maxSelections != 1:
            o = ListField(object, self, pos=wxPoint(x, nextY), size=size)
          else:
            o = ComboField(object, self, pos=wxPoint(x, nextY), size=size)
        else:
          o = TextField(object, self, pos=wxPoint(x, nextY), size=size)


        self.editorMappings[o] = object
        MyId = len(self.textctrlList)
        self.textctrlList.append(o)

        nextY = nextY + ySpacing + o.GetSize().y

        EVT_CHAR(o, FieldHandler(self, MyId).fieldEventTrap)

        if self.parser.template.variables.has_key(object.variable):
          o.set(self.parser.template.variables[object.variable])

    # Add our buttons to the focus list
    if self.prevStep:
      self.textctrlList.append(self.parser.prevButton)
    self.textctrlList.append(self.parser.nextButton)
    self.textctrlList.append(self.parser.cancelButton)

    # Set focus to be the first input widget
    self.textctrlList[0].SetFocus()




class TextField(wxTextCtrl):
  def __init__(self, source, parent,
               pos=wxDefaultPosition, size=wxDefaultSize):
    wxTextCtrl.__init__(self, parent, -1, pos=pos, size=size)
    self.source = source
    self.parent = parent

    self.forceupper = self.source.forceupper
    self.forcelower = self.source.forcelower

    if self.forceupper:
      EVT_CHAR(wxTextCtrl, self.OnCharUpper)
    if self.forcelower:
      EVT_CHAR(wxTextCtrl, self.OnCharLower)


  def get(self):
    return self.GetValue()

  def set(self, value):
    self.SetValue(value)

  # TODO: This EVT_CHAR should convert the text to uppercase
  def OnCharUpper(self, event):
    event.Skip()

  # TODO: This EVT_CHAR should convert the text to lowercase
  def OnCharLower(self, event):
    event.Skip()



class ComboField(wxComboBox):
  def __init__(self, source, parent, pos=wxDefaultPosition, size=wxDefaultSize):
    wxComboBox.__init__(self, parent, -1, pos=pos, size=size)
    self.source = source
    self.parent = parent
    self.mapping = {}
    self.lookup = []

    i = 0
    for choice in source.set:
      key, descr = choice
      self.mapping[key] = i
      self.lookup.append(key)
      self.Append(descr)
      i = i + 1

  def get(self):
    return self.lookup[self.GetSelection()]

  def set(self, value):
    self.SetSelection(self.mapping[value])



class ListField(wxListBox):
  def __init__(self, source, parent, pos=wxDefaultPosition, size=wxDefaultSize):
    if source.maxSelections != 1:
      style = wxLB_MULTIPLE
    else:
      style = wxLB_SINGLE

    wxListBox.__init__(self, parent, -1, pos=pos, size=size, \
            style=style|wxLB_HSCROLL|wxLB_NEEDED_SB)

    self.source = source
    self.parent = parent
    self.mapping = {}
    self.lookup = []

    #print source.set

    i = 0

    # TODO: This [:2000] is a temporary fix since GTK limits to 2000 entries
    for choice in source.set[:500]:
      key, descr = choice
      self.mapping[key] = i
      self.lookup.append(key)
      self.Append(descr)
      i = i + 1

  def get(self):
    if self.source.maxSelections == 1:
      return self.lookup[self.GetSelection()]
    else:
      rv = []
      for i in self.GetSelections():
        rv.append(self.lookup[i])

  def set(self, value):
    if type(value) in ( type([]), type(()) ) :
      set = []
      for v in value:
        set.append(self.mapping[v])
      self.SetSelections(set)
    else:
      try: 
        self.SetSelection(self.mapping[value])
      except KeyError: 
        pass


class SortableListField(wxPanel):
  def __init__(self, source, parent, pos=wxDefaultPosition,
              size=wxDefaultSize):

    wxPanel.__init__(self, parent, -1, pos=pos, size=size,
              style=wxSIMPLE_BORDER)

    self.source = source
    self.parent = parent

    self.addButton = wxButton(self, -1, _("  Add >"))
    self.delButton = wxButton(self, -1, _("< Remove  "))
    self.upButton = wxButton(self, -1, _("Move Up"))
    self.downButton = wxButton(self, -1, _("Move Down"))
    self.selectAllButton = wxButton(self, -1, _("Select All"))

    self.addButton.SetSize(self.downButton.GetSize())
    self.delButton.SetSize(self.downButton.GetSize())
    self.upButton.SetSize(self.downButton.GetSize())
    self.selectAllButton.SetSize(self.downButton.GetSize())

    self.addButton.Enable(0)
    self.delButton.Enable(0)
    self.upButton.Enable(0)
    self.downButton.Enable(0)
    self.selectAllButton.Enable(1)

    self.list1 =     wxListBox(self, -1, wxPoint(6,6), \
            style=wxLB_MULTIPLE|wxLB_HSCROLL|wxLB_NEEDED_SB)

    self.list2 =     wxListBox(self, -1, \
            style=wxLB_MULTIPLE|wxLB_HSCROLL|wxLB_NEEDED_SB)

    self.included = []
    self.excluded = []

    self.includedMap = {}
    self.excludedMap = {}

    bw, bh = self.delButton.GetSizeTuple()
    lw, lh = self.list1.GetSizeTuple()

    self.addButton.SetPosition(wxPoint(lw + 12, 12))
    self.delButton.SetPosition(wxPoint(lw + 12, 16 + bh))
    self.upButton.SetPosition(wxPoint(lw + 12, 26 + bh*2))
    self.downButton.SetPosition(wxPoint(lw + 12, 30 + bh*3))
    self.selectAllButton.SetPosition(wxPoint(lw + 12, 34 + bh*4))

    self.list1.SetSize(wxSize(self.list1.GetSizeTuple()[0],
                              50 + bh*4))
    self.list2.SetSize(self.list1.GetSize())

    self.list2.SetPosition(wxPoint(lw+bw+18,6))
    lw, lh = self.list1.GetSizeTuple()

    self.SetSize(wxSize(lw*2+bw+25, lh + 13))

    EVT_BUTTON(self, self.addButton.GetId(), self.OnAdd)
    EVT_BUTTON(self, self.delButton.GetId(), self.OnRemove)
    EVT_BUTTON(self, self.upButton.GetId(), self.OnMoveUp)
    EVT_BUTTON(self, self.downButton.GetId(), self.OnMoveDown)
    EVT_BUTTON(self, self.selectAllButton.GetId(), self.OnSelectAll)

    EVT_LISTBOX(self, self.list1.GetId(), self.OnSelectLeft)
    EVT_LISTBOX(self, self.list2.GetId(), self.OnSelectRight)

    for choice in source.set:
      key, descr = choice
      self.excluded.append(key)
      self.list1.Append(descr)

  def get(self):
    return self.included[:]

  def set(self, value):
    map = {}
    self.included = []
    self.excluded = []
    self.list1.Clear()
    self.list2.Clear()
    for choice in self.source.set:
      key, descr = choice
      if key in value:
        map[key] = descr
      else:
        self.excluded.append(key)
        self.list1.Append(descr)

    for key in value:
       self.included.append(key)
       self.list2.Append(map[key])

  def OnSelectLeft(self, event):
    if len(self.list1.GetSelections()):
      self.addButton.Enable(1)
    else:
      self.addButton.Enable(0)
    self.delButton.Enable(0)
    self.upButton.Enable(0)
    self.downButton.Enable(0)
    for i in self.list2.GetSelections():
      self.list2.SetSelection(i,0)

  def OnSelectRight(self, event):
    sel = self.list2.GetSelections()
    if len(sel):
      self.delButton.Enable(1)

      if 0 in sel: 
        self.upButton.Enable(0)
      else: 
        self.upButton.Enable(1)

      if self.list2.Number()-1 in sel:
        self.downButton.Enable(0)
      else: 
        self.downButton.Enable(1)
    else:
      self.delButton.Enable(0)
      self.upButton.Enable(0)
      self.downButton.Enable(0)
    

    self.addButton.Enable(0)
    for i in self.list1.GetSelections():
      self.list1.SetSelection(i,0)

  def OnSelectAll(self, event):
    
    count = self.list1.Number()
    for count in range(count):
      self.list1.SetSelection(count)
    self.addButton.Enable(1)

  def OnAdd(self, event):
    sel = list(self.list1.GetSelections())
    sel.sort(lambda a,b : cmp (b,a))
    appendages1 = []
    appendages2 = []
    for i in sel:
      appendages1.insert(0,self.list1.GetString(i))
      appendages2.insert(0,self.excluded[i])
      self.excluded.pop(i)
      self.list1.Delete(i)

    for a in appendages1:
      self.list2.Append(a)

    for a in appendages2:
      self.included.append(a)

    self.OnSelectRight(None)


  def OnRemove(self, event): 
    sel = list(self.list2.GetSelections())
    sel.sort(lambda a,b : cmp (b,a))
    appendages1 = []
    appendages2 = []
    for i in sel: 
      appendages1.insert(0,self.list2.GetString(i))
      appendages2.insert(0,self.included[i])
      self.included.pop(i)
      self.list2.Delete(i)

    for a in appendages1:
      self.list1.Append(a)

    for a in appendages2:
      self.excluded.append(a)

    self.OnSelectLeft(None)


  def OnMoveUp(self, event):
    sel = self.list2.GetSelections()
    if 0 in sel:
      return

    for i in sel:
      k = i - 1
      prevKey = self.included[k]
      prevText = self.list2.GetString(k)
      self.included.pop(k)
      self.included.insert(i,prevKey)
      self.list2.Delete(k)
      self.list2.InsertItems([prevText],i)

  def OnMoveDown(self, event):
    sel = list(self.list2.GetSelections())
    if self.list2.Number()-1 in sel:
      return

    sel.sort(lambda a,b: cmp(b,a))

    for i in sel:
      k = i + 1
      prevKey = self.included[k]
      prevText = self.list2.GetString(k)
      self.included.pop(k)
      self.included.insert(i,prevKey)
      self.list2.Delete(k)
      self.list2.InsertItems([prevText],i)



class WizardRunner:
  def __init__(self, template, instance):
     self.template = template
     self.instance = instance
     instance.registerEventListeners(
         { 'Wizard:%s' % template['BaseID']: self.run,
           'Cancel:Wizard:%s' % template['BaseID']: self.cancel } )

  def run(self, event, *args, **parms):
    self.instance.dispatchEvent('BeginWizard', template=self.template)
    self.instance.dispatchEvent('BeginUndoGroup')
    self.parser = TemplateParser(self.instance, self.instance.rootObject,
          self.instance, self.template)
    self.parser.run(*args, **parms)
    self.instance.dispatchEvent('EndUndoGroup')
    self.instance.dispatchEvent('EndWizard', template=self.template)

  def cancel(self, event=None):
    self.parser.cancel()

  def finalize(self):
    pass

class WrappedStaticText(wxStaticText):
  def __init__(self, parent, id, label, width, *args, **params):
    wxStaticText.__init__(self, parent, id, "bah!", *args, **params)


    textSoFar = ""
    thisLine = ""
    for part in string.split(label,'\n'):
      for word in string.split(part):
        self.SetLabel(thisLine + word)
        if self.GetSize().width > width:
          textSoFar += thisLine + " \n"
          thisLine = word + " "
        else:
          thisLine += word + " "

      textSoFar += thisLine + " \n"
      thisLine = ""

    if len(textSoFar):
      self.SetLabel(string.replace(textSoFar,' \n','\n')[:-1])
    else:
      self.SetLabel("")


#
# FieldHandler
#
# enables the user to press tab/return and have it jump to the next box
#
class FieldHandler:
  def __init__(self, app, seq):
    self.app = app
    self.seq = seq

  def fieldEventTrap(self, event):
     keycode = event.KeyCode()
     if keycode in (WXK_RETURN, WXK_TAB):
       if self.seq < len(self.app.textctrlList) - 1:
         self.app.textctrlList[self.seq+1].SetFocus()
       else:
         if keycode == WXK_TAB:
           if event.ShiftDown():
             self.app.textctrlList[self.seq-1].SetFocus()
     elif keycode == WXK_ESCAPE:
       self.app.OnCancel(None)
     else:
      event.Skip()

