#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can` redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# Copyright 2001-2004 Free Software Foundation
#
# FILE:
# SchemaViewer.py
#
# DESCRIPTION:
# Class that implements a "Schema Viewer" (introspection) Tool Panel
#
# NOTES:
#

import sys, os, cPickle, traceback, StringIO, string
from wxPython.wx import *
from gnue.common.apps import GDebug
from gnue.common.datasources import GDataSource
from gnue.common.apps import RuntimeSettings
from gnue.common import events
from gnue.designer.base.ToolBase import *

class SchemaViewer(ToolBase):

  runtime_section = "SchemaViewer"

  def init(self):
    self.connections = self.instance._app.connections

    self.connectionName = None
    self.connectionNames = self.connections.getConnectionNames()
    self.connectionNames.sort()
    self.connectionCache = {}

##    self.toolBar = ToolBar(self, parentWindow)

    self.splitter = wxSplitterWindow(self, -1, style=wxSP_3D, point=wxPoint(0,32))

    self.list = wxListCtrl(self.splitter, -1, pos=wxPoint(0,0),
             style=wxLC_REPORT|
                   wxLC_HRULES|wxLC_VRULES|wxLC_SINGLE_SEL)

    self.list.InsertColumn(0,_('Name'))
    self.list.InsertColumn(1,_('Type'))

    self.schemaPanel = ChildSchemaViewer(self, self.splitter)

    self.splitter.SplitHorizontally(self.list, self.schemaPanel,200)

    EVT_SIZE(self, self.OnSize)

    self.current = None

    # EventAware provided by ToolBase
    self.registerEventListeners({
                       'ObjectSelected'      : self.onSetCurrentObject,
                       'ObjectCreated'       : self.onCreateObject,
                       'ObjectModified'      : self.onModifyObject,
                       'ObjectDeleted'       : self.onDeleteObject,
                      })


    #
    # Set up the "selection" panel
    #
    self.combopanel = wxPanel(self, -1, pos=wxPoint(0,0), size=wxSize(32,32),
         style=wxRAISED_BORDER|wxCLIP_CHILDREN)


    self.connLabel = wxStaticText(self.combopanel, -1, _("Connection: "),
         pos=wxPoint(4,6))
    self.connCombo = wxComboBox(self.combopanel, -1,
         pos=wxPoint(self.connLabel.GetSize().width + 10,4),
#         size=self.connLabel.GetSize(),
         style=wxCB_READONLY)
    self.typeLabel = wxStaticText(self.combopanel, -1, _("Show: "),
         pos=wxPoint(self.connCombo.GetPosition().x + \
                     self.connCombo.GetSize().width + 20,6))
    self.typeCombo = wxComboBox(self.combopanel, -1,
         pos=wxPoint(self.typeLabel.GetPosition().x + \
                     self.typeLabel.GetSize().width + 10,4),
         style=wxCB_READONLY)

    self.combopanel.SetSize(wxSize(32, self.typeCombo.GetSize().height+12))
    self.splitter.SetPosition((0,self.combopanel.GetSize().height))

    self.connCombo.Append('')


    for name in self.connectionNames:
      self.connCombo.Append(name)

    EVT_COMBOBOX(self, self.connCombo.GetId(), self.OnConnectionSelected)


    EVT_LIST_BEGIN_DRAG(self, self.list.GetId(), self.OnBeginDrag)
    EVT_LIST_ITEM_SELECTED(self, self.list.GetId(), self.OnTableSelected)

    self.finalized = 0
    self.splitter.SetSize(self.GetSize())


  def fillList(self):

    try:
      conn = self.connectionName
      if conn is None or conn == "":
        return

      if not self.connections.isConnectionActive(conn):
        return


      # TODO: Much error catching... any DB errors should be
      # TODO: trapped so the designing session can continue,
      # TODO: albeit without schema support.


      # Create a cache by connection to conserve database calls
      try:
        schemaTypes, schemas = self.connectionCache[conn]
      except KeyError:
        dataObject = self.connections.getDataObject(conn, "object")

        self.connections.requestConnection(dataObject, conn)

        schemaTypes = dataObject.types
        schemas = dataObject.find()
        schemas.sort()

        self.connectionCache[conn] = (schemaTypes, schemas)


      index = 0
      self.list.DeleteAllItems()

      for schema in schemas:

        self.list.InsertStringItem(index, schema.name)
        self.list.SetStringItem(index, 1,
           schema.type)
        self.list.SetItemData(index, index)
        index += 1


    except:

      buffer = StringIO.StringIO()
      traceback.print_exc(file=buffer)

      GDebug.printMesg(5,"Exception when retrieving schema (last 5 lines):")
      GDebug.printMesg(5,"%s"%buffer.getvalue())
      buffer.close()



  def onSetCurrentObject (self, event):
    object = event.object
    handler = event.originator

  def onCreateObject (self, event):
    object = event.object
    handler = event.originator

  def onModifyObject (self, event):
    object = event.object
    handler = event.originator

  def onDeleteObject (self, event):
    object = event.object
    handler = event.originator

  def OnSize(self, event):
    if not self.finalized:
      self.finalize()
    self.splitter.SetSize(self.GetSize())

    w,h = self.GetClientSizeTuple()
    self.combopanel.SetSize(wxSize(w,self.combopanel.GetSize().height))
    self.splitter.SetSize(wxSize(w,h-self.combopanel.GetSize().height))


  # Runtime Setting Support
  def finalize(self):

    self.finalized = 1

    self.SetPosition((
       RuntimeSettings.getint(self.runtime_section, 'x', -1),
       RuntimeSettings.getint(self.runtime_section, 'y', -1)))

    self.splitter.SetSashPosition(
       RuntimeSettings.getint(
           self.runtime_section, 'sash', 100) or 100)

    for i in range(2):  # Should be the number of columns in the list
      self.list.SetColumnWidth(i,
        RuntimeSettings.getint(
           self.runtime_section, "col%s" % i, -2))

    for i in range(SCH_COLUMNS):
      self.schemaPanel.list.SetColumnWidth(i,
        RuntimeSettings.getint(
           self.runtime_section, "fieldCol%s" % i, -2))

    self.list.Refresh()
    self.Refresh()


  def saveRuntimeSettings(self):
    sash = self.GetPositionTuple()

    settings = {
              'sash': self.splitter.GetSashPosition(),
             }

    # Save the top-level list column sizes
    for i in range(2):  # Should be the number of columns in the list
      settings["col%s" % i] = self.list.GetColumnWidth(i)

    # Save the schema viewer column sizes
    for i in range(SCH_COLUMNS):
      settings["fieldCol%s" % i] = self.schemaPanel.list.GetColumnWidth(i)

    return ( self.runtime_section,
             settings )


  # A connection was selected in the dropdown
  def OnConnectionSelected(self, event):

    self.connectionName = self.connCombo.GetString(self.connCombo.GetSelection())
    self.typeCombo.Clear()
    self.typeCombo.Append('All')

    self.schemaPanel.clear()
    self.fillList()

    try:
      for type in self.connectionCache[self.connectionName][0]:
        self.typeCombo.Append(type[1]+ 's')
    except KeyError:
      pass


  # A Datasource was selected in the menu
  def OnTableSelected(self, event):
    self.current = self.connectionCache[self.connectionName][1][event.GetData()]
    self.schemaPanel.OnTableSelected(self.current)


  def OnBeginDrag(self, event):

    object = self.connectionCache[self.connectionName][1]\
           [self.list.HitTest(event.GetPoint())[0]]

    do = wxCustomDataObject(wxCustomDataFormat("application/x-gnue-designer"))
    do.SetData(cPickle.dumps(self.buildDataObject(object),1))

    dropSource = wxDropSource(self)
    dropSource.SetData(do)
    result = dropSource.DoDragDrop(false)


  def buildDataObject(self, object):

    basename =     string.join(string.split(string.capwords( \
                   string.replace(
                     string.replace(object.name,'.',''),'_',' '))),'')

    dname = self.instance.getUniqueName("dts%s" % basename)
    bname = self.instance.getUniqueName("blk%s" % basename)

    data = [ { "Type" : "datasource",
               "Attributes": {
                     "name": dname,
                     "table": object.name,
                     "connection": self.connectionName } },
             { "Type" : "block",
               "Attributes": {
                     "name": bname,
                     "datasource": dname } } ]

    return data



#############################################################################
#
# Schema Viewer Tab
#

SCH_FIELD = 0
SCH_TYPE = 1
SCH_SIZE = 2
SCH_NATIVE = 3
SCH_REQ = 4
SCH_COLUMNS = SCH_REQ + 1

class ChildSchemaViewer(wxPanel):

  def __init__(self, editor, parentWindow):
    wxPanel.__init__(self, parentWindow, -1)
    self.editor = editor

    self.list = wxListCtrl(self, -1, pos=wxPoint(0,0),
             style=wxLC_REPORT|wxLC_HRULES|wxLC_VRULES)

    self.list.InsertColumn(SCH_FIELD, _('Field'))
    self.list.InsertColumn(SCH_TYPE,  _('Base Type'))
    self.list.InsertColumn(SCH_SIZE,  _('Size'), wxLIST_FORMAT_RIGHT)
    self.list.InsertColumn(SCH_NATIVE,_('Native Type'))
    self.list.InsertColumn(SCH_REQ,   _('Required'))

    EVT_SIZE(self, self.OnSize)
    EVT_LIST_ITEM_SELECTED(self, self.list.GetId(), self.OnFieldSelected)
    EVT_LIST_BEGIN_DRAG(self, self.list.GetId(), self.OnBeginDrag)


  def clear(self):
    self.list.DeleteAllItems()


  def fillList(self):

    try:

      # TODO: Much error catching... any DB errors should be
      # TODO: trapped so the designing session can continue,
      # TODO: albeit without schema support.

      index = 0
      self.list.DeleteAllItems()

      # Cache the child schema definitions (fields)
      try:
        self.table.__childCache
      except AttributeError:
        self.table.__childCache = self.table.fields()

      for field in self.table.__childCache:

        self.list.InsertStringItem(index, field.name)
        self.list.SetStringItem(index, SCH_TYPE,
           string.upper(field.datatype[0]) + field.datatype[1:])
        self.list.SetStringItem(index, SCH_NATIVE,
           hasattr(field, 'nativetype') and field.nativetype or '(unknown)')
        self.list.SetStringItem(index, SCH_REQ,
           field.required and "Yes" or "No")
        if hasattr(field,'length'):
          if hasattr(field,'precision') and field.precision > 0:
            self.list.SetStringItem(index, SCH_SIZE, "%s;%s" % (
                  field.length, field.precision))
          else:
            self.list.SetStringItem(index, SCH_SIZE, "%s" % field.length)
        else:
          self.list.SetStringItem(index, SCH_SIZE,'-')
        self.list.SetItemData(index, index)
        index += 1

    except:

      buffer = StringIO.StringIO()
      traceback.print_exc(file=buffer)

      GDebug.printMesg(5,"Exception when retrieving schema (last 5 lines):")
      GDebug.printMesg(5,"%s"%buffer.getvalue())
      buffer.close()


  def OnTableSelected(self, object):
    self.table = object
    self.fillList()


  # A field was selected in the dropdown
  def OnFieldSelected(self, event):
    self.current = self.table.__childCache[event.GetData()]


  def OnSize(self, event):
    self.list.SetSize(self.GetSize())


  def OnBeginDrag(self, event):

    #object = self.schemaMap[event.GetIndex()] # Bah! this always returns 0
    object = self.table.__childCache[self.list.HitTest(event.GetPoint())[0]]

    data = self.editor.buildDataObject(self.table)

    item = -1
    # Cycle through each selected item
    while 1:
      item = self.list.GetNextItem(item,
                                   wxLIST_NEXT_ALL,
                                   wxLIST_STATE_SELECTED);
      if item == -1:
        break

      object = self.table.__childCache[item]

      attributes = {
          "block": data[1]['Attributes']['name'],
          "name": self.editor.instance.getUniqueName("fld%s" % (
                     string.join(string.split(string.capwords( \
                     string.replace(object.name,'_',' '))),''))),
          "field": object.name,
          "datatype": object.datatype,
          "required" : object.required,
        }

      if hasattr(object,'length') and object.length:
        attributes['max_length'] = object.length
        attributes['width'] = object.length < 30 and object.length or 30

      try: 
        defaulttype = object.defaulttype
        defaultval = object.defaultval
      except AttributeError:
        defaulttype = None
        defaultval = None

      if defaulttype == 'constant':
        if defaultval[:1] in "\"'":
          defaultval = defaultval[1:-2]
        attributes['default'] = defaultval
        defaulttype = None
        defaultval = None


      if hasattr(object, 'label') and len(object.label):
        label = object.label
      elif len(object.name) == 1:
        label = string.upper(object.name)
      else:
        label = string.join(string.split(string.capwords( \
                     string.replace(object.name,'_',' '))),' ')

      data.append( { "Type" : "entry",
                     "Attributes": attributes,
                     "Label": label,
                     "DefaultTrigger": defaulttype,
                     "DefaultInfo": defaultval
                   } )

    do = wxCustomDataObject(wxCustomDataFormat("application/x-gnue-designer"))
    do.SetData(cPickle.dumps(data,1))

    dropSource = wxDropSource(self)
    dropSource.SetData(do)
    result = dropSource.DoDragDrop(false)


# Get around a few wx 2.2.x shortcomings:
try:
  wxLC_HRULES
  wxLC_VRULES
except NameError:
  wxLC_HRULES=0
  wxLC_VRULES=0

