#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# Copyright 2001-2004 Free Software Foundation
#
# FILE:
# forms/wizards/AddDropdown.py
#
# DESCRIPTION:
# Add a dropdown entry to a form, creating any datasources if necessary.
#
# NOTES:
#

from gnue.designer.forms.TemplateSupport import *
from gnue.designer import VERSION
import string

#
# Steps
#
# 0. Get id/field/[Existing Datasource/New Datasource /Static Set]
# 1. Select / create datasource
# 2. Select key field + description field
#

class AddDropDownWizard(FormPrePositioningTemplate):


  # The first step in our wizard.
  # The template parser will initially
  # call GetStep(FIRST_STEP).
  FIRST_STEP = '0'


  ###############
  #
  # Initialize any runtime variables
  #
  def StartWithPositioning(self, root, current,
           x=0, y=0, width=10, height=1):
    self.form = root
    self.current = current
    self.x = x
    self.y = y
    self.width = width
    self.height = height

    if not self.current['block']:
      raise InsufficientInformation, _("There are no blocks in this form.")

    self.__getExistingDatasources()


  ###############
  #
  # Return the markup for a specific page
  #
  def GetStep(self, stepCode):

    if stepCode == '1':
      if self.variables['datasource'] == 0:
        stepCode = '1:new'
      elif self.variables['datasource'] == 1:
        stepCode = '1:reuse'
      else:
        stepCode = '1:fixed'

    #
    # Step #1 / Get Title, et al
    #
    if stepCode == '0':

      if self.__firstConnection:
        self.variables['connection'] = self.__firstConnection

      if len (self.__datasourceMap):
          set=((0,'Create a new datasource'),
               (1,'Use an existing datasource'),
               (2,'Specify a static list of valid choices' )
               )
      else:
          set=((0,'Create a new datasource'),
               (2,'Specify a static list of valid choices' )
               )


      return   { 'title': 'Add Dropdown Entry',
                 'content': (WizardText('To create your dropdown, I need to know some basic information. '
                                        'First, what field will store the dropdown\'s value?'),
                             WizardInput('name', label='Field name:', required=1,
                                         size=40),
                             WizardText('Where will this dropdown pull its data?'),
                             WizardInput('datasource', label='Source:', required=1,
                                         maxSelections=1,
                                         set=set)
                            ),
                 'prev': None,
                 'next': '1' }

    #
    # Step #2 [New Datasource] / Get Connection
    #
    elif stepCode == '1:new':
      return   { 'title': 'Datasource Connection',
                 'content': (WizardText('What connection should this form use to connect to the \ndatabase?'),
                             WizardInput('connection',label='Connection:', required=1,
                                         set=self.GetAvailableConnections()),
                             WizardText('You may be asked to login to this connection.')),
                 'prev': '0',
                 'next': '2:new' }

    #
    # Step #3 [New Datasource] / Get Table
    #
    elif stepCode == '2:new':
      return   { 'title': 'Select Validator Table/Source',
                 'content': (WizardText('Now, please select the table the dropdown will\nuse to populate its data set.'),
                             WizardInput('table', label='Table:', required=1, lines=5,
                                         set=self.GetAvailableSources(self.variables['connection'])), ),
                 'prev': '1:new',
                 'next': '3:new' }


    #
    # Step #4 [New Datasouce] / Get key/description fields
    #
    elif stepCode == '3:new':

      return   { 'title': 'Select Key/Description fields',
                 'content': (WizardText('Which field will be the "key"? (i.e., which\nfield contains the value that will\nbe stored in the database?)'),
                             WizardInput('key', label='Key Field:', required=1,
                                         maxSelections=1,
                                         set=self.GetAvailableFields( \
                                                self.variables['connection'],
                                                self.variables['table'])),
                             WizardText('Which field will be the "description" value?'),
                             WizardInput('descr', label='Description:', required=1,
                                         maxSelections=1,
                                         set=self.GetAvailableFields( \
                                                self.variables['connection'],
                                                self.variables['table']))),
                 'prev': '2:new',
                 'next': None }

    #
    # Step #2 [reuse Datasource] / Get DataSource
    #
    elif stepCode == '1:reuse':
      return   { 'title': 'Datasource Connection',
                 'content': (WizardText('Which existing datasource should be used?'),
                             WizardInput('source',label='Datasource:', required=1,
                                         set=self.__getExistingDatasources())),
                 'prev': '0',
                 'next': '2:reuse' }

    #
    # Step #3 [Reuse Datasouce] / Get key/description fields
    #
    elif stepCode == '2:reuse':

      datasource = self.__datasourceMap[self.variables['source']]


      return   { 'title': 'Select Key/Description fields',
                 'content': (WizardText('Which field will be the "key"? (i.e., which\nfield contains the value that will\nbe stored in the database?)'),
                             WizardInput('key', label='Key Field:', required=1,
                                         maxSelections=1,
                                         set=self.GetAvailableFields( \
                                                datasource.connection,
                                                datasource.table)),
                             WizardText('Which field will be the "description" value?'),
                             WizardInput('descr', label='Description:', required=1,
                                         maxSelections=1,
                                         set=self.GetAvailableFields( \
                                                datasource.connection,
                                                datasource.table))),
                 'prev': '1:reuse',
                 'next': None }


    #
    # Step #2 [Static set] / Get Data Set
    #
    elif stepCode == '1:fixed':

      return   { 'title': 'Enter data set',
                 'content': (WizardText('Please enter the values the dropdown box\nshould use. Enter in the format: val1:Display1|val2:display2|...\n(For example, Y:Yes|N:No|M:Maybe'),
                             WizardInput('dataset', label='Data Set:', required=1, lines=5,
                                         size=40) ),
                 'prev': '0',
                 'next': None }





  ###############
  #
  # Verify contents of current step
  # Return None if no problems, otherwise
  # return a tuple of error message strings
  #
  def ValidateStep(self, stepCode):

    # TODO: This should validate step 1:fixed to verify the dataset is valid
    return None



  ###############
  #
  # We have all the data, so generate our form. This
  # is called after the user clicks the "Finish" button.
  # No more user input is allowed at this point.
  #
  def Finalize(self):

    #
    # Set up or retrieve our datasource
    #

    # Create new datasource
    if self.variables['datasource'] == 0:
      datasource = self.AddElement('datasource', self.form,
        { 'name': self.GetUniqueName("dtsDrp_%s_%s" % (self.variables['table'],
                                     self.variables['name'])),
          'connection': self.variables['connection'],
          'prequery': 1,
          'table': self.variables['table'] } )
      keyfield = self.variables['key']
      descrfield = self.variables['descr']

    # Reuse existing datasource
    elif self.variables['datasource'] == 1:
      datasource = self.__datasourceMap[self.variables['source']]
      keyfield = self.variables['key']
      descrfield = self.variables['descr']

    # Create a "static" datasource
    else:
      datasource = self.AddElement('datasource', self.form,
        { 'name': self.GetUniqueName("dtsDrpStatic%s" % self.variables['name']),
          'type': "static" } )
      keyfield = 'key'
      descrfield = 'descr'
      # Add staticset
      ss = self.AddElement( 'staticset', datasource,
                            { 'fields': 'key,descr' } )
      # And each row...
      for r in string.split(self.variables['dataset'],'|'):
        try:
          key, descr = string.split(r,':',1)
        except ValueError:
          pass
        ssr = self.AddElement('staticsetrow', ss)
        self.AddElement('staticsetfield',ssr,{'name': 'key', 'value': key})
        self.AddElement('staticsetfield',ssr,{'name': 'descr','value': descr})


    #
    # Create the dropdown
    #

    # Make sure we find a Block
    block = self.current['block']
    page = self.current['page']

    field = self.AddElement('field', block,
                    { 'name': self.GetUniqueName('fld%s' % self.variables['name'], block),
                      'field': self.variables['name'],
                      'fk_source': datasource.name,
                      'fk_key': keyfield,
                      'fk_description': descrfield } )


    self.AddElement('entry', page,
                    { 'name': self.GetUniqueName('ntry%s' % self.variables['name'], page),
                      'field': field.name,
                      'block': block.name,
                      'style': 'dropdown',
                      'Char:x': self.x,
                      'Char:y': self.y,
                      'Char:width': self.width or 10,
                      'Char:height': 1 } )


    return 1


  ###############
  #
  # Internal stuff
  # Get a list of all existing datasources
  #
  def __getExistingDatasources(self):
    self.__firstConnection = None
    self.__datasourceSet = []
    self.__datasourceMap = {}
    for child in self.form.findChildrenOfType('GFDataSource'):
      if hasattr(child,'table'):
        self.__datasourceSet.append((child.name,"%s (%s)" % (child.name, child.table)))
        self.__datasourceMap[child.name] = child
        if not self.__firstConnection:
          self.__firstConnection = child.connection
    return self.__datasourceSet





############
#
# Basic information about this template
#
TemplateInformation = {
    'Product': 'forms',
    'BaseID' : 'AddDropDown',
    'BaseClass' : AddDropDownWizard,
    'Name' : 'Add Bound Drop Down',
    'Description' : 'Inserts a bound dropdown',
    'Version' : VERSION,
    'Author' : 'The GNUe Designer Team',
    'Behavior': WIZARD,
    'MenuLocation' : ('Insert',_('Dropdown'), 501.2)
}

