#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# Copyright 2001-2005 Free Software Foundation
#
# FILE:
# Simple.py
#
# DESCRIPTION:
# Implements a basic form template
#
# NOTES:
# While functional, the primary purpose of this wizard is
# as a "learning-by-example" tool.


from gnue.designer.forms.TemplateSupport import *

import string

# NOTE: It is VERY important that in any references to a
#    "<insert type here> wizard", the word "wizard" must be
#    in lower case, as many proper names such as "Foobar Wizard"
#    have been trademarked. (No, I'm not kidding :)
#
# TODO: Make sure "Foobar Wizard" is not trademarked.
# TODO: If not, should we?
#


class SimpleFormTemplate (FormTemplate):

  TYPEMAP    = {'date'    : 'date',
                'time'    : 'date',
                'datetime': 'date',
                'boolean' : 'boolean',
                'number'  : 'number',
                'string'  : 'text'}

  # The first step in our wizard.
  # The template parser will initially
  # call GetStep(FIRST_STEP).
  FIRST_STEP = '0'


  ###############
  #
  # Initialize any runtime variables
  #
  def Start(self, form, current):
    self.form = form
    self.current = current



  ###############
  #
  # Return the markup for a specific page
  #
  def GetStep(self, stepCode):

    #
    # Step #1 / Get Title, et al
    #
    if stepCode == '0':
      return   { 'title': 'Basic Form Information',
                 'content': (WizardText('Welcome to the sample form wizard.'),
                             WizardText('To create your form, I need to know some basic information.\n\n'
                                        'First, what shall I call your form? This name will appear in '
                                        'the title bar.'),
                             WizardInput('title', label='Form Title:', required=1,
                                         size=40),
                             WizardText('What connection should this form use to connect to the database?'),
                             WizardInput('connection',label='Connection:', required=1,
                                         set=self.GetAvailableConnections()),
                             WizardText('You may be asked to login to this connection.'),
                            ),
                 'prev': None,
                 'next': '1' }


    #
    # Step #2 / Get Base Table
    #
    elif stepCode == '1':

      # TODO: If the connection changed between steps,
      # TODO: variables['table'] and variables['fields']
      # TODO: should be cleared.

      return   { 'title': 'Select Base Table/Source',
                 'content': (WizardText('Now, please select the base table for your form.'),
                             WizardInput('table', label='Base Table:', required=1, lines=5,
                                         set=self.GetAvailableSources(self.variables['connection'])), ),
                 'prev': '0',
                 'next': '2' }


    #
    # Step #3 / Get Columns to Include
    #
    elif stepCode == '2':

      # TODO: If the table changed between steps,
      # TODO: variables['fields'] should be cleared.

      return   { 'title': 'Select Fields to Include',
                 'content': (WizardText('Which fields shall I include in your form?'),
                             WizardInput('fields', label='Columns:', required=1,
                                         maxSelections=-1, orderable=1,
                                         set=self.GetAvailableFields( \
                                                self.variables['connection'],
                                                self.variables['table'])),
                             WizardInput('arrangement', label='Arrangement Method:',
                                         required=1,
                                         set=(('left','Single Record with Labels to Left'),
                                              ('above','Single Record with Labels Above'),
                                              ('grid','Grid Format')))),
                 'prev': '1',
                 'next': None }



  ###############
  #
  # Verify contents of current step
  # Return None if no problems, otherwise
  # return a tuple of error message strings
  #
  def ValidateStep(self, stepCode):

    # The Simple wizard uses basic "required"
    # settings in the page markup, so doesn't
    # need any special validation.
    return None



  ###############
  #
  # We have all the data, so generate our form. This
  # is called after the user clicks the "Finish" button.
  # No more user input is allowed at this point.
  #
  def Finalize(self):
    self.instance.wizardName = TemplateInformation['Name']
    
    # We will use the table name as the basis for all our
    # object names. We will add a prefix based on the object
    # type to the table name. Capitalize the first letter and
    # strip all spaces...
    if len(self.variables['table']) == 1:
      tableKey = string.upper(self.variables['table'])
    else:
      tableKey = string.upper(self.variables['table'][0]) + \
                    string.replace(self.variables['table'][1:],' ','_')

    # Is this a grid style form?
    multirecord = self.variables['arrangement'] == 'grid'

    # Will labels appear above or to the left of the entries?
    # Note: this results in false for both "grid" and "above"
    leftlabels = self.variables['arrangement'] == 'left'

    # Set the basic attributes of the form
    self.form['title'] = self.variables['title']

    # Create a single datasource based on user's input
    datasource = self.AddElement('datasource', self.form,
       {  'connection': self.variables['connection'],
          'table': self.variables['table'],
          'type': 'object',
          'name': 'dts%s' %  tableKey })

    logic = self.current['logic']
    layout = self.current['layout']
    page = self.current['page'] or self.AddElement('page', layout,
         {  'name': 'pg%s' % tableKey })


    # We will need a block to hold our entries...

    # The basic attributes for a block
    attrs = {'name' : 'blk%s' % tableKey,
             'datasource': datasource.name}

    # If this is a multirecord form, set rows at the block level.
    if multirecord:
      attrs['rows'] = 10

    # Create the block
    block = self.AddElement('block', logic, attrs)

    schema = self.GetSourceSchema(self.variables['connection'],
                         self.variables['table'])


    # Create the entries and labels...


    # Make a map of all the field schemas we will need
    # We will not actually create the entries at this point
    # because we want to keep in the order that the user
    # specified
    fields = {}
    for field in schema.findChildrenOfType ('GSField', False, True):
      if field.name in self.variables['fields']:
        fields[field.name] = field

    entryQueue = []
    labelQueue = []
    largestField = 0
    largestLabel = 0

    # First, let's make all the fields
    for name in self.variables['fields']:

      field = fields[name]

      # We will use the field name as the basis for all our
      # entry and label names.  Capitalize the first letter
      # and strip all spaces...
      if len(field.name) == 1:
        fieldKey = string.upper(field.name)
      else:
        fieldKey = string.join(string.split(string.capwords( \
                     string.replace(field.name,'_',' '))),'')

      # Create a label. If the schema specified a label for a field,
      # use that as is.  If not, use the field name, replacing any '_'
      # with spaces and tacking on a colon.
      text = hasattr(field,'label') and label or \
             string.capwords(string.replace(field.name,'_',' ')) + ':'

      # Add text, x, y to labelQueue
      # The x,y will be replaced by logic later on...
      labelQueue.append([text,0,0])

      # Keep track of the greatest label width
      largestLabel = max(len(text),largestLabel)

      # Create an entry for this field.

      attrs={ 'name': "fld%s" % fieldKey,
              'field': field.name,
              'typecast': self.TYPEMAP.get (field.type, 'text')}

      # If we have a length for the field, use this as the maxLength
      # for the entry. Also, adjust the display width if necessary.
      if hasattr(field,'length'):
        ln = max(min(field.length, 40),0) or 10

        ##ln = min(field.length, 40)
        attrs['maxLength'] = ln
        largestField = max(largestField, ln)
      else:
        largestField = max(largestField, 10)

      # Create the entry element
      fld = self.AddElement('field', block, attrs)

      # and queue it so we can create entry's later
      entryQueue.append([fld, 0, 0] )


    #
    # Rearrange the fields and labels to snugly fit the screen
    # based on the layout arrangement selected by the user.
    #

    # Grid/multirecord layout
    if multirecord:

      height = 13
      width = 1
      x = 1

      for i in range(len(entryQueue)):
        field = entryQueue[i][0]
        text = labelQueue[i][0]
        textLen = len(text)

        labelQueue[i][1] = x
        labelQueue[i][2] = 1
        entryQueue[i][1] = x
        entryQueue[i][2] = 2

        # If label width is larger than entry width, center the entry
        try:
          entryWidth = min(field['maxLength'],40)
        except KeyError:
          entryWidth = 10
        if entryWidth < textLen:
          entryQueue[i][1] += int((textLen - entryWidth)/2)

        # Calculate the starting x for the next label/entry
        dx = max(entryWidth, textLen) + 1

        # Increase the form width accordingly
        width += dx
        x += dx

    # Single Record layout (regardless of label location)
    else:

      # Assign the starting (x,y) values for labels (l) and fields (f)
      # as well as the delta (x,y) values.
      if leftlabels:
        # Labels to the left of the entries
        lx, ly, ldx, ldy = (1, 1, 0, 1)
        fx, fy, fdx, fdy = (largestLabel + 2, 1, 0, 1)

        # Set form width
        width = largestField + largestLabel + 3

      else:
        # Labels above the entries
        lx, ly, ldx, ldy = (1, 1, 0, 3)
        fx, fy, fdx, fdy = (1, 2, 0, 3)

        # Set the form width
        width = 2 + max(largestField, largestLabel)

      # Rearrange
      height = fy+2-fdy
      for i in range(len(entryQueue)):
        labelQueue[i][1] = lx
        labelQueue[i][2] = ly
        entryQueue[i][1] = fx
        entryQueue[i][2] = fy

        ly = ly + ldy
        lx = lx + ldx
        fy = fy + fdy
        fx = fx + fdx
        height += fdy

    #
    # Resize the layout screen to fit our form
    #
    layout['Char:width'] = width
    layout['Char:height'] = height


    #
    # Finally, add the visual elements...
    #
    for i in range(len(entryQueue)):

      #
      # First, the label
      #
      text, x, y = labelQueue[i]
      self.AddElement( 'label', page,
                  {'Char:x': x,
                   'Char:y': y,
                   'name': "lbl%s" % (text[-1]==':' and text[:-1] or text),
                   'text': text,
                   'Char:width': len(text)})

      #
      # And the entry...
      #
      field, x, y = entryQueue[i]
      attrs={'name': "ent%s" % field['name'][3:],
             'field': field['name'],
             'block': block['name'],
             'Char:x': x,
             'Char:y': y,
             'Char:width': 10 }

      # If we have a length for the field, use this as the maxLength
      # for the entry. Also, adjust the display width if necessary.
      try:
        attrs['Char:width'] = min(field['maxLength'], 40)
      except KeyError:
        pass

      # Create the entry element
      self.AddElement('entry', page, attrs)


    # That's it... we're done.
    return 1



############
#
# Basic information about this template
#
from gnue.designer import VERSION

TemplateInformation = {
    'Product': 'forms',
    'BaseID' : 'Simple',
    'BaseClass' : SimpleFormTemplate,
    'Name' : 'Simple form wizard',
    'Description' : 'Creates a simple single-source entry table.',
    'Version' : VERSION,
    'Author' : 'The GNUe Designer Team',
    'Behavior': WIZARD
}

