#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# Copyright 2001-2005 Free Software Foundation
#
# FILE:
# ToolDock.py
#
# DESCRIPTION:
# Provides "docks" for the main application window
# Handles resizing, etc.
#
# NOTES:
#

__all__ = ['Docker']

import sys
from wxPython.wx import *
from wxPython.wx import __version__ as wxversion
from gnue.common.apps import RuntimeSettings, GDebug
from FloatingPane import FloatingPane
from SplitDock import SplitDock
from DockedPane import DockedPane

class Docker(wxPanel):
  def __init__(self, parent, instance, tools):
    # on win32 a toolbar doesn't reduce the client rectangle height
    # wx doesn't handle this when calculates default pos :(
    if sys.platform == "win32":
      if wxversion[:3] == '2.6':
       pos = (0, 0)
      else:
       pos = (0, 36)
    else:
      pos = (-1,-1)

    wxPanel.__init__(self, parent, -1, pos=pos)
    self.parent = parent
    self.instance = instance

    self.horizSplitter = horizSplitter = wxSplitterWindow(self, -1, style=wxSP_3D)
    self.leftDock = leftWin = SplitDock(horizSplitter, self, 'left')
    self.rightDock = rightWin = SplitDock(horizSplitter, self,'right')
    EVT_SIZE(parent, self.__OnSize)

    instance.registerEventListeners({
                   'Docker:Undock': self.undock,
                   'Docker:Dock': self.dock})

    # Tell RuntimeSettings that we have information to save
    self.runtime_section = runtime_section = '%s-Docker' % instance.properties.nickname
    RuntimeSettings.registerRuntimeSettingHandler(self.instance, self)

    self.tools = []

    #
    # Add each tool
    #
    for id, title, baseclass, hotkey, group in tools:

      GDebug.printMesg(2,'Adding tool %s [%s]' % (title,id))

      indx = 0
      pos = baseclass.default_dock
      if '-' in pos:
        pos, indx = pos.split('-')
        indx = int(indx)

      tool = baseclass(id, title, instance, instance.rootObject, self)

      tool._docked = RuntimeSettings.getint(self.runtime_section, '%s-docked' % id, pos != 'frame')
      tool._dock_location = RuntimeSettings.get(self.runtime_section, '%s-location' % id, (pos == 'frame' and 'left' or pos))
      tool._dock_index = RuntimeSettings.getint(self.runtime_section, '%s-index' % id, indx)
      tool._visible = RuntimeSettings.getint(self.runtime_section, '%s-visible' % id, tool.default_visible)
      tool._frame_x = RuntimeSettings.getint(self.runtime_section, '%s-frame-x' % id, -1)
      tool._frame_y = RuntimeSettings.getint(self.runtime_section, '%s-frame-y' % id, -1)
      tool._frame_width = RuntimeSettings.getint(self.runtime_section, '%s-frame-width' % id, 200)
      tool._frame_height = RuntimeSettings.getint(self.runtime_section, '%s-frame-height' % id, 150)

      appd = (id, title, baseclass, hotkey, pos)

      if tool._docked:
        # Docked type
        if tool._dock_location == 'right':
          dock = rightWin
        else:
          dock = leftWin
        dock.add(DockedPane(dock, instance, tool), tool._dock_index)
      else:
        # Frame type
        FloatingPane(instance.uiinstance, instance, tool)

      # Keep track of the tools we created
      self.tools.append(tool)

      # Add ourselves to the Window menu
      instance.menubar.addAction(location='View', text=title, event='Tool:' + id,
                              grouping=group, canToggle=1, canDisable=0,
                              icon=None, hotkey=hotkey)


      instance.__dict__[id] = tool

      if tool._visible:
        instance.dispatchEvent('Show:Tool:%s' % id)
      else:
        instance.dispatchEvent('Hide:Tool:%s' % id)

    leftWin.refresh()
    rightWin.refresh()
    horizSplitter.SplitVertically(leftWin, rightWin,
              RuntimeSettings.getint(self.runtime_section, 'DockSplitter-TopBottom',500))

  def __OnSize(self, event):
    self.SetSize(self.parent.GetClientSize())
    self.horizSplitter.SetSize(self.GetClientSize())

  def saveRuntimeSettings(self):
    settings = {'docksplitter-topbottom': self.horizSplitter.GetSashPosition()}
    for tool in self.tools:
      tid = tool.id
      tool.dockpane.closing()
      settings['%s-docked' % tid] = tool._docked
      settings['%s-dock-location' % tid] = tool._dock_location
      settings['%s-dock-location' % tid] = tool._dock_location
      settings['%s-dock-index' % tid] = tool._dock_index
      settings['%s-frame-x' % tid] = tool._frame_x
      settings['%s-frame-y' % tid] = tool._frame_y
      settings['%s-frame-width' % tid] = tool._frame_width
      settings['%s-frame-height' % tid] = tool._frame_height
      settings['%s-visible' % tid] = tool._visible

    settings.update(self.leftDock.saveSettings())
    settings.update(self.rightDock.saveSettings())

    return (self.runtime_section, settings)

  def undock(self, event):
    tool = event.tool
    instance = self.instance
    parent = self.parent
    dockpane = tool.dockpane
    panel = dockpane.parent.remove(dockpane)
    FloatingPane(parent, instance, tool)
    tool._docked = 0
    dockpane.Destroy()
    instance.dispatchEvent('Show:Tool:%s' % tool.id)

  def dock(self, event):
    tool = event.tool
    if not tool._docked:
      frame = tool.dockpane
      frame.closing()
      tool._docked = 1
      if tool._dock_location == 'right':
        dock = self.rightDock
      else:
        dock = self.leftDock
      dock.add(DockedPane(dock, self.instance, tool), tool._dock_index - .5)
      frame.Destroy()
    self.instance.dispatchEvent('Show:Tool:%s' % tool.id)
    dock.refresh()
