
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# Copyright 2002-2004 Free Software Foundation
#
# FILE:
# GFDisplayHandler.py
#
# DESCRIPTION:
"""
DisplayHandler classes for Forms input validation
"""
# NOTES:
#

import sys, string, time, types

from gnue.common.utils.FileUtils import openResource
from gnue.common.formatting import DateMask, TextMask, NumberMask
from gnue.common.apps import GDebug
from gnue.common import events

try:
  import Image
except ImportError:
  Image = None

try:
  from mx.DateTime import DateTime, mktime
except ImportError:
  print """
   This GNUe tool requires mxDateTime to be installed

   You can find the mxDateTime at

   http://www.lemburg.com/files/python/mxDateTime.html

"""
  import sys
  sys.exit()

#############################################################################
#
# Base handler
#
# Do not use this handler directly.
# You should be using one of the
# other types based on this one.
#
class FieldDisplayHandler(events.EventAware):


  #####################
  #
  # General methods
  #

  def setValue(self, value):
    self.modified = False
    self.value = value
    self._buildDisplay()


  def getValue(self):
    return self.value


##   def getDisplay(self):
##     return self.display


  def getDisplayFiller(self, value):
    return self._buildDisplayHelper(value, False)


  def generateRefreshEvent(self):
    # TODO: this should probably actually happen in UIwxpython!
    if (self.handleCR and type(self.display)=='str'):
      self.dispatchEvent(events.Event('updateEntryEditor',
           object = self.field,
           display=self.display, # string.replace(self.display,'\n','\r\n'),
           cursor=self.cursor + len(string.split(self.display[:self.cursor+1],'\n'))-1,
           selection=self.getSelectionArea(),
         ))
    else:
      self.dispatchEvent(events.Event('updateEntryEditor',
           object = self.entry,
           display=self.display,
           cursor=self.cursor,
           selection=self.getSelectionArea(),
         ))

  def isPending(self):
    return self.modified

  #####################
  #
  # Editing methods
  #

  #
  # beginEdit
  # called when a widget first gets focus
  #
  def beginEdit(self, event):
    """
    Called when a widget first gets focus.
    """
    self.editing = True
    self.modified = False

    # TODO: Replace with formatter
    self.setValue(self.field.getValue())
    self.work = self._buildDisplayHelper(self.value, False)
    self._buildDisplay()

    self.cursor = len(self.display)
    # Ensure cursor is properly placed.
    self.generateRefreshEvent()

  #
  # endEdit
  # called when a widget loses focus or
  # when enter is hit
  #
  def endEdit(self, event):
    """
    Called when a widget loses focus or when ENTER is hit.
    """
    if not self.editing:
      return

    if not self._loadedAllowedValues:
      self.field.allowedValues()
      self._loadedAllowedValues = True


    # If this event returns __error__, then
    # the input was invalid and the entry is
    # still in editing mode.  __errortext__
    # will contain an appropriate error message.
    #
    # If no error, then event.modified is true
    # if the field was modified and the event's
    # result is the final entry value.

    self.selection1 = None

    if self.modified:
      if self._buildValue():
        if self.field._allowedValues and \
           not self.field._allowedValues.has_key("%s" % self.value):
          self.work = ""
          event.__error__ = True
          event.__errortext__ = _("Invalid value '%s' for field") % self.value
          return

        self.editing = False
        event.__results__ = self.value
        event.modified = self.modified
        self.field.setValue(self.value)

        self._buildDisplay()

      else:
        self.dispatchEvent(events.Event('formALERT',
          u_("Invalid input: '%s'") % self.work,_form=event._form))


  def addText(self, event):

    value = event.text

    if (self.field.readonly and not self.field._block.mode=='query'):
      event.__error__ = True
      event.__errortext__ = _("Cannot modify field.  Form is read only")
      return
    elif self.field.editOnNull and \
         self.value != None and \
         self.value != "" and \
         not self.field._block._resultSet.current.isPending():
      event.__error__ = True
      event.__errortext__ = _("Cannot modify field after initial creation.")
      return

    # Do some appropriate checks on the text
    # TODO: This will be moved to the mask system!!

    if self.field._lowercase:
      value = string.lower(event.text)
    if self.field._uppercase:
      value = string.upper(event.text)

    if hasattr(self.field,'max_length') and \
       len(self.work)  + len(value) > self.field.max_length:
      return

    if ( self.field._numeric and \
         self.field._block.mode == 'normal' ):
      for char in value:
        if not (char in string.digits or char in '.-') :
          return

    # To do overstrike, we'll fudge by first "highlighting"
    # the character to replace, then use the selection logic.

    if  hasattr(event,'overstrike') and event.overstrike and \
        self.selection1 == None:
      self.selection1 = self.cursor
      self.selection2 = self.selection1 + 1

    if self.selection1 != None:
      # If text is selected, then we will replace

      s1 = min(self.selection1, self.selection2)
      s2 = max(self.selection1, self.selection2)

      self.work = self.work[:s1]  \
                   + value        \
                   + self.work[s2:]

      self.selection1 = None
      self.cursor = s1 + len(value)

    else:
      # Otherwise just graft the new text in place

      self.work = self.work[:self.cursor] \
                   + value                \
                   + self.work[self.cursor:]

      self.cursor += len(value)


    event.__dropped__ = True
    event.refreshDisplay = True
    self.modified = True
    self._buildDisplay()

  # Insert text at defined position
  def insertTextAt(self, event):
    # set cursor position
    self.cursor = event.position

    # add the event text
    self.addText(event)

  # Delete Text Range
  def deleteRange(self, event):
    self.selection1 = event.start_pos
    self.selection2 = event.end_pos
    self.cursor = event.position
    self.delete(event)

  # Delete backwards one character
  def backspace(self, event):

    # If we have a "selection",
    # then act like a "delete"
    if self.selection1 != None:
      self.delete(event)
      return

    precurs = self.cursor
    self.moveCursorLeft(event)

    if self.cursor != precurs:
      event.overstrike = True
      event.text = ""

      self.addText(event)

  # Delete forward one character
  def delete(self, event):
    event.overstrike = True
    event.text = ""

    self.addText(event)


  #####################
  #
  # Cursor movement
  #

  def moveCursorRelative(self, relative):
    pass


  def moveCursor(self, event, selecting=False):
    if not selecting:
      self.selection1 = None

    self.cursor = min(event.position, len(self.display))
    event.refreshDisplay = True


  def moveCursorLeft(self, event, selecting=False):
    if not selecting:
      self.selection1 = None

    if self.cursor > 0:
      self.cursor -= 1
      event.refreshDisplay = True

  def moveCursorRight(self, event, selecting=False):
    if not selecting:
      self.selection1 = None

    if self.cursor < len(self.display):
      self.cursor += 1
      event.refreshDisplay = True

  def moveCursorToEnd(self, event, selecting=False):
    if not selecting:
      self.selection1 = None

    self.cursor = len(self.display)
    event.refreshDisplay = True


  def moveCursorToBegin(self, event, selecting=False):
    if not selecting:
      self.selection1 = None

    self.cursor = 0
    event.refreshDisplay = True


  #####################
  #
  # Selection stuff
  #

  # Set the selection area
  def setSelectionArea(self, cursor1, cursor2):
    self.selection1 = min(cursor1, cursor2)
    self.selection2 = max(cursor1, cursor2)


  # Return the selected area as a tuple (or
  # None if no selection)
  def getSelectionArea(self):
    if self.selection1 == None:
      return None
    else:
      return ( min(self.selection1, self.selection2),
               max(self.selection1, self.selection2) )


  # Select text with mouse and move the cursor
  def selectWithMouse(self, event):

    self.selection1 = event.position1
    self.selection2 = event.position2
    if self.cursor == self.selection2:
      event.position = self.selection1
    else:
      event.position = self.selection2
    self.moveCursor(event, True)


  # Select the entire text of the entry and move
  # the cursor to the end
  def selectAll(self, event):

    self.selection1 = 0
    self.moveCursorToEnd(event, True)
    self.selection2 = self.cursor


  # Move the selection cursor to the left one unit
  def selectLeft(self, event):

    if self.selection1 == None:
      self.selection1 = self.cursor

    self.moveCursorLeft(event, True)
    self.selection2 = self.cursor


  # Move the selection cursor to the right one unit
  def selectRight(self, event):

    if self.selection1 == None:
      self.selection1 = self.cursor

    self.moveCursorRight(event, True)
    self.selection2 = self.cursor


  # Select to the beginning of the entry
  def selectToBegin(self, event):

    if self.selection1 == None:
      self.selection1 = self.cursor

    self.moveCursorToBegin(event, True)
    self.selection2 = self.cursor


  # Select to the beginning of the entry
  def selectToEnd(self, event):

    if self.selection1 == None:
      self.selection1 = self.cursor

    self.moveCursorToEnd(event, True)
    self.selection2 = self.cursor


  # Copy to the clipboard
  def clipboardCopy(self, event):
    if self.selection1 != None:
      sel1, sel2 = self.getSelectionArea()
      self.dispatchEvent(events.Event('setCLIPBOARD',
               text=self.display[sel1:sel2]))

    event.refreshDisplay = False


  def clipboardCut(self, event):
    self.clipboardCopy(event)
    edevent = events.Event("requestKEYPRESS", text="", _form=event._form)
    self.dispatchEvent(edevent)


  def clipboardPaste(self, event):
    event.text = self.dispatchEvent(events.Event('getCLIPBOARD'))
    if event.text != None:
      self.addText(event)


  #####################
  #
  # Internal methods
  #
  def __init__(self, entry, eventHandler, formatter=None):
    events.EventAware.__init__(self, eventHandler)

    self.entry = entry            # Our entry
    self.field = entry._field
    self.editing = False          # In editing mode??
    self.modified = False         # Have we been modified??
    self.value = None             # The latest db-compat value
    self.work = ""                # Our working value
    self.display = ""             # The latest display-formatted value
    self.selection1 = None        # Start of highlight
    self.selection2 = None        # End of highlight
    self.cursor = 0               # Cursor position
    self._loadedAllowedValues = False # the allowed values already been loaded

    # TODO: replace w/an event that asks the
    # TODO: UIdriver if this should happen!
    self.handleCR = sys.platform == "win32"

    # Needs to be set by the child classes
    self.formatter = formatter

    self.subevents = events.EventController()
    self.subevents.registerEventListeners( {

                 # "Entry" events
                 'requestKEYPRESS'     : self.addText,
                 'requestCURSORLEFT'   : self.moveCursorLeft,
                 'requestCURSORRIGHT'  : self.moveCursorRight,
                 'requestCURSOREND'    : self.moveCursorToEnd,
                 'requestCURSORHOME'   : self.moveCursorToBegin,
                 'requestCURSORMOVE'   : self.moveCursor,
                 'requestBACKSPACE'    : self.backspace,
                 'requestDELETE'       : self.delete,
                 'beginEDITMODE'       : self.beginEdit,
                 'endEDITMODE'         : self.endEdit,

                 # Selection/clipboard events
                 'requestSELECTWITHMOUSE' : self.selectWithMouse,
                 'requestSELECTALL'    : self.selectAll,
                 'requestSELECTTOHOME' : self.selectToBegin,
                 'requestSELECTTOEND'  : self.selectToEnd,
                 'requestSELECTLEFT'   : self.selectLeft,
                 'requestSELECTRIGHT'  : self.selectRight,
                 'requestCOPY'         : self.clipboardCopy,
                 'requestCUT'          : self.clipboardCut,
                 'requestPASTE'        : self.clipboardPaste,

                 # Request for direct buffer manipulation
                 'requestINSERTAT'     : self.insertTextAt,
                 'requestDELETERANGE'  : self.deleteRange,
        })


  def _buildValue(self):
    self.value = self.work
    return True

  def _buildDisplayHelper(self, value, editing):
    if value == None:
      return ""
    else:
      return "%s" % value

  def _buildDisplay(self):
    if self.editing:
      self.display = self._buildDisplayHelper(self.work, True)
    else:
      self.display = self._buildDisplayHelper(self.value, False)



#############################################################################
#
#
#
class TextDisplayHandler(FieldDisplayHandler):
  def __init__(self, entry, eventHandler, displayMask, inputMask):
##    FieldDisplayHandler.__init__(self, entry, eventHandler,
##            TextMask.TextMask(displayMask, inputMask, formatMask))
    FieldDisplayHandler.__init__(self, entry, eventHandler, None)


#############################################################################
#
#
#
class PasswordDisplayHandler(TextDisplayHandler):
  def __init__(self, entry, eventHandler, displayMask, inputMask):
##    FieldDisplayHandler.__init__(self, entry, eventHandler,
##            TextMask.TextMask(displayMask, inputMask, formatMask))
    FieldDisplayHandler.__init__(self, entry, eventHandler, None)

  #
  # beginEdit
  # called when a widget first gets focus
  #
  def beginEdit(self, event):
    """
    Called when a widget first gets focus.
    """
    self.editing = True
    self.modified = False

    # TODO: Replace with formatter
    self.setValue(self.field.getValue())
    self.work = self.value
    self._buildDisplay()

    self.cursor = len(self.display)
    # Ensure cursor is properly placed.
    self.generateRefreshEvent()
    
    
  def _buildDisplayHelper(self, value, editing):
    if value == None:
      return ""
    else:
      return "*" * len(str(value))

  # These are no-no's for password-type fields
  def clipboardCopy(self, event):
    pass

  def clipboardCut(self, event):
    pass


#############################################################################
#
# Handler for Numeric types
#
class NumberDisplayHandler(FieldDisplayHandler):
  def __init__(self, entry, eventHandler, displayMask, inputMask):
##    FieldDisplayHandler.__init__(self, entry, eventHandler,
##            DateMask.DateMask(displayMask, inputMask, displayMask))
    FieldDisplayHandler.__init__(self, entry, eventHandler, None)


  def setValue(self, value):
    if value in (None,""):
      value = None
    else:
      value = float(value)

    return FieldDisplayHandler.setValue(self, value)


  # TODO: Replace with format mask
  def _buildDisplayHelper(self, val, editing):
    try:
      if val in (None,""):
        return ""
    except TypeError:
      pass

    if editing and type(val) == types.StringType:
      return val

    try:
      value = float(val)
    except ValueError:
      return str(val)

    try:
      if int(value) == value:
        value = "%d" % value
      else:
        value = "%s" % value
    except OverflowError:
      value = "%s" % value

#     if not editing:
#       # Comma-fy (Temporary!)
#       if value[:1] == '-':
#         minus = "-"
#         value = value[1:]
#       else:
#         minus = ""
#
#       try:
#         whole, decimal = string.split(value,'.')
#         value = ".%s" % decimal
#       except ValueError:
#         whole = value
#         value = ""
#
#       while len(whole) > 3:
#         value = ",%s%s" % (whole[-3:], value)
#         whole = whole[:-3]
#
#       value = minus + whole + value

    return value


  def _buildValue(self):
    if not len(self.work):
      self.value = None
      return True

    try:
      self.value = float(self.work)
      return True
    except ValueError:
      return False




#############################################################################
#
# Handler for Date types
#
class DateDisplayHandler(FieldDisplayHandler):
  def __init__(self, entry, eventHandler, displayMask, inputMask):
##    FieldDisplayHandler.__init__(self, entry, eventHandler,
##            DateMask.DateMask(displayMask, inputMask, displayMask))
    FieldDisplayHandler.__init__(self, entry, eventHandler, None)
    self.__displayMask = displayMask
    self.__inputMask = inputMask


  def setValue(self, value):
    return FieldDisplayHandler.setValue(self, value)


  # TODO: Replace with format mask
  def _buildDisplayHelper(self, value, editing):
    if editing:
      if self.__inputMask:
        format = self.__inputMask
      else:
        format = "%m/%d/%Y"
    else:
      if self.__displayMask:
        format = self.__displayMask
      else:
        format = "%m/%d/%y"

##    print "format=%s, value=%s, type=%s" % (format, value, type(value))
    if value in (None,""):
      return ""
    try:
      return value.strftime(format)
    except AttributeError:
      return str(value)


  def _buildValue(self):

    if not len(self.work):
      self.value = None
      return 1

    # TODO: Replace with format mask
    if self.__inputMask:
      try:
        # mx.DateTime.strptime is not available under windows
        self.value = mktime (time.strptime (self.work, self.__inputMask))
      except:
        return 0
      return 1
    try:
      value = string.replace(string.replace(self.work,'.','/'),'-','/')

      # Support for quick entry like '123102' for '12/31/02'
      if len(value) in (6,8) and string.find(value,'/') == -1:
        month = value[:2]
        day = value[2:4]
        year = value[4:]
      else:
        month, day, year = string.split(value,'/')

      year = int(year)
      if year < 100:
        if year > 50:
          year = year + 1900
        else:
          year = year + 2000

      self.value = DateTime(year, int(month), int(day))

      return 1
    except:
      return 0



#############################################################################
#
# Handler for dropdowns
#
class DropdownDisplayHandler(FieldDisplayHandler):

  def __init__(self, *args, **params):

    FieldDisplayHandler.__init__(self, *args, **params)

    # My events...
    self.subevents.registerEventListeners( {
           'requestREPLACEVALUE' : self.replaceText,
           'requestCOMBODROPPED' : self.beginEdit } )


  def beginEdit(self, event):
    if self.editing == True and self.modified == False:
      return

    if not self._loadedAllowedValues and not hasattr(self.field,'_allowedValues'):
      self.field.allowedValues()
      self._loadedAllowedValues = True

    self.editing = True
    self.modified = False

    # TODO: Replace with formatter
    self.setValue(self.field.getValue())

    if self.value == None:
      self.work = ""
    else:
      try:
        self.work = self.field._allowedValues ["%s" % self.value]
      except KeyError:
        self.work = ""
        event.__error__ = True
        event.__errortext__ = _("Invalid value '%s' for keyed pull-down field") % self.value

    self._buildDisplay()

    self.cursor = len(self.display)


  # TODO: Replace with format mask
  def _buildDisplayHelper(self, value, editing):
    if value in (None,""):
      return ""

    if editing:
      val = string.lower(value)
      for disp in self.field._allowedValuesDescr:
        if string.lower(disp[:len(val)]) == val:
          display = self.field._allowedValues[self.field._allowedValuesReverse[disp]]
          return display
      return value

    if self.field._allowedValues.has_key ("%s" % value):
      return self.field._allowedValues ["%s" % value]
    else:
      return ""

  def _buildDisplay(self):
    if self.editing:
      self.display = self._buildDisplayHelper(self.work, True)
      if self.cursor > len(self.work):
        self.work = self.display
    else:
      self.display = self._buildDisplayHelper(self.value, False)


  def replaceText(self, event):
    self.selection1 = 0
    self.selection2 = len(self.display)
    self.addText(event)

# TODO: Kills dropdown handling with keyboard on win32
# TODO: needs more work :(
#    self.endEdit(event) # Hack: allows dropdowns to change field
                        # value immediately (causes post-change
                        # on the field to fire).
                        # I don't think it'll hurt anything.
                        # if it does then we'll need to pull
                        # the logic from FieldDisplayHandler.endEdit
                        # and add here.  - jamest


  def _buildValue(self):
    if self.work == "":
      self.value = '' # None
    else:
      try:
        self.value = self.field._allowedValuesReverse[self.display]
      except KeyError:
        return False
    return True



#############################################################################
#
# Handler for Checkbox types
#
class CheckboxDisplayHandler(FieldDisplayHandler):

  def __init__(self, *args, **params):

    self.trueValue = gConfigForms("checkboxTrue")
    self.falseValue = gConfigForms("checkboxFalse")

    self.trueValues =  ('Y','y','T','t','1','x','X', self.trueValue)
    self.falseValues = ('N','n','F','f','0','', ' ', self.falseValue)

    FieldDisplayHandler.__init__(self, *args, **params)

    # My events...
    self.subevents.registerEventListeners( {
           'requestTOGGLECHKBOX' : self.toggle  } )


  def setValue(self, value):
    # Force to 0 or 1
    self.value = self._sanitizeValue(value)

    self.modified = False
    self._buildDisplay()


  def _sanitizeValue(self, value):
    if ("%s" % value)[:1] in self.trueValues:
      return True
    elif ("%s" % value)[:1] in self.falseValues:
      return False
    else:
      return value and True or False

  def _buildValue(self):
    self.value = self.work
    return True


  # TODO: Replace with format mask
  def _buildDisplayHelper(self, value, editing):
    return self._sanitizeValue(value)

##  def getDisplay(self):
##    return self.display


  def toggle(self, event):
    if event.data == None:
      # saving our checkboxes from Python 2.3 boolean type (it kills sql update)
      self.work = int(not self.work)
    else:
      self.work = self._sanitizeValue(event.data[1])
    self.modified = True
    self._buildDisplay()
    event.refreshDisplay=True


  def beginEdit(self, event):

    self.editing = True
    self.modified = False

    self.setValue(self.field.getValue())

    self.work = self.value

    self._buildDisplay()

    self.cursor = 0


  # Delete backwards one character
  def backspace(self, event):
    return

  # Delete forward one character
  def delete(self, event):
    return

  def moveCursorRelative(self, relative):
    return


  def moveCursor(self, event, selecting=False):
    return


  def moveCursorLeft(self, event, selecting=False):
    return


  def moveCursorRight(self, event, selecting=False):
    return


  def moveCursorToEnd(self, event, selecting=False):
    return


  def moveCursorToBegin(self, event, selecting=False):
    return


  #####################
  #
  # Selection stuff
  #

  # Set the selection area
  def setSelectionArea(self, cursor1, cursor2):
    return


  # Return the selected area as a tuple (or
  # None if no selection)
  def getSelectionArea(self):
    return


  # Select the entire text of the entry and move
  # the cursor to the end
  def selectAll(self, event):
    return


  # Move the selection cursor to the left one unit
  def selectLeft(self, event):
    return


  # Move the selection cursor to the right one unit
  def selectRight(self, event):
    return


  # Select to the beginning of the entry
  def selectToBegin(self, event):
    return


  # Select to the beginning of the entry
  def selectToEnd(self, event):
    return


  # Copy to the clipboard
  def clipboardCopy(self, event):
    return self.work


  def clipboardCut(self, event):
    return self.work


  def clipboardPaste(self, event):
    event.text = self.dispatchEvent(events.Event('getClipboard'))
    if event.text != None:
      self.work = self._sanitizeValue(event.data)
      self.modified = True
      self._buildDisplay()



#############################################################################
#
# ImageDiplayHandler
#
class ImageDisplayHandler(FieldDisplayHandler):
  def __init__(self, entry, eventHandler):
    if not Image:
      raise "Form contains a <image> and image support not loaded"
    FieldDisplayHandler.__init__(self, entry, eventHandler, None)

  def _buildDisplayHelper(self, value, editing):
    if string.lower(self.entry.type) == 'url':
      try:
        # TODO: PIL doesn't like our openResource
        #im = Image.open(openResource(value))
        im = Image.open(value)
      except IOError:
        im = Image.new("RGB",(1,1,))

    return im


#############################################################################
#
# ComponentDiplayHandler
#
class ComponentDisplayHandler(FieldDisplayHandler):
  def __init__(self, entry, eventHandler):
    FieldDisplayHandler.__init__(self, entry, eventHandler, None)

  def _buildDisplayHelper(self, value, editing):
    if string.lower(self.entry.type) == 'url':
      return value
