#
# This file is part of GNU Enterprise.
#
# GNU Enterprise is free software; you can redistribute it
# and/or modify it under the terms of the GNU General Public
# License as published by the Free Software Foundation; either
# version 2, or (at your option) any later version.
#
# GNU Enterprise is distributed in the hope that it will be
# useful, but WITHOUT ANY WARRANTY; without even the implied
# warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
# PURPOSE. See the GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public
# License along with program; see the file COPYING. If not,
# write to the Free Software Foundation, Inc., 59 Temple Place
# - Suite 330, Boston, MA 02111-1307, USA.
#
# Copyright 2000-2004 Free Software Foundation
#
# FILE:
# MenuBar.py
#
# DESCRIPTION:
# A generic UIdriver base for common message-based GUI toolkits.
#
# NOTES:
#

import string
from types import StringType

from gnue.forms.uidrivers._base import UserActions

class MenuBar:


  ####################################################################
  #
  # These methods should be superclassed
  # by the toolkit-specific MenuBar
  #
  
  # Smile.  It's menu-building time!
  def init(self): 
    pass

  # Add a (sub)menu
  def addMenu(self, name, parent):
    pass
  
  # Add a menu item (action)
  def addAction(self, name, parent, userAction):
    pass
    
  # Add a separator
  def addSeparator(self, parent): 
    pass
    
  # Enable a menu item
  def enableItem(self, item):
    pass
  
  # Disable a menu item
  def disableItem(self, item): 
    pass


  ####################################################################
  #
  # These methods should not be touched
  # by the toolkit-specific MenuBar.
  # 
  # Or, in the words of the great poet, 
  # M.C. Hammer,  "Can't touch this!"
  #
      
  def __init__(self, uidriver, container, form): 
    self.driver = uidriver
    self.container = container
    self.form = form
    self.enabled = 1
    self.__actionMap = {}
    
    self.finalize()
    
  def disable(self): 
    self.enabled = 0
    
  def finalize(self): 
    if self.enabled: 

      # Let specific drivers create themselves
      m = self.init()

      self.__createmenu(_DEFAULT_MENU,m)


  def __createmenu(self, contents, parent):
    for item in contents:
      if item == None:
        self.addSeparator(parent)
      else:
        name, contents = item
        if type(contents) == StringType: 
          userAction = UserActions.userActionMap[contents]
          self.__actionMap[userAction.event] = self.addAction(name, parent, userAction)
          self.driver.registerEventListeners( 
              {'can%s' %  userAction.event: self.__canRequestEvent,
               'cannot%s' %  userAction.event: self.__cannotRequestEvent })

        else: 
          m = self.addMenu(name, parent)
          self.__createmenu(contents, m)
        
  def __canRequestEvent(self, event): 
    item = self.__actionMap[event.__event__[3:]]
    self.enableItem(item)

  def __cannotRequestEvent(self, event): 
    item = self.__actionMap[event.__event__[6:]]
    self.disableItem(item)




####################################################################
#
#

_DEFAULT_MENU =  (
   ( _('&File'), (
       (_('&Save Changes'), "COMMIT"),
       (_('&Revert Changes'), "ROLLBACK"),
       None,
       (_('&Print'), "PRINTOUT"),
       None,
       (_('&Close'), "EXIT") ) ),

   ( _('&Edit'), (
       (_('C&ut'), "CUT"),
       (_('&Copy'), "COPY"),
#TODO: POST 0.5.0#       ('C&opy Special', (
#TODO: POST 0.5.0#          ('&Record as Text', 'COPYRECORD'),
#TODO: POST 0.5.0#          ('&Screen as Text', 'COPYSCREEN' ) ) ),
       (_('&Paste'), "PASTE"),
       (_('Select &All'), "SELECTALL"),
       None,
       (_('&Insert New Record'),"NEWRECORD"),
       None,
       (_('&Delete Current Record'),'MARKFORDELETE') ) ),

   ( _('&Navigation'), (
       (_('&Next Entry'), "NEXTENTRY"),
       (_('&Previous Entry'), "PREVENTRY"),
       None,
       (_('Next &Record'), "NEXTRECORD"),
       (_('Previous R&ecord'), "PREVRECORD"),
       (_('&First Record'), 'FIRSTRECORD'),
       (_('&Last Record'), 'LASTRECORD'),
       (_('&Go to record...'), 'JUMPPROMPT'),
       None,
       (_('Next &Block'), 'NEXTBLOCK'),
       (_('Previous Bl&ock'), 'PREVBLOCK'),
       None,
       (_('Next P&age'), 'NEXTPAGE'),
       (_('Pre&vious Page'), 'PREVPAGE') ) ),

   ( _('&Search'), (
       (_('Enter Query Mode'), "ENTERQUERY"),
       (_('Cancel Query Mode'), "CANCELQUERY"),       
       None, 
       (_('Repeat Last Query'), "COPYQUERY"),
       None,
       (_('Perform Query'), "EXECQUERY") ) ),

   ( _('&Help'), (
#TODO: POST 0.5.0#       (_('Keystrokes...'), "HELPKEYS"),
       (_('About...'), "ABOUT"), ) ),
  )

