//////////////////////////////////////////////////////////////////
//
// gkauth.cxx
//
// This work is published under the GNU Public License (GPL)
// see file COPYING for details.
// We also explicitely grant the right to link this code
// with the OpenH323 library.
//
// History:
//      2001/09/19      initial version (Chih-Wei Huang)
//
//////////////////////////////////////////////////////////////////

#if (_MSC_VER >= 1200)
#pragma warning( disable : 4786 ) // warning about too long debug symbol off
#pragma warning( disable : 4800 ) // warning about forcing value to bool
#endif

#include <ptlib.h>
#include <h323pdu.h>
#include <h225.h>
#include <h235.h>
#include <h235auth.h>
#include "gkauth.h"
#include "gk_const.h"
#include "h323util.h"
#include "stl_supp.h"
#include "RasTbl.h"
#include "Toolkit.h"

#ifdef P_SOLARIS
#define map stl_map
#endif

#include <map>
#include <list>

using std::map;
using std::list;

const char *GkAuthSectionName = "Gatekeeper::Auth";

GkAuthenticator *GkAuthenticator::head = 0;

static GkAuthInit<GkAuthenticator> _defaultGKA_("default");

//////////////////////////////////////////////////////////////////////
// Definition of authentication rules

class AliasAuth : public GkAuthenticator {
public:
	AliasAuth(PConfig *, const char *);

protected:
	virtual int Check(const H225_GatekeeperRequest &, unsigned &);
	virtual int Check(H225_RegistrationRequest &, unsigned &);
	virtual int Check(const H225_UnregistrationRequest &, unsigned &);
	virtual int Check(const H225_AdmissionRequest &, unsigned &);
	virtual int Check(const H225_BandwidthRequest &, unsigned &);
	virtual int Check(const H225_DisengageRequest &, unsigned &);
	virtual int Check(const H225_LocationRequest &, unsigned &);
	virtual int Check(const H225_InfoRequest &, unsigned &);

	virtual bool doCheck(const H225_ArrayOf_TransportAddress &, const PString &);
	virtual bool AuthCondition(const H225_TransportAddress & SignalAdr, const PString &);

private:
	/** @returns the value for a given alias from section #RasSrv::RRQAuth# 
	    in ini-file
	 */
	virtual PString GetConfigString(const PString & alias) const;
};

static GkAuthInit<AliasAuth> A_A("AliasAuth");

#ifdef HAS_MYSQL

class MySQLAuthBase;

class MySQLPasswordAuth : public SimplePasswordAuth {
public:
	MySQLPasswordAuth(PConfig *, const char *);
	~MySQLPasswordAuth();

private:
	virtual PString GetPassword(const PString &);

	MySQLAuthBase *mysqlconn;
};

static GkAuthInit<MySQLPasswordAuth> M_P_A("MySQLPasswordAuth");

class MySQLAliasAuth : public AliasAuth {
public:
	MySQLAliasAuth(PConfig *, const char *);
	~MySQLAliasAuth();

private:
	virtual PString GetConfigString(const PString & alias) const;

	MySQLAuthBase *mysqlconn;
	CacheManager *cache;
};

static GkAuthInit<MySQLAliasAuth> M_A_A("MySQLAliasAuth");

#endif // HAS_MYSQL

#if ((defined(__GNUC__) && __GNUC__ <= 2) && !defined(WIN32))
#include <unistd.h>
#include <procbuf.h>

class ExternalPasswordAuth : public SimplePasswordAuth {
public:
	ExternalPasswordAuth(PConfig *, const char *);

	virtual PString GetPassword(const PString &);
private:
	bool ExternalInit();

	PString Program;
};

static GkAuthInit<ExternalPasswordAuth> E_P_A("ExternalPasswordAuth");

#endif

// LDAP authentification
#if defined(HAS_LDAP)		// shall use LDAP

#include "gkldap.h"

class LDAPPasswordAuth : public SimplePasswordAuth {
public:
  LDAPPasswordAuth(PConfig *, const char *);
  virtual ~LDAPPasswordAuth();

  virtual PString GetPassword(const PString &alias);
  
  virtual int Check(H225_RegistrationRequest & rrq, unsigned & reason);
};

// ISO 14882:1998 (C++), ISO9899:1999 (C), ISO9945-1:1996 (POSIX) have a
// very clear oppinion regarding user symbols starting or ending with '_'
static GkAuthInit<LDAPPasswordAuth> L_P_A("LDAPPasswordAuth");

class LDAPAliasAuth : public AliasAuth {
public:
	LDAPAliasAuth(PConfig *, const char *);
	virtual ~LDAPAliasAuth();
	
	virtual int Check(H225_RegistrationRequest & rrq, unsigned &);

private:
	/** Searchs for an alias in LDAP and converts it to a valid config
	    string (the expected return value from LDAP is only an IP-address!).
	    @returns config-string (format: see description in ini-file)
	 */
	virtual PString GetConfigString(const PString &alias) const;
};

static GkAuthInit<LDAPAliasAuth> L_A_A("LDAPAliasAuth");

#endif // HAS_LDAP


// Initial author: Michael Rubashenkkov  2002/01/14 (GkAuthorize)
// Completely rewrite by Chih-Wei Huang  2002/05/01
class AuthObj;
class AuthRule;
class PrefixAuth : public GkAuthenticator {
public:
	PrefixAuth(PConfig *, const char *);
	~PrefixAuth();

	typedef std::map< PString, AuthRule *, greater<PString> > Rules;

private:
	virtual int Check(const H225_GatekeeperRequest &, unsigned &);
	virtual int Check(H225_RegistrationRequest &, unsigned &);
	virtual int Check(const H225_UnregistrationRequest &, unsigned &);
	virtual int Check(const H225_AdmissionRequest &, unsigned &); 
	virtual int Check(const H225_BandwidthRequest &, unsigned &);
	virtual int Check(const H225_DisengageRequest &, unsigned &);
	virtual int Check(const H225_LocationRequest &, unsigned &);
	virtual int Check(const H225_InfoRequest &, unsigned &);

	virtual int doCheck(const AuthObj &);
	
	Rules prefrules;
};

static GkAuthInit<PrefixAuth> PF_A("PrefixAuth");


class CacheManager {
public:
	CacheManager(int t) : ttl(t) {}

	bool Retrieve(const PString & key, PString & value);
	void Save(const PString & key, const PString & value);

private:
	map<PString, PString> cache;
	map<PString, PTime> ctime;
	// 0 means don't cache, -1 means never expires
	int ttl; // miliseconds

	CacheManager(const CacheManager &);
	CacheManager & operator=(const CacheManager &);
};      

bool CacheManager::Retrieve(const PString & key, PString & value)
{
	std::map<PString, PString>::iterator iter = cache.find(key);
	if (iter == cache.end())
		return false;
	if (ttl > 0) {
		std::map<PString, PTime>::iterator i = ctime.find(key);
		if ((PTime() - i->second) > ttl)
			return false; // cache expired
	}
	value = iter->second;
	PTRACE(5, "GkAuth\tCache found for " << key);
	return true;
}

void CacheManager::Save(const PString & key, const PString & value)
{
	if (ttl != 0) {
		cache[key] = value;
		ctime[key] = PTime();
	}
}

//////////////////////////////////////////////////////////////////////

GkAuthenticator::GkAuthenticator(PConfig *cfg, const char *authName) 
	: config(cfg), h235Authenticators(NULL), name(authName), checkFlag(e_ALL)
{
	PStringArray control(config->GetString(GkAuthSectionName, name, "").Tokenise(";,"));
	if (PString(name) == "default")
		controlFlag = e_Sufficient,
		defaultStatus = Toolkit::AsBool(control[0]) ? e_ok : e_fail;
	else if (control[0] *= "optional")
		controlFlag = e_Optional, defaultStatus = e_next;
	else if (control[0] *= "required")
		controlFlag = e_Required, defaultStatus = e_fail;
	else
		controlFlag = e_Sufficient, defaultStatus = e_fail;

	if (control.GetSize() > 1) {
		checkFlag = 0;
		map<PString, int> rasmap;
		rasmap["GRQ"] = e_GRQ, rasmap["RRQ"] = e_RRQ,
		rasmap["URQ"] = e_URQ, rasmap["ARQ"] = e_ARQ,
		rasmap["BRQ"] = e_BRQ, rasmap["DRQ"] = e_DRQ,
		rasmap["LRQ"] = e_LRQ, rasmap["IRQ"] = e_IRQ,
		rasmap["Setup"] = e_Setup;
		for (PINDEX i=1; i < control.GetSize(); ++i) {
			if (rasmap.find(control[i]) != rasmap.end())
				checkFlag |= rasmap[control[i]];
		}
	}
	
	next = head;
	head = this;

	PTRACE(1, "GkAuth\tAdd " << name << " rule with flag " << hex << checkFlag << dec);
}

GkAuthenticator::~GkAuthenticator()
{
	PTRACE(1, "GkAuth\tRemove " << name << " rule");
	delete next;  // delete whole list recursively
	delete h235Authenticators;
}

bool GkAuthenticator::CheckRas(
	PBYTEArray &rawPDU, 
	H225_RegistrationRequest& rrq, 
	unsigned& rejectReason,
	const PIPSocket::Address& rxaddr
	)
{
	setLastReceivedRawPDU(rawPDU);
	if (checkFlag & RasValue(rrq)) {
		int r = Check(rrq, rejectReason);
		if (r == e_ok) {
			PTRACE(4, "GkAuth\t" << name << " check ok");
			if (controlFlag != e_Required)
				return true;
		} else if (r == e_fail) {
			PTRACE(2, "GkAuth\t" << name << " check failed");
			return false;
		}
	}
	// try next rule
	return next == NULL || next->CheckRas(rawPDU,rrq,rejectReason,rxaddr);
}

bool GkAuthenticator::CheckRas(
	PBYTEArray &rawPDU, 
	const H225_AdmissionRequest& req, 
	unsigned& rejectReason,
	long& callDurationLimit
	)
{
	callDurationLimit = -1;
	setLastReceivedRawPDU(rawPDU);
	if (checkFlag & RasValue(req)) {
		int r = Check(req, rejectReason, callDurationLimit);
		if( callDurationLimit == 0 ) {
			PTRACE(2,"GkAuth\t" << name << " - call duration limit is 0");
			return false;
		}
		if (r == e_ok) {
			PTRACE(4, "GkAuth\t" << name << " check ok");
			if (controlFlag != e_Required)
				return true;
		} else if (r == e_fail) {
			PTRACE(2, "GkAuth\t" << name << " check failed");
			return false;
		}
	}
	// try next rule
	if( next ) {
		long limit = -1;
		if( !next->CheckRas(rawPDU,req,rejectReason,limit) )
			return false;
		if( callDurationLimit >= 0 && limit >= 0 )
			callDurationLimit = PMIN(limit,callDurationLimit);
		else
			callDurationLimit = PMAX(limit,callDurationLimit);
	}
	if( callDurationLimit == 0 ) {
		PTRACE(2,"GkAuth\t" << name << " - call duration limit is 0");
		return false;
	}
	return true;
}
		
bool GkAuthenticator::CheckSig( 
	/// received Q.931 Setup message
	const Q931& q931pdu, 
	/// received H.225 Setup message
	const H225_Setup_UUIE& setup, 
	/// CallRec for the call being authenticated
	const callptr& call,
	/// Q.931 cause to set, if authentication fails
	unsigned& releaseCompleteCause, 
	/// call duration limit (should be set initially to -1,
	/// what means no limit)
	long& callDurationLimit
	)
{
	callDurationLimit = -1;
	if( checkFlag & e_Setup ) {
		const int r = Check( q931pdu, setup, call, releaseCompleteCause, callDurationLimit );
		if( callDurationLimit == 0 ) {
			PTRACE(2,"GkAuth\t"<<name<<" - call duration limit is 0");
			return false;
		}
		if( r == e_ok ) {
			PTRACE(4,"GkAuth\t"<<name<<" Q.931 Setup check ok");
			if( controlFlag != e_Required )
				return true;
		} else if( r == e_fail ) {
			PTRACE(2,"GkAuth\t"<<name<<" Q.931 Setup check failed");
			return false;
		}
	}
	
	// try next rule
	if( next ) {
		long limit = -1;
		if( !next->CheckSig( q931pdu, setup, call, releaseCompleteCause, limit ) )
			return false;
		if( callDurationLimit >= 0 && limit >= 0 )
			callDurationLimit = PMIN(limit,callDurationLimit);
		else
			callDurationLimit = PMAX(limit,callDurationLimit);
	}
	if( callDurationLimit == 0 ) {
		PTRACE(2,"GkAuth\t"<<name<<" - call duration limit is 0");
		return false;
	}
	return true;
}

int GkAuthenticator::Check(const H225_GatekeeperRequest &, unsigned &)
{
	return defaultStatus;
}

int GkAuthenticator::Check(H225_RegistrationRequest &, unsigned &)
{
	return defaultStatus;
}

int GkAuthenticator::Check(const H225_UnregistrationRequest &, unsigned &)
{
	return defaultStatus;
}

int GkAuthenticator::Check(const H225_AdmissionRequest &, unsigned &)
{
	return defaultStatus;
}

int GkAuthenticator::Check(
	const H225_AdmissionRequest& request, 
	unsigned& rejectReason,
	long& callDurationLimit
	)
{
	// for backward compatibility with derived authenticators
	// that do not understand call duration limit feature
	return Check(request,rejectReason);
}

int GkAuthenticator::Check(const H225_BandwidthRequest &, unsigned &)
{
	return defaultStatus;
}

int GkAuthenticator::Check(const H225_DisengageRequest &, unsigned &)
{
	return defaultStatus;
}

int GkAuthenticator::Check(const H225_LocationRequest &, unsigned &)
{
	return defaultStatus;
}

int GkAuthenticator::Check(const H225_InfoRequest &, unsigned &)
{
	return defaultStatus;
}

int GkAuthenticator::Check( const Q931&, const H225_Setup_UUIE&, const callptr&, unsigned&, long& )
{
	return defaultStatus;
}

BOOL GkAuthenticator::GetH235Capability(
	H225_ArrayOf_AuthenticationMechanism& mechanisms,
	H225_ArrayOf_PASN_ObjectId& algorithmOIDs
	) const
{
	if( h235Authenticators == NULL )
		return FALSE;
		
	for( int i = 0; i < h235Authenticators->GetSize(); i++ )
		(*h235Authenticators)[i].SetCapability(mechanisms,algorithmOIDs);
		
	return TRUE;
}		

BOOL GkAuthenticator::IsH235Capability(
	const H235_AuthenticationMechanism& mechanism,
	const PASN_ObjectId& algorithmOID
	) const
{
	if( h235Authenticators == NULL )
		return FALSE;
		
	for( int i = 0; i < h235Authenticators->GetSize(); i++ )
		if( (*h235Authenticators)[i].IsCapability(mechanism,algorithmOID) )
			return TRUE;
		
	return FALSE;
}

BOOL GkAuthenticator::IsH235Capable() const
{
	return (h235Authenticators != NULL) && (h235Authenticators->GetSize() > 0);
}

static GkAuthInit<SimplePasswordAuth> S_P_A("SimplePasswordAuth");

const char *passwdsec = "Password";

// SimplePasswordAuth
SimplePasswordAuth::SimplePasswordAuth(PConfig *cfg, const char *authName)
      : GkAuthenticator(cfg, authName), aliases(0)
{
	filled = config->GetInteger(passwdsec, "KeyFilled", 0);
	checkid = Toolkit::AsBool(config->GetString(passwdsec, "CheckID", "0"));
	cache = new CacheManager(config->GetInteger(passwdsec, "PasswordTimeout", -1) * 1000);
	
	if( h235Authenticators == NULL )
		h235Authenticators = new H235Authenticators;
		
	H235Authenticator* authenticator;
	
	authenticator = new H235AuthSimpleMD5;
	authenticator->SetLocalId("dummy");
	authenticator->SetRemoteId("dummy");
	authenticator->SetPassword("dummy");
	h235Authenticators->Append(authenticator);
#ifdef OPENH323_NEWVERSION
	authenticator = new H235AuthCAT;
	authenticator->SetLocalId("dummy");
	authenticator->SetRemoteId("dummy");
	authenticator->SetPassword("dummy");
	h235Authenticators->Append(authenticator);
#endif
#ifdef P_SSL
	authenticator = new H235AuthProcedure1;
	authenticator->SetLocalId("dummy");
	authenticator->SetRemoteId("dummy");
	authenticator->SetPassword("dummy");
	h235Authenticators->Append(authenticator);
#endif
}

SimplePasswordAuth::~SimplePasswordAuth()
{
	delete cache;
}

int SimplePasswordAuth::Check(const H225_GatekeeperRequest & grq, unsigned &)
{
	return doCheck(grq);
}

int SimplePasswordAuth::Check(H225_RegistrationRequest & rrq, unsigned &)
{
	if (checkid) {
		if (!rrq.HasOptionalField(H225_RegistrationRequest::e_terminalAlias))
			return e_fail;
		aliases = &rrq.m_terminalAlias;
	}
	return doCheck(rrq);
}

int SimplePasswordAuth::Check(const H225_UnregistrationRequest & urq, unsigned &)
{
	return doCheck(urq);
}

int SimplePasswordAuth::Check(const H225_AdmissionRequest & arq, unsigned &)
{
	return doCheck(arq);
}

int SimplePasswordAuth::Check(const H225_BandwidthRequest & brq, unsigned &)
{
	return doCheck(brq);
}

int SimplePasswordAuth::Check(const H225_DisengageRequest & drq, unsigned &)
{
	return doCheck(drq);
}

int SimplePasswordAuth::Check(const H225_LocationRequest & lrq, unsigned &)
{
	return doCheck(lrq);
}

int SimplePasswordAuth::Check(const H225_InfoRequest & drq, unsigned &)
{
	return doCheck(drq);
}

PString SimplePasswordAuth::GetPassword(const PString & id)
{
	return Toolkit::CypherDecode(id, config->GetString(passwdsec, id, ""), filled);
}

bool SimplePasswordAuth::InternalGetPassword(const PString & id, PString & passwd)
{
	bool result = cache->Retrieve(id, passwd);
	if (!result)
		passwd = GetPassword(id);
	return result;
}

bool SimplePasswordAuth::CheckAliases(const PString & id)
{
	bool r = false;
	for (PINDEX i = 0; i < aliases->GetSize(); i++)
		if (H323GetAliasAddressString((*aliases)[i]) == id) {
			r = true;
			break;
		}
	aliases = 0;
	return r;
}

static const char OID_CAT[] = "1.2.840.113548.10.1.2.1";

bool SimplePasswordAuth::CheckTokens(const H225_ArrayOf_ClearToken & tokens)
{
	for (PINDEX i=0; i < tokens.GetSize(); ++i) {
		H235_ClearToken & token = tokens[i];
#ifdef OPENH323_NEWVERSION
		// check for Cisco Access Token
		if( token.m_tokenOID == OID_CAT )
		{
			if( !token.HasOptionalField(H235_ClearToken::e_generalID) )
				return false;
				
			PString id = token.m_generalID;
			if (aliases && !CheckAliases(id))
				return false;
				
			PString passwd;
			bool cached = InternalGetPassword(id, passwd);

			H235AuthCAT authCAT;
			authCAT.SetLocalId(id);
			authCAT.SetPassword(passwd);
			if (authCAT.ValidateClearToken(token) == H235Authenticator::e_OK) {
				PTRACE(4, "GkAuth\t" << id << " password match (CAT)");
				if (!cached)
					cache->Save(id, passwd);
				return true;
			}
			return false;
		}
#endif
		if (token.HasOptionalField(H235_ClearToken::e_generalID) &&
		    token.HasOptionalField(H235_ClearToken::e_password)) {
			PString id = token.m_generalID;
			if (aliases && !CheckAliases(id))
				return false;
			PString passwd, tokenpasswd = token.m_password;
			bool cached = InternalGetPassword(id, passwd);
			if (passwd == tokenpasswd) {
				PTRACE(4, "GkAuth\t" << id << " password match");
				if (!cached)
					cache->Save(id, passwd);
				return true;
			}
		}
	}
	return false;
}

bool SimplePasswordAuth::CheckCryptoTokens(const H225_ArrayOf_CryptoH323Token & tokens)
{
	for (PINDEX i = 0; i < tokens.GetSize(); ++i) {
		if (tokens[i].GetTag() == H225_CryptoH323Token::e_cryptoEPPwdHash) {
			H225_CryptoH323Token_cryptoEPPwdHash & pwdhash = tokens[i];
			PString id = AsString(pwdhash.m_alias, FALSE);
			if (aliases && !CheckAliases(id))
				return false;
			PString passwd;
			bool cached = InternalGetPassword(id, passwd);

			H235AuthSimpleMD5 authMD5;
			authMD5.SetLocalId(id);
			authMD5.SetPassword(passwd);
			PBYTEArray nullPDU;
#ifdef OPENH323_NEWVERSION
			if (authMD5.ValidateCryptoToken(tokens[i], nullPDU) == H235Authenticator::e_OK) {
#else
			if (authMD5.VerifyToken(tokens[i], nullPDU) == H235Authenticator::e_OK) {
#endif
				PTRACE(4, "GkAuth\t" << id << " password match (MD5)");
				if (!cached)
					cache->Save(id, passwd);
				return true;
			}
#ifdef P_SSL
		} else if (tokens[i].GetTag() == H225_CryptoH323Token::e_nestedcryptoToken){
			H235_CryptoToken & nestedCryptoToken = tokens[i];
			H235_CryptoToken_cryptoHashedToken & cryptoHashedToken = nestedCryptoToken;
			H235_ClearToken & clearToken = cryptoHashedToken.m_hashedVals;
			PString gk_id = clearToken.m_generalID;
			//assumption: sendersID == endpoint alias (RRQ)
			PString ep_alias = clearToken.m_sendersID; 
			if (aliases && !CheckAliases(ep_alias))
				return false;
			PString passwd;
			bool cached = InternalGetPassword(ep_alias, passwd);
			//if a password is not found: senderID == endpointIdentifier?
			if (passwd.IsEmpty()){
			 	//get endpoint by endpointIdentifier
				H225_EndpointIdentifier ep_id;
				ep_id = clearToken.m_sendersID;
				endptr ep = RegistrationTable::Instance()->FindByEndpointId(ep_id);
				if(!ep){
					return false;
				}
				//check all endpoint aliases for a password
				H225_ArrayOf_AliasAddress ep_aliases = ep->GetAliases();
				for (PINDEX i = 0; i < ep_aliases.GetSize(); i++){
					ep_alias = H323GetAliasAddressString(ep_aliases[i]);
					cached = InternalGetPassword(ep_alias, passwd);
					if (!passwd)
						break;
				}
			}
			H235AuthProcedure1 authProcedure1;
			authProcedure1.SetLocalId(gk_id);
			authProcedure1.SetPassword(passwd);
#ifdef OPENH323_NEWVERSION
			if (authProcedure1.ValidateCryptoToken(tokens[i], getLastReceivedRawPDU()) == H235Authenticator::e_OK) {
#else
			if (authProcedure1.VerifyToken(tokens[i], getLastReceivedRawPDU()) == H235Authenticator::e_OK) {
#endif
				PTRACE(4, "GkAuth\t" << ep_alias << " password match (SHA-1)");
				if (!cached)
					cache->Save(ep_alias, passwd);
				return true;
			}
#endif
		}
	}
	return false;
}


#ifdef HAS_MYSQL
#define MYSQL_NO_SHORT_NAMES  // use long names
#include <mysql++>

class MySQLAuthBase {
public:
	MySQLAuthBase(PConfig *,
		const char *section,
		const char *host,
		const char *dbname,
		const char *user,
		const char *passwd,
		const char *table,
		const char *alias,
		const char *query,
		const char *extra
	);
	~MySQLAuthBase();
	bool Exec(const PString &, MysqlRes &);
	PString GetString(const PString &);

private:
	bool MySQLInit();
	void Cleanup();

	MysqlConnection *mysql_connection;
	MysqlQuery *mysql_query;

	PConfig *config;
	const char *section_n;
	const char *host_n, *dbname_n, *user_n, *passwd_n;
	const char *table_n, *alias_n, *query_n, *extra_n;
};

MySQLAuthBase::MySQLAuthBase(PConfig *cfg,
		const char *section,
		const char *host,
		const char *dbname,
		const char *user,
		const char *passwd,
		const char *table,
		const char *alias,
		const char *query,
		const char *extra
	) : mysql_connection(0), mysql_query(0),
	    config(cfg), section_n(section),
	    host_n(host), dbname_n(dbname), user_n(user), passwd_n(passwd),
	    table_n(table), alias_n(alias), query_n(query), extra_n(extra)
{
	MySQLInit();
}

MySQLAuthBase::~MySQLAuthBase()
{
	Cleanup();
}

bool MySQLAuthBase::Exec(const PString & id, MysqlRes & result)
{
	if (mysql_connection || MySQLInit()) {
		try {
			result = mysql_query->store(SQLString(id));
			return true;
		} catch (MysqlBadQuery er) {
			PTRACE(1, "MySQL\tBadQuery: " << er.error);
			Cleanup();
		} catch (MysqlBadConversion er) {
			PTRACE(1,  "MySQL\tBadConversion: Tried to convert \"" << er.data << "\" to a \"" << er.type_name << "\".");
		} 
	}
	return false;
}

PString MySQLAuthBase::GetString(const PString & id)
{
	PString str;
	MysqlRes result;
	if (Exec(id, result) && !result.empty())
		str = (*result.begin())[0].c_str();
	return str;
}

bool MySQLAuthBase::MySQLInit()
{
	try {
		PString host = config->GetString(section_n, host_n, "localhost");
		PString dbname = config->GetString(section_n, dbname_n, "billing");
		PString user = config->GetString(section_n, user_n, "cwhuang");
		PString passwd = config->GetString(section_n, passwd_n, "123456");

		PString table = config->GetString(section_n, table_n, "customer");
		PString alias = config->GetString(section_n, alias_n, "IPN");
		PString query = config->GetString(section_n, query_n, "Password");
		PString extra = config->GetString(section_n, extra_n, "");

		mysql_connection = new MysqlConnection(mysql_use_exceptions);
		mysql_connection->connect(dbname, host, user, passwd);

		PTRACE(2, "MySQL\tConnect to server " << host << ", database " << dbname);
		mysql_query = new MysqlQuery(mysql_connection, true);
		PString select_clause(PString::Printf,
			"select %s from %s where %s = '%%0:id'",
			(const char *)query,
			(const char *)table,
			(const char *)alias
		);
		if (!extra)
			select_clause += " and " + extra;
		PTRACE(2, "MySQL\t" << select_clause);

		*mysql_query << select_clause;
		mysql_query->parse();
		PTRACE(1, "MySQL\tReady for query");
		return true;
	} catch (MysqlBadQuery er) { // any error?
		PTRACE(1, "MySQL\tError: " << er.error);
		Cleanup();
		return false;
	}
}

void MySQLAuthBase::Cleanup()
{
	delete mysql_query;
	mysql_query = 0;
	delete mysql_connection;
	mysql_connection = 0; // disable the authenticator
}

// MySQLPasswordAuth
MySQLPasswordAuth::MySQLPasswordAuth(PConfig *cfg, const char *authName)
	: SimplePasswordAuth(cfg, authName)
{
	mysqlconn = new MySQLAuthBase(cfg, "MySQLAuth",
				"Host", "Database", "User", "Password",
				"Table", "IDField", "PasswordField", "ExtraCriterion"
			);
}

MySQLPasswordAuth::~MySQLPasswordAuth()
{
	delete mysqlconn;
}

PString MySQLPasswordAuth::GetPassword(const PString & id)
{
	return mysqlconn->GetString(id);
}

// MySQLAliasAuth
MySQLAliasAuth::MySQLAliasAuth(PConfig *cfg, const char *authName)
	: AliasAuth(cfg, authName)
{
	const char *secname = "MySQLAliasAuth";
	mysqlconn = new MySQLAuthBase(cfg, secname,
				"Host", "Database", "User", "Password",
				"Table", "IDField", "IPField", "ExtraCriterion"
			);
	cache = new CacheManager(config->GetInteger(secname, "CacheTimeout", -1) * 1000);
}

MySQLAliasAuth::~MySQLAliasAuth()
{
	delete mysqlconn;
	delete cache;
}

PString MySQLAliasAuth::GetConfigString(const PString & alias) const
{
	PString result;
	if (!cache->Retrieve(alias, result)) {
		result = mysqlconn->GetString(alias);
		if (!result)
			cache->Save(alias, result);
	}
	return result;
}

#endif // HAS_MYSQL

#if ((defined(__GNUC__) && __GNUC__ <= 2) && !defined(WIN32))
// ExternalPasswordAuth

ExternalPasswordAuth::ExternalPasswordAuth(PConfig * cfg, const char * authName)
      : SimplePasswordAuth(cfg, authName)
{
	ExternalInit();
}

bool ExternalPasswordAuth::ExternalInit()
{
	const char *ExternalSec = "ExternalAuth";

	// Read the configuration
	Program = config->GetString(ExternalSec, "PasswordProgram", "");
	
	return true;
}

PString ExternalPasswordAuth::GetPassword(const PString & id)
{
	const int BUFFSIZE = 256;
	char buff[BUFFSIZE] = "";
	if (!Program) {
		procbuf proc(Program + " " + id, ios::in);
		istream istr(&proc);
		istr.getline(buff, BUFFSIZE);
	} else {
		PTRACE(2, "GkAuth\tProgram is not defined");
	}
	return PString(buff);
}

#endif // class ExternalPasswordAuth

// LDAP authentification
#if defined(HAS_LDAP)

LDAPPasswordAuth::LDAPPasswordAuth(PConfig * cfg, const char * authName)
  : SimplePasswordAuth(cfg, authName)
{
}

LDAPPasswordAuth::~LDAPPasswordAuth()
{
}

PString LDAPPasswordAuth::GetPassword(const PString & alias)
{
	PStringList attr_values;
	using namespace lctn; // LDAP config tags and names
	// get pointer to new answer object
	if(GkLDAP::Instance()->getAttribute(alias, H235PassWord, attr_values) && 
	   !attr_values.IsEmpty()){
		return attr_values[0];
	}
	return "";
}  

int LDAPPasswordAuth::Check(H225_RegistrationRequest & rrq, unsigned & reason)
{
	int result = SimplePasswordAuth::Check(rrq, reason);
	if(result == e_ok) {
		// check if all aliases in RRQ exists in LDAP entry
		const H225_ArrayOf_AliasAddress & aliases = rrq.m_terminalAlias;
		if(!GkLDAP::Instance()->validAliases(aliases)) {
			result = e_fail;
		}
	}
	return result;
}


LDAPAliasAuth::LDAPAliasAuth(PConfig *cfg, const char *authName) : AliasAuth(cfg, authName)
{
}

LDAPAliasAuth::~LDAPAliasAuth()
{
}

PString LDAPAliasAuth::GetConfigString(const PString &alias) const
{
	PStringList attr_values;
	using namespace lctn; // LDAP config tags and names
	// get pointer to new answer object
	if (GkLDAP::Instance()->getAttribute(alias, IPAddress, attr_values) && (!attr_values.IsEmpty())) {
		PString ip = attr_values[0];
    		if(!ip.IsEmpty()){
      			PString port = GK_DEF_ENDPOINT_SIGNAL_PORT;    
			return "sigip:" + ip + ":" + port;
		}
	}
	return "";
}

int LDAPAliasAuth::Check(H225_RegistrationRequest & rrq, unsigned & reason)
{
	int result = AliasAuth::Check(rrq, reason);
	if(result == e_ok) {
		// check if all aliases in RRQ exists in LDAP entry
		const H225_ArrayOf_AliasAddress & aliases = rrq.m_terminalAlias;
		if(!GkLDAP::Instance()->validAliases(aliases)) {
      			result = e_fail;
    		}
	}
  	return result;
}

#endif // HAS_LDAP


// AliasAuth
AliasAuth::AliasAuth(PConfig *cfg, const char *authName) : GkAuthenticator(cfg, authName)
{
}

int AliasAuth::Check(const H225_GatekeeperRequest &, unsigned &)
{
	return e_next;
}

int AliasAuth::Check(H225_RegistrationRequest & rrq, unsigned &)
{
	bool AliasFoundInConfig = false;

	if (!rrq.HasOptionalField(H225_RegistrationRequest::e_terminalAlias))
		return defaultStatus;

	const H225_ArrayOf_AliasAddress & NewAliases = rrq.m_terminalAlias;

	// alias is the config file entry of this endpoint
	for (PINDEX i = 0; i <= NewAliases.GetSize(); ++i) {
		const PString alias = (i < NewAliases.GetSize()) ? AsString(NewAliases[i], FALSE) : PString("default");
		const PString cfgString = GetConfigString(alias);
		if (!cfgString) {
			if (doCheck(rrq.m_callSignalAddress, cfgString)) {
				AliasFoundInConfig = true;
				break;
			} else {
				PTRACE(4, "Gk\tRRQAuth condition '" << cfgString << "' rejected endpoint " << alias);
				return e_fail;
			}
		}
	}
	return (AliasFoundInConfig) ? e_ok : defaultStatus;
}

int AliasAuth::Check(const H225_UnregistrationRequest &, unsigned &)
{
	return e_next;
}

int AliasAuth::Check(const H225_AdmissionRequest &, unsigned &)
{
	return e_next;
}

int AliasAuth::Check(const H225_BandwidthRequest &, unsigned &)
{
	return e_next;
}

int AliasAuth::Check(const H225_DisengageRequest &, unsigned &)
{
	return e_next;
}

int AliasAuth::Check(const H225_LocationRequest &, unsigned &)
{
	return e_next;
}

int AliasAuth::Check(const H225_InfoRequest &, unsigned &)
{
	return e_next;
}

PString AliasAuth::GetConfigString(const PString & alias) const
{
	return config->GetString("RasSrv::RRQAuth", alias, "");
}

bool AliasAuth::doCheck(const H225_ArrayOf_TransportAddress & addrs, const PString & cfgString)
{
	const PStringArray conditions(cfgString.Tokenise("&|", FALSE));
	for (PINDEX i = 0; i < conditions.GetSize(); ++i) {
		for (PINDEX j = 0; j < addrs.GetSize(); ++j) {
			if (AuthCondition(addrs[j], conditions[i])) {
				PTRACE(4, "Gk\tRRQAuth condition '" << conditions[i] << "' applied successfully for endpoint " << AsDotString(addrs[j]));
				return true;
			}
		}
	}
	return false;
}

bool AliasAuth::AuthCondition(const H225_TransportAddress & SignalAdr, const PString & Condition)
{
	const bool ON_ERROR = false; // return value on parse error in condition

	const PStringArray rule = Condition.Tokenise(":", FALSE);
	if (rule.GetSize() < 1) {
		PTRACE(1, "Errornous RRQAuth rule: " << Condition);
		return ON_ERROR;
	}
	
	// 
	// condition = rule[0]:rule[1]... = rName:params...
	//
	
	const PString &rName = rule[0];

 	if (rName=="confirm" || rName=="allow") {
 		return true;
 	}
 	else if (rName=="reject" || rName=="deny" || rName=="forbid") {
 		return false;
 	}
	//
	// condition 'sigaddr' example:
	//   sigaddr:.*ipAddress .* ip = .* c3 47 e2 a2 .*port = 1720.*
	//
	else if (rName=="sigaddr") {
		if(rule.GetSize() < 2)
			return false;
		return Toolkit::MatchRegex(AsString(SignalAdr), rule[1]) != 0;
	}
	//
	// condition 'sigip' example:
	//   sigip:195.71.129.69:1720
	//
	else if (rName=="sigip") {
		if (rule.GetSize() < 2)
			return false;
		PIPSocket::Address ip;
		PIPSocket::GetHostAddress(rule[1], ip);
		WORD port = (rule.GetSize() < 3) ? GK_DEF_ENDPOINT_SIGNAL_PORT : rule[2].AsInteger();
		return (SignalAdr == SocketToH225TransportAddr(ip, port));
	} else {
		PTRACE(4, "Unknown RRQAuth condition: " << Condition);
		return ON_ERROR;
	}

	// not reached...
	return false;
}

// Help classes for PrefixAuth
static const char* const prfflag="prf:";
static const char* const allowflag="allow";
static const char* const denyflag="deny";
static const char* const ipflag="ipv4:";
static const char* const aliasflag="alias:";

class AuthObj { // abstract class
public:
	virtual ~AuthObj() {}

	virtual bool IsValid() const { return true; }

	virtual PStringArray GetPrefixes() const = 0;

	virtual PIPSocket::Address GetIP() const = 0;
	virtual PString GetAliases() const = 0;
};

class RRQAuthObj : public AuthObj {
public:
	RRQAuthObj(const H225_RegistrationRequest & ras) : rrq(ras) {}

	virtual PStringArray GetPrefixes() const;

	virtual PIPSocket::Address GetIP() const;
	virtual PString GetAliases() const;

private:
	const H225_RegistrationRequest & rrq;
};

class ARQAuthObj : public AuthObj {
public:
	ARQAuthObj(const H225_AdmissionRequest & ras);

	virtual bool IsValid() const { return ep; }

	virtual PStringArray GetPrefixes() const;

	virtual PIPSocket::Address GetIP() const;
	virtual PString GetAliases() const;

private:
	const H225_AdmissionRequest & arq;
	endptr ep;
};

ARQAuthObj::ARQAuthObj(const H225_AdmissionRequest & ras) : arq(ras)
{
	ep = RegistrationTable::Instance()->FindByEndpointId(arq.m_endpointIdentifier);
}

PStringArray ARQAuthObj::GetPrefixes() const
{
	PStringArray array;
	if (arq.HasOptionalField(H225_AdmissionRequest::e_destinationInfo))
		if (PINDEX ss = arq.m_destinationInfo.GetSize() > 0) {
			array.SetSize(ss);
			for (PINDEX i = 0; i < ss; ++i)
				array[i] = AsString(arq.m_destinationInfo[i], FALSE);
		}
	if (array.GetSize() == 0)
		// let empty destinationInfo match the ALL rule
		array.SetSize(1);

	return array;
}

PIPSocket::Address ARQAuthObj::GetIP() const
{
	PIPSocket::Address result;
	const H225_TransportAddress & addr = (arq.HasOptionalField(H225_AdmissionRequest::e_srcCallSignalAddress)) ?
		arq.m_srcCallSignalAddress : ep->GetCallSignalAddress();
	GetIPFromTransportAddr(addr, result);
	return result;
}

PString ARQAuthObj::GetAliases() const
{
	return AsString(ep->GetAliases());
}

class LRQAuthObj : public AuthObj {
public:
	LRQAuthObj(const H225_LocationRequest & ras);

	virtual PStringArray GetPrefixes() const;

	virtual PIPSocket::Address GetIP() const;
	virtual PString GetAliases() const;

private:
	const H225_LocationRequest & lrq;
	PIPSocket::Address ipaddress;
};

LRQAuthObj::LRQAuthObj(const H225_LocationRequest & ras) : lrq(ras)
{
	GetIPFromTransportAddr(lrq.m_replyAddress, ipaddress);
}

PStringArray LRQAuthObj::GetPrefixes() const
{
	PStringArray array;
	if (PINDEX ss = lrq.m_destinationInfo.GetSize() > 0) {
		array.SetSize(ss);
		for (PINDEX i = 0; i < ss; ++i)
			array[i] = AsString(lrq.m_destinationInfo[i], FALSE);
	}
	return array;
}

PIPSocket::Address LRQAuthObj::GetIP() const
{
	return ipaddress;
}

PString LRQAuthObj::GetAliases() const
{
	return (lrq.HasOptionalField(H225_LocationRequest::e_sourceInfo)) ? AsString(lrq.m_sourceInfo) : PString();
}


class AuthRule {
public:
	enum Result {
		e_nomatch,
		e_allow,
		e_deny
	};

	AuthRule(Result f, bool r) : priority(1000), fate(f), inverted(r), next(0) {}
	virtual ~AuthRule() { delete next; }

	virtual bool Match(const AuthObj &) = 0;
	int Check(const AuthObj &);
	
	bool operator<(const AuthRule & o) const { return priority < o.priority; }
	void SetNext(AuthRule *n) { next = n; }

//	virtual PString GetName() const { return PString(); }

protected:
	int priority; // the lesser the value, the higher the priority

private:
	Result fate;
	bool inverted;
	AuthRule *next;
};

int AuthRule::Check(const AuthObj & aobj)
{
//	PTRACE(3, "auth\t" << GetName());
	return (Match(aobj) ^ inverted) ? fate : (next) ? next->Check(aobj) : e_nomatch;
}

inline void delete_rule(PrefixAuth::Rules::value_type r)
{
	delete r.second;
}

class NullRule : public AuthRule {
public:
	NullRule() : AuthRule(e_nomatch, false) {}
	virtual bool Match(const AuthObj &) { return false; }
};

class IPv4AuthRule : public AuthRule {
public:
	IPv4AuthRule(Result, const PString &, bool);

private:
	virtual bool Match(const AuthObj &);
//	virtual PString GetName() const { return network.AsString() + "/" + PString(PString::Unsigned, 32-priority); }

	PIPSocket::Address network, netmask;
};

IPv4AuthRule::IPv4AuthRule(Result f, const PString & cfg, bool r) : AuthRule(f, r)
{
	Toolkit::GetNetworkFromString(cfg, network, netmask);
	DWORD n = ~PIPSocket::Net2Host(DWORD(netmask));
	for (priority = 0; n; n >>= 1)
		++priority;
}

bool IPv4AuthRule::Match(const AuthObj & aobj)
{
	return ((aobj.GetIP() & netmask) == network);
}

class AliasAuthRule : public AuthRule {
public:
	AliasAuthRule(Result f, const PString & cfg, bool r) : AuthRule(f, r), pattern(cfg) { priority = -1; }

private:
	virtual bool Match(const AuthObj &);
//	virtual PString GetName() const { return pattern; }

	PString pattern;
};

bool AliasAuthRule::Match(const AuthObj & aobj)
{
	return (aobj.GetAliases().FindRegEx(pattern) != P_MAX_INDEX);
}

inline bool is_inverted(const PString & cfg, PINDEX p)
{
	return (p > 1) ? cfg[p-1] == '!' : false;
}

inline bool comp_authrule_priority(AuthRule *a1, AuthRule *a2)
{
	return *a1 < *a2;
}

// PrefixAuth
PrefixAuth::PrefixAuth(PConfig *cfg, const char *authName)
      : GkAuthenticator(cfg, authName)
{
	int ipfl = strlen(ipflag), aliasfl = strlen(aliasflag);
	PStringToString cfgs=cfg->GetAllKeyValues("PrefixAuth");
	for (PINDEX i = 0; i < cfgs.GetSize(); ++i) {
		PString key = cfgs.GetKeyAt(i);
		if (key *= "default") {
			defaultStatus = Toolkit::AsBool(cfgs.GetDataAt(i)) ? e_ok : e_fail;
			continue;
		} else if (key *= "ALL") {
			// use space (0x20) as the key so it will be the last resort
			key = " ";
		}
		if (prefrules.find(key) != prefrules.end())
			continue; //rule already exists? ignore

		PStringArray rules = cfgs.GetDataAt(i).Tokenise("|", FALSE);
		PINDEX sz = rules.GetSize();
		if (sz < 1)
			continue;
		//AuthRule *rls[sz];
		AuthRule **rls = new AuthRule *[sz];
		for (PINDEX j = 0; j < sz; ++j) {
			PINDEX pp;
			// if not allowed, assume denial
			AuthRule::Result ft = (rules[j].Find(allowflag) != P_MAX_INDEX) ? AuthRule::e_allow : AuthRule::e_deny;
			if ((pp=rules[j].Find(ipflag)) != P_MAX_INDEX)
				rls[j] = new IPv4AuthRule(ft, rules[j].Mid(pp+ipfl), is_inverted(rules[j], pp));
			else if ((pp=rules[j].Find(aliasflag)) != P_MAX_INDEX)
				rls[j] = new AliasAuthRule(ft, rules[j].Mid(pp+aliasfl), is_inverted(rules[j], pp));
			else
				rls[j] = new NullRule;
		}

		// sort the rules by priority
		stable_sort(rls, rls + sz, comp_authrule_priority);
		for (PINDEX k = 1; k < sz; ++k)
			rls[k-1]->SetNext(rls[k]);
		prefrules[key] = rls[0];
		delete [] rls;
	}
}

PrefixAuth::~PrefixAuth()
{
	for_each(prefrules.begin(), prefrules.end(), delete_rule);
}

int PrefixAuth::Check(const H225_GatekeeperRequest &, unsigned &)
{
	return e_next;
}

int PrefixAuth::Check(H225_RegistrationRequest & rrq, unsigned &)
{
	return e_next;
}

int PrefixAuth::Check(const H225_UnregistrationRequest &, unsigned &)
{
	return e_next;
}

int PrefixAuth::Check(const H225_AdmissionRequest & arq, unsigned &)
{
	return CallTable::Instance()->FindCallRec(arq.m_callIdentifier) ? e_ok : doCheck(ARQAuthObj(arq));
}

int PrefixAuth::Check(const H225_BandwidthRequest &, unsigned &)
{
	return e_next;
}

int PrefixAuth::Check(const H225_DisengageRequest &, unsigned &)
{
	return e_next;
}

int PrefixAuth::Check(const H225_LocationRequest & lrq, unsigned &)
{
	return doCheck(LRQAuthObj(lrq));
}

int PrefixAuth::Check(const H225_InfoRequest &, unsigned &)
{
	return e_next;
}

struct comp_pref { // function object
	comp_pref(const PString & s) : value(s) {}
	bool operator()(const PrefixAuth::Rules::value_type & v) const;
	const PString & value;
};

inline bool comp_pref::operator()(const PrefixAuth::Rules::value_type & v) const
{
	return (value.Find(v.first) == 0) || (v.first *= " ");
}

int PrefixAuth::doCheck(const AuthObj & aobj)
{
	if (!aobj.IsValid())
		return e_fail;
	PStringArray ary(aobj.GetPrefixes());
	for (PINDEX i = 0; i < ary.GetSize(); ++i) {
		// find the first match rule
		// since prefrules is descendently sorted
		// it must be the most specific prefix
		for (Rules::iterator j = prefrules.begin(); j != prefrules.end(); ++j) {
			Rules::iterator iter = find_if(j, prefrules.end(), comp_pref(ary[i]));
			if (iter == prefrules.end())
				break;
			switch (iter->second->Check(aobj))
			{
				case AuthRule::e_allow:
					return e_ok;
				case AuthRule::e_deny:
					return e_fail;
				default: // try next prefix...
					j = iter;
			}
		}
	}
	return defaultStatus;
}


static list<GkAuthInitializer *> *AuthNameList;

GkAuthInitializer::GkAuthInitializer(const char *n) : name(n)
{
	static list<GkAuthInitializer *> aList;
	AuthNameList = &aList;

	AuthNameList->push_back(this);
}

GkAuthInitializer::~GkAuthInitializer()
{
}

bool GkAuthInitializer::Compare(PString n) const
{
	return n == name;
}

GkAuthenticatorList::GkAuthenticatorList(PConfig *cfg)
{
	PStringList authList(cfg->GetKeys(GkAuthSectionName));

	for (PINDEX i=authList.GetSize(); i-- > 0; ) {
		PString authName(authList[i]);
		std::list<GkAuthInitializer *>::iterator Iter =
			find_if(AuthNameList->begin(), AuthNameList->end(),
				bind2nd(mem_fun(&GkAuthInitializer::Compare), authName));
		if (Iter != AuthNameList->end())
			(*Iter)->CreateAuthenticator(cfg);
#if PTRACING
		else
			PTRACE(1, "GkAuth\tUnknown auth " << authName << ", ignore!");
#endif
	}
	
	m_mechanisms = new H225_ArrayOf_AuthenticationMechanism;
	m_algorithmOIDs = new H225_ArrayOf_PASN_ObjectId;
	GkAuthenticator* authenticator = GetHead();
	BOOL found = FALSE;
			
	// scan all authenticators that are either "required" or "sufficient"
	// (skip "optional") and fill #mechanisms# and #algorithmOIDs# arrays
	// with H.235 capabilities that are supported by all these authenticators
	while( authenticator )
	{
		if( authenticator->IsH235Capable() 
			&& ((authenticator->GetControlFlag() == GkAuthenticator::e_Required)
				|| (authenticator->GetControlFlag() == GkAuthenticator::e_Sufficient)) )
		{
			if( m_mechanisms->GetSize() == 0 )
			{
				// append H.235 capability to empty arrays
				authenticator->GetH235Capability(
					*m_mechanisms, *m_algorithmOIDs
					);
				// should never happen, but we should check just for a case				
				if( m_algorithmOIDs->GetSize() == 0 )
					m_mechanisms->RemoveAll();
				else
					found = TRUE;
				authenticator = authenticator->GetNext();
				continue;
			}
			
			// Already have H.235 capabilities - check the current
			// authenticator if it supports any of the capabilities.
			// Remove capabilities that are not supported
			
			H225_ArrayOf_AuthenticationMechanism matchedMechanisms;
			
			int i, j, k;
			
			for( i = 0; i < m_algorithmOIDs->GetSize(); i++ )
			{
				BOOL matched = FALSE;
			
				for( j = 0; j < m_mechanisms->GetSize(); j++ )
					if( authenticator->IsH235Capability(
							(*m_mechanisms)[j], (*m_algorithmOIDs)[i]
							) )
					{
						for( k = 0; k < matchedMechanisms.GetSize(); k++ )
							if( matchedMechanisms[k].GetTag() == (*m_mechanisms)[j].GetTag() )
								break;
						if( k == matchedMechanisms.GetSize() )
						{
							matchedMechanisms.SetSize(k+1);
							matchedMechanisms[k].SetTag((*m_mechanisms)[j].GetTag());
						}
						matched = TRUE;
					}
				
				if( !matched )
				{
					PTRACE(5,"GkAuth\tRemoved from GCF list algorithm OID: "<<(*m_algorithmOIDs)[i]);
					m_algorithmOIDs->RemoveAt(i);
					i--;
				}
			}
			
			for( i = 0; i < m_mechanisms->GetSize(); i++ )
			{
				for( j = 0; j < matchedMechanisms.GetSize(); j++ )
					if( (*m_mechanisms)[i].GetTag() == matchedMechanisms[j].GetTag() )
						break;
				if( j == matchedMechanisms.GetSize() )
				{
					PTRACE(5,"GkAuth\tRemoved from GCF list mechanism: "<<(*m_mechanisms)[i]);
					m_mechanisms->RemoveAt(i);
					i--;
				}
			}
			
			if( (m_mechanisms->GetSize() == 0) || (m_algorithmOIDs->GetSize() == 0) )
			{
				PTRACE(4,"GkAuth\tConflicting H.235 capabilities are active"
					<<" - GCF will not select any particular capability"
					);
				m_mechanisms->RemoveAll();
				m_algorithmOIDs->RemoveAll();
				break;
			}
		}
		authenticator = authenticator->GetNext();
	}

	// Scan "optional" authenticators if the above procedure has not found
	// any H.235 capabilities or has found more than one
	if( (!found) || (m_mechanisms->GetSize() > 1) || (m_algorithmOIDs->GetSize() > 1) )
	{
		authenticator = GetHead();
		while( authenticator )
		{
			if( authenticator->IsH235Capable() 
				&& (authenticator->GetControlFlag() == GkAuthenticator::e_Optional) )
			{
				if( m_mechanisms->GetSize() == 0 )
				{
					authenticator->GetH235Capability(
						(*m_mechanisms), (*m_algorithmOIDs)
						);
					if( m_algorithmOIDs->GetSize() == 0 )
						m_mechanisms->RemoveAll();
					else
						found = TRUE;
					authenticator = authenticator->GetNext();
					continue;
				}
			
				H225_ArrayOf_AuthenticationMechanism matchedMechanisms;
			
				int i, j, k;
			
				for( i = 0; i < m_algorithmOIDs->GetSize(); i++ )
				{
					BOOL matched = FALSE;
			
					for( j = 0; j < m_mechanisms->GetSize(); j++ )
						if( authenticator->IsH235Capability(
								(*m_mechanisms)[j], (*m_algorithmOIDs)[i]
								) )
						{
							for( k = 0; k < matchedMechanisms.GetSize(); k++ )
								if( matchedMechanisms[k].GetTag() == (*m_mechanisms)[j].GetTag() )
									break;
							if( k == matchedMechanisms.GetSize() )
							{
								matchedMechanisms.SetSize(k+1);
								matchedMechanisms[k].SetTag((*m_mechanisms)[j].GetTag());
							}
							matched = TRUE;
						}
				
					if( !matched )
					{
						PTRACE(5,"GkAuth\tRemoved from GCF list algorithm OID: "<<(*m_algorithmOIDs)[i]);
						m_algorithmOIDs->RemoveAt(i);
						i--;
					}
				}
			
				for( i = 0; i < m_mechanisms->GetSize(); i++ )
				{
					for( j = 0; j < matchedMechanisms.GetSize(); j++ )
						if( (*m_mechanisms)[i].GetTag() == matchedMechanisms[j].GetTag() )
							break;
					if( j == matchedMechanisms.GetSize() )
					{
						PTRACE(5,"GkAuth\tRemoved from GCF list mechanism: "<<(*m_mechanisms)[i]);
						m_mechanisms->RemoveAt(i);
						i--;
					}
				}
			
				if( (m_mechanisms->GetSize() == 0) || (m_algorithmOIDs->GetSize() == 0) )
				{
					PTRACE(4,"GkAuth\tConflicting H.235 capabilities are active"
						<<" - GCF will not select any particular capability"
						);
					m_mechanisms->RemoveAll();
					m_algorithmOIDs->RemoveAll();
					break;
				}
			}
			authenticator = authenticator->GetNext();
		}
	}
	
	if( PTrace::CanTrace(5) )
		if( (m_mechanisms->GetSize() > 0) && (m_algorithmOIDs->GetSize() > 0) )
		{
			ostream& strm = PTrace::Begin(5,__FILE__,__LINE__);
			strm <<"GkAuth\tH.235 capabilities selected for GCF:\n";
			strm <<"\tAuthentication mechanisms: \n";
			int i;
			for( i = 0; i < m_mechanisms->GetSize(); i++ )
				strm << "\t\t" << (*m_mechanisms)[i] << '\n';
			strm <<"\tAuthentication algorithm OIDs: \n";
			for( i = 0; i < m_algorithmOIDs->GetSize(); i++ )
				strm << "\t\t" << (*m_algorithmOIDs)[i] << '\n';
			PTrace::End(strm);
		}
}

GkAuthenticatorList::~GkAuthenticatorList()
{	
	delete GkAuthenticator::head;
	GkAuthenticator::head = 0;
	
	delete m_mechanisms;
	delete m_algorithmOIDs;
}

void GkAuthenticatorList::GetH235Capabilities(
	H225_ArrayOf_AuthenticationMechanism& mechanisms,
	H225_ArrayOf_PASN_ObjectId& algorithmOIDs
	) const
{
	mechanisms = *m_mechanisms;
	algorithmOIDs = *m_algorithmOIDs;
}
