/*
**  MailboxManagerController.m
**
**  Copyright (C) 2001, 2002, 2003
**
**  Author: Ludovic Marcotte <ludovic@Sophos.ca>
**
**  This program is free software; you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation; either version 2 of the License, or
**  (at your option) any later version.
**
**  This program is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with this program; if not, write to the Free Software
**  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "MailboxManagerController.h"

#include "ConsoleWindowController.h"
#include "Constants.h"
#include "EditWindowController.h"
#include "ExtendedOutlineView.h"
#include "GNUMail.h"
#include "GNUMail+TaskManager.h"

#ifndef MACOSX
#include "ImageTextCell.h"
#include "MailboxManager.h"
#endif

#include "FolderNode.h"
#include "MailboxManagerCache.h"
#include "MailWindowController.h"
#include "MessageViewWindowController.h"
#include "NewMailboxPanelController.h"
#include "NSUserDefaults+Extensions.h"
#include "PasswordPanelController.h"
#include "Utilities.h"

#include <Pantomime/Constants.h>
#include <Pantomime/Flags.h>
#include <Pantomime/FolderInformation.h>
#include <Pantomime/IMAPCacheManager.h>
#include <Pantomime/IMAPFolder.h>
#include <Pantomime/IMAPStore.h>
#include <Pantomime/LocalFolder.h>
#include <Pantomime/LocalStore.h>
#include <Pantomime/Message.h>
#include <Pantomime/NSData+Extensions.h>
#include <Pantomime/NSString+Extensions.h>
#include <Pantomime/TCPConnection.h>
#include <Pantomime/URLName.h>


static MailboxManagerController *singleInstance = nil;

// 
// Here's how it does work:
//
// allFolders (NSArray) -> localNodes (FolderNode * - name == _("Local"))
//
//                      -> IMAP FolderNode 1 (FoderNode * - name == "username @ imap.server1.com")  
//                           
//                      -> IMAP FolderNode 2 (FoderNode * - name == "username @ imap.server2.com")
// 
//                      -> ...   
//                                 
//
@implementation MailboxManagerController

#ifndef MACOSX
- (id) initWithWindowNibName: (NSString *) windowNibName
{
  id aCell;

  MailboxManager *theWindow;
  
  theWindow = [[MailboxManager alloc] initWithContentRect: NSMakeRect(200,200,334,435)
				      styleMask: (NSClosableWindowMask |
						  NSTitledWindowMask |
						  NSMiniaturizableWindowMask |
						  NSResizableWindowMask)
				      backing: NSBackingStoreBuffered
				      defer: YES];

  self = [super initWithWindow: theWindow];

  [theWindow layoutWindow];
  [theWindow setDelegate: self];

  // We link our outlets
  move = theWindow->move;
  copy = theWindow->copy;
  outlineView = theWindow->outlineView;
  scrollView = theWindow->scrollView;

  RELEASE(theWindow);

  // We set the title of our window (causing it to be loaded under OS X)
  [[self window] setTitle: _(@"Mailboxes")];

  // We now set our data cell for the "Mailbox" column
  aCell =  [[ImageTextCell alloc] init];
  [[outlineView tableColumnWithIdentifier: @"Mailbox"] setDataCell: aCell];
  AUTORELEASE(aCell);

  // We register the outline view for dragged types
  [outlineView registerForDraggedTypes: [NSArray arrayWithObject: MessagePboardType]];

  // We set our autosave window frame name and restore the one from the user's defaults.
  [[self window] setFrameAutosaveName: @"MailboxManager"];
  [[self window] setFrameUsingName: @"MailboxManager"]; 

  // We set our autosave name for our outline view
  [outlineView setAutosaveName: @"MailboxManager"];
  [outlineView setAutosaveTableColumns: YES];

  // We set our outline view background color
  if ( [[NSUserDefaults standardUserDefaults] colorForKey: @"MAILBOXMANAGER_OUTLINE_COLOR"] )
    {
      [outlineView setBackgroundColor: [[NSUserDefaults standardUserDefaults]
					 colorForKey: @"MAILBOXMANAGER_OUTLINE_COLOR"]];
      [scrollView setBackgroundColor: [[NSUserDefaults standardUserDefaults]
					colorForKey: @"MAILBOXMANAGER_OUTLINE_COLOR"]];
    }

  return self;
}
#else
- (id) init
{
  self = [super init];
  
  // We initialize some ivars
  [self windowDidLoad];
  
  return self;
}
#endif


//
//
//
- (void) dealloc
{
  NSDebugLog(@"MailboxManagerController: -dealloc");

  [[NSNotificationCenter defaultCenter]
    removeObserver: self
    name: AccountsHaveChanged
    object: nil];
  
#ifndef MACOSX
  [[self window] setDelegate: nil];
#else
  RELEASE(menu);
#endif

  RELEASE(localNodes);
  RELEASE(mailboxManagerCache);
  RELEASE(allFolders);
  RELEASE(allStores);
  
  [super dealloc];
}


//
// Datasource methods for the outline view
//
- (id) outlineView: (NSOutlineView *) outlineView
	     child: (int) index
	    ofItem: (id) item
{
  if ( !item || item == allFolders )
    {
      return [allFolders objectAtIndex: index];
    }

  if ( [item isKindOfClass: [FolderNode class]] )
    {
      return [(FolderNode *)item childAtIndex: index];
    }
 
  return nil;
}


//
//
//
- (BOOL) outlineView: (NSOutlineView *) outlineView
    isItemExpandable: (id) item
{
  if ( item == allFolders || [allFolders containsObject: item] )
    {
      return YES;
    }
  
  if ( [item isKindOfClass: [FolderNode class]] )
    {
      if ( [(FolderNode *)item childCount] > 0 )
	{
	  return YES;
	}
      else
	{
	  return NO;
	}
    }

  return NO;
}


//
//
//
- (int)        outlineView: (NSOutlineView *) outlineView 
    numberOfChildrenOfItem: (id) item
{
  // The root.
  if ( !item || item == allFolders )
    {
      return [allFolders count];
    }
  
  // Children of our root, the Local folder and all the IMAP folders, subfolders, etc.
  if ( [item isKindOfClass: [FolderNode class]] )
    {
      return [(FolderNode *)item childCount];
    }
  
  return 0;
}


//
//
//
- (id)         outlineView: (NSOutlineView *) outlineView 
 objectValueForTableColumn: (NSTableColumn *) tableColumn 
		    byItem: (id) item
{
  if ( [[[tableColumn headerCell] stringValue] isEqual: _(@"Mailbox")] )
    {
      if ( [item isKindOfClass: [FolderNode class]] )
	{
	  return [(FolderNode *)item name];
	}
    }
  
  if ( [item isKindOfClass: [FolderNode class]] && [item parent] )
    {
      BOOL b;
      
      b = [[Utilities completePathForFolderNode: item  separator: @"/"]
	    hasPrefix: [NSString stringWithFormat: @"/%@", _(@"Local")]];
      
      if ( [item childCount] == 0 || !b )
	{
	  int nbOfMessages, nbOfUnreadMessages;
	  
	  [self _nbOfMessages: &nbOfMessages
		nbOfUnreadMessages: &nbOfUnreadMessages
		forItem: item];
	  
	  // If we have an IMAP folder AND the could is 0 AND it has children, do nothing.
	  if (!b && nbOfMessages == 0 && [item childCount] > 0) return nil;

	  if ( [[[tableColumn headerCell] stringValue] isEqual: _(@"Messages")] )
	    {
	      return [NSString stringWithFormat: _(@"%d (%d unread)"), nbOfMessages, nbOfUnreadMessages];
	    }
	}
    }
  
  return nil;
}


//
//
//
- (void) outlineView: (NSOutlineView *) aOutlineView
     willDisplayCell: (id) aCell
      forTableColumn: (NSTableColumn *) aTableColumn
                item: (id) item
{
  // We set our default node icon, if we need to.
  if ( [[[aTableColumn headerCell] stringValue] isEqual: _(@"Mailbox")] )
    {
      int level;
      
      level = [aOutlineView levelForItem: item];
      
      if ( level > 0 && [(FolderNode *)item childCount] == 0 )
	{
	  NSString *aString;
	  id aStore;
	  
	  aString = [self _stringValueOfURLNameFromItem: item
			  store: &aStore];

	  if ( [Utilities stringValueOfURLName: aString  isEqualTo: @"TRASHFOLDERNAME"] )
	    {
	      [aCell setImage: [NSImage imageNamed: @"delete_16"]];
	    }
	  else if ( [Utilities stringValueOfURLName: aString  isEqualTo: @"SENTFOLDERNAME"] )
	    {
	      [aCell setImage: [NSImage imageNamed: @"send_16"]];
	    }
	  else if ( [Utilities stringValueOfURLName: aString  isEqualTo: @"DRAFTSFOLDERNAME"] )
	    {
	      [aCell setImage: [NSImage imageNamed: @"create_16"]];
	    }
	  else if ( [Utilities stringValueOfURLName: aString  isEqualTo: @"INBOXFOLDERNAME"] )
	    { 
	      [aCell setImage: [NSImage imageNamed: @"retrieve_16"]];
	    }
	  else
	    {
	      [aCell setImage: [NSImage imageNamed: @"mailboxes_16"]];
	    }
	} 
      else 
	{ 
	  [aCell setImage: nil];
	}
    }

  //
  //
  //
  if ( [item isKindOfClass: [FolderNode class]] && [item parent] )
    {
      int nbOfMessages, nbOfUnreadMessages;
      
      [self _nbOfMessages: &nbOfMessages
	    nbOfUnreadMessages: &nbOfUnreadMessages
	    forItem: item];

      if ( nbOfUnreadMessages > 0 )
	{
#ifdef MACOSX
	  [aCell setFont: [NSFont boldSystemFontOfSize: [NSFont smallSystemFontSize]]];
#else
	  [aCell setFont: [NSFont boldSystemFontOfSize: 0]];
#endif
	  return;
	}
    }
  
  // We set our default font.
#ifdef MACOSX
  [aCell setFont: [NSFont systemFontOfSize: [NSFont smallSystemFontSize]]];
#else
  [aCell setFont: [NSFont systemFontOfSize: 0]];
#endif

  // We set the right text aligment
  if ( [[[aTableColumn headerCell] stringValue] isEqual: _(@"Mailbox")] )
    {
      [aCell setAlignment: NSLeftTextAlignment];
    }
  else
    {
      [aCell setAlignment: NSRightTextAlignment];
    }   
}


//
//
//
#ifdef MACOSX
- (void)    outlineView: (NSOutlineView *) aOutlineView 
 willDisplayOutlineCell: (id) aCell 
	 forTableColumn: (NSTableColumn *) aTbleColumn 
		   item: (id)item
{
  [aCell setImage: [NSImage imageNamed: @"closedFolder"]];
  [aCell setAlternateImage: [NSImage imageNamed: @"openFolder"]];
  [aCell setImagePosition: NSImageAbove];
}


- (NSMenu *) outlineView: (NSOutlineView *) aOutlineView
      contextMenuForItem: (id) item
{
  int row, level;
  BOOL aBOOL;
  
  row = [aOutlineView selectedRow];
  level = [aOutlineView levelForItem: item];
  
  aBOOL = (row > 0 && [aOutlineView numberOfSelectedRows] == 1 && level >= 1);
  
  [[menu itemWithTitle: _(@"Delete Mailbox...")] setEnabled: aBOOL];
  [[menu itemWithTitle: _(@"Rename Mailbox...")] setEnabled: aBOOL];
  [[menu itemWithTitle: _(@"Move Messages")] setEnabled: aBOOL];
  [[menu itemWithTitle: _(@"Copy Messages")] setEnabled: aBOOL];
  
  aBOOL = (row >= 0 && [aOutlineView numberOfSelectedRows] == 1 && level >= 0);
  
  [[menu itemWithTitle: _(@"Create Mailbox...")] setEnabled: aBOOL];

  [menu update];
  return menu;
}
#endif


//
// Delegate methods
//
- (BOOL) outlineView: (NSOutlineView *) outlineView
  shouldCollapseItem: (id) item
{
  if ( [allFolders containsObject: item] &&
       item != localNodes )
    {
      NSString *aUsername, *aServerName;
      IMAPStore *aStore;

      BOOL shouldCloseStore;

      [Utilities storeKeyForFolderNode: item
		 serverName: &aServerName
		 username: &aUsername];
      
      aStore = [self storeForName: aServerName
		     username: aUsername];
      shouldCloseStore = NO;
      
      if ( [aStore isConnected] )
	{
	  int choice;

	  choice = NSRunAlertPanel( [NSString stringWithFormat: _(@"Closing connection on %@ @ %@..."), aUsername, aServerName],
				    _(@"Would you like to close the connection with this IMAP server?"),
				    _(@"Close"), // default
				    _(@"No"),    // alternate
				    NULL );
	  
	  if ( choice == NSAlertDefaultReturn )
	    {
	      shouldCloseStore = YES;
	    }
	}
      
      //
      // The store is already disconnected or the user wants to close it.
      // Let's close its window and remove it from the list of opened stores.
      //
      if ( ![aStore isConnected] || shouldCloseStore )
	{
	  id aWindow;
	  
	  // We MUST "flush" the folder associated to the window if we are re-using it since
	  // it's currently displaying an IMAP folder of this store (if it hasn't been closed).
	  // We do that ONLY if the store of the displayed folder IS this store.
	  // For example, one could have 'expanded' the IMAP node but not opened any 
	  // IMAP folder so we must NOT automatically "flush" the folder associated with the 
	  // MailWindowController since it could be displaying a local folder!
	  aWindow = [Utilities windowForFolderName: nil  store: (Store *)aStore];
	  
	  if ( aWindow )
	    {
	      // Flush the folder...
	      [[aWindow windowController] setFolder: nil];
	    }
	  
	  // This does nothing if it's already disconnected...
	  [aStore close];
	  
	  // This causes problems under OS X. Anyway, the children will be released
	  // if we re-open the connection to the IMAP server.
#ifndef MACOSX	      
	  [item setChildren: nil];
#endif
	  [allStores removeObjectForKey: [Utilities storeKeyForFolderNode: item
						    serverName: NULL
						    username: NULL]];
	  
	  [[ConsoleWindowController singleInstance] addConsoleMessage: [NSString stringWithFormat: _(@"Closing IMAP connection on %@."),
										 aServerName]];
	}
      
    }

  return YES;
}


//
//
//
- (BOOL) outlineView: (NSOutlineView *) outlineView
    shouldExpandItem: (id) item
{
  if ( item == allFolders || item == localNodes )
    {
      return YES;
    }
  
  if ( [allFolders containsObject: item] )
    {
      NSString *aString;

      aString = [(FolderNode *)item name];    // returns the account name

      return [self _initializeIMAPStoreWithAccountName: aString];
    }
  
  return YES;
}


//
// NSOutlineViewDataSource Drag and drop
//
- (NSDragOperation) outlineView: (NSOutlineView*) outlineView
		   validateDrop: (id <NSDraggingInfo>) info
		   proposedItem: (id) item
	     proposedChildIndex: (int) index
{
  if ( ![item respondsToSelector: @selector(childCount)] ||
       [(FolderNode*)item childCount]  > 0)
    {
      return NSDragOperationNone;
    }
  
  if ([info draggingSourceOperationMask] & NSDragOperationGeneric)
    {
      return NSDragOperationGeneric;
    }
  else if ([info draggingSourceOperationMask] & NSDragOperationCopy)
    {
      return NSDragOperationCopy;
    }
  else
    {
      return NSDragOperationNone;
    }
}


//
// NSOutlineViewDataSource Drag and drop
//
- (BOOL) outlineView: (NSOutlineView*) outlineView
	  acceptDrop: (id <NSDraggingInfo>) info
		item: (id) item
	  childIndex: (int) index
{
  NSArray *propertyList;
  int count, i;
  
  FolderNode *aFolderNode;
  NSString *aFolderName;

  Folder *sourceFolder, *destinationFolder;
  Store *sourceStore, *destinationStore;
  MailWindowController *aMailWindowController;

  aFolderNode = (FolderNode *)item;

  // We get our store
  if ( [[Utilities completePathForFolderNode: aFolderNode  separator: @"/"]
	 hasPrefix: [NSString stringWithFormat: @"/%@", _(@"Local")]] )
    {
      destinationStore = [self storeForName: @"GNUMAIL_LOCAL_STORE"
			       username: NSUserName()];
    }
  else
    {
      NSString *aServerName, *aUsername;
      
      [Utilities storeKeyForFolderNode: aFolderNode
		 serverName: &aServerName
		 username: &aUsername];
      
      destinationStore = [self storeForName: aServerName
			       username: aUsername];
    }

  aFolderName = [Utilities pathOfFolderFromFolderNode: aFolderNode
			   separator: [(id<Store>)destinationStore folderSeparator]];

  // We get the MailWindowController source
  aMailWindowController = (MailWindowController *)[[info draggingSource] delegate];
  
  if ( !aMailWindowController || ![aMailWindowController isKindOfClass: [MailWindowController class]] ||
       !aFolderName || [aFolderName length] == 0 )
    {
      NSBeep();
      return NO;
    }
  
  // We verify if we aren't trying to transfer to the current mbox!
  sourceFolder = [aMailWindowController folder];
  sourceStore = [sourceFolder store];
  
  if ( sourceStore == destinationStore && [[sourceFolder name] isEqualToString: aFolderName] )
    {
      NSRunInformationalAlertPanel(_(@"Transfer error!"),
				   _(@"You cannot transfer a message inside the same mailbox!"),
				   _(@"OK"),
				   NULL, 
				   NULL,
				   NULL);
      return NSDragOperationNone;      
    }

  // We get a reference to our destination folder, w/o parsing it if it's not already open.
  // or w/o selecting it if it's an IMAP store.
  if ( [(id<NSObject>)destinationStore isKindOfClass: [IMAPStore class]] )
    {
      destinationFolder = (Folder *)[(IMAPStore *)destinationStore folderForName: aFolderName
						  select: NO];
    }
  else
    {
      destinationFolder = (Folder *)[(LocalStore *)destinationStore folderForName: aFolderName];
    }

  propertyList = [[info draggingPasteboard] propertyListForType: MessagePboardType];
  count = [propertyList count];

  for (i = 0; i < count; i++)
    {
      NSDictionary *aDictionary;
      Flags *flags;
      NSData *rawSource;
      
      aDictionary = (NSDictionary*)[propertyList objectAtIndex: i];
      
      rawSource = [aDictionary objectForKey: @"message"];
      flags = (Flags*)[NSUnarchiver unarchiveObjectWithData: (NSData*)[aDictionary objectForKey: @"flags"]];

      if (rawSource && flags)
	{
	  if ( ![self transferMessageFromRawSource: rawSource
		      flags: flags
		      toFolderWithName: aFolderName
		      orToFolder: destinationFolder
		      store: destinationStore] )
	    {
	      // The transfer failed; we exit the loop
	      break;
	    }
	}
    }

  // If the message is moved (and not copied),
  // we inform the dragging source that the selected messages have been transferred
  if ( count > 0 && 
       count == i && 
       [info draggingSourceOperationMask] & NSDragOperationGeneric )
    {
      // We mark messages as transferred if append succeed
      [aMailWindowController setSelectedMessagesAsTransferred];
    }

  // If the folder was not open, we just close it.
  if ( destinationFolder )
    {
      [destinationFolder close];
    }

  return YES;
}


//
//
//
- (void) windowDidLoad
{
#ifdef MACOSX
  NSMenuItem *aMenuItem;
  
  menu = [[NSMenu alloc] init];
  [menu setAutoenablesItems: NO];
  
  aMenuItem = [[NSMenuItem alloc] initWithTitle: _(@"Delete") action: @selector(delete:)  keyEquivalent: @""];
  [aMenuItem setTarget: self];
  [menu addItem: aMenuItem];
  RELEASE(aMenuItem);
  
  aMenuItem = [[NSMenuItem alloc] initWithTitle: _(@"Create") action: @selector(create:)  keyEquivalent: @""];
  [aMenuItem setTarget: self];
  [menu addItem: aMenuItem];
  RELEASE(aMenuItem);
  
  aMenuItem = [[NSMenuItem alloc] initWithTitle: _(@"Rename") action: @selector(rename:)  keyEquivalent: @""];
  [aMenuItem setTarget: self];
  [menu addItem: aMenuItem];
  RELEASE(aMenuItem);
  
  [menu addItem: [NSMenuItem separatorItem]];
  
  aMenuItem = [[NSMenuItem alloc] initWithTitle: _(@"Move Messages") action: @selector(transfer:)  keyEquivalent: @""];
  [aMenuItem setTarget: self];
  [aMenuItem setTag: MOVE_MESSAGES];
  [menu addItem: aMenuItem];
  RELEASE(aMenuItem);
  
  aMenuItem = [[NSMenuItem alloc] initWithTitle: _(@"Copy Messages") action: @selector(transfer:)  keyEquivalent: @""];
  [aMenuItem setTarget: self];
  [aMenuItem setTag: COPY_MESSAGES];
  [menu addItem: aMenuItem];
  RELEASE(aMenuItem);
#endif

  [self setMailboxManagerCache: [MailboxManagerCache mailboxManagerCacheFromDisk]];

  // We initialize our array containing all Stores and we load of folders
  allFolders = [[NSMutableArray alloc] init];
  
  // We initialize our dictionary containing all openend IMAPStores
  allStores = [[NSMutableDictionary alloc] init];

  [[NSNotificationCenter defaultCenter]
    addObserver: self
    selector: @selector(_accountsHaveChanged:)
    name: AccountsHaveChanged
    object: nil];
}


//
// action methods
//
- (IBAction) open: (id) sender
{
  id item;
  int row, level;
  
  row = [outlineView selectedRow];

  if ( row < 0 )
    {
      NSBeep();
      return;
    }

  item = [outlineView itemAtRow: row];
  level = [outlineView levelForItem: item];
  
  //
  // We must verify that:
  //
  // a) we have at least one selected row
  // b) we haven't selected our root, or a store (local or IMAP)
  //
  if ( [outlineView numberOfSelectedRows] != 1 )
    {
      NSRunInformationalAlertPanel(_(@"Mailbox error!"),
				   _(@"You must select a valid mailbox to open!"),
				   _(@"OK"),
				   NULL, 
				   NULL,
				   NULL);
      return;
    }
  else if ( level < 1 )
    {
      if ( [outlineView isItemExpanded: item] )
	{
	  [outlineView collapseItem: item];
	}
      else
	{
	  [outlineView expandItem: item];
	}
      return;
    }
 
  
  // We verify if it's a local folder
  if ( [[Utilities completePathForFolderNode: item  separator: @"/"] 
	 hasPrefix: [NSString stringWithFormat: @"/%@", _(@"Local")]] )
    { 
      NSString *aString;
      
      aString = [Utilities pathOfFolderFromFolderNode: (FolderNode *)item
			   separator: @"/"];

      [self _openLocalFolderWithName: aString  sender: sender];
    }
  // It's an IMAP folder...
  else
    {
      NSString *aServerName, *aUsername, *aString;
      IMAPStore *aStore;

      [Utilities storeKeyForFolderNode: item
      		 serverName: &aServerName
      		 username: &aUsername];

      aStore = (IMAPStore *)[self storeForName: aServerName
				  username: aUsername];
      
      aString = [Utilities pathOfFolderFromFolderNode: (FolderNode *)item
			   separator: [aStore folderSeparator]];
      
      [self _openIMAPFolderWithName: aString
	    store: aStore
	    sender: sender];
    }
}


//
//
//
- (IBAction) transfer: (id) sender
{
  Folder *destinationFolder, *sourceFolder;
  id destinationStore, sourceStore;

  MailWindowController *aMailWindowController;
  NSString *aFolderName;
  
  NSArray *selectedMessages;
  
  int row, level, nbOfTransferredMessages;
  id item;
  
  nbOfTransferredMessages = 0;
  row = [outlineView selectedRow];
  
  if ( row < 0 )
    {
      NSBeep();
      return;
    }

  item = [outlineView itemAtRow: row];
  level = [outlineView levelForItem: item];
  
  if ( [outlineView numberOfSelectedRows] != 1 ||
       level < 1 )
    {
      NSRunInformationalAlertPanel(_(@"Mailbox error!"),
				   _(@"You must select a valid mailbox to transfer this E-Mail to!"),
				   _(@"OK"),
				   NULL, 
				   NULL,
				   NULL);
      return;
    }  

  // We get our store!
  if ( [[Utilities completePathForFolderNode: item  separator: @"/"]
	 hasPrefix: [NSString stringWithFormat: @"/%@", _(@"Local")]] )
    {
      destinationStore = [self storeForName: @"GNUMAIL_LOCAL_STORE"
			       username: NSUserName()];
    }
  else
    {
      NSString *aServerName, *aUsername;
      
      [Utilities storeKeyForFolderNode: item
		 serverName: &aServerName
		 username: &aUsername];

      destinationStore = [self storeForName: aServerName
			       username: aUsername];
    }
  
  // We get our folder name, respecting the folder separator.
  aFolderName = [Utilities pathOfFolderFromFolderNode: (FolderNode *) item
			   separator: [(id<Store>)destinationStore folderSeparator]];
  
  aMailWindowController = (MailWindowController *)[[GNUMail lastMailWindowOnTop] delegate];
  
  if ( !aMailWindowController ) 
    {
      NSBeep();
      return;
    }
  
  // We verify if we aren't trying to transfer to the current mbox!
  sourceFolder = [aMailWindowController folder];
  sourceStore = [sourceFolder store];
  
  if ( sourceStore == destinationStore &&
       [[sourceFolder name] isEqualToString: aFolderName] )
    {
      NSRunInformationalAlertPanel(_(@"Transfer error!"),
				   _(@"You cannot transfer messages inside the same mailbox!"),
				   _(@"OK"),
				   NULL, 
				   NULL,
				   NULL);
      return;      
    }
  
  //
  // Now let's verify if we can actually transfer messages to this folder.
  // If it's a local mailbox AND it has the PantomimeHoldsFolders flag, we can't.
  // If it's an IMAP mailbox AND it has the PantomimeNoSelect flag, we can't.
  // Otherwise, we might be able, let's give it a try.
  //
  if ( ([destinationStore isKindOfClass: [LocalStore class]] &&
	([destinationStore folderTypeForFolderName: aFolderName] & PantomimeHoldsFolders) == PantomimeHoldsFolders)
       ||
       ([destinationStore isKindOfClass: [IMAPStore class]] &&
	([destinationStore folderTypeForFolderName: aFolderName] & PantomimeNoSelect) == PantomimeNoSelect) )
    {
      NSRunInformationalAlertPanel(_(@"Transfer error!"),
				   _(@"You cannot transfer messages inside this mailbox!"),
				   _(@"OK"),
				   NULL, 
				   NULL,
				   NULL);
      return;
    }
       
  // We get a reference to our destination folder, w/o parsing it if it's not already open.
  // or w/o selecting it if it's an IMAP store.
  if ( [destinationStore isKindOfClass: [IMAPStore class]] )
    {
      destinationFolder = (Folder *)[(IMAPStore *)destinationStore folderForName: aFolderName
						  select: NO];
    }
  else
    {
      destinationFolder = (Folder *)[(LocalStore *)destinationStore folderForName: aFolderName];
    }
    
  

  // We transfer all the selected messages  
  selectedMessages = [aMailWindowController selectedMessages];
  
  if ( selectedMessages && [selectedMessages count] )
    {
      // If we are transferring messages from an IMAPFolder to an IMAPFolder on the SAME
      // IMAPStore, let's use Pantomime's IMAPFolder: -copyMessage: toFolder: method
      // since the operation is gonna be server-side - so MUCH FASTER.
      if ( [sourceStore isKindOfClass: [IMAPStore class]] &&
	   sourceStore == destinationStore )
	{
	  int nbOfUnreadTransferredMessages;

	  nbOfUnreadTransferredMessages = 0;

	  NS_DURING
	    {
	      [(IMAPFolder *)sourceFolder copyMessages: selectedMessages
			     toFolder: aFolderName];
	      nbOfTransferredMessages = [selectedMessages count];
	      
	      // If we are moving the messages, mark them as deleted.
	      if ( [sender tag] == MOVE_MESSAGES )
		{
		  Message *aMessage;
		  Flags *theFlags;
		  int i;
		  
		  for (i = 0; i < [selectedMessages count]; i++)
		    {
		      aMessage = [selectedMessages objectAtIndex: i];
		      theFlags = [[aMessage flags] copy];

		      // If we are transferring an unread message, we increment our counter
		      if ( ![theFlags contain: SEEN] ) nbOfUnreadTransferredMessages += 1;

		      [theFlags add: DELETED];
		      [aMessage setFlags: theFlags];
		      RELEASE(theFlags);
		    }
		}
	    }
	  NS_HANDLER
	    {
	      nbOfTransferredMessages = 0;
	    }
	  NS_ENDHANDLER;

	  // We now update our MailboxManagerController cache. Since the operation
	  // was server-side, we must obtain the current values in the cache and
	  // increment them according to what has been transferred.
	  if ( nbOfTransferredMessages > 0 )
	    {
	      int nbOfMessages, nbOfUnreadMessages;
	      
	      [mailboxManagerCache allValuesForStoreName: [(IMAPStore *)sourceStore name]
				   folderName: [aFolderName stringByReplacingOccurrencesOfString: [(IMAPStore *)sourceStore folderSeparator]
							    withString: @"/"]
				   username: [(IMAPStore *)sourceStore username]
				   nbOfMessages: &nbOfMessages
				   nbOfUnreadMessages: &nbOfUnreadMessages];
	      
	      nbOfUnreadMessages += nbOfUnreadTransferredMessages;
	      nbOfMessages += nbOfTransferredMessages;
	      
	      [mailboxManagerCache setAllValuesForStoreName: [(IMAPStore *)sourceStore name]
				   folderName: [aFolderName stringByReplacingOccurrencesOfString: [(IMAPStore *)sourceStore folderSeparator]
							    withString: @"/"]
				   username: [(IMAPStore *)sourceStore username]
				   nbOfMessages: nbOfMessages
				   nbOfUnreadMessages: nbOfUnreadMessages];
	    }
	}
      else
	{
	  NSAutoreleasePool *pool;
	  Message *aMessage;
	  NSData *aData;
	  int i;
	  
	  for (i = 0; i < [selectedMessages count]; i++)
	    {
	      pool = [[NSAutoreleasePool alloc] init];

	      aMessage = [selectedMessages objectAtIndex: i];
	      aData = [aMessage rawSource];
	      
	      if ( aData )
		{
		  Flags *theFlags;
		  
		  // We get our flags but we remove the DELETED flag from them
		  theFlags = [[aMessage flags] copy];
		  [theFlags remove: DELETED];
		  
		  if ( [self transferMessageFromRawSource: aData
			     flags: AUTORELEASE([theFlags copy])
			     toFolderWithName: aFolderName
			     orToFolder: destinationFolder
			     store: destinationStore] )
		    {
		      // We verify if we flag it as DELETED or not (in case of a
		      // copy operation).
		      if ( [sender tag] == MOVE_MESSAGES )
			{
			  [theFlags add: DELETED];
			  [aMessage setFlags: theFlags];
			  nbOfTransferredMessages++;
			}
		    }
		  RELEASE(theFlags);
		}
	      
	      RELEASE(pool);
	    }
	}
    }
  
  // We need to reload our tableView's data if we transferred some messages
  if ( nbOfTransferredMessages > 0 )
    {
      // We force the reload of our cache in Folder
      [[aMailWindowController folder] updateCache];
      [aMailWindowController dataViewShouldReloadData];
      [aMailWindowController updateStatusLabel];
    }

  // If our folder wasn't open, we just close it
  if ( destinationFolder )
    {
      [destinationFolder close];
    }
}


//
//
//
- (IBAction) create: (id) sender
{
  NewMailboxPanelController *theController;
  Store *aStore;
  id item;

  int row, level, result;
  
  row = [outlineView selectedRow];

  if ( row < 0 )
    {
      NSBeep();
      return;
    }

  item = [outlineView itemAtRow: row];
  level = [outlineView levelForItem: item];
  
  if ( [outlineView numberOfSelectedRows] != 1 )
    {
      NSRunInformationalAlertPanel(_(@"Mailbox error!"),
				   _(@"You must select a valid root where to create this new mailbox."),
				   _(@"OK"),
				   NULL, 
				   NULL,
				   NULL);
      return;
    }
  
  
  // We create our NewMailboxPanelController object. It'll be automatically deallocated when the 
  // window will be closed.
  theController = [[NewMailboxPanelController alloc] initWithWindowNibName: @"NewMailboxPanel"
						     folderNode: (FolderNode *)item];
  [theController setMailboxManagerController: self];

  // Disable our mailbox type popup button if we are creating an IMAP folder.
  // We also get the right store.
  if ( [[Utilities completePathForFolderNode: (FolderNode *)item  separator: @"/"]
	 hasPrefix: [NSString stringWithFormat: @"/%@", _(@"Local")]] )
    {
      // Creating a local folder
      [[theController popUpButton] setEnabled: YES];
      aStore = [self storeForName: @"GNUMAIL_LOCAL_STORE"
		     username: NSUserName()];
    }
  else
    {
      NSString *aServerName, *aUsername;
      
      [[theController popUpButton] setEnabled: NO];

      [Utilities storeKeyForFolderNode: item
		 serverName: &aServerName
		 username: &aUsername];
      
      aStore = [self storeForName: aServerName
		     username: aUsername];
    }
  
  result = [NSApp runModalForWindow: [theController window]];
 
  if ( IMAPSTORE_IS_DISCONNECTED(aStore) )
    {
      [self setStore: nil
	    name: [(IMAPStore *)aStore name]
	    username: [(IMAPStore *)aStore username]];
    }
  else if ( result == NSRunStoppedResponse )
    {
      NSString *aString;
      
      aString = [NSString stringWithFormat: @"%@/%@", [Utilities completePathForFolderNode: item
								 separator: @"/"],
			  [[[theController mailboxNameField] stringValue] stringByTrimmingWhiteSpaces] ];
      
      [self _reloadFoldersAndExpandParentsFromNode: item
	    selectNodeWithPath: aString];
    }
 
#ifndef MACOSX
  [[self window] makeKeyAndOrderFront: self];
#endif

  RELEASE(theController);
}


//
//
//
- (IBAction) delete: (id) sender
{
  int choice, row, level;
  id item;
  
  row = [outlineView selectedRow];

  if ( row < 0 )
    {
      NSBeep();
      return;
    }

  item = [outlineView itemAtRow: row];
  level = [outlineView levelForItem: item];
  
  if ( [outlineView numberOfSelectedRows] != 1 ||
       level < 1 )
    {
      NSRunInformationalAlertPanel(_(@"Mailbox error!"),
				   _(@"Please select the mailbox you would like to delete."),
				   _(@"OK"),
				   NULL, 
				   NULL,
				   NULL);
      return;
    }
  
  
  // We show our prompt panel
  choice = NSRunAlertPanel(_(@"Delete..."),
			   _(@"Are you sure you want to delete this mailbox?"),
			   _(@"Delete"),  // default
			   _(@"Cancel"),  // alternate
			   nil);
  
  if ( choice == NSAlertDefaultReturn )
    {
      NSString *aDefaultMailbox, *aFolderName, *aString;
      id aStore;
             
      aString = [self _stringValueOfURLNameFromItem: item
		      store: &aStore];
      
      // We get our folder name, respecting the folder separator
      aFolderName = [Utilities pathOfFolderFromFolderNode: (FolderNode *)item
			       separator: [(id<Store>)aStore folderSeparator]];
      
      if ( [self _deletingDefaultMailbox: &aDefaultMailbox
		 usingURLNameAsString: aString] )
	{
	  NSRunAlertPanel(_(@"Error while deleting!"),
  			  _(@"You can't delete your default %@ mailbox. Use the Mailboxes tab in the\nAccount Preferences panel to change it before trying again."),
			  _(@"OK"),   // default
			  NULL,       // alternate
  			  NULL,
			  aDefaultMailbox);
	  return;
	}
      
      if ( [aStore folderForNameIsOpen: aFolderName] )
	{
	  NSRunAlertPanel(_(@"Error while deleting!"),
  			  _(@"You can't delete an opened mailbox! Close it first."),
			  _(@"OK"),   // default
			  NULL,       // alternate
  			  NULL); 
	  return;
	}
      else
	{
	  BOOL aBOOL;

	  aBOOL = [aStore deleteFolderWithName: aFolderName];
	  
	  if ( IMAPSTORE_IS_DISCONNECTED(aStore) )
	    {
	      [self setStore: nil
		    name: [(IMAPStore *)aStore name]
		    username: [(IMAPStore *)aStore username]];
	    }

	  if ( aBOOL )
	    {
	      // Delete cache files, ONLY if this is an IMAP folder!
	      if ( [aStore isKindOfClass: [IMAPStore class]] )
		{
		  NSString *aKey, *cacheFilePath;
		  FolderNode *node;
		  int i;

		  aKey = [NSString stringWithFormat: @"%@ @ %@", 
				   [(IMAPStore *)aStore username], 
				   [(IMAPStore *)aStore name]];
		  
		  cacheFilePath = [NSString stringWithFormat: @"%@/IMAPCache_%@_%@", 
					    GNUMailUserLibraryPath(),
					    [Utilities flattenPathFromString: aKey
						       separator: @"/"],
					    [Utilities flattenPathFromString: 
							 [Utilities pathOfFolderFromFolderNode: (FolderNode *)item
								    separator: [(id<Store>)aStore folderSeparator]]
						       separator: @"/"]];
		  
		  // We remove the file
		  NS_DURING
		    [[NSFileManager defaultManager] removeFileAtPath: cacheFilePath
						    handler: nil];
		  NS_HANDLER
		    // Under GNUstep, if we pass something that can't be converted to a cString
		    // to -removeFileAtPath, it throws an exception.
		    NSDebugLog(@"Exception occured while removing the cache file.");
		  NS_ENDHANDLER

		  // We remove the cache file of the children of this folder, if any.
		  for (i = 0; i < [(FolderNode *)item childCount]; i++)
		    {
		      node = [(FolderNode *)item childAtIndex: i];
		      cacheFilePath = [NSString stringWithFormat: @"%@/IMAPCache_%@_%@", 
						GNUMailUserLibraryPath(),
						[Utilities flattenPathFromString: aKey
							   separator: @"/"],
						[Utilities flattenPathFromString: 
							     [Utilities pathOfFolderFromFolderNode: node
									separator: [(id<Store>)aStore folderSeparator]]
							   separator: @"/"]];
		      
		      NS_DURING
			[[NSFileManager defaultManager] removeFileAtPath: cacheFilePath
							handler: nil];
		      NS_HANDLER
			// Under GNUstep, if we pass something that can't be converted to a cString
			// to -removeFileAtPath, it throws an exception.
			NSDebugLog(@"Exception occured while removing the cache file.");
		      NS_ENDHANDLER
		    }
		}
	      
	      [self _reloadFoldersAndExpandParentsFromNode: [item parent]
		    selectNodeWithPath: [Utilities completePathForFolderNode: [item parent]
						   separator: @"/"]];
	    }
	  else
	    {
	      NSRunInformationalAlertPanel(_(@"Error!"),
					   _(@"The mailbox delete operation failed."),
					   _(@"OK"),
					   NULL,
					   NULL,
					   NULL);
	    }
	  
	}
    }
}


//
//
//
- (IBAction) rename: (id) sender
{
  NewMailboxPanelController *theController;
  Store *aStore;
  
  NSString *aFolderName;
  id item;

  int row, level, result;
  
  row = [outlineView selectedRow];

  if ( row < 0 )
    {
      NSBeep();
      return;
    }

  item = [outlineView itemAtRow: row];
  level = [outlineView levelForItem: item];
  
  if ( [outlineView numberOfSelectedRows] != 1 ||
       level < 1 )
    {
      NSRunInformationalAlertPanel(_(@"Mailbox error!"),
				   _(@"You must select a valid mailbox to rename!"),
				   _(@"OK"),
				   NULL, 
				   NULL,
				   NULL);
      return;
    }
     
  // We get our right store
  if ( [[Utilities completePathForFolderNode: item  separator: @"/"]
	 hasPrefix: [NSString stringWithFormat: @"/%@", _(@"Local")]] )
    {
      aStore = [self storeForName: @"GNUMAIL_LOCAL_STORE"
		     username: NSUserName()];
    }
  else
    {
      NSString *aServerName, *aUsername;
      
      [Utilities storeKeyForFolderNode: item
		 serverName: &aServerName
		 username: &aUsername];
      
      aStore = [self storeForName: aServerName
		     username: aUsername];
    }
  
  // We get our foldername, respecting the folder separator
  aFolderName = [Utilities pathOfFolderFromFolderNode: item
			   separator: [(id<Store>)aStore folderSeparator]];
  
  // We verify that the mailbox we want to rename is not open
  if ( [(id<Store>)aStore folderForNameIsOpen: aFolderName] )
    {
      NSRunAlertPanel(_(@"Error!"),
		      _(@"You can't rename an opened mailbox! Close it first."),
		      _(@"OK"),   // default
		      NULL,       // alternate
		      NULL); 
      return;
    }

  
  // We create our NewMailboxPanelController object. It'll be automatically 
  // deallocated when the window will be closed.
  theController = [[NewMailboxPanelController alloc] initWithWindowNibName: @"NewMailboxPanel"
						     folderNode: item];
  [theController setMailboxManagerController: self];
  [theController setName: [(FolderNode*)item name]];
  
  // We set the right window tile and label content
  [[theController window] setTitle: _(@"Edit Mailbox")];
  [[theController mailboxNameLabel] setStringValue: _(@"Please enter the new name of the Mailbox:")];

  // Disable our mailbox type popup button.
  [[theController popUpButton] setEnabled: NO];

  result = [NSApp runModalForWindow: [theController window]];

  if ( IMAPSTORE_IS_DISCONNECTED(aStore) )
    {
       [self setStore: nil
	     name: [(IMAPStore *)aStore name]
	     username: [(IMAPStore *)aStore username]];
    }
  else if ( result == NSRunStoppedResponse )
    {
      NSString *aString;
      
      aString = [NSString stringWithFormat: @"%@/%@", [Utilities completePathForFolderNode: [item parent]
								 separator: @"/"],
			  [[[theController mailboxNameField] stringValue] stringByTrimmingWhiteSpaces] ];
      
      [self _reloadFoldersAndExpandParentsFromNode: [item parent]
	    selectNodeWithPath: aString];
    }


#ifndef MACOSX
  [[self window] makeKeyAndOrderFront: self];
#endif

  RELEASE(theController);
}


//
// access / mutation methods
//
- (NSOutlineView *) outlineView
{
  return outlineView;
}


//
//
//
- (id) storeForName: (NSString *) theName
	   username: (NSString *) theUsername
{
  NSString *aString;
  
  aString = [NSString stringWithFormat: @"%@ @ %@", theUsername, theName];
  
  return [allStores objectForKey: aString];
}


//
//
//
- (id) storeForURLName: (URLName *) theURLName
{
  id aStore;
  
  if ( [[theURLName protocol] caseInsensitiveCompare: @"LOCAL"] == NSOrderedSame )
    {
      aStore = [self storeForName: @"GNUMAIL_LOCAL_STORE"
		     username: NSUserName()];
    }
  else
    {
      if ( [self _initializeIMAPStoreWithAccountName: [Utilities accountNameForServerName: [theURLName host]
								 username: [theURLName username]]] )
	{
	  aStore = [self storeForName: [theURLName host]
			 username: [theURLName username]];
	}
      else
	{
	  aStore = nil;
	}
    }
  
  return aStore;
}


//
//
//
- (void) setStore: (id) theStore
	     name: (NSString *) theName
	 username: (NSString *) theUsername
{
  NSString *aString;

  aString = [NSString stringWithFormat: @"%@ @ %@", theUsername, theName];

  // We verify if we want to remove an opened store.
  if ( !theStore && theName && theUsername )
    {
      FolderNode *aFolderNode;
      int row;

      // For an IMAP store, we remove all children of our root node
      aFolderNode = [self _storeFolderNodeForName: [Utilities accountNameForServerName: theName  username: theUsername]];
#ifndef MACOSX
      [aFolderNode setChildren: nil];
#endif
      [outlineView collapseItem: aFolderNode];

      row = [outlineView rowForItem: aFolderNode];

      if ( row >= 0 && row < [outlineView numberOfRows] )
	{
	  [outlineView selectRow: row
		       byExtendingSelection: NO];
	}

      [allStores removeObjectForKey: aString];
      return;
    }
  
  // We always first "remove" the object in case we call this method
  // multiple times with the same object.
  RETAIN(theStore);
  [allStores removeObjectForKey: aString];
      
  // We {re}add it to our dictionary.
  [allStores setObject: theStore
	     forKey: aString];
  RELEASE(theStore);
}


//
//
//
- (MailboxManagerCache *) mailboxManagerCache
{
  return mailboxManagerCache;
}


//
//
//
- (void) setMailboxManagerCache: (MailboxManagerCache *) theMailboxManagerCache
{
  if ( theMailboxManagerCache )
    {
      RETAIN(theMailboxManagerCache);
      RELEASE(mailboxManagerCache);
      mailboxManagerCache = theMailboxManagerCache;
    }
  else
    {
      DESTROY(mailboxManagerCache);
    }
}


//
//
//
- (void) addMessage: (NSDictionary *) theDictionary
{
  [self addMessage: [theDictionary objectForKey: @"Message"]
	toFolder: [theDictionary objectForKey: @"URLName"]];
}


//
// This method appends a message to the folder specified in theURLName.
//
- (void) addMessage: (NSData *) theMessage
	   toFolder: (URLName *) theURLName
{
  NSString *aFolderName;
  Store *aStore;
 
  BOOL storeWasOpen;
   
  // We verify if out store was open. If it wasn't, we will close the store at the end of this method.
  storeWasOpen = [self storeWithURLNameIsOpen: theURLName];

  aStore = [self storeForURLName: theURLName];
  
  if ( !aStore )
    {
      NSRunAlertPanel(_(@"Error!"),
		      _(@"An error occured while attempting to obtain the message store for the %@ folder."),
		      _(@"OK"),
		      NULL,
		      NULL,
		      [theURLName foldername]);
      return; 
    }
  
  aFolderName = [theURLName foldername];
	  
  if ( ![self transferMessageFromRawSource: theMessage
	      flags: nil
	      toFolderWithName: aFolderName
	      orToFolder: nil
	      store: aStore] )
    {
      NSRunAlertPanel(_(@"Error!"),
		      _(@"An error occured while adding the message to the %@ folder."),
		      _(@"OK"),
		      NULL,
		      NULL,
		      [theURLName foldername]);
    }

  // We verify if we must close the store. That could only happen with IMAP stores since the
  // local store is always open.
  if ( !storeWasOpen )
    {
      [(IMAPStore *)aStore close];
      [self setStore: nil
	    name: [theURLName host]
	    username: [theURLName username]];
    }
}


//
// 
//
- (Message *) messageFromDraftsFolder
{
  id aMailWindowController;
  Message *aMessage;
  
  aMailWindowController = [[GNUMail lastMailWindowOnTop] delegate];

  // We first verify if current folder is a Drafts folder.
  if ( aMailWindowController && [aMailWindowController isKindOfClass: [MailWindowController class]] )
    {
      if ( ![Utilities stringValueOfURLName: [Utilities stringValueOfURLNameFromFolder: 
							  [aMailWindowController folder]]
		       isEqualTo: @"DRAFTSFOLDERNAME"] )
	{
	  return nil;
	}
    }
   
  if ( [[aMailWindowController folder] count] > 0 &&
       [aMailWindowController selectedMessage])
    {
      aMessage = [aMailWindowController selectedMessage];
    }
  else
    {
      aMessage = nil;
    }

  return aMessage;
}


//
//
//
- (NSDictionary *) allStores
{
  return [NSDictionary dictionaryWithDictionary: allStores];
}

 
//
// This method is used under OS X since we must "swap" the current
// outline view in the Mailbox Manager to match the one currently
// on top in the NSDrawer. This is needed so -open: and other methods
// can work properly with the "current" outline view.
//
#ifdef MACOSX
- (void) setCurrentOutlineView: (id) theOutlineView
{
  outlineView = theOutlineView;
}
#endif


//
//
//
- (void) updateFolderInformation: (NSDictionary *) theInformation
{
  FolderInformation *aFolderInformation;

  NSString *aFolderName;
  
  aFolderInformation = [theInformation objectForKey: @"FOLDER_INFORMATION"];
  aFolderName = [theInformation objectForKey: @"FOLDER_NAME"];

  [[self mailboxManagerCache] setAllValuesForStoreName: [theInformation objectForKey: @"STORE_NAME"]
			      folderName: [aFolderName stringByReplacingOccurrencesOfString: [theInformation objectForKey: @"FOLDER_SEPARATOR"]
						       withString: @"/"]
			      username: [theInformation objectForKey: @"USERNAME"]
			      nbOfMessages: [aFolderInformation nbOfMessages]
			      nbOfUnreadMessages: [aFolderInformation nbOfUnreadMessages]];

  [outlineView setNeedsDisplay: YES];
}


//
// class methods
//
+ (id) singleInstance
{
  if ( !singleInstance )
    {
#ifdef MACOSX
      singleInstance = [[MailboxManagerController alloc] init];
#else
      singleInstance = [[MailboxManagerController alloc] initWithWindowNibName: @"MailboxManager"];
#endif
    }
  
  return singleInstance;
}


//
// Other methods
//
- (void) openFolderWithURLName: (URLName *) theURLName
			sender: (id) theSender
{
  if ( [[theURLName protocol] caseInsensitiveCompare: @"LOCAL"] == NSOrderedSame )
    {
      [self _openLocalFolderWithName: [theURLName foldername]
	    sender: theSender];
    }
  else if ( [[theURLName protocol] caseInsensitiveCompare: @"IMAP"] == NSOrderedSame )
    {
      if ( [self _initializeIMAPStoreWithAccountName: [Utilities accountNameForServerName: [theURLName host]
								 username: [theURLName username]]] )
	{
	  [self _openIMAPFolderWithName: [theURLName foldername]
		store: (IMAPStore *)[self storeForName: [theURLName host]  username: [theURLName username]]
		sender: theSender];
	}
    }
}


//
//
//
- (void) reloadAllFolders
{
  DESTROY(localNodes);

  // We remove all our elements
  [allFolders removeAllObjects];

  // We add our local folder, if we need to
  localNodes = [Utilities folderNodesFromFolders: [[self storeForName: @"GNUMAIL_LOCAL_STORE"
							 username: NSUserName()] folderEnumerator]
			  separator: @"/"];

  [localNodes setName: _(@"Local")];
  [localNodes setParent: nil];

  if ( [localNodes childCount] > 0 )
    {
      [allFolders addObject: localNodes];
    }

  RETAIN(localNodes);

  // We verify if the ACCOUNTS preferences have been defined.
  if ( [[NSUserDefaults standardUserDefaults] objectForKey: @"ACCOUNTS"] )
    {
      NSMutableDictionary *allAccounts;
      NSDictionary *allValues;
      NSEnumerator *allKeys;
      NSString *aKey;

      allAccounts = [[NSMutableDictionary alloc] init];
      [allAccounts addEntriesFromDictionary: [Utilities allEnabledAccounts]];
      
      allKeys = [[[allAccounts allKeys] sortedArrayUsingSelector: @selector(compare:)] objectEnumerator];
      
      // We build a correct subset of all our IMAP servers defined in all accounts
      while ( (aKey = [allKeys nextObject]) )
	{
	  allValues = [[allAccounts objectForKey: aKey] objectForKey: @"RECEIVE"];  
	  
	  // We add it only if it's an IMAP server AND if we receive mails either
	  // manuall or automatically
	  if ( [[allValues objectForKey: @"SERVERTYPE"] intValue] == IMAP &&
	       [[allValues objectForKey: @"RETRIEVEMETHOD"] intValue] != NEVER )
	    {
	      NSString *aServerName, *aUsername;
	      FolderNode *aFolderNode;
	      
	      aServerName = [allValues objectForKey: @"SERVERNAME"];
	      aUsername = [allValues objectForKey: @"USERNAME"];
	      
	      aFolderNode = [FolderNode folderNodeWithName: [NSString stringWithFormat: @"%@", aKey]
					parent: nil];
	      
	      [allFolders addObject: aFolderNode];
	      
	      // If our IMAP Store has been previously initialized, we re-initialize it in order to get
	      // the most recent values for the subscribed folders
	      if ( [self storeForName: aServerName
			 username: aUsername] )
		{
		  [self _reloadFoldersForIMAPStoreWithName: aServerName
			username: aUsername
			account: aKey];
		}
	    }
	}
      
      RELEASE(allAccounts);
    }
  
  // We inform our outline view to reload its data.
  [outlineView reloadData];

  // We alwas expand our root item
  [outlineView expandItem: allFolders];
  
  // We now select and expand the 'Local' folder if there's no IMAP folders defined
  if ( [allFolders count] == 1 )
    {
      [outlineView expandItem: localNodes];
      [outlineView selectRow: [outlineView rowForItem: localNodes]
  		   byExtendingSelection: NO];
   }
}


//
// This method is used to transfer a message (from raw source) to
// the specified folder.
//
// If the folder isn't opened, it is opened w/o parsing.
// If it was opened, we refresh the dataView that shows
// the messages to the user.
//
// Note: This method MUST NOT be called in a thread other than the
//       main thread.
//
- (BOOL) transferMessageFromRawSource: (NSData *) theRawSource
				flags: (Flags *) theFlags
		     toFolderWithName: (NSString *) theFolderName
			   orToFolder: (Folder *) theFolder
				store: (Store *) theStore
{
  NSString *aStoreName, *aUsername;
  Folder *aFolder;
  Flags *flags;
  id aWindow;

  // We first verify if theRawSource is nil, if it is, we simply return NO.
  if ( !theRawSource || [theRawSource length] == 0 )
    {
      return NO;
    }
  
  aWindow = nil;

  // We get the store name
  if ( [(id<NSObject>)theStore isKindOfClass: [LocalStore class]] )
    {
      aStoreName = @"GNUMAIL_LOCAL_STORE";
      aUsername = NSUserName();
    }
  else
    {
      aStoreName = [(IMAPStore *)theStore name];
      aUsername = [(IMAPStore *)theStore username];
    }


  // We get a reference to our folder, w/o parsing it if it's not already open.
  if ( !theFolder )
    {
      if ( [(id<NSObject>)theStore isKindOfClass: [IMAPStore class]] )
	{
	  aFolder = [(IMAPStore *)theStore folderForName: theFolderName
				  select: NO];
	}
      else
	{
	  aFolder = [(id<Store>)theStore folderForName: theFolderName];
	}
    }
  else
    {
      aWindow = [Utilities windowForFolderName: [theFolder name]
			   store: theStore]; 
      aFolder = theFolder;
    }

  // Our folder is opened, let's get it from the list of opened windows.
  if ( !aFolder )
    {
      aWindow = [Utilities windowForFolderName: theFolderName
			   store: theStore]; 
      aFolder = (LocalFolder *)[[aWindow windowController] folder];
    } 
  
  // We transfer the new message. If we are transferring to an Sent folder, mark it as read.
  flags = theFlags;

  if ( [Utilities stringValueOfURLName: [Utilities stringValueOfURLNameFromFolder: aFolder]  
		  isEqualTo: @"SENTFOLDERNAME"] )
    {
      flags = [[Flags alloc] initWithFlags: SEEN];
      AUTORELEASE(flags);
    }

  NS_DURING
    {
      [aFolder appendMessageFromRawSource: theRawSource
	       flags: flags];

      // We update the Mailbox Manager cache, IIF the folder wasn't opened initially.
      // If it was open, we had its window and we are gonna refresh its cache in
      // -updateStatusLabel.
      if ( aWindow == nil )
	{
	  NSString *aFolderName;
	  int nbOfMessages, nbOfUnreadMessages;
	  
	  aFolderName = [aFolder name];

	  if ( [aFolder isKindOfClass: [LocalFolder class]] )
	    {
	      nbOfMessages = [aFolder count];
	      nbOfUnreadMessages = [aFolder numberOfUnreadMessages];
	    }
	  else
	    {      
	      // If we are transferring to an IMAP folder, we must obtain the
	      // current values of the target folder since we haven't SELECT'ed it
	      // (so [aFolder count] and [aFolder numberOfUnreadMessages] return 0.
	      [mailboxManagerCache allValuesForStoreName: aStoreName
				   folderName: [aFolderName stringByReplacingOccurrencesOfString: [(IMAPStore *)[aFolder store] folderSeparator]
							    withString: @"/"]
				   username: aUsername
				   nbOfMessages: &nbOfMessages
				   nbOfUnreadMessages: &nbOfUnreadMessages];

	      if ( flags && ![flags contain: SEEN] ) nbOfUnreadMessages += 1;
	      nbOfMessages += 1;
	    }

	  [mailboxManagerCache setAllValuesForStoreName: aStoreName
			       folderName: [aFolderName stringByReplacingOccurrencesOfString: [(IMAPStore *)[aFolder store] folderSeparator]
							withString: @"/"]
			       username: aUsername
			       nbOfMessages: nbOfMessages
			       nbOfUnreadMessages: nbOfUnreadMessages];

	  [outlineView setNeedsDisplay: YES];
	}

      [[self mailboxManagerCache] synchronize];
    }
  NS_HANDLER
    {
      NSRunAlertPanel(_(@"Error!"),
		      _(@"A fatal error occured when appending the message to the %@ folder."),
		      _(@"OK"),
		      NULL,
		      NULL,
		      [aFolder name]);

      return NO;
    }
  NS_ENDHANDLER
  
  // If the folder was open, we refresh its view. Otherwise, we just close it.
  if ( aWindow )
    {
      [[aWindow delegate] dataViewShouldReloadData];
      [[aWindow delegate] updateStatusLabel];
    }
  else
    {
      // If our folder wasn't first openend
      if ( !theFolder )
	{
	  [aFolder close];
	}
    }

  return YES;
}


//
// This method is used to verify if a folder with name "theFolderName"
// exists in the specified local store.
//
- (BOOL) folderNameExists: (NSString *) theFolderName
		 forStore: (Store *) theStore
{
  NSEnumerator *anEnumerator;
  NSString *aString;

  anEnumerator = [(id<Store>)theStore folderEnumerator];

  while ( (aString = [anEnumerator nextObject]) )
    {
      if ( [aString isEqualToString: theFolderName] )
	{
	  return YES;
	}
    }
  
  return NO;
}


//
// This method is called if we've tried to send an IMAP command
// to an IMAP store and the connection was lost to that store.
//
- (void) connectionWasLost: (id) sender
{
  NSEnumerator *theEnumerator;
  IMAPFolder *aFolder;
  IMAPStore *aStore;
  NSString *aKey;

  aStore = (IMAPStore *)sender;
  aKey = [NSString stringWithFormat: @"%@ @ %@", [aStore username], [aStore name]];

  NSRunCriticalAlertPanel(_(@"Error!"),
			  _(@"The connection was lost to the %@ IMAP server."),
			  _(@"OK"),
			  NULL,
			  NULL,
			  aKey);

  // We find the window and we close it, if we need to.
  theEnumerator = [aStore openedFoldersEnumerator];

  while ( (aFolder = [theEnumerator nextObject]) )
    {
      id aWindow;

      aWindow = [Utilities windowForFolderName: [aFolder name]
			   store: (Store *)aStore];
      
      if ( aWindow )
	{
	  MailWindowController *aMailWindowController;
	  
	  aMailWindowController = [aWindow windowController];
	  [aMailWindowController close];
	}
    }
}


//
//
//
- (BOOL) storeWithURLNameIsOpen: (URLName *) theURLName
{
  // The local store is always open.
  if ( [[theURLName protocol] caseInsensitiveCompare: @"LOCAL"] == NSOrderedSame )
    {
      return YES;
    }
  
  if ( [self storeForName: [theURLName host]  username: [theURLName username]] )
    {
      return YES;
    }
  
  return NO;
}


//
//
//
- (void) saveMessageInDraftsFolderForController: (EditWindowController *) theEditWindowController
{
  NSString *theAccountName;
  URLName *theURLName;
  
  // We first update the current message content with the current content of the view
  // and we synchronize our popup button.
  [theEditWindowController updateMessageContentFromTextView];
  [[theEditWindowController accountPopUpButton] synchronizeTitleAndSelectedItem];

  // We get our account name
  theAccountName = [Utilities accountNameForItemTitle: [[theEditWindowController accountPopUpButton] titleOfSelectedItem]];

  // We finally get our URLName object.
  theURLName = [[URLName alloc] initWithString: [[[[[NSUserDefaults standardUserDefaults] objectForKey: @"ACCOUNTS"] objectForKey: theAccountName] 
						   objectForKey: @"MAILBOXES"] objectForKey: @"DRAFTSFOLDERNAME"]
				path: [[NSUserDefaults standardUserDefaults] objectForKey: @"LOCALMAILDIR"]];
  
  [[MailboxManagerController singleInstance] addMessage: [[theEditWindowController message] dataValue]
					     toFolder: theURLName];

  // If this message is already in the Drafts folder, set the "deleted" flag 
  // of the original message.
  if ( [theEditWindowController editingDraftMessage] )
    {
      Flags *theFlags;
      
      theFlags = [[[theEditWindowController message] flags] copy];
      [theFlags add: DELETED];
      [[theEditWindowController message] setFlags: theFlags];
      RELEASE(theFlags);

      // We post our notifications
      [[NSNotificationCenter defaultCenter] postNotificationName: SelectionOfMessageHasChanged
					    object: nil
					    userInfo: nil];
      
      [[NSNotificationCenter defaultCenter] postNotificationName: @"ReloadMessageList"
					    object: nil
					    userInfo: nil];
  }
  
  // We mark the window's document as non-edited
  [[theEditWindowController window] setDocumentEdited: NO];

  RELEASE(theURLName);
}

@end


//
// Private methods
//
@implementation MailboxManagerController (Private)

- (void) _accountsHaveChanged: (id) sender
{
  [self reloadAllFolders];
}


//
//
//
- (BOOL) _deletingDefaultMailbox: (NSString **) theMailboxName
	    usingURLNameAsString: (NSString *) theURLNameAsString
{
  if ( [Utilities stringValueOfURLName: theURLNameAsString  isEqualTo: @"INBOXFOLDERNAME"] )
    {
      *theMailboxName = _(@"Inbox");
      return YES;
    }
  else if ( [Utilities stringValueOfURLName: theURLNameAsString  isEqualTo: @"SENTFOLDERNAME"] )
    {
      *theMailboxName = _(@"Sent");
      return YES;
    }
  else if ( [Utilities stringValueOfURLName: theURLNameAsString  isEqualTo: @"DRAFTSFOLDERNAME"] )
    {
      *theMailboxName = _(@"Drafts");
      return YES;
    }
  else if ( [Utilities stringValueOfURLName: theURLNameAsString  isEqualTo: @"TRASHFOLDERNAME"] )
    {
      *theMailboxName = _(@"Trash");
      return YES;
    }
  
  return NO;
}


//
//
//
- (BOOL) _initializeIMAPStoreWithAccountName: (NSString *) theAccountName
{
  NSString *aServerName, *aUsername, *aPassword, *aMechanism;
  NSNumber *serverTypeValue, *portValue;  
  NSDictionary *allValues;
  
  // We begin by searching in our ACCOUNTS values for the right account.
  // Now, let's get all the receive values
  allValues = [[[[NSUserDefaults standardUserDefaults] objectForKey: @"ACCOUNTS"]
		 objectForKey: theAccountName] objectForKey: @"RECEIVE"];
  
  serverTypeValue = [allValues objectForKey: @"SERVERTYPE"];
  portValue =  [allValues objectForKey: @"PORT"];
	  
  // We use the default IMAP port if it's not defined.
  if ( !portValue )
    {
      portValue = [NSNumber numberWithInt: 143];
    }
  
  // We get our username
  aUsername = [allValues objectForKey: @"USERNAME"];

  // We get our servername
  aServerName = [allValues objectForKey: @"SERVERNAME"];
  
  // We get our authentication mechanism
  aMechanism = [allValues objectForKey: @"AUTH_MECHANISM"];

  if ( aMechanism && [aMechanism isEqualToString: @"Password"] )
    {
      aMechanism = nil;
    }

  // We first verify if we haven't already cached our store. If so,
  // we simply return since the Store has already been initialized.
  if ( [self storeForName: aServerName
	     username: aUsername] )
    {
      return YES;
    }

  // We get our password. We could prompt for it in case we need to.
  aPassword = [Utilities passwordForKey: theAccountName
			 type: IMAP
			 prompt: YES];
  
  if ( aPassword )
    {		
      IMAPStore *aStore;
     
      // We verify if we must initiate a TCP/SSL or a normal TCP connectin.
      if ( [allValues objectForKey: @"USESECURECONNECTION"] &&
	   [[allValues objectForKey: @"USESECURECONNECTION"] intValue] == NSOnState )
	{
	  [[ConsoleWindowController singleInstance] addConsoleMessage: [NSString stringWithFormat: 
										   _(@"Connecting to IMAP server %@ using SSL"), 
										 aServerName]];
	  
	  aStore = [[IMAPStore alloc] initSSLWithName: aServerName
				      port: [portValue intValue]];
	}
      else
	{
	  [[ConsoleWindowController singleInstance] addConsoleMessage: [NSString stringWithFormat: 
										   _(@"Connecting to IMAP server %@"),
										 aServerName]];
	  
	  aStore = [[IMAPStore alloc] initWithName: aServerName
				      port: [portValue intValue]];
	}

      if ( !aStore )
	{
	  NSRunInformationalAlertPanel(_(@"Error!"),
				       _(@"Unable to communicate with the IMAP server (%@:%i)."),
				       _(@"OK"),
				       NULL,
				       NULL,
				       aServerName, 
				       [portValue intValue]);
	  return NO;
 	}
        else
	  {
	    // We set the store's delegate so we can be informed right away if we lose
	    // the connection to it.
	    [aStore setDelegate: self];

	    [[ConsoleWindowController singleInstance] addConsoleMessage: [NSString stringWithFormat: 
										     _(@"Authenticating using %@..."),
										   aUsername]];
	    
	    // We verify if the authentication is successful
	    if ( [aStore authenticate: aUsername
			 password: aPassword
			 mechanism: aMechanism] )
	      {
		// We cache our folder
		[self setStore: aStore
		      name: aServerName
		      username: aUsername];
		
		[self _reloadFoldersForIMAPStoreWithName: aServerName
		      username: aUsername
		      account: theAccountName];

		RELEASE(aStore);
		
		[[ConsoleWindowController singleInstance] addConsoleMessage: [NSString stringWithFormat: 
											 _(@"Connected to %@!"),
										       aServerName]];

		return YES;
	      }
	    else
	      {
		[[NSApp delegate] authenticationFailed: [NSDictionary dictionaryWithObjectsAndKeys: aServerName, @"Server",
								      aUsername, @"Username",
								      [NSNumber numberWithInt: IMAP], @"Type", nil]];
		[aStore close];
		RELEASE(aStore);
	      }

	  } // else of if ( !aStore ) ...
	
#ifndef MACOSX
      if ( [[self window] isVisible] )
	{
	  [[self window] makeKeyAndOrderFront: self];
	}
#endif
    } // if (password )
	
  return NO;
}


//
//
//
- (void) _nbOfMessages: (int *) theNbOfMessages
    nbOfUnreadMessages: (int *) theNbOfUnreadMessages
               forItem: (id) theItem
{
  NSString *aString, *aStoreName, *aFolderName, *aUsername;
  
  aString = [Utilities completePathForFolderNode: theItem
		       separator: @"/"];
  
  if ( [aString hasPrefix: [NSString stringWithFormat: @"/%@", _(@"Local")]] )
    {
      aStoreName = @"GNUMAIL_LOCAL_STORE";
      aFolderName = [Utilities pathOfFolderFromFolderNode: (FolderNode *)theItem
			       separator: @"/"];
      aUsername = NSUserName();
    }
  else
    {
      [Utilities storeKeyForFolderNode: theItem
		 serverName: &aStoreName
		 username: &aUsername];
      
      aFolderName = [Utilities pathOfFolderFromFolderNode: (FolderNode *)theItem
			       separator: @"/"];
    }
  
  [mailboxManagerCache allValuesForStoreName: aStoreName
		       folderName: aFolderName
		       username: aUsername
		       nbOfMessages: theNbOfMessages
		       nbOfUnreadMessages: theNbOfUnreadMessages];
}


//
//
//
- (void) _openLocalFolderWithName: (NSString *) theFolderName
			   sender: (id) theSender
{
  MailWindowController *aMailWindowController;
  LocalStore *localStore;
  LocalFolder *aFolder;
  
  BOOL reusingLastMailWindowOnTop, aMask;
  
  // We get out local store and our folder.
  localStore = [self storeForName: @"GNUMAIL_LOCAL_STORE"
		     username: NSUserName()];
  
  // We first verify if the folder is still valid. For example, it could have been
  // deleted (the file) manually while GNUMail.app was running.
  if ( ![[NSFileManager defaultManager] 
	  fileExistsAtPath: [[localStore path] stringByAppendingPathComponent: theFolderName]] )
    {
      NSRunInformationalAlertPanel(_(@"Mailbox error!"),
				   _(@"The local mailbox %@ does not exist!"),
				   _(@"OK"),
				   NULL, 
				   NULL,
				   theFolderName);
      return;
    }

  // We now verify if it's not a directory (a folder holding folders)
  if ( ([localStore folderTypeForFolderName: theFolderName] & PantomimeHoldsFolders) == PantomimeHoldsFolders)
    {
      id item;

      item = [outlineView itemAtRow: [outlineView selectedRow]];

      if ( [outlineView isItemExpanded: item] )
	{
	  [outlineView collapseItem: item];
	}
      else
	{
	  [outlineView expandItem: item];
	}
      return;
    }

  // If the folder is already open, we "focus" that window
  if ( [localStore folderForNameIsOpen: theFolderName] )
    {
      NSWindow *aWindow;

      aWindow = (NSWindow *)[Utilities windowForFolderName: theFolderName 
				       store: (Store *)localStore];
      if ( aWindow )
	{
	  [aWindow orderFrontRegardless];
	}
      
      return;
    }
  
  // The folder is not open, so we get it!
  aFolder = [localStore folderForName: theFolderName];

#ifdef MACOSX
  aMask = ([[NSApp currentEvent] modifierFlags] & NSAlternateKeyMask) == NSAlternateKeyMask;
#else
  aMask = ([[NSApp currentEvent] modifierFlags] & NSControlKeyMask) == NSControlKeyMask;
#endif

  // If we reuse our window controller...
  if ( [theSender isKindOfClass: [NSMenuItem class]] || 
       [GNUMail lastMailWindowOnTop] == nil || 
       theSender == [NSApp delegate] ||
       aMask )
  {
      aMailWindowController = [[MailWindowController alloc] initWithWindowNibName: @"MailWindow"];
      reusingLastMailWindowOnTop = NO;
    }
  else
    {      
      aMailWindowController = [[GNUMail lastMailWindowOnTop] windowController];
      reusingLastMailWindowOnTop = YES;
      
      // We must NOT assume that we got a MailWindowController
      if ( [aMailWindowController isKindOfClass: [MessageViewWindowController class]] )
	{
	  aMailWindowController = [(MessageViewWindowController *)aMailWindowController mailWindowController];
	}

      // We close the previous folder. No need to handle the IMAP timeout
      // as it's handled in IMAPFolder: -close.
      [[aMailWindowController folder] close];
    }
      
  // We update the show/hide deleted/read
  [self _updateShowOrHideReadAndDeletedForFolder: aFolder];
  
  // We set the new folder
  [aMailWindowController setFolder: aFolder];

  // We we are reusing our window controller, we must always reload the table view
  if ( reusingLastMailWindowOnTop && [GNUMail lastMailWindowOnTop] )
    {
      [aMailWindowController dataViewShouldReloadData];
    }

  // And we show the window.. 
  [[aMailWindowController window] orderFrontRegardless];
  [[aMailWindowController window] makeKeyAndOrderFront: nil];

  [[ConsoleWindowController singleInstance] addConsoleMessage: [NSString stringWithFormat: 
									   _(@"Local folder %@ opened."), 
									 theFolderName]];

  // If the "Local" node was collapsed in our MailboxManager, we now expend it
  if ( ![outlineView isItemExpanded: [self _storeFolderNodeForName: _(@"Local")]] )
    {
      [outlineView expandItem: [self _storeFolderNodeForName: _(@"Local")]];
    }
}


//
//
//
- (void) _openIMAPFolderWithName: (NSString *) theFolderName
			   store: (IMAPStore *) theStore
			  sender: (id) theSender
{
  MailWindowController *aMailWindowController;
  IMAPCacheManager *anIMAPCacheManager;
  IMAPFolder *aFolder;
  NSString *aKey;
  
  BOOL reusingLastMailWindowOnTop, aMask;

#ifdef MACOSX
  aMask = ([[NSApp currentEvent] modifierFlags] & NSAlternateKeyMask) == NSAlternateKeyMask;
#else
  aMask = ([[NSApp currentEvent] modifierFlags] & NSControlKeyMask) == NSControlKeyMask;
#endif

  // Using IMAP, we currently only allow the user to have one folder opened
  // at the time on the same IMAPStore.
  if ( [[[theStore openedFoldersEnumerator] allObjects] count] > 0 )
    {
      id aWindow;

      // We search for one opened window (so folder) on the IMAP store
      aWindow = [Utilities windowForFolderName: nil  store: (Store *)theStore];

      // If the folder that the window is 'using' is the same as the one we are trying to open,
      // we simply make this window the key one and order it front. There's no need to try
      // to reopen that folder!
      if ( [[[[aWindow windowController] folder] name] isEqualToString: theFolderName] )
	{
	  [aWindow makeKeyAndOrderFront: self];
	  return;
	}

      // If we are trying to open a new MailWindow using the menu item or if we are reusing
      // a MailWindow but the current on top isn't a window 'using' our IMAP store...
      if ( [theSender isKindOfClass: [NSMenuItem class]] ||
	   aMask ||
	   ([[GNUMail allMailWindows] count] > 1 && [GNUMail lastMailWindowOnTop] != aWindow) )
	{
	  NSRunInformationalAlertPanel(_(@"Mailbox error!"),
				       _(@"A folder (%@) is already open. Please close it first."),
				       _(@"OK"),
				       NULL, 
				       NULL,
				       [(IMAPFolder *)[[theStore openedFoldersEnumerator] nextObject] name]);
	  return;
	}
    }
  
  

  // If we must reuse our window controller...
  if ( [theSender isKindOfClass: [NSMenuItem class]] || 
       [GNUMail lastMailWindowOnTop] == nil || 
       theSender == [NSApp delegate] ||
       aMask )
    {
      aMailWindowController = [[MailWindowController alloc] initWithWindowNibName: @"MailWindow"];
      reusingLastMailWindowOnTop = NO;
    }
  else
    {
      aMailWindowController = [[GNUMail lastMailWindowOnTop] windowController];
      reusingLastMailWindowOnTop = YES;
      
      // We must NOT assume that we got a MailWindowController
      if ( [aMailWindowController isKindOfClass: [MessageViewWindowController class]] )
	{
	  aMailWindowController = [(MessageViewWindowController *)aMailWindowController mailWindowController];
	}
      
      // We close the previous folder. No need to handle the IMAP timeout
      // as it's handled in IMAPFolder: -close.
      [[aMailWindowController folder] close];
    }

  // We send our message to the console saying we are about to open the IMAP folder
  [[ConsoleWindowController singleInstance] addConsoleMessage: [NSString stringWithFormat: 
									   _(@"Opening IMAP folder %@ on %@..."), 
									 theFolderName, [theStore name]]];
  
  // We get our cache manager for this server / folder
  aKey = [NSString stringWithFormat: @"%@ @ %@", [theStore username], [theStore name]];
  anIMAPCacheManager = [IMAPCacheManager cacheFromDiskWithPath:
					   [NSString stringWithFormat: @"%@/IMAPCache_%@_%@",
						     GNUMailUserLibraryPath(),
						     [Utilities flattenPathFromString: aKey
								separator: @"/"],
						     [Utilities flattenPathFromString: theFolderName
								separator: [theStore folderSeparator]]]];
  //
  // We obtain our folder from the IMAP store. We might be in the following situation:
  //
  // a) everything is alright so we proceed by opening our folder.
  //
  // b) we lost the connection to the IMAP store and we're trying to open
  //    an other folder on the same server
  //
  // c) we lost the connection to the IMAP store we want to open a folder to
  //    
  //

  // We verify for b) and c).
  if ( IMAPSTORE_IS_DISCONNECTED(theStore) )
    {
      // We warn that we lost our connection;.
      NSRunCriticalAlertPanel(_(@"Error!"),
			      _(@"The connection was lost to the %@ IMAP server."),
			      _(@"OK"),
			      NULL,
			      NULL,
			      [NSString stringWithFormat: @"%@ @ %@", [theStore username], [theStore name]]);
      [aMailWindowController setFolder: nil];
      [aMailWindowController close];
      
      // We close our store
      [self setStore: nil
            name: [theStore name]
            username: [theStore username]];
      return;
    }

  aFolder = (IMAPFolder *)[theStore folderForName: theFolderName
				    mode: PantomimeReadWriteMode
				    prefetch: NO];  

  // We verify if the folder can be opened. It could have been a \NoSelect folder.
  if ( !aFolder)
    {
      NSRunInformationalAlertPanel(_(@"Mailbox error!"),
				   _(@"You must select a valid mailbox to open!"),
				   _(@"OK"),
				   NULL,
				   NULL,
				   NULL);
      return;
    }

  // We set the cache manager and we prefetch our messages
  [aFolder setCacheManager: anIMAPCacheManager];
  [aFolder prefetch];
  

  // We update the show/hide deleted/read
  [self _updateShowOrHideReadAndDeletedForFolder: aFolder];
  
  // We set the folder
  [aMailWindowController setFolder: aFolder];
  
  // We are reusing our window controller, we must always reload the table view
  if ( reusingLastMailWindowOnTop && [GNUMail lastMailWindowOnTop] )
    {
      [aMailWindowController dataViewShouldReloadData];
    }

  [[aMailWindowController window] orderFrontRegardless];
  [[aMailWindowController window] makeKeyAndOrderFront: nil];

  [[ConsoleWindowController singleInstance] addConsoleMessage: [NSString stringWithFormat:
									   _(@"IMAP folder %@ on %@ opened."), 
									 theFolderName, [theStore name]]];

  // If the "IMAP" node was collapsed in our MailboxManager, we now expand it
  if ( ![outlineView isItemExpanded: [self _storeFolderNodeForName: 
					     [Utilities accountNameForServerName: [theStore name]
							username: [theStore username]]] ])
    {
      [outlineView expandItem: [self _storeFolderNodeForName: 
				       [Utilities accountNameForServerName: [theStore name]
						  username: [theStore username]]] ];
    }
}


//
//
//
- (void) _reloadFoldersForIMAPStoreWithName: (NSString *) theName
				   username: (NSString *) theUsername
				    account: (NSString *) theAccountName
{
  FolderNode *aFolderNode;
  IMAPStore *aStore;
  
  aStore = [self storeForName: theName  username: theUsername];
  aFolderNode = [self _storeFolderNodeForName: theAccountName];
  
  if ( aFolderNode )
    {
      NSMutableDictionary *allAccounts, *allValues, *theAccount;
      NSArray *subscribedFolders;
      NSNumber *aNumber;
      
      NSString *theSeparator;
      FolderNode *nodes;

      Task *aTask;    

      // We handle the IMAP disconnection.
      if ( IMAPSTORE_IS_DISCONNECTED(aStore) )
	{
          // Close any associated windows and flush our IMAPStore object.
          [self connectionWasLost: aStore];
	  [self setStore: nil
		name: [aStore name]
		username: [aStore username]];
	  return;
	}

      aNumber = [[[[[NSUserDefaults standardUserDefaults] objectForKey: @"ACCOUNTS"]
		    objectForKey: theAccountName] objectForKey: @"RECEIVE"] objectForKey: @"SHOW_WHICH_MAILBOXES"];

      if ( aNumber && [aNumber intValue] == IMAP_SHOW_SUBSCRIBED_ONLY )
	{
	  subscribedFolders = [NSArray arrayWithArray: [[aStore subscribedFolderEnumerator] allObjects]];
	}
      else
	{
	  subscribedFolders =  [NSArray arrayWithArray: [[aStore folderEnumerator] allObjects]];
	}
      
      // When using IMAP, we query the server for folders's status information.
      // We do that in our worker thread.
      aTask = [[Task alloc] init];
      [aTask setOp: RECEIVE_IMAP];
      [aTask setSubOp: IMAP_STATUS];
      [aTask setKey: theAccountName];
      [aTask setOrigin: ORIGIN_STARTUP];  // We MUST delay the task's execution otherwise GNUMail.app
      [[NSApp delegate] addTask: aTask];  // will freeze upon startup under GNUstep.
      RELEASE(aTask);

      theSeparator = RETAIN([aStore folderSeparator]);

      nodes = [Utilities folderNodesFromFolders: [subscribedFolders objectEnumerator]
			 separator: theSeparator];
      RETAIN(nodes);
      
      [aFolderNode setChildren: [nodes children]];

      RELEASE(theSeparator);
      RELEASE(nodes);
      
#ifdef MACOSX
      [outlineView reloadItem: aFolderNode];
#else
      [outlineView reloadData];
#endif

      // FIXME - can we optimize this? Ie., write directly to the RECEIVE dictionary.
      // We finally cache our subscribed folders in the user's defaults for this server.
      allAccounts = [[NSMutableDictionary alloc] initWithDictionary: [[NSUserDefaults standardUserDefaults] 
								       objectForKey: @"ACCOUNTS"]];
      theAccount = [[NSMutableDictionary alloc] initWithDictionary: [allAccounts objectForKey: theAccountName]];
      allValues = [[NSMutableDictionary alloc] initWithDictionary: [theAccount objectForKey: @"RECEIVE"]];
      
      // We write back the info
      [allValues setObject: subscribedFolders  forKey: @"SUBSCRIBED_FOLDERS"];
      [theAccount setObject: allValues  forKey: @"RECEIVE"];
      [allAccounts setObject: theAccount  forKey: theAccountName];
      [[NSUserDefaults standardUserDefaults] setObject: allAccounts  forKey: @"ACCOUNTS"];
      [[NSUserDefaults standardUserDefaults] synchronize];

      RELEASE(allValues);
      RELEASE(theAccount);
      RELEASE(allAccounts);
    }
  else
    {
      NSDebugLog(@"Error getting the folder's root for IMAP folders tree.");
    }
}


//
//
//
- (FolderNode *) _storeFolderNodeForName: (NSString *) theName
{
  int i;

  for (i = 0; i < [allFolders count]; i++)
    {
      FolderNode *aFolderNode;
      
      aFolderNode = [allFolders objectAtIndex: i];

      if ( [theName isEqualToString: [aFolderNode name]] )
	{
	  return aFolderNode;
	}
    }

  return nil;
}


//
//
//
- (void) _reloadFoldersAndExpandParentsFromNode: (FolderNode *) theNode
			     selectNodeWithPath: (NSString *) thePath
{ 
  NSString *aString, *aServerName, *aUsername;
  NSMutableArray *nodesToExpand;
  
  id aParent, aRootNode;
  int i, aRow; 
  
  aString = [Utilities storeKeyForFolderNode: theNode
		       serverName: &aServerName
		       username: &aUsername];

  // We must refresh our outline view by reload its content
  [self reloadAllFolders];
  
  // We first get our root node
  if ( [thePath hasPrefix: [NSString stringWithFormat: @"/%@", _(@"Local")]] )
    {
      aRootNode = localNodes;
    }
  else
    {
      aRootNode = [self _storeFolderNodeForName: [Utilities accountNameForServerName: aServerName  username: aUsername]];
    }


  // We get our new node in our tree and also our new row index
  aParent = [Utilities folderNodeForPath: [thePath stringByDeletingLastPathComponent]
		       using: aRootNode
		       separator: @"/"];

  nodesToExpand = [[NSMutableArray alloc] init];

  // We expand all our parent, to make the row visible.
  while ( aParent )
    {
      [nodesToExpand addObject: aParent];
      aParent = [aParent parent];
    }
  
  // We must expand our nodes starting from the root to the children and not
  // the other way around. Otherwise, the NSOutlineView just borks.
  for (i = ([nodesToExpand count] - 1); i >= 0; i--)
    {
      [outlineView expandItem: [nodesToExpand objectAtIndex: i]];
    }

  RELEASE(nodesToExpand);

  // We now get our new node node (renamed or created). Since it's now shown on the screen,
  // we can now obtain the row for it and select it.
  aParent = [Utilities folderNodeForPath: thePath
		       using: aRootNode
		       separator: @"/"];

  aRow = [outlineView rowForItem: aParent];


  if ( aRow >= 0 && aRow < [outlineView numberOfRows] )
    {
      [outlineView selectRow: aRow
      		   byExtendingSelection: NO];
      [outlineView scrollRowToVisible: aRow];
    }
}


//
//
//
- (FolderNode *) _rootFolderNodeForNode: (FolderNode *) theNode
{
  FolderNode *aNode;

  aNode = theNode;

  while ( [aNode parent] )
    {
      aNode = [aNode parent];
    }

  return aNode;
}


//
//
//
- (NSString *) _stringValueOfURLNameFromItem: (id) theItem
				       store: (Store **) theStore
{
  NSMutableString *aMutableString;
  NSString *aString;
  
  aMutableString = [[NSMutableString alloc] init];
  
  // We verify if it's a local folder
  if ( [[Utilities completePathForFolderNode: theItem  separator: @"/"] 
	 hasPrefix: [NSString stringWithFormat: @"/%@", _(@"Local")]] )
    {
      [aMutableString appendFormat: @"local://%@", [[NSUserDefaults standardUserDefaults] 
						     objectForKey: @"LOCALMAILDIR"]];
      *theStore = [self storeForName: @"GNUMAIL_LOCAL_STORE"
			username: NSUserName()];
    }
  else
    {
      NSString *aServerName, *aUsername;
      
      [Utilities storeKeyForFolderNode: theItem
		 serverName: &aServerName 
		 username: &aUsername];
      *theStore = [self storeForName: aServerName
			username: aUsername];
  
      [aMutableString appendFormat: @"imap://%@@%@", aUsername, aServerName];
    }
  
  // We get our folder name, respecting the folder separator
  aString = [Utilities pathOfFolderFromFolderNode: (FolderNode *)theItem
		       separator: [(id<Store>)*theStore folderSeparator]];
  
  [aMutableString appendFormat: @"/%@", aString];

  return AUTORELEASE(aMutableString);
}


//
//
//
- (void) _updateShowOrHideReadAndDeletedForFolder: (Folder *) theFolder
{
  
  // We now set the folder to show or hide our deleted messages from the folder
  // Hiding them is now the default behaviour (in 1.1.0pre2)
  if ( [[NSUserDefaults standardUserDefaults] objectForKey: @"HIDE_DELETED_MESSAGES"] &&
       [[[NSUserDefaults standardUserDefaults] objectForKey: @"HIDE_DELETED_MESSAGES"] intValue] == NSOffState)
    {
      [theFolder setShowDeleted: YES];
      [[NSApp delegate] updateShowOrHideDeletedMenuItem: YES];
    }
  else
    {
      [theFolder setShowDeleted: NO];
      [[NSApp delegate] updateShowOrHideDeletedMenuItem: NO];
    }

  // We now set the folder to show or hide read messages from the folder. Showing them
  // is the default behavior
  if ( ![[NSUserDefaults standardUserDefaults] objectForKey: @"HIDE_READ_MESSAGES"] ||
       [[[NSUserDefaults standardUserDefaults] objectForKey: @"HIDE_READ_MESSAGES"] intValue] == NSOffState)
    {
      [theFolder setShowRead: YES];
      [[NSApp delegate] updateShowOrHideReadMenuItem: YES];
    }
  else
    {
      [theFolder setShowRead: NO];
      [[NSApp delegate] updateShowOrHideReadMenuItem: NO];
    }
}

@end








