/*
**  ExtendedOutlineView.m
**
**  Copyright (c) 2002-2004
**
**  Author: Francis Lachapelle <francis@Sophos.ca>
**          Ludovic Marcotte <ludovic@Sophos.ca>
**
**  This program is free software; you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation; either version 2 of the License, or
**  (at your option) any later version.
**
**  This program is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with this program; if not, write to the Free Software
**  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "ExtendedOutlineView.h"

#include "Constants.h"
#include "MailboxManagerController.h"
#include "MailWindowController.h"


/*!
 * @class ExtendedOutlineView
 * @abstract This class overwrites some methods of NSOutlineView.
 * @discussion This class is used by the MailboxManager class so
 *             all items look like folders un Mac OS X and items
 *             can be dragged on items and not inbetween nodes.
 *             
 */

@implementation ExtendedOutlineView: NSOutlineView

- (BOOL) acceptsFirstMouse: (NSEvent *) theEvent
{
  return YES;
}


//
//
//
- (NSMenu *) menuForEvent: (NSEvent *) theEvent
{
  int row;
  id item;
  
#ifndef MACOSX
  if (![[NSUserDefaults standardUserDefaults] boolForKey: @"EnableContextMenus"])
    {
      return nil;
    }
#endif
  
  row = [self rowAtPoint: [self convertPoint: [theEvent locationInWindow] fromView: nil]];
  
  if (row >= 0)
    {
      [self abortEditing];
      
      item = [self itemAtRow: row];
      
      if (item)
	{
	  id delegate;
	  
	  delegate = [self delegate];
	  
	  if ( [self numberOfSelectedRows] <= 1 )
	    {
	      [self selectRow: row  byExtendingSelection: NO];
	    }
	  
	  if ( [delegate respondsToSelector: @selector(outlineView:contextMenuForItem:)] ) 
	    {
	      return [delegate outlineView: self  contextMenuForItem: item];
	    }
	  else if ( [delegate respondsToSelector: @selector(dataView:contextMenuForRow:)] ) 
	    {
	      return [delegate dataView: self  contextMenuForRow: row];
	    }
	}
    }
  else
    {
      [self deselectAll: self];
      return [self menu];
    }

  return nil;
}


/*!
 * @method shouldCollapseAutoExpandedItemsForDeposited:
 * @abstract Indicate if auto expanded items should return to
 *           their original collapsed state. 
 * @discussion
 * @param deposited Tells whether or not the drop terminated due
 *                  to a successful drop.
 * @result Always returns NO, so that items never return to
 *         their original collapsed state.
 */
- (BOOL) shouldCollapseAutoExpandedItemsForDeposited: (BOOL) deposited
{
  return NO;
}


//
//
//
- (NSImage *) dragImageForRows: (NSArray *) dragRows
			 event: (NSEvent *) dragEvent 
	       dragImageOffset: (NSPointPointer) dragImageOffset
{ 
  if ([dragRows count] > 1)
    {
      return [NSImage imageNamed: @"drag_mails.tiff"];
    }
  
  return [NSImage imageNamed: @"drag_mail.tiff"];
}
@end
