/*
**  FindWindowController.m
**
**  Copyright (c) 2001-2005
**
**  Author: Ludovic Marcotte <ludovic@Sophos.ca>
**
**  This program is free software; you can redistribute it and/or modify
**  it under the terms of the GNU General Public License as published by
**  the Free Software Foundation; either version 2 of the License, or
**  (at your option) any later version.
**
**  This program is distributed in the hope that it will be useful,
**  but WITHOUT ANY WARRANTY; without even the implied warranty of
**  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
**  GNU General Public License for more details.
**
**  You should have received a copy of the GNU General Public License
**  along with this program; if not, write to the Free Software
**  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "FindWindowController.h"

#ifndef MACOSX
#include "FindWindow.h"
#endif

#include "GNUMail.h"
#include "ConsoleWindowController.h"
#include "Constants.h"
#include "MailboxManagerController.h"
#include "MailWindowController.h"
#include "Utilities.h"

#include <Pantomime/CWConstants.h>
#include <Pantomime/CWContainer.h>
#include <Pantomime/CWFolder.h>
#include <Pantomime/CWIMAPStore.h>
#include <Pantomime/CWMessage.h>
#include <Pantomime/NSString+Extensions.h>

static FindWindowController *singleInstance = nil;

//
// FindWindowController Private interface
//
@interface FindWindowController (Private)
- (void) _folderCloseCompleted: (NSNotification *) theNotification;
- (void) _folderSearchCompleted: (NSNotification *) theNotification;
- (void) _folderSearchFailed: (NSNotification *) theNotification;
- (void) _selectIndexesFromResults: (NSArray *) theResults
                        controller: (MailWindowController *) theMailWindowController;
- (void) _setState: (BOOL) theState;
@end


//
//
//
@implementation FindWindowController

- (id) initWithWindowNibName: (NSString *) windowNibName
{
#ifdef MACOSX
  
  self = [super initWithWindowNibName: windowNibName];
 
#else
  FindWindow *theWindow;
  
  theWindow = [[FindWindow alloc] initWithContentRect: NSMakeRect(200,200,340,175)
				  styleMask: NSTitledWindowMask|NSClosableWindowMask|
				  NSMiniaturizableWindowMask|NSResizableWindowMask
				  backing: NSBackingStoreBuffered
				  defer: YES];
  
  [super initWithWindow: theWindow];

  [theWindow layoutWindow];
  [theWindow setDelegate: self];
  [theWindow setMinSize: [theWindow frame].size];

  // We link our outlets
  findField = theWindow->findField;
  foundLabel = theWindow->foundLabel;
  ignoreCaseButton = theWindow->ignoreCaseButton;
  regularExpressionButton = theWindow->regularExpressionButton;
  matrix = theWindow->matrix;
  findAllButton = theWindow->findAllButton;
  nextButton = theWindow->nextButton;
  previousButton = theWindow->previousButton;

  RELEASE(theWindow);
#endif

  [[self window] setTitle: _(@"Find")];

  // We finally set our autosave window frame name and restore the one from the user's defaults.
  [[self window] setFrameAutosaveName: @"FindWindow"];
  [[self window] setFrameUsingName: @"FindWindow"];
  
  [[NSNotificationCenter defaultCenter]
    addObserver: self
    selector: @selector(_folderCloseCompleted:)
    name: @"PantomimeFolderCloseCompleted"
    object: nil];

  [[NSNotificationCenter defaultCenter]
    addObserver: self
    selector: @selector(_folderSearchCompleted:)
    name: @"PantomimeFolderSearchCompleted"
    object: nil];

  [[NSNotificationCenter defaultCenter]
    addObserver: self
    selector: @selector(_folderSearchFailed:)
    name: @"PantomimeFolderSearchFailed"
    object: nil];

  return self;
}


//
//
//
- (void) dealloc
{
  [[NSNotificationCenter defaultCenter] removeObserver: self];
  RELEASE(_indexes);
  RELEASE(_folder);
  [super dealloc];
}


//
// action methods
//
- (IBAction) findAll: (id) sender
{
  NSString *aString;

  aString = [findField stringValue];
  _location = 0;
  
  if ([GNUMail lastMailWindowOnTop] && 
      ([[aString stringByTrimmingWhiteSpaces] length] > 0)) 
    {
      MailWindowController *aMailWindowController;
      CWFolder *aFolder;
      int mask, options;
      
      aMailWindowController = (MailWindowController *)[[GNUMail lastMailWindowOnTop] delegate];
      aFolder = [aMailWindowController folder];
      
      ADD_CONSOLE_MESSAGE(_(@"Searching for %@..."), aString);
      
      //
      // We get our mask
      //
      if ([[matrix cellAtRow: 0  column: 0] state] == NSOnState)
	{
	  mask = PantomimeFrom;
	}
      else if ([[matrix cellAtRow: 1  column: 0] state] == NSOnState)
	{
	  mask = PantomimeTo;
	}
      else if ([[matrix cellAtRow: 1  column: 1] state] == NSOnState)
	{
	  mask = PantomimeContent;
	}
      else
	{
	  mask = PantomimeSubject;
	}

      //
      // We get our options
      //
      options = 0;

      if ([ignoreCaseButton state] == NSOnState)
	{
	  options = options|PantomimeCaseInsensitiveSearch;
	}
      
      if ([regularExpressionButton state] == NSOnState)
	{
	  options = options|PantomimeRegularExpression;
	}

      [aFolder search: aString  mask: mask  options: options];

      //
      // We must only start the animation for a IMAP folder
      // since for a local folder, we will receive the PantomimeFolderSearchCompleted
      // notification BEFORE starting the actual animation!
      //
      if ([aFolder isKindOfClass: [CWIMAPFolder class]])
	{
	  [foundLabel setStringValue: _(@"Searching...")];
	  [aMailWindowController startAnimation];
	  [self _setState: NO];
	}
    }
  else
    {
      NSBeep();
    }
}


//
//
//
- (IBAction) nextMessage: (id) sender
{
  if ([GNUMail lastMailWindowOnTop])
    {
      MailWindowController *aMailWindowController;
      id dataView; 

      aMailWindowController = (MailWindowController *)[[GNUMail lastMailWindowOnTop] delegate];
      
      dataView = [aMailWindowController dataView];
      
      if ([_indexes count] < 2)
	{
	  NSBeep();
	  return;
	}
      else
	{
	  [dataView selectRow: [[_indexes objectAtIndex: _location] intValue] 
		    byExtendingSelection: NO];
	  [dataView scrollRowToVisible: [[_indexes objectAtIndex: _location] intValue]];
	  _location += 1;
	  
	  if (_location == [_indexes count])
	    {
	      _location = 0;
	    }
	  
	  [dataView setNeedsDisplay:YES];
	}
    }
}


//
//
//
- (IBAction) previousMessage: (id) sender
{  
  if ([GNUMail lastMailWindowOnTop])
    {
      MailWindowController *aMailWindowController;
      id dataView; 
      
      aMailWindowController = (MailWindowController *)[[GNUMail lastMailWindowOnTop] delegate];
      
      dataView = [aMailWindowController dataView];
      
      if ([_indexes count] < 2)
	{
	  NSBeep();
	  return;
	}
      else
	{
	  [dataView selectRow: [[_indexes objectAtIndex: _location] intValue] 
		    byExtendingSelection: NO];
	  [dataView scrollRowToVisible: [[_indexes objectAtIndex: _location] intValue]];
	  _location -= 1;
	  
	  if (_location < 0)
	    {
	      _location = [_indexes count]-1;
	    }

	  [dataView setNeedsDisplay:YES];
	}
    }
}


//
// delegate methods
//
- (void) windowDidLoad
{
  _indexes = [[NSMutableArray alloc] init];
  _location = 0;  
  _folder = nil;
}


//
// access / mutation
//
- (NSTextField *) findField
{
  return findField;
}

//
// class methods
//
+ (id) singleInstance
{
  if (!singleInstance)
    {
      singleInstance = [[FindWindowController alloc] initWithWindowNibName: @"FindWindow"];
    }
  
  return singleInstance;
}

@end


//
// FindWindowController Private implementation
//
@implementation FindWindowController (Private)

- (void) _folderCloseCompleted: (NSNotification *) theNotification
{
  if ([[theNotification userInfo] objectForKey: @"Folder"] == _folder)
    {
      [foundLabel setStringValue: @""];
      [_indexes removeAllObjects];
      DESTROY(_folder);
      _location = 0;
    }
}


//
//
//
- (void) _folderSearchCompleted: (NSNotification *) theNotification
{
  MailWindowController *aMailWindowController;
  NSArray *allResults;
  id o;

  allResults = [[theNotification userInfo] objectForKey: @"Results"];
  o = [theNotification object];

  ASSIGN(_folder, (CWFolder *)[[theNotification userInfo] objectForKey: @"Folder"]);

  aMailWindowController = [[Utilities windowForFolderName: [_folder name]  store: o] delegate];

  if (!aMailWindowController)
    {
      DESTROY(_folder);
      return;
    }

  if ([[aMailWindowController folder] isKindOfClass: [CWIMAPFolder class]])
    {
      [aMailWindowController stopAnimation];
      [self _setState: YES];
    }
  
  // We get all the indexes from our messages found
  if ([allResults count])
    {
      id dataView;
      
      int previous_row_selected;
      
      dataView = [aMailWindowController dataView];	  
      previous_row_selected = [dataView selectedRow];
      
      [dataView deselectAll: nil];
      [_indexes removeAllObjects];
      
      // We add the index of our rows.. and we select the rows..
      [[FindWindowController singleInstance] _selectIndexesFromResults: allResults
					     controller: aMailWindowController];
      
      // If we found only one result, we automatically scroll to that row
      if ([allResults count] == 1 && [_indexes count] > 0)
	{
	  [dataView scrollRowToVisible: [[_indexes objectAtIndex: 0] intValue]];
	}
      
      [dataView setNeedsDisplay: YES];
    }
  else
    {
      NSBeep();
    }

  [foundLabel setStringValue: [NSString stringWithFormat: @"%i found", [allResults count]]];
  ADD_CONSOLE_MESSAGE(_(@"Done searching. %d results found."), [allResults count]);
}


//
//
//
- (void) _folderSearchFailed: (NSNotification *) theNotification
{
  [foundLabel setStringValue: _(@"Search failed.")];
  [self _setState: YES];
}


//
//
//
- (void) _selectIndexesFromResults: (NSArray *) theResults
                        controller: (MailWindowController *) theMailWindowController
{
  NSArray *allMessages;

  int i, index, count;
  id tableView;
 
  tableView = [theMailWindowController dataView];

  allMessages = [theMailWindowController allMessages];
  count = [theResults count];
  
  for (i = 0; i < count; i++)
    {
      index = [allMessages indexOfObject: [theResults objectAtIndex: i]];
      
      if ( index != NSNotFound )
	{
	  [_indexes addObject: [NSNumber numberWithInt: index]];
	  [tableView selectRow: index  byExtendingSelection: YES];
	}
    }
}

//
//
//
- (void) _setState: (BOOL) theState
{
  [findAllButton setEnabled: theState];
  [nextButton setEnabled: theState];
  [previousButton setEnabled: theState];
}

@end
