/*
     This file is part of GNUnet.
     (C) 2005, 2006 Christian Grothoff (and other contributing authors)

     GNUnet is free software; you can redistribute it and/or modify
     it under the terms of the GNU General Public License as published
     by the Free Software Foundation; either version 2, or (at your
     option) any later version.

     GNUnet is distributed in the hope that it will be useful, but
     WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
     General Public License for more details.

     You should have received a copy of the GNU General Public License
     along with GNUnet; see the file COPYING.  If not, write to the
     Free Software Foundation, Inc., 59 Temple Place - Suite 330,
     Boston, MA 02111-1307, USA.
*/

/**
 * @file src/plugins/fs/download.c
 * @brief code for downloading with gnunet-gtk
 * @author Christian Grothoff
 */

#include "platform.h"
#include "gnunetgtk_common.h"
#include "download.h"
#include "search.h"
#include "fs.h"
#include <extractor.h>

typedef struct DL {
  struct DL * next;
  struct ECRS_URI * uri;
  char * filename;
  char * finalName;
  GtkTreeRowReference * rr;
  GtkTreeModel * model;
} DownloadList;

static DownloadList * head;

static GtkTreeStore * summary;

static int addFilesToDirectory
  (const ECRS_FileInfo * fi,
   const HashCode512 * key,
   int isRoot,
   void * closure) {
  struct ECRS_URI * uri = closure;
  DownloadList * pos;

  if (isRoot == YES)
    return OK;
  FSUI_trackURI(fi);
  pos = head;
  while (pos != NULL) {
    if (ECRS_equalsUri(uri,
                       pos->uri))
      break;
    pos = pos->next;
  }
  if (pos == NULL) {
    BREAK(); /* odd */
  } else {
    GtkTreeIter iter;
    GtkTreeIter child;
    int i;
    GtkTreePath * path;

    if (! gtk_tree_row_reference_valid(pos->rr))
      return SYSERR;
    path = gtk_tree_row_reference_get_path(pos->rr);
    gtk_tree_model_get_iter(GTK_TREE_MODEL(pos->model),
                            &iter,
                            path);
    gtk_tree_path_free(path);
    for (i=gtk_tree_model_iter_n_children(pos->model,
                                          &iter)-1;i>=0;i--) {
      if (TRUE == gtk_tree_model_iter_nth_child(pos->model,
                                                &child,
                                                &iter,
                                                i)) {
        struct ECRS_URI * uri;
        uri = NULL;
        gtk_tree_model_get(pos->model,
                           &child,
                           SEARCH_URI, &uri,
                           -1);
        if ( (uri != NULL) &&
             (ECRS_equalsUri(uri,
                             fi->uri)) )
          return OK;
      }
    }
    gtk_tree_store_append(GTK_TREE_STORE(pos->model),
                          &child,
                          &iter);
    addEntryToSearchTree(GTK_TREE_STORE(pos->model),
                         &child,
                         fi->uri,
                         fi->meta);
  }
  return OK;
}

static void initiateDownload(GtkTreeModel * model,
                             GtkTreePath * path,
                             GtkTreeIter * iter,
                             gpointer unused) {
  struct ECRS_URI * uri;
  struct ECRS_MetaData * meta;
  char * uri_name;
  char * final_download_destination;
  char * final_download_dir;
  DownloadList * list;
  GtkTreeIter iiter;
  GtkWidget * spin;
  const char * name;
  const char * mime;
  unsigned int anon;
  GtkTreePath *dirTreePath;
  char *dirPath;
  unsigned int dirPathLen;

  uri = NULL;
  meta = NULL;
  name = NULL;
  mime = NULL;
  gtk_tree_model_get(model,
                     iter,
                     SEARCH_NAME, &name,
                     SEARCH_URI, &uri,
                     SEARCH_META, &meta,
                     SEARCH_MIME, &mime,
                     -1);
  if (uri == NULL) {
    BREAK();
    return;
  }
  if (! ECRS_isFileUri(uri)) {
    if (ECRS_isNamespaceUri(uri)) {
      /* start namespace search; would probably be better
	 to add this as a subtree, but for simplicity
	 we'll just add it as a new tab for now */
      if (OK == FSUI_startSearch(ctx,
				 anon,
				 uri))
	openTabForSearch(NULL,
			 uri,
			 anon,
			 0,
			 NULL);
      return;
    } else {
      BREAK(); /* unsupported URI type (i.e. ksk or loc) */
      return;
    }
  }


  uri_name = ECRS_uriToString(uri);
  if ( (uri_name == NULL) ||
       (strlen(uri_name) <
        strlen(ECRS_URI_PREFIX) +
        strlen(ECRS_FILE_INFIX)) ) {
    BREAK();
    FREENONNULL(uri_name);
    return;
  }

  if (name == NULL)
    name = uri_name;

  final_download_dir = getFileName("FS",
				   "INCOMINGDIR",
				   _("You must specify a directory in the configuration"
				     " in section `%s' under `%s'."));
  mkdirp(final_download_dir);


  /* If file is inside a directory, get the full path */
  dirTreePath = gtk_tree_path_copy(path);
  dirPath = MALLOC(1);
  dirPath[0] = '\0';
  dirPathLen = 0;
  while (gtk_tree_path_get_depth(dirTreePath) > 1) {
    const char * dirname;
    char * new;

    if (! gtk_tree_path_up(dirTreePath))
      break;

    if (!gtk_tree_model_get_iter(model,
				 &iiter,
				 dirTreePath))
      break;
    gtk_tree_model_get(model,
                       &iiter,
                       SEARCH_NAME, &dirname,
                       -1);
    dirPathLen = strlen(dirPath) + strlen(dirname) + strlen(DIR_SEPARATOR_STR) + 1;
    new = MALLOC(dirPathLen + 1);
    strcpy(new, dirname);
    if (new[strlen(new)-1] != DIR_SEPARATOR)
      strcat(new, DIR_SEPARATOR_STR);
    strcat(new, dirPath);
    FREE(dirPath);
    dirPath = new;
  }
  gtk_tree_path_free(dirTreePath);


  /* construct completed/directory/real-filename */
  final_download_destination = MALLOC(strlen(final_download_dir) + 2 +
               strlen(name) + strlen(GNUNET_DIRECTORY_EXT) +
	       strlen(dirPath));
  strcpy(final_download_destination, final_download_dir);
  if (final_download_destination[strlen(final_download_destination)-1] != DIR_SEPARATOR)
    strcat(final_download_destination,
           DIR_SEPARATOR_STR);
  strcat(final_download_destination, dirPath);
  mkdirp(final_download_destination);
  strcat(final_download_destination, name);
  if ( (final_download_destination[strlen(final_download_destination) - 1] == '/') ||
       (final_download_destination[strlen(final_download_destination) - 1] == '\\') )
    final_download_destination[strlen(final_download_destination) - 1] = '\0'; 
  /* append ".gnd" if needed (== directory and .gnd not present) */
  if ( (mime != NULL) && 
       (0 == strcmp(mime, GNUNET_DIRECTORY_MIME)) &&
       ( (strlen(final_download_destination) < strlen(GNUNET_DIRECTORY_EXT)) ||
	 (0 != strcmp(&final_download_destination[strlen(final_download_destination) - strlen(GNUNET_DIRECTORY_EXT)],
		      GNUNET_DIRECTORY_EXT)) ) )
    strcat(final_download_destination, GNUNET_DIRECTORY_EXT);
    
  /* setup visualization */
  list = MALLOC(sizeof(DownloadList));
  list->next = head;
  list->rr = NULL;
  list->model = NULL;
  if (YES == ECRS_isDirectory(meta)) {
    list->rr = gtk_tree_row_reference_new(model, path);
    list->model = model;
  }
  list->uri = ECRS_dupUri(uri);
  list->filename = final_download_destination;
  list->finalName = MALLOC(strlen(final_download_dir) + strlen(dirPath) + strlen(name) + 2);
  strcpy(list->finalName, final_download_dir);
  if (final_download_dir[strlen(final_download_dir)-1] != DIR_SEPARATOR)
    strcat(list->finalName, DIR_SEPARATOR_STR);
  strcat(list->finalName, dirPath);
  mkdirp(list->finalName);
  strcat(list->finalName, name);
  head = list;
  gtk_tree_store_insert(summary,
                        &iiter,
                        NULL,
                        0);
  gtk_tree_store_set(summary,
                     &iiter,
                     DOWNLOAD_FILENAME, final_download_destination,
                     DOWNLOAD_SHORTNAME, name,
                     DOWNLOAD_SIZE, ECRS_fileSize(uri),
                     DOWNLOAD_PROGRESS, 0, /* progress */
                     DOWNLOAD_URISTRING, uri_name,
                     DOWNLOAD_URI, ECRS_dupUri(uri),
                     DOWNLOAD_TREEPATH, list->rr, /* internal: row reference! */
                     DOWNLOAD_DIRPATH, dirPath,                     
                     -1);
  FREE(uri_name);
  FREE(dirPath);
  FREENONNULL(final_download_dir);


  spin = getAnonymityButtonFromTM(model);
  if (spin == NULL) {
    BREAK();
    anon = 1;
  } else {
    anon = gtk_spin_button_get_value_as_int
      (GTK_SPIN_BUTTON(spin));
  }

  addLogEntry(_("Downloading `%s'"), name);
  FSUI_startDownload(ctx,
                     anon,
                     uri,
                     final_download_destination);
}

void on_downloadButton_clicked(GtkWidget * treeview,
                               GtkWidget * downloadButton) {
  GtkTreeSelection * selection;

  selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(treeview));
  gtk_tree_selection_selected_foreach
    (selection,
     &initiateDownload,
     NULL);
}


/**
 */
void displayDownloadUpdate(const struct ECRS_URI * uri,
                           unsigned long long completed,
                           const char * data,
                           unsigned int size) {
  GtkTreeIter iter;
  unsigned int val;
  unsigned long long total;
  struct ECRS_URI * u;
  struct ECRS_MetaData * meta;


  if (gtk_tree_model_get_iter_first(GTK_TREE_MODEL(summary),
                                    &iter)) {
    do {
      gtk_tree_model_get(GTK_TREE_MODEL(summary),
                         &iter,
                         DOWNLOAD_SIZE, &total,
                         DOWNLOAD_URI, &u,
                         -1);
      if (u == NULL)
        return;
      if (ECRS_equalsUri(u, uri)) {
        if (total != 0)
          val = completed * 100 / total;
        else
          val = 100;
        gtk_tree_store_set(summary,
                           &iter,
                           DOWNLOAD_PROGRESS, val,
                           -1);
        break;
      }
    } while (gtk_tree_model_iter_next(GTK_TREE_MODEL(summary),
                                      &iter));
  }
  meta = NULL;
  ECRS_listDirectory(data,
                     size,
                     &meta,
                     &addFilesToDirectory,
                     (void*)uri);
  if (meta != NULL)
    ECRS_freeMetaData(meta);
}

/**
 */
void displayDownloadComplete(const struct ECRS_URI * uri,
                             const char * filename) {
  unsigned long long size;
  char * data;
  int fd;
  struct ECRS_MetaData * meta;
  DownloadList * pos;

  LOG(LOG_DEBUG,
      "Download '%s' complete\n",
      filename);
  pos = head;
  while (pos != NULL) {
    if (ECRS_equalsUri(uri,
                       pos->uri))
      break;
    pos = pos->next;
  }
  
  /* Not available for resumed downloads */
  if (pos != NULL) {
    if ( (pos->rr != NULL) &&
         (gtk_tree_row_reference_valid(pos->rr)) ) { 
      /* update directory view (if applicable!) */
      if (OK == getFileSize(filename, &size)) {
	LOG(LOG_DEBUG,
	    "Updating directory view of '%s'\n",
	    filename);

	meta = NULL;
	fd = fileopen(filename, O_RDONLY);
	if (fd == -1) {
	  LOG_FILE_STRERROR(LOG_ERROR, "open", filename);
	} else {
	  data = MMAP(NULL,
		      size,
		      PROT_READ,
		      MAP_SHARED,
		      fd,
		      0);
	  if (data == MAP_FAILED) {
	    LOG_FILE_STRERROR(LOG_ERROR, "mmap", filename);
	  } else {
	    if (data != NULL) {
	      ECRS_listDirectory(data,
				 size,
				 &meta,
				 &addFilesToDirectory,
				 (void*)uri);
	      MUNMAP(data, size);
	    }
	  }
	  CLOSE(fd);
	}
	if (meta != NULL)
	  ECRS_freeMetaData(meta);	
      }
    }
  }
}

static int delDownloadView(void * cls,
                           const struct FSUI_DownloadList * pos,
                           const char * filename,
                           const struct ECRS_URI * uri,
                           unsigned long long filesize,
                           unsigned long long bytesCompleted,
                           int isRecursive,
                           unsigned int anonymityLevel) {
  GtkTreeIter iter;
  char * f;
  char * fn;
  struct ECRS_URI * u;

  if (gtk_tree_model_get_iter_first(GTK_TREE_MODEL(summary),
                                    &iter)) {
    do {
      gtk_tree_model_get(GTK_TREE_MODEL(summary),
                         &iter,
                         DOWNLOAD_FILENAME, &f,
                         DOWNLOAD_URI, &u,
                         -1);
                         
      f = strrchr(f, DIR_SEPARATOR);
      fn = strrchr(filename, DIR_SEPARATOR);
                         
      if ( (ECRS_equalsUri(u, uri)) &&
           (0 == strcmp(f, fn)) ) {
        gtk_tree_store_remove(summary,
                              &iter);
        break;
      }
    } while (gtk_tree_model_iter_next(GTK_TREE_MODEL(summary),
                                      &iter));
  }
  return OK;
}

void on_clearCompletedDownloadsButton_clicked(void * unused,
                                              GtkWidget * clearButton) {
  FSUI_clearCompletedDownloads(ctx,
                               &delDownloadView,
                               NULL);
}

struct aDC_closure {
  struct ECRS_URI * u;
  char * fn;
};

static void * shutdownCode(void * c) {
  struct aDC_closure * cls = c;
  FSUI_stopDownload(ctx,
                    cls->u,
                    cls->fn);
  return NULL;
}

static void abortDownloadCallback(GtkTreeModel * model,
                                  GtkTreePath * path,
                                  GtkTreeIter * iter,
                                  GtkTreeStore * tree) {
  struct aDC_closure cls;

  GNUNET_ASSERT(model == GTK_TREE_MODEL(summary));
  gtk_tree_model_get(model,
                     iter,
                     DOWNLOAD_URI, &cls.u,
                     DOWNLOAD_FILENAME, &cls.fn,
                     -1);
  run_with_save_calls(&shutdownCode,
                      &cls);
  gtk_tree_store_remove(summary,
                        iter);
  if (cls.u != NULL)
    ECRS_freeUri(cls.u);
}

void on_abortDownloadButton_clicked(void * unused,
                                    GtkWidget * clearButton) {
  GtkTreeSelection * selection;
  GtkWidget * downloadList;

  downloadList = glade_xml_get_widget(getMainXML(),
                                      "activeDownloadsList");
  selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(downloadList));
  gtk_tree_selection_selected_foreach
    (selection,
     (GtkTreeSelectionForeachFunc) &abortDownloadCallback,
     NULL);
}

static int addDownloadView(void * cls,
                           const struct FSUI_DownloadList * pos,
                           const char * filename,
                           const struct ECRS_URI * uri,
                           unsigned long long filesize,
                           unsigned long long bytesCompleted,
                           int isRecursive,
                           unsigned int anonymityLevel) {
  GtkTreeIter iiter;
  int progress;
  char * uriname;
  const char * sname;

  if (filesize != 0)
    progress = bytesCompleted * 100 / filesize;
  else
    progress = 100;
  uriname = ECRS_uriToString(uri);
  gtk_tree_store_insert(summary,
                        &iiter,
                        NULL,
                        0);
  sname = &filename[strlen(filename)-1];
  while ( (sname > filename) &&
	  (sname[-1] != '/') &&
	  (sname[-1] != '\\') )
    sname--;
  gtk_tree_store_set(summary,
                     &iiter,
                     DOWNLOAD_FILENAME, filename,
                     DOWNLOAD_SHORTNAME, sname,
                     DOWNLOAD_SIZE, filesize,
                     DOWNLOAD_PROGRESS, progress,
                     DOWNLOAD_URISTRING, uriname,
                     DOWNLOAD_URI, ECRS_dupUri(uri),
                     DOWNLOAD_TREEPATH, NULL,
                     -1);
  FREE(uriname);
  return OK;
}


void fs_download_start() {
  GtkWidget * downloadList;
  GtkCellRenderer * renderer;
  int col;


  downloadList = glade_xml_get_widget(getMainXML(),
                                      "activeDownloadsList");
  summary =
    gtk_tree_store_new(DOWNLOAD_NUM,
                       G_TYPE_STRING, /* name (full-path file name) */
                       G_TYPE_STRING, /* name (user-friendly name) */
                       G_TYPE_UINT64,  /* size */
                       G_TYPE_INT,  /* progress */
                       G_TYPE_STRING, /* uri */
                       G_TYPE_POINTER,  /* url */
                       G_TYPE_POINTER, /* internal: gtk tree path / NULL */
                       G_TYPE_STRING); /* directory path if file is inside a dir */
  gtk_tree_view_set_model(GTK_TREE_VIEW(downloadList),
                          GTK_TREE_MODEL(summary));
  renderer = gtk_cell_renderer_progress_new();
  col = gtk_tree_view_insert_column_with_attributes(GTK_TREE_VIEW(downloadList),
                                              -1,
                                              _("Name"),
                                              renderer,
                                              "value", DOWNLOAD_PROGRESS,
                                              "text", DOWNLOAD_SHORTNAME,
                                              NULL);
  gtk_tree_view_column_set_resizable(gtk_tree_view_get_column(GTK_TREE_VIEW(downloadList),
                                              col - 1),
                                              TRUE);
  renderer = gtk_cell_renderer_text_new();
  col = gtk_tree_view_insert_column_with_attributes(GTK_TREE_VIEW(downloadList),
                                              -1,
                                              _("Size"),
                                              renderer,
                                              "text", DOWNLOAD_SIZE,
                                              NULL);
  gtk_tree_view_column_set_resizable(gtk_tree_view_get_column(GTK_TREE_VIEW(downloadList),
                                              col - 1),
                                              TRUE);
  renderer = gtk_cell_renderer_text_new();
  col = gtk_tree_view_insert_column_with_attributes(GTK_TREE_VIEW(downloadList),
                                              -1,
                                              _("URI"),
                                              renderer,
                                              "text", DOWNLOAD_URISTRING,
                                              NULL);
  gtk_tree_view_column_set_resizable(gtk_tree_view_get_column(GTK_TREE_VIEW(downloadList),
                                              col - 1),
                                              TRUE);
  FSUI_listDownloads(ctx,
                     NULL,
                     &addDownloadView,
                     NULL);
}


void fs_download_stop() {
  GtkTreeIter iter;
  struct ECRS_URI * u;
  DownloadList * pos;
  char *dirPath;

  /* free URIs in summary model */
  if (! gtk_tree_model_get_iter_first(GTK_TREE_MODEL(summary),
                                      &iter))
    return;
  do {
    gtk_tree_model_get(GTK_TREE_MODEL(summary),
                       &iter,
                       DOWNLOAD_URI, &u,
                       DOWNLOAD_DIRPATH, &dirPath,
                       -1);
    gtk_tree_store_set(summary,
                       &iter,
                       DOWNLOAD_URI, NULL,
                       -1);
    if (u != NULL)
      ECRS_freeUri(u);
    FREENONNULL(dirPath);
  } while (gtk_tree_model_iter_next(GTK_TREE_MODEL(summary),
                                    &iter));
  while (head != NULL) {
    pos = head->next;
    ECRS_freeUri(head->uri);
    FREE(head->filename);
    gtk_tree_row_reference_free(head->rr);
    FREE(head);
    head = pos;
  }
}


/* end of download.c */
