/*
     This file is part of GNUnet.
     (C) 2005, 2006, 2007 Christian Grothoff (and other contributing authors)

     GNUnet is free software; you can redistribute it and/or modify
     it under the terms of the GNU General Public License as published
     by the Free Software Foundation; either version 2, or (at your
     option) any later version.

     GNUnet is distributed in the hope that it will be useful, but
     WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
     General Public License for more details.

     You should have received a copy of the GNU General Public License
     along with GNUnet; see the file COPYING.  If not, write to the
     Free Software Foundation, Inc., 59 Temple Place - Suite 330,
     Boston, MA 02111-1307, USA.
*/

/**
 * @file src/plugins/fs/namespace_search.c
 * @brief code for namespace-related operations on the search frame
 * @author Christian Grothoff
 */

#include "platform.h"
#include "gnunetgtk_common.h"
#include "fs.h"
#include "helper.h"
#include "meta.h"
#include <GNUnet/gnunet_util_crypto.h>
#include <GNUnet/gnunet_uritrack_lib.h>
#include <GNUnet/gnunet_namespace_lib.h>
#include <extractor.h>


/**
 * The spin button giving the rating for a particular namespace
 * has been changed.  Store the new rating for the namespace.
 */
void
on_namespaceRatingSpinButton_changed_fs (GtkWidget * dummy,
                                         GtkWidget * dummy2)
{
  GtkWidget *spin;
  GtkWidget *ncbe;
  GtkTreeModel *model;
  GtkTreeIter iter;
  char *encStr;
  char *description;
  int rating;
  int newrating;

  DEBUG_BEGIN ();
  spin = glade_xml_get_widget (getMainXML (), "namespaceRatingSpinButton");
  ncbe = glade_xml_get_widget (getMainXML (), "searchNamespaceComboBoxEntry");
  model = gtk_combo_box_get_model (GTK_COMBO_BOX (ncbe));
  description = NULL;
  encStr = NULL;
  if (TRUE == gtk_combo_box_get_active_iter (GTK_COMBO_BOX (ncbe), &iter))
    {
      gtk_tree_model_get (model,
                          &iter,
                          NS_SEARCH_DESCRIPTION, &description,
                          NS_SEARCH_ENCNAME, &encStr,
                          NS_SEARCH_RATING, &rating, -1);
      if ((description != NULL) && (0 == strcmp (description, _("globally"))))
        {
          /* just to be sure */
          gtk_widget_set_sensitive (spin, FALSE);
        }
      else
        {
          if (encStr != NULL)
            {
              newrating = gtk_spin_button_get_value (GTK_SPIN_BUTTON (spin));
              rating = NS_rankNamespace (ectx,
                                         cfg, encStr, newrating - rating);
              if (rating != newrating)
                {
                  /* concurrent modification? */
                  gtk_spin_button_set_value (GTK_SPIN_BUTTON (spin), rating);
                  GE_BREAK (ectx, 0);
                }
              gtk_list_store_set (GTK_LIST_STORE (model),
                                  &iter, NS_SEARCH_RATING, rating, -1);
            }
        }
    }
  else
    {
      /* FIXME: if enc2hash succeeds, we may want to keep this
         active */
      gtk_spin_button_set_value (GTK_SPIN_BUTTON (spin), 0);
      gtk_widget_set_sensitive (spin, FALSE);
    }
  if (description != NULL)
    free (description);
  if (encStr != NULL)
    free (encStr);
  DEBUG_END ();
}


/**
 * The namespace in the search window has changed.
 * Update the trust level (possibly changing sensitivity)
 * and set the search string to the root (if available).
 */
void
on_searchNamespaceComboBoxEntry_changed_fs (GtkWidget * dummy,
                                            GtkWidget * dummy2)
{
  GtkWidget *keyword;
  GtkWidget *spin;
  GtkWidget *ncbe;
  GtkTreeModel *model;
  GtkTreeIter iter;
  int rating;
  char *encStr;
  char *descStr;
  HashCode512 ns;
  HashCode512 root;
  EncName enc;

  DEBUG_BEGIN ();
  spin = glade_xml_get_widget (getMainXML (), "namespaceRatingSpinButton");
  ncbe = glade_xml_get_widget (getMainXML (), "searchNamespaceComboBoxEntry");
  model = gtk_combo_box_get_model (GTK_COMBO_BOX (ncbe));
  descStr = NULL;
  encStr = NULL;
  if (TRUE == gtk_combo_box_get_active_iter (GTK_COMBO_BOX (ncbe), &iter))
    {
      gtk_tree_model_get (model,
                          &iter,
                          NS_SEARCH_DESCRIPTION, &descStr,
                          NS_SEARCH_ENCNAME, &encStr,
                          NS_SEARCH_RATING, &rating, -1);
      if ((descStr != NULL) && (0 == strcmp (descStr, _("globally"))))
        {
          gtk_spin_button_set_value (GTK_SPIN_BUTTON (spin), 0);
          gtk_widget_set_sensitive (spin, FALSE);
        }
      else if (encStr != NULL)
        {
          enc2hash (encStr, &ns);
          gtk_spin_button_set_value (GTK_SPIN_BUTTON (spin), rating);
          gtk_widget_set_sensitive (spin, TRUE);
          keyword
            = glade_xml_get_widget (getMainXML (),
                                    "fssearchKeywordComboBoxEntry");
          if (OK == NS_getNamespaceRoot (ectx, cfg, encStr, &root))
            {
              hash2enc (&root, &enc);
              gtk_entry_set_text (GTK_ENTRY
                                  (gtk_bin_get_child (GTK_BIN (keyword))),
                                  (const gchar *) &enc);
            }
          else
            {
              gtk_entry_set_text (GTK_ENTRY
                                  (gtk_bin_get_child (GTK_BIN (keyword))),
                                  "");
            }
        }
    }
  else
    {
      gtk_spin_button_set_value (GTK_SPIN_BUTTON (spin), 0);
      gtk_widget_set_sensitive (spin, FALSE);
    }
  if (descStr != NULL)
    free (descStr);
  if (encStr != NULL)
    free (encStr);
  DEBUG_END ();
}

struct NewNamespaceInfo
{
  const char *namespaceName;
  const HashCode512 *namespaceId;
  const struct ECRS_MetaData *md;
  int rating;
};

static void *
saveDiscovery (void *cls)
{
  struct NewNamespaceInfo *nni = cls;

  GtkListStore *model;
  GtkTreeIter iter;
  EncName enc;
  char *name;
  struct ECRS_MetaData *dmd;
  GtkWidget *ncbe;
  char *desc;
  size_t n;
  int found;
  char *encName;

  DEBUG_BEGIN ();
  ncbe = glade_xml_get_widget (getMainXML (), "searchNamespaceComboBoxEntry");
  model = GTK_LIST_STORE (gtk_combo_box_get_model (GTK_COMBO_BOX (ncbe)));
  hash2enc (nni->namespaceId, &enc);
  if (nni->md == NULL)
    {
      dmd = NULL;
      desc = STRDUP ("");
    }
  else
    {
      dmd = ECRS_dupMetaData (nni->md);
      desc = ECRS_getFirstFromMetaData (dmd,
                                        EXTRACTOR_DESCRIPTION,
                                        EXTRACTOR_TITLE,
                                        EXTRACTOR_AUTHOR,
                                        EXTRACTOR_GENRE,
                                        EXTRACTOR_SUBJECT,
                                        EXTRACTOR_CREATOR,
                                        EXTRACTOR_PRODUCER,
                                        EXTRACTOR_GROUP,
                                        EXTRACTOR_CREATED_FOR,
                                        EXTRACTOR_SUMMARY,
                                        EXTRACTOR_OWNER, -1);
      if (desc == NULL)
        desc = STRDUP ("");
    }

  n = strlen (desc) + 64;
  name = MALLOC (n);
  SNPRINTF (name, n, "%s: %.*s", desc, 20, &enc);
  /* check if present! */
  found = NO;
  if (gtk_tree_model_get_iter_first (GTK_TREE_MODEL (model), &iter))
    {
      do
        {
          gtk_tree_model_get (GTK_TREE_MODEL (model),
                              &iter, NS_SEARCH_ENCNAME, &encName, -1);
          if ((encName != NULL) &&
              (0 == strcmp (encName, (const char *) &enc)))
            found = YES;
          FREENONNULL (encName);
        }
      while ((found == NO) &&
             (gtk_tree_model_iter_next (GTK_TREE_MODEL (model), &iter)));
    }
  if (found == NO)
    {
      infoMessage (YES, _("Namespace found: %s - %.*s\n"), desc, 20, &enc);
      gtk_list_store_append (model, &iter);
      gtk_list_store_set (model,
                          &iter,
                          NS_SEARCH_DESCRIPTION, name,
                          NS_SEARCH_ENCNAME, &enc,
                          NS_SEARCH_METADATA, dmd,
                          NS_SEARCH_RATING, nni->rating, -1);
    }
  FREE (name);
  DEBUG_END ();

  return NULL;
}



/**
 * Call this function to inform the user about
 * newly found namespaces and to add them to the
 * list of available namespaces in the search
 * dialog.
 *
 * @param cls pass NULL
 * @param rating the local rating of the namespace
 * @return OK (always)
 */
int
namespace_discovered_cb (void *cls,
                         const char *namespaceName,
                         const HashCode512 * namespaceId,
                         const struct ECRS_MetaData *md, int rating)
{
  struct NewNamespaceInfo nni;
  nni.namespaceName = namespaceName;
  nni.namespaceId = namespaceId;
  nni.md = md;
  nni.rating = rating;
  gtkSaveCall (&saveDiscovery, &nni);
  return OK;
}
