/*
     This file is part of gnunet-qt.
     (C) 2007 Nils Durner (and other contributing authors)

     gnunet-qt is free software; you can redistribute it and/or modify
     it under the terms of the GNU General Public License as published
     by the Free Software Foundation; either version 2, or (at your
     option) any later version.

     gnunet-qt is distributed in the hope that it will be useful, but
     WITHOUT ANY WARRANTY; without even the implied warranty of
     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
     General Public License for more details.

     You should have received a copy of the GNU General Public License
     along with gnunet-qt; see the file COPYING.  If not, write to the
     Free Software Foundation, Inc., 59 Temple Place - Suite 330,
     Boston, MA 02111-1307, USA.
*/

/**
 * @file src/plugins/fs/searchController.cc
 * @brief Controller for *all* searches
 * @author Nils Durner
 */

#include <QDir>
#include <QFile>

#include "gnunet_qt_common.h"
#include "searchController.h"

GFSSearchController::GFSSearchController(class GFSPlugin *fs)
{
  this->fs = fs;
  searchSummaryCntrl = new GFSSearchSummaryController(fs);
}

GFSSearchController::~GFSSearchController()
{
  delete searchSummaryCntrl;
}

static int insertMetaData(EXTRACTOR_KeywordType type, const char *data,
  void *cls)
{
  QAbstractItemModel *model;
  QModelIndex *rowIndex, idx;
  QString content;

  rowIndex = (QModelIndex *) cls;
  model = (QAbstractItemModel *) rowIndex->model();
  idx = model->index(rowIndex->row(), MODEL_IDX(type), rowIndex->parent());
  
  content = model->data(idx).toString();

  if (content != "")
    content += "\n";
  content += QString::fromUtf8(data);
  
  model->setData(idx, QVariant(content), Qt::DisplayRole);

  return OK;
}

GFSSearchInfo *GFSSearchController::started(struct FSUI_SearchList *list,
  const struct ECRS_URI *uri, unsigned int resultCount, const ECRS_FileInfo *results)
{
  GEvent *event;
  GFSSearch *view = NULL;
  GFSNewSearchInfo info;
  GFSSearchInfo *searchInfo;
  QSemaphore sem;


  info.model = new QStandardItemModel;
  info.uri = uri;
  event = new GEvent((QEvent::Type) GFSPlugin::NewSearch, &info, (void **) &view, &sem);
  
  GEventDispatcher::postEvent(fs, event);
  sem.acquire(1); // wait until event is processed
  
  searches.insert(info.uri, list);
  
  connect(view, SIGNAL(closeSearchWnd(GFSEcrsUri &)), this, SLOT(closed(GFSEcrsUri &)));
  connect(view,
    SIGNAL(download(QStandardItemModel *, GFSEcrsUri &, QModelIndexList, int, bool)),
    this,
    SLOT(download(QStandardItemModel *, GFSEcrsUri &, QModelIndexList, int, bool)));
  
  searchSummaryCntrl->searchStarted(list, uri);
  
  // insert search results from last session
  while (resultCount > 0)
  {
    result(info.model, list, &results[resultCount - 1]);
    resultCount--;
  }
  
  searchInfo = new GFSSearchInfo;
  searchInfo->model = info.model;
  searchInfo->searchWindow = view;
  searchInfo->handle = list;

  return searchInfo;
}

void GFSSearchController::addSearchResult(QStandardItemModel *model,
  QModelIndex parent, const ECRS_FileInfo *info)
{
  int row;
  GFSEcrsUri ecrsUri;
  GFSEcrsMetaData meta;
  size_t thumbSize;
  unsigned char *thumb;
  QStandardItem *child;
  QModelIndex childIdx, itemIdx;
  
  child = new QStandardItem;
  child->setColumnCount(SEARCH_MAX_COLUMN + 1);
  row = model->rowCount(parent);

  if (parent.isValid())
    model->itemFromIndex(parent)->appendRow(child);
  else
    model->appendRow(child);

  childIdx = model->index(row, 0, parent);

  // insert flat meta data
  ECRS_getMetaData(info->meta, &insertMetaData, &childIdx);
  
  // insert thumbnail data
  thumbSize = ECRS_getThumbnailFromMetaData(info->meta, &thumb);
  if (thumbSize)
  {
    QByteArray data((char *) thumb, thumbSize);
    
    itemIdx = model->index(row, MODEL_IDX(EXTRACTOR_THUMBNAIL_DATA), parent);
    model->setData(itemIdx, QVariant(data));
    FREE(thumb);
  }

  // make directories expandable
  itemIdx = model->index(row, MODEL_IDX(EXTRACTOR_MIMETYPE), parent);
  if (model->data(itemIdx).toString() == "application/gnunet-directory")
  {
    int row;
    QStandardItem *item;

    item = new QStandardItem;
    item->setColumnCount(SEARCH_MAX_COLUMN + 1);
    row = model->rowCount(childIdx);
    child->appendRow(item);

    itemIdx = model->index(0, 0, childIdx);
    model->setData(itemIdx, QVariant(1), Qt::UserRole);
  }

  // insert serialized URI (used by download)
  ecrsUri = info->uri;
  itemIdx = model->index(row, SEARCH_URI_COLUMN, parent);
  model->setData(itemIdx, QVariant(ecrsUri.serialized()));

  // insert serialized meta data (used by download)
  meta = info->meta;
  itemIdx = model->index(row, SEARCH_META_COLUMN, parent);
  model->setData(itemIdx, QVariant(meta.serialized()));
}

/**
 * @brief Add a search result to the model
 * @param searchInfo information about the search
 * @param info file information
 * @notice called by FSUI
 */
void GFSSearchController::result(GFSSearchInfo *searchInfo,
  const ECRS_FileInfo *info)
{
  result(searchInfo->model, searchInfo->handle, info);
}

/**
 * @brief Adds a search result to the model
 * @param model the model
 * @param list FSUI search handle
 * @param info file information
 * @notice called by started() if there are results from the last session
 */
void GFSSearchController::result(QStandardItemModel *model,
  const struct FSUI_SearchList *list, const ECRS_FileInfo *info)
{
  addSearchResult(model, QModelIndex(), info);

  searchSummaryCntrl->searchResult(list);
}

void GFSSearchController::stopped(GFSSearchInfo *info)
{
  GEvent *event;
  QSemaphore sem;
  void *lastTab;
  
  event = new GEvent((QEvent::Type) GFSPlugin::CloseSearch, info->searchWindow,
    (void **) &lastTab, &sem);
  GEventDispatcher::postEvent(fs, event);
  sem.acquire(1); // wait until event is processed
  
  searchSummaryCntrl->searchStopped(info->handle);
  
  if (!lastTab)
    info->model->deleteLater();
    
  delete info;
}

void GFSSearchController::state(GFSSearchInfo *info, FSUI_EventType event)
{
  searchSummaryCntrl->searchState(info->handle, event);
}

void GFSSearchController::closed(GFSEcrsUri &uri)
{
  FSUI_SearchList *list = searches[uri];

  FSUI_abortSearch(fs->context(), list);
  FSUI_stopSearch(fs->context(), list);
}

void GFSSearchController::download(QStandardItemModel *model, GFSEcrsUri &uri,
  QModelIndexList indexes, int anonymity, bool recurse)
{
  QModelIndexList::iterator it;

  for (it = indexes.begin(); it != indexes.end(); it++)
  {
    QModelIndex idx;
    FSUI_SearchList *handle;
    QPersistentModelIndex persistIdx(*it);
    QString gnPath;
    
    // get URI
    idx = model->index(it->row(), SEARCH_URI_COLUMN, it->parent());
    GFSEcrsUri fileUri(model->data(idx).toString());
    
    // get meta data
    idx = model->index(it->row(), SEARCH_META_COLUMN, it->parent());
    GFSEcrsMetaData meta(model->data(idx).toByteArray());
    
    // get filename
    idx = model->index(it->row(), MODEL_IDX(EXTRACTOR_FILENAME), it->parent());
    QString file = model->data(idx).toString();
    
    // get GNUnet path
    idx = idx.parent();
    while (idx.isValid())
    {
      gnPath = model->data(idx).toString() + gnPath;
      idx = idx.parent();
    }
    
    // start download
    handle = searches[uri];
    GNUNETQT_ASSERT(handle);
    
    fs->download(persistIdx, handle, fileUri, meta, gnPath, file, anonymity, recurse);
  }
}

/**
 * @brief Callback that adds content of a directory to the search list
 */
static int addFilesToDirectory(const ECRS_FileInfo *fi, const HashCode512 *key,
  int isRoot, void *closure)
{
  Q_UNUSED(key)
  
  QPersistentModelIndex *pIdx;
  QStandardItemModel *model;
  QModelIndex parentIdx, child;

  if (isRoot == YES)
    return OK;
  
  GFSEcrsUri uri(fi->uri);

  pIdx = (QPersistentModelIndex *) closure;
  model = (QStandardItemModel *) pIdx->model();
  parentIdx = model->index(pIdx->row(), 0, pIdx->parent());
  
  /* check for existing entry -- this function maybe called multiple
     times for the same directory entry */
  child = parentIdx.child(0, SEARCH_URI_COLUMN);
  while (child.isValid())
  {
    if (GFSEcrsUri(model->data(child).toString()) == uri)
      return OK;
    
    child = child.sibling(child.row() + 1, SEARCH_URI_COLUMN);
  }
  
  GFSSearchController::addSearchResult(model, parentIdx, fi);
  
  return OK;
}

void GFSSearchController::downloadCompleted(QPersistentModelIndex &idx, GString file)
{
  QModelIndex index;
  QAbstractItemModel *model;
  struct ECRS_MetaData *meta;
  
  model = (QAbstractItemModel *) idx.model();

  index = model->index(idx.row(), MODEL_IDX(EXTRACTOR_MIMETYPE), idx.parent());
  if (model->data(index).toString() == "application/gnunet-directory")
  {
    // remove dummy child
    index = model->index(idx.row(), 0, idx.parent());
    model->removeRow(0, index);
    
    // read directory
    if (file.endsWith("/") || file.endsWith("\\"))
      file = file.left(file.length() - 1);
    file += ".gnd";
  
    QFile gnDir(file);
    gnDir.open(QIODevice::ReadOnly);
    QByteArray data = gnDir.readAll();
    
    if (!data.size())
      return;
      
    meta = NULL;
    ECRS_listDirectory(fs->errorContext(), data.data(), data.size(), &meta,
      &addFilesToDirectory, &idx);
  
    if (meta)
      ECRS_freeMetaData(meta);
  }
}

/* end of searchController.cc */
