
// created 06.2003 by Stefan Kleine Stegemann
// 
// licensed under GPL


#include "XPDFBridge.h"
#include "PDFDoc.h"
#include "Outline.h"
#include "SplashBitmap.h"
#include "SplashOutputDev.h"
#include "TextOutputDev.h"
#include "GString.h"
#include "GlobalParams.h"
#include "GList.h"
#include "UnicodeMap.h"
#include "Link.h"

#include <stdlib.h>
#include <string.h>

#include <typeinfo>



/* ----------------------------------------------------------- */

/*
 * Initialization and destruction.
 */
 
static int    initialized = 0;
static double dpi         = 0.0;

void XPDF_Initialize(double _dpi)
{
   globalParams = new GlobalParams(NULL);
   dpi          = _dpi;

   initialized = 1;
}


void XPDF_Destroy(void)
{
   initialized = 0;
   delete globalParams;
}


int  XPDF_IsInitialized(void)
{
   return initialized;
}


double XPDF_DPI()
{
   return dpi;
}


/* 
 * PDFDoc class 
 */

#define TO_PDFDoc(object) (static_cast<PDFDoc*>(object))

XPDFObject PDFDoc_create(const char* filename,
                         const char* ownerPassword,
                         const char* userPassword)
{
   GString* gsFilename;
   GString* gsOwnerPassword;
   GString* gsUserPassword;
   PDFDoc*  doc;

   gsFilename = new GString(filename);
   gsOwnerPassword = (ownerPassword != NULL ? new GString(ownerPassword) : NULL);
   gsUserPassword  = (userPassword != NULL ? new GString(userPassword) : NULL);

   doc = new PDFDoc(gsFilename, gsOwnerPassword, gsUserPassword);

   return (XPDFObject)doc;
}


void PDFDoc_delete(XPDFObject pdfDoc)
{
   delete TO_PDFDoc(pdfDoc);
}


int PDFDoc_isOk(XPDFObject pdfDoc)
{
   return TO_PDFDoc(pdfDoc)->isOk();
}


int PDFDoc_getErrorCode(XPDFObject pdfDoc)
{
   return TO_PDFDoc(pdfDoc)->getErrorCode();
}


double PDFDoc_getPageWidth(XPDFObject pdfDoc, int pageNum)
{
   return TO_PDFDoc(pdfDoc)->getPageWidth(pageNum);
}


double PDFDoc_getPageHeight(XPDFObject pdfDoc, int pageNum)
{
   return TO_PDFDoc(pdfDoc)->getPageHeight(pageNum);
}


int PDFDoc_getPageRotate(XPDFObject pdfDoc, int pageNum)
{
   return TO_PDFDoc(pdfDoc)->getPageRotate(pageNum);
}


int PDFDoc_getNumPages(XPDFObject pdfDoc)
{
   return TO_PDFDoc(pdfDoc)->getNumPages();
}

 
const char* PDFDoc_getMetaData(XPDFObject pdfDoc)
{
   return (const char*)TO_PDFDoc(pdfDoc)->readMetadata();
}


/* ----------------------------------------------------------- */

/*
 * Seraching in PDFDocs.
 */
int PDFUtil_FindText(XPDFObject pdfDoc,
                     const char* text,
                     int* pageA,
                     int toPage,   // -1 for whole document
                     double *xMin,
                     double* yMin,
                     double* xMax,
                     double *yMax,
                     char** textContext,
                     int* textContextLength)
{
   XPDF_AcquireLock();

   PDFDoc* doc = TO_PDFDoc(pdfDoc);
   int found = 0;
   int startPage = *pageA;
   GBool top;

   TextOutputDev* textOut = new TextOutputDev(NULL, gTrue, gFalse, gFalse);
   if (!textOut->isOk())
   {
      delete textOut;
      fprintf(stderr, "unable to create TextOutputDev\n");
      fflush(stderr);
      XPDF_ReleaseLock();
      return -1;
   }


   int len = strlen(text);
   Unicode* u = (Unicode *)gmalloc(len * sizeof(Unicode));
   for (int i = 0; i < len; ++i)
   {
      u[i] = (Unicode)(text[i] & 0xff);
   }

   
   if ((*xMin == 0) && (*yMin == 0) && (*xMax == 0) && (*yMax == 0))
   {
      top = gTrue;
   }
   else
   {
      top = gFalse;
   }

   //fprintf(stderr, "start search at %f, %f, %f, %f\n", *xMin, *yMin, *xMax, *yMax);
   //fflush(stderr);

   // first search forward
   int page;
   int maxPage = (toPage != -1 ? toPage : doc->getNumPages());
   for (page = startPage; page <= maxPage; page++)
   {
      doc->displayPage(textOut, page, XPDF_DPI(), XPDF_DPI(), 0, gFalse, gTrue);

      if (page == startPage)
      {
         found =
            textOut->findText(u, len, top, gTrue, gFalse, gFalse, xMin, yMin, xMax, yMax);
      }
      else
      {
         found =
            textOut->findText(u, len, gTrue, gTrue, gFalse, gFalse, xMin, yMin, xMax, yMax);
      }

      if (found)
      {
         *pageA  = page;
         break;
      }
   }

   // continue search from the beginning of the document
   // if the whole document is searched
   if ((!found) && (toPage == -1))
   {
      for (page = 1; page <= startPage; page++)
      {
         doc->displayPage(textOut, page, XPDF_DPI(), XPDF_DPI(), 0, gFalse, gTrue);

         if (page == startPage)
         {
            found =
               textOut->findText(u, len, gTrue, gTrue, gFalse, gFalse, xMin, yMin, xMax, yMax);
         }
         else
         {
            found =
               textOut->findText(u, len, gTrue, gFalse, gFalse, gFalse, xMin, yMin, xMax, yMax);
         }

         if (found)
         {
            *pageA = page;
            break;
         }
      }
   }

   if (found)
   {
      /*
      fprintf(stderr, "%s found at %f, %f, %f, %f\n", text, *xMin, *yMin, *xMax, *yMax);
      fflush(stderr);
      */
      // Get the context of the found text (some text around the
      // position where the text occured)
      double ctxXMin = *xMin - FOUND_CONTEXT_LEFT;
      double ctxXMax = *xMax + FOUND_CONTEXT_RIGHT;

      if (ctxXMin < 0)
      {
         ctxXMax += ctxXMin * (-1);
         if (ctxXMax > doc->getPageWidth(*pageA))
         {
            ctxXMax = doc->getPageWidth(*pageA);
         }
      }

      if (ctxXMax > doc->getPageWidth(*pageA))
      {
         ctxXMin -= ctxXMax - doc->getPageWidth(*pageA);
         if (ctxXMin < 0)
         {
            ctxXMin = 0;
         }
      }

      GString *textContextStr = textOut->getText(ctxXMin, *yMin, ctxXMax, *yMax);
      if (textContextStr)
      {
         *textContext = (char*)calloc(1, textContextStr->getLength() * sizeof(char));
         memcpy(*textContext, textContextStr->getCString(), textContextStr->getLength());
         *textContextLength = textContextStr->getLength();
         delete textContextStr;
      }
      else
      {
         *textContext = NULL;
         *textContextLength = 0;
      }
   }

   delete textOut;

   XPDF_ReleaseLock();
   return found;
}


/* ----------------------------------------------------------- */


/*
 * Accessing the pdf content.
 */

void PDFUtil_GetText(XPDFObject pdfDoc,
                     int page,
                     double xMin,
                     double yMin,
                     double xMax,
                     double yMax,
                     char** textA,
                     int* length)
{
   XPDF_AcquireLock();

   PDFDoc* doc = TO_PDFDoc(pdfDoc);

   TextOutputDev* textOut = new TextOutputDev(NULL, gTrue, gFalse, gFalse);
   if (!textOut->isOk())
   {
      delete textOut;
      fprintf(stderr, "unable to create TextOutputDev\n");
      fflush(stderr);
      XPDF_ReleaseLock();
      return;
   }
   
   doc->displayPage(textOut, page, XPDF_DPI(), XPDF_DPI(), 0, gFalse, gTrue);

   GString* text = textOut->getText(xMin, yMin, xMax, yMax);
   if (text)
   {
      *textA = (char*)calloc(1, text->getLength() * sizeof(char));
      memcpy(*textA, text->getCString(), text->getLength());
      *length = text->getLength();
      delete text;
   }
   else
   {
      *textA  = NULL;
      *length = 0;
   }

   delete textOut;

   XPDF_ReleaseLock();
}


/* ----------------------------------------------------------- */

/*
 * Outline.
 */


int PDFOutline_HasOutline(XPDFObject pdfDoc)
{
   return (TO_PDFDoc(pdfDoc)->getOutline()->getItems() != NULL);
}


XPDFObject PDFOutline_GetOutlineItems(XPDFObject pdfDoc)
{
   return TO_PDFDoc(pdfDoc)->getOutline()->getItems();
}


int PDFOutline_CountItems(XPDFObject outlineItems)
{
   return static_cast<GList*>(outlineItems)->getLength();
}


XPDFObject PDFOutline_ItemAt(XPDFObject outlineItems, int index)
{
   OutlineItem* item = (OutlineItem*)static_cast<GList*>(outlineItems)->get(index);
   return item;
}


void PDFOutline_ItemOpen(XPDFObject outlineItem)
{
   static_cast<OutlineItem*>(outlineItem)->open();
}


void PDFOutline_ItemClose(XPDFObject outlineItem)
{
   static_cast<OutlineItem*>(outlineItem)->close();
}


int PDFOutline_HasKids(XPDFObject outlineItem)
{
   return static_cast<OutlineItem*>(outlineItem)->hasKids();
}


XPDFObject PDFOutline_GetKids(XPDFObject outlineItem)
{
   return static_cast<OutlineItem*>(outlineItem)->getKids();
}


// Note that the returned character buffer must be freed
// when no longer used
char* PDFOutline_GetTitle(XPDFObject outlineItem)
{
   OutlineItem* item = static_cast<OutlineItem*>(outlineItem);
   GString* enc = new GString("Latin1");
   UnicodeMap* uMap = globalParams->getUnicodeMap(enc);
   delete enc;

   GString* title = new GString();
   char buf[8];
   int i, n;
   for (i = 0; i < item->getTitleLength(); i++)
   {
      n = uMap->mapUnicode(item->getTitle()[i], buf, sizeof(buf));
      title->append(buf, n);
   }

   char* result = (char*)malloc((sizeof(char) * title->getLength()) + 1);
   strcpy(result, title->getCString());

   delete title;
   uMap->decRefCnt();

   return result;
}


int PDFOutline_GetTargetPage(XPDFObject outlineItem, XPDFObject pdfDoc)
{
   int page = 0;
   LinkAction* action = static_cast<OutlineItem*>(outlineItem)->getAction();
   LinkDest* dest = 0;

   if (action && (action->getKind() == actionGoTo))
   {
      if (static_cast<LinkGoTo*>(action)->getDest())
      {
         dest = static_cast<LinkGoTo*>(action)->getDest();
      }
      else
      {
         dest = 
            TO_PDFDoc(pdfDoc)->findDest(static_cast<LinkGoTo*>(action)->getNamedDest());
      }
   }

   if (dest)
   {
      if (!dest->isPageRef())
      {
         page = dest->getPageNum();
      }
      else
      {
         Ref ref = dest->getPageRef();
         page = TO_PDFDoc(pdfDoc)->findPage(ref.num, ref.gen);
      }
   }

   return page;
}

/* ----------------------------------------------------------- */

/*
 * Font Management.
 */

void PDFFont_AddDisplayFont(const char* fontName,
                            const char* fontFile,
                            DisplayFontType type)
{
   DisplayFontParam* dfp;

   dfp = new DisplayFontParam(new GString(fontName),
                              (type == T1DisplayFont ? displayFontT1 : displayFontTT));
   switch (type)
   {
      case T1DisplayFont:
      {
         dfp->t1.fileName = new GString(fontFile);
         break;
      }
      case TTDisplayFont:
      {
         dfp->tt.fileName = new GString(fontFile);
         break;
      }
      default:
      {
         delete dfp;
         fprintf(stderr, "invalid font type for %s\n", fontName); fflush(stderr);
         return;
      }
   }

   globalParams->addDisplayFont(dfp);
}


void PDFFont_GetDisplayFont(const char* fontName,
                            const char** fontFile,
                            DisplayFontType* type)
{
   DisplayFontParam* dfp;

   *fontFile = NULL;

   dfp = globalParams->getDisplayFont(new GString(fontName));
   if  (dfp)
   {
      switch (dfp->kind)
      {
         case displayFontT1:
         {
            *type = T1DisplayFont;
            *fontFile = dfp->t1.fileName->getCString();
            break;
         }
         case displayFontTT:
         {
            *type = TTDisplayFont;
            *fontFile = dfp->tt.fileName->getCString();
            break;
         }
      }
   }
}


/* ----------------------------------------------------------- */

/*
 * Renderer
 */

#define TO_SplashOutputDev(object) (static_cast<SplashOutputDev*>(object))
#define TO_SplashBitmap(object) (static_cast<SplashBitmap*>(object))


XPDFObject PDFRender_CreateOutputDevice(XPDFObject pdfDoc)
{
   SplashOutputDev* outputDev;
   SplashColor      paperColor;
   
   fprintf(stderr, "DEBUG: creating output device\n"); fflush(stderr);

   paperColor.rgb8 = splashMakeRGB8(255, 255, 255);
   
   outputDev = new SplashOutputDev(splashModeRGB8, gFalse, paperColor);
   outputDev->startDoc(TO_PDFDoc(pdfDoc)->getXRef());

   return outputDev;
}


void PDFRender_DestroyOutputDevice(XPDFObject device)
{
   fprintf(stderr, "DEBUG: destroy output device\n"); fflush(stderr);
   delete TO_SplashOutputDev(device);
}


XPDFObject PDFRender_RenderPage(XPDFObject pdfDoc,
                                XPDFObject device,
                                int page,
                                double dpi,
                                int rotate)
{
   XPDF_AcquireLock();

   if ((page < 0) || (page > TO_PDFDoc(pdfDoc)->getNumPages()))
   {
      fprintf(stderr, "page %d out of range\n", page); fflush(stderr);
      return NULL;
   }

   TO_PDFDoc(pdfDoc)->displayPage(TO_SplashOutputDev(device),
                                  page, dpi, dpi, rotate, gFalse, gFalse);
   
   XPDF_ReleaseLock();

   return (XPDFObject)TO_SplashOutputDev(device)->getBitmap();
}


void PDFRender_GetBitmapSize(XPDFObject bitmap, int* width, int* height)
{
   *width  = TO_SplashBitmap(bitmap)->getWidth();
   *height = TO_SplashBitmap(bitmap)->getHeight();
}


void PDFRender_GetRGB(XPDFObject bitmap, unsigned char** buffer)
{
   SplashRGB8*     rgb8;
   unsigned char   r, g, b;
   unsigned char*  bufferPtr;

   rgb8 = TO_SplashBitmap(bitmap)->getDataPtr().rgb8;

   bufferPtr = *buffer;
   for (int row = 0; row < TO_SplashBitmap(bitmap)->getHeight(); row++)
   {
      for (int col = 0; col < TO_SplashBitmap(bitmap)->getWidth(); col++)
      {
         *bufferPtr++ = splashRGB8R(*rgb8);
         *bufferPtr++ = splashRGB8G(*rgb8);
         *bufferPtr++ = splashRGB8B(*rgb8);
         ++rgb8;
      }
   }
}
