package commands

import (
	"context"
	"flag"
	"io/ioutil"
	"os"
	"strconv"
	"time"

	"github.com/google/subcommands"
	c "github.com/kotakanbe/go-cve-dictionary/config"
	db "github.com/kotakanbe/go-cve-dictionary/db"
	log "github.com/kotakanbe/go-cve-dictionary/log"
	"github.com/kotakanbe/go-cve-dictionary/nvd"
	"github.com/kotakanbe/go-cve-dictionary/util"
)

// FetchNvdCmd is Subcommand for fetch Nvd information.
type FetchNvdCmd struct {
	debug    bool
	debugSQL bool
	quiet    bool
	logDir   string

	dbpath string
	dbtype string

	last2Y   bool
	modified bool
	years    bool

	httpProxy string
}

// Name return subcommand name
func (*FetchNvdCmd) Name() string { return "fetchnvd" }

// Synopsis return synopsis
func (*FetchNvdCmd) Synopsis() string { return "Fetch Vulnerability dictionary from NVD" }

// Usage return usage
func (*FetchNvdCmd) Usage() string {
	return `fetchnvd:
	fetchnvd
		[-last2y]
		[-modified]
		[-years] 2015 2016 ...
		[-dbtype=mysql|postgres|sqlite3|redis]
		[-dbpath=$PWD/cve.sqlite3 or connection string]
		[-http-proxy=http://192.168.0.1:8080]
		[-debug]
		[-debug-sql]
		[-quiet]
		[-log-dir=/path/to/log]

For the first time, run the blow command to fetch data for entire period. (It takes about 10 minutes)
   $ for i in ` + "`seq 2002 $(date +\"%Y\")`;" + ` do go-cve-dictionary fetchnvd -years $i; done

`
}

// SetFlags set flag
func (p *FetchNvdCmd) SetFlags(f *flag.FlagSet) {
	f.BoolVar(&p.debug, "debug", false, "debug mode")
	f.BoolVar(&p.debugSQL, "debug-sql", false, "SQL debug mode")
	f.BoolVar(&p.quiet, "quiet", false, "quiet mode (no output)")

	defaultLogDir := util.GetDefaultLogDir()
	f.StringVar(&p.logDir, "log-dir", defaultLogDir, "/path/to/log")

	pwd := os.Getenv("PWD")
	f.StringVar(&p.dbpath, "dbpath", pwd+"/cve.sqlite3",
		"/path/to/sqlite3 or SQL connection string")

	f.StringVar(&p.dbtype, "dbtype", "sqlite3",
		"Database type to store data in (sqlite3, mysql, postgres or redis supported)")

	f.BoolVar(&p.last2Y, "last2y", false,
		"Refresh NVD data in the last two years.")

	f.BoolVar(&p.modified, "modified", false,
		"Refresh NVD data in the last 8 days.")

	f.BoolVar(&p.years, "years", false,
		"Refresh NVD data of specific years.")

	f.StringVar(
		&p.httpProxy,
		"http-proxy",
		"",
		"http://proxy-url:port (default: empty)",
	)
}

// Execute execute
func (p *FetchNvdCmd) Execute(_ context.Context, f *flag.FlagSet, _ ...interface{}) subcommands.ExitStatus {
	c.Conf.Quiet = p.quiet
	if c.Conf.Quiet {
		log.Initialize(p.logDir, ioutil.Discard)
	} else {
		log.Initialize(p.logDir, os.Stderr)
	}

	c.Conf.DebugSQL = p.debugSQL
	c.Conf.Debug = p.debug
	if c.Conf.Debug {
		log.SetDebug()
	}

	c.Conf.DBPath = p.dbpath
	c.Conf.DBType = p.dbtype
	c.Conf.HTTPProxy = p.httpProxy

	if !c.Conf.Validate() {
		return subcommands.ExitUsageError
	}

	years := []int{}
	thisYear := time.Now().Year()

	switch {
	case p.last2Y:
		for i := 0; i < 2; i++ {
			years = append(years, thisYear-i)
		}
	case p.modified:
		// It would be better to change from int to string
		years = append(years, -8)
	case p.years:
		if len(f.Args()) == 0 {
			log.Errorf("Specify years to fetch (from 2002 to %d)", thisYear)
			return subcommands.ExitUsageError
		}
		for _, arg := range f.Args() {
			year, err := strconv.Atoi(arg)
			if err != nil || year < 2002 || time.Now().Year() < year {
				log.Errorf("Specify years to fetch (from 2002 to %d), arg: %s", thisYear, arg)
				return subcommands.ExitUsageError
			}
			found := false
			for _, y := range years {
				if y == year {
					found = true
					break
				}
			}
			if !found {
				years = append(years, year)
			}
		}
	default:
		log.Errorf("specify -last2y or -modified or -years")
		return subcommands.ExitUsageError
	}

	entries, err := nvd.FetchFiles(years)
	if err != nil {
		log.Error(err)
		return subcommands.ExitFailure
	}
	log.Infof("Fetched %d CVEs", len(entries))

	var driver db.DB
	if driver, err = db.NewDB(c.Conf.DBType, c.Conf.DBPath, c.Conf.DebugSQL); err != nil {
		log.Error(err)
		return subcommands.ExitFailure
	}
	defer driver.CloseDB()

	log.Infof("Inserting NVD into DB (%s).", driver.Name())
	if err := driver.InsertNvd(entries); err != nil {
		log.Errorf("Failed to insert. dbpath: %s, err: %s",
			c.Conf.DBPath, err)
		return subcommands.ExitFailure
	}

	return subcommands.ExitSuccess
}
