package db

import (
	"fmt"
	"strconv"
	"strings"
	"time"

	jvn "github.com/kotakanbe/go-cve-dictionary/jvn"
	log "github.com/kotakanbe/go-cve-dictionary/log"
	"github.com/kotakanbe/go-cve-dictionary/models"
	"github.com/kotakanbe/go-cve-dictionary/nvd"
)

// DB is interface for a database driver
type DB interface {
	Name() string
	CloseDB() error
	Get(string) *models.CveDetail
	GetMulti([]string) map[string]*models.CveDetail
	GetByCpeName(string) []*models.CveDetail
	InsertJvn([]jvn.Item) error
	InsertNvd([]nvd.Entry) error
	CountNvd() (int, error)
}

// NewDB return DB accessor.
func NewDB(dbType, dbpath string, debugSQL bool) (DB, error) {
	switch dbType {
	case dialectSqlite3, dialectMysql, dialectPostgreSQL:
		return NewRDB(dbType, dbpath, debugSQL)
	case dialectRedis:
		return NewRedis(dbType, dbpath, debugSQL)
	}
	return nil, fmt.Errorf("Invalid database dialect, %s", dbType)
}

// convertNvd converts Nvd structure(got from NVD) to model structure.
func convertNvd(entries []nvd.Entry) (cves []models.CveDetail) {
	for _, entry := range entries {
		//  if entry.Cvss.Score == "0" || len(entry.Cvss.Score) == 0 {
		//      log.Warnf("Skip. CVSS Score is zero. JvnID: %s", entry.CveID)
		//      //ignore invalid item
		//      continue
		//  }

		// References
		refs := []models.Reference{}
		for _, r := range entry.References {
			ref := models.Reference{
				Source: r.Source,
				Link:   r.Link.Href,
			}
			refs = append(refs, ref)
		}

		//  // Cpes
		cpes := []models.Cpe{}
		for _, c := range entry.Products {
			cpe, err := parseCpe(c)
			if err != nil {
				panic(err)
			}
			cpes = append(cpes, cpe)
		}

		cve := models.CveDetail{
			CveID: entry.CveID,
			Nvd: models.Nvd{
				CveID:                 entry.CveID,
				Summary:               entry.Summary,
				Score:                 stringToFloat(entry.Cvss.Score),
				AccessVector:          entry.Cvss.AccessVector,
				AccessComplexity:      entry.Cvss.AccessComplexity,
				Authentication:        entry.Cvss.Authentication,
				ConfidentialityImpact: entry.Cvss.ConfidentialityImpact,
				IntegrityImpact:       entry.Cvss.IntegrityImpact,
				AvailabilityImpact:    entry.Cvss.AvailabilityImpact,
				CweID:                 entry.Cwe.ID,
				PublishedDate:         entry.PublishedDate,
				LastModifiedDate:      entry.LastModifiedDate,

				Cpes:       cpes,
				References: refs,
			},
		}
		cves = append(cves, cve)
	}
	return
}

// ConvertJvn converts Jvn structure(got from JVN) to model structure.
func convertJvn(items []jvn.Item) (cves []models.CveDetail) {
	for _, item := range items {

		// TODO support V3
		var cvssV2 jvn.Cvss
		for _, cvss := range item.Cvsses {
			if cvss.Version == "3.0" {
				continue
			}
			if cvss.Score == "0" || len(cvss.Score) == 0 {
				log.Debugf("Skip. CVSS Score V2 is zero. JvnID: %s", item.Identifier)
				//ignore invalid item
				continue
			}
			cvssV2 = cvss
		}

		//  References
		refs := []models.Reference{}
		for _, r := range item.References {
			ref := models.Reference{
				Source: r.Source,
				Link:   r.URL,
			}
			refs = append(refs, ref)
		}

		cpes := []models.Cpe{}
		for _, cpe := range item.Cpes {
			cpe, err := parseCpe(cpe.Value)
			if err != nil {
				panic(err)
			}
			cpes = append(cpes, cpe)
		}

		var publish, modified time.Time
		var err error
		if publish, err = toJvnTime(item.Issued); err != nil {
			panic(err)
		}
		if modified, err = toJvnTime(item.Issued); err != nil {
			panic(err)
		}

		cveIDs := getCveIDs(item)
		if len(cveIDs) == 0 {
			log.Debugf("No CveIDs in references. JvnID: %s, Link: %s",
				item.Identifier, item.Link)
			// ignore this item
			continue

		}

		for _, cveID := range cveIDs {
			cve := models.CveDetail{
				CveID: cveID,

				// JVN
				Jvn: models.Jvn{
					Title:   strings.Replace(item.Title, "\r", "", -1),
					Summary: strings.Replace(item.Description, "\r", "", -1),
					JvnLink: item.Link,
					JvnID:   item.Identifier,
					CveID:   cveID,

					Score:    stringToFloat(cvssV2.Score),
					Severity: cvssV2.Severity,
					Vector:   cvssV2.Vector,
					//  Version:  item.Cvss.Version,

					References: refs,
					Cpes:       cpes,

					PublishedDate:    publish,
					LastModifiedDate: modified,
				},
			}
			cves = append(cves, cve)
		}
	}
	return
}

func chunkSlice(l []models.CveDetail, n int) chan []models.CveDetail {
	ch := make(chan []models.CveDetail)

	go func() {
		for i := 0; i < len(l); i += n {
			fromIdx := i
			toIdx := i + n
			if toIdx > len(l) {
				toIdx = len(l)
			}
			ch <- l[fromIdx:toIdx]
		}
		close(ch)
	}()
	return ch
}

func getCveIDs(item jvn.Item) []string {
	cveIDsMap := map[string]bool{}
	for _, ref := range item.References {
		switch ref.Source {
		case "NVD", "CVE":
			cveIDsMap[ref.ID] = true
		}
	}
	var cveIDs []string
	for cveID := range cveIDsMap {
		cveIDs = append(cveIDs, cveID)
	}
	return cveIDs
}

func stringToFloat(str string) float64 {
	if len(str) == 0 {
		return 0
	}
	var f float64
	var ignorableError error
	if f, ignorableError = strconv.ParseFloat(str, 64); ignorableError != nil {
		log.Errorf("Failed to cast CVSS score. score: %s, err; %s",
			str,
			ignorableError,
		)
		f = 0
	}
	return f
}

// TODO move to jvn
func parseCpe(cpeName string) (models.Cpe, error) {
	s := strings.Split(cpeName, "cpe:/")
	if len(s) != 2 {
		return models.Cpe{}, fmt.Errorf("Unknow format: cpeName: %s", cpeName)
	}
	splitted := strings.Split(s[1], ":")
	items := []string{
		"", // Part
		"", // Vendor
		"", // Product
		"", // Version
		"", // VendorUpdate
		"", // Edition
		"", // Language
	}
	for i, item := range splitted {
		items[i] = item
	}
	cpe := models.Cpe{
		CpeName:      cpeName,
		Part:         items[0],
		Vendor:       items[1],
		Product:      items[2],
		Version:      items[3],
		VendorUpdate: items[4],
		Edition:      items[5],
		Language:     items[6],
	}
	return cpe, nil
}

// convert string time to time.Time
// JVN : "2016-01-26T13:36:23+09:00",
// NVD : "2016-01-20T21:59:01.313-05:00",
func toJvnTime(strtime string) (t time.Time, err error) {
	layout := "2006-01-02T15:04-07:00"
	t, err = time.Parse(layout, strtime)
	if err != nil {
		return t, fmt.Errorf("Failed to parse time, time: %s, err: %s", strtime, err)
	}
	return
}
